% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/neighbourhood.R
\name{cir}
\alias{cir}
\title{Identify pixels in a circle or ring (donut) around an object.}
\usage{
cir(
  landscape,
  coords,
  loci,
  maxRadius = ncol(landscape)/4,
  minRadius = maxRadius,
  allowOverlap = TRUE,
  allowDuplicates = FALSE,
  includeBehavior = "includePixels",
  returnDistances = FALSE,
  angles = NA_real_,
  returnAngles = FALSE,
  returnIndices = TRUE,
  closest = FALSE,
  simplify = TRUE
)
}
\arguments{
\item{landscape}{Raster on which the circles are built.}

\item{coords}{Either a matrix with 2 (or 3) columns, x and y (and id), representing the
coordinates (and an associated id, like cell index),
or a \code{SpatialPoints*} object around which to make circles. Must be same
coordinate system as the \code{landscape} argument. Default is missing,
meaning it uses the default to \code{loci}}

\item{loci}{Numeric. An alternative to \code{coords}. These are the indices on
\code{landscape} to initiate this function. See \code{coords}. Default is one
point in centre of \code{landscape}..}

\item{maxRadius}{Numeric vector of length 1 or same length as coords}

\item{minRadius}{Numeric vector of length 1 or same length as \code{coords}. Default is
\code{maxRadius}, meaning return all cells that are touched
by the narrow ring at that exact radius. If smaller than \code{maxRadius},
then this will create a buffer or donut or ring.}

\item{allowOverlap}{Logical. Should duplicates across id be removed or kept. Default TRUE.}

\item{allowDuplicates}{Logical. Should duplicates within id be removed or kept. Default FALSE.
This is useful if the actual x, y coordinates are desired, rather
than the cell indices. This will increase the size of the returned
object.}

\item{includeBehavior}{Character string. Currently accepts only "includePixels", the default,
and "excludePixels". See details.}

\item{returnDistances}{Logical. If TRUE, then a column will be added to the returned
data.table that reports the distance from \code{coords} to every
point that was in the circle/donut surrounding \code{coords}. Default
FALSE, which is faster.}

\item{angles}{Numeric. Optional vector of angles, in radians, to use. This will create
"spokes" outward from coords. Default is NA, meaning, use internally
derived angles that will "fill" the circle.}

\item{returnAngles}{Logical. If TRUE, then a column will be added to the returned
data.table that reports the angle from \code{coords} to every
point that was in the circle/donut surrounding \code{coords}. Default
FALSE.}

\item{returnIndices}{Logical or numeric. If \code{1} or \code{TRUE}, will
return a \code{data.table} with indices and values of
successful spread events.
If \code{2}, it will simply return a vector of pixel indices of
all cells that were touched. This will be the fastest option. If
\code{FALSE}, then it will return a raster with
values. See Details.}

\item{closest}{Logical. When determining non-overlapping circles, should the function
give preference to the closest \code{loci} or the first one (much faster).
Default is FALSE, meaning the faster, though maybe not desired behaviour.}

\item{simplify}{logical. If TRUE, then all duplicate pixels are removed. This means
that some x, y combinations will disappear.}
}
\value{
A \code{matrix} with 4 columns, \code{id}, \code{indices},
\code{x}, \code{y}. The \code{x} and \code{y} indicate the exact coordinates of
the \code{indices} (i.e., cell number) of the \code{landscape}
associated with the ring or circle being identified by this function.
}
\description{
Identify the pixels and coordinates that are at a (set of) buffer distance(s)
of the objects passed into \code{coords}.
This is similar to \code{rgeos::gBuffer} but much faster and without
the geo referencing information.
In other words, it can be used for similar problems, but where speed is important.
This code is substantially adapted from \code{PlotRegionHighlighter::createCircle}.
}
\details{
This function identifies all the pixels as defined by a donut
with inner radius \code{minRadius} and outer radius of \code{maxRadius}.
The \code{includeBehavior} defines whether the cells that intersect the radii
but whose centres are not inside the donut are included \code{includePixels}
or not \code{excludePixels} in the returned pixels identified.
If this is \code{excludePixels}, and if a \code{minRadius} and
\code{maxRadius} are equal, this will return no pixels.
}
\examples{
library(data.table)
library(sp)
library(raster)
library(quickPlot)

set.seed(1462)

# circle centred
ras <- raster(extent(0, 15, 0, 15), res = 1, val = 0)
middleCircle <- cir(ras)
ras[middleCircle[, "indices"]] <- 1
circlePoints <- SpatialPoints(middleCircle[, c("x", "y")])
if (interactive()) {
  clearPlot()
  Plot(ras)
  Plot(circlePoints, addTo = "ras")
}

# circles non centred
ras <- randomPolygons(ras, numTypes = 4)
n <- 2
agent <- SpatialPoints(coords = cbind(x = stats::runif(n, xmin(ras), xmax(ras)),
                                      y = stats::runif(n, xmin(ras), xmax(ras))))

cirs <- cir(ras, agent, maxRadius = 15, simplify = TRUE) ## TODO: empty with some seeds! e.g. 1642
cirsSP <- SpatialPoints(coords = cirs[, c("x", "y")]) ## TODO: error with some seeds! e.g. 1642
cirsRas <- raster(ras)
cirsRas[] <- 0
cirsRas[cirs[, "indices"]] <- 1

if (interactive()) {
  clearPlot()
  Plot(ras)
  Plot(cirsRas, addTo = "ras", cols = c("transparent", "#00000055"))
  Plot(agent, addTo = "ras")
  Plot(cirsSP, addTo = "ras")
}

# Example comparing rings and cir
a <- raster(extent(0, 30, 0, 30), res = 1)
hab <- gaussMap(a, speedup = 1) # if raster is large (>1e6 pixels) use speedup > 1
radius <- 4
n <- 2
coords <- SpatialPoints(coords = cbind(x = stats::runif(n, xmin(hab), xmax(hab)),
                                       y = stats::runif(n, xmin(hab), xmax(hab))))

# cirs
cirs <- cir(hab, coords, maxRadius = rep(radius, length(coords)), simplify = TRUE)

# rings
loci <- cellFromXY(hab, coordinates(coords))
cirs2 <- rings(hab, loci, maxRadius = radius, minRadius = radius - 1, returnIndices = TRUE)

# Plot both
ras1 <- raster(hab)
ras1[] <- 0
ras1[cirs[, "indices"]] <- cirs[, "id"]

ras2 <- raster(hab)
ras2[] <- 0
ras2[cirs2$indices] <- cirs2$id
if (interactive()) {
  clearPlot()
  Plot(ras1, ras2)
}

a <- raster(extent(0, 100, 0, 100), res = 1)
hab <- gaussMap(a, speedup = 1)
cirs <- cir(hab, coords, maxRadius = 44, minRadius = 0)
ras1 <- raster(hab)
ras1[] <- 0
cirsOverlap <- data.table(cirs)[, list(sumIDs = sum(id)), by = indices]
ras1[cirsOverlap$indices] <- cirsOverlap$sumIDs
if (interactive()) {
  clearPlot()
  Plot(ras1)
}

# Provide a specific set of angles
ras <- raster(extent(0, 330, 0, 330), res = 1)
ras[] <- 0
n <- 2
coords <- cbind(x = stats::runif(n, xmin(ras), xmax(ras)),
                y = stats::runif(n, xmin(ras), xmax(ras)))
circ <- cir(ras, coords, angles = seq(0, 2 * pi, length.out = 21),
            maxRadius = 200, minRadius = 0, returnIndices = FALSE,
            allowOverlap = TRUE, returnAngles = TRUE)
}
\seealso{
\code{\link{rings}} which uses \code{spread} internally.
\code{cir} tends to be faster when there are few starting points, \code{rings}
tends to be faster when there are many starting points. \code{cir} scales with
\code{maxRadius} ^ 2 and \code{coords}. Another difference
between the two functions is that \code{rings} takes the centre of the pixel
as the centre of a circle, whereas \code{cir} takes the exact coordinates.
See example. For the specific case of creating distance surfaces from specific
points, see \code{\link{distanceFromEachPoint}}, which is often faster.
For the more general GIS buffering, see \code{rgeos::gBuffer}.
}
