% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/batty_karlstrom_entropy.R
\name{karlstrom}
\alias{karlstrom}
\title{Karlstrom and Ceccato's entropy.}
\usage{
karlstrom(
  data,
  category = 1,
  cell.size = 1,
  partition = 10,
  neigh = 4,
  method = "number"
)
}
\arguments{
\item{data}{If data are lattice, a data matrix, which can be numeric, factor, character, ...
If the dataset is a point pattern, \code{data} is a \code{ppp} object.}

\item{category}{A single value matching the data category of interest for computing Batty's entropy.
Default to 1. If the dataset is an unmarked point pattern, this argument must not be changed from the default.}

\item{cell.size}{A single number. If data are lattice, the length of the side of each pixel.
Default to 1. Ignored if data are points.}

\item{partition}{Input defining the partition into subareas. If an integer, it defines the
number of sub-areas that are randomly generated by \link{areapart}; if a two column matrix
with coordinates, they are the centroids of the subareas built by \link{areapart}. Alternatively,
it can be the output of \link{areapart}, or a \code{tess} object built by the user.
The default option is \code{partition=areapart(data, G=10)}, which generates 10 random sub-areas.}

\item{neigh}{A single number. It can be either the number of neighbours for each sub-area
(including the area itself). or the Euclidean distance to define which sub-areas are neighbours,
based on their centroids. Default to 4 neighbours.}

\item{method}{Character, it guides the interpretation of \code{neigh}. Either "number" (the default)
or "distance".}
}
\value{
A list of four elements:
\itemize{
\item \code{area.tess} a \code{tess} object with the area partition
\item \code{areas.freq} a dataframe giving, for each sub-area, the absolute and relative frequency of
the points/pixels of interest, the weighted probabilities of the neighbours and the sub-area size
for each sub-area
\item \code{karlstrom} Karlstrom and Ceccato's entropy
\item \code{rel.karl} Karlstrom and Ceccato's entropy divided by \eqn{\log(G)} (number og sub-areas) for comparison across observation areas.
}
}
\description{
This function computes Karlstrom and Ceccato's spatial entropy for a
chosen neighbourhood distance,
following Karlstrom and Ceccato (2002), see also Altieri et al (2017) and following works
(references are under the topic \code{\link{SpatEntropy}}).
}
\details{
Karlstrom and Ceccato's spatial entropy measures the heterogeneity in the spatial distribution
of a phenomenon of interest, with regard to an area partition and accounting for the neighbourhood.
It is similar to Batty's entropy (see \link{batty}) discarding the sub-area size,
with the difference that the probability of occurrence of the phenomenon over area \eqn{g}
is actually a weighted sum of the neighbouring probabilities.
\deqn{H_{KC}=\sum p_g \log(1/ \tilde{p}_g)}
where \eqn{p_g} is the probability of occurrence of the phenomenon over sub-area \eqn{g},
and \eqn{\tilde{p}_g} is the averaged probability over the neighbouring areas (including the g-th area itself).
When data are categorical, the phenomenon of interest corresponds to
one category, which must be specified. If data are an unmarked
point pattern, a fake mark vector is be created with the same category for all points.
For comparison purposes, the relative version of Karlstrom and Ceccato's entropy is also returned, i.e.
Karlstrom and Ceccato's entropy divided by its maximum log(number of sub-areas).
The function is able to work with grids containing missing data, specified as NA values.
All NAs are ignored in the computation.
}
\examples{
#LATTICE DATA

data=matrix((sample(c("a","b","c"), 100, replace=TRUE)), nrow=10)
KC.entropy=karlstrom(data, category="a")
KC.entropy=karlstrom(data, category="a", neigh=3.5, method="distance")
##to plot
data.binary=matrix(as.numeric(data=="a"), nrow(data))
plot(as.im(data.binary, W=KC.entropy$areas.tess$window), main="",
     col=gray(seq(1,0,l=length(unique(c(data.binary))))), ribbon=FALSE)
plot(KC.entropy$areas.tess, add=TRUE, border=2)

#POINT DATA

#unmarked pp
data=ppp(x=runif(100, 0, 10), y=runif(100, 0, 10), window=square(10))
KC.entropy=karlstrom(data)
##to plot
plot(data)
plot(KC.entropy$areas.tess, add=TRUE, border=2)

#marked pp
data=ppp(x=runif(100, 0, 10), y=runif(100, 0, 10), window=square(10),
         marks=(sample(1:5, 100, replace=TRUE)))
#if you want to compute the entropy on all points
KC.entropy=karlstrom(unmark(data))
#if you want to compute the entropy on a category, say 3
KC.entropy=karlstrom(data, category=3)
##to plot using the selected category
ind=which(marks(data)==3)
data.binary=unmark(data[ind])
plot(data.binary)
plot(KC.entropy$areas.tess, add=TRUE, border=2)

}
