#'@title Gillespie Stochastic Simulation Algorithm single trajectory
#'
#'@description
#'\code{ssaSingle} Run a single SSA trajectory and write output
#'data to \code{outputFile}. Stores a row of data for every reaction event.
#'
#'@param modelFile Character string with path to StochKit2 .xml model file
#'@param outputFile Character string with path to output file.
#'@param startTime Simulation start time
#'@param endTime Simulation end time
#'@param seed Seed the random number generator. By default the seed is determined by the R random number generator, so the seed can also be set by calling \code{set.seed} in R immediately before calling \code{ssaSingle}
#'@return NULL
#'@examples
#'\dontrun{
#'#example using included dimer_decay.xml file
#'#output written to file single_output.txt (created in current working directory)
#'ssaSingle(system.file("dimer_decay.xml",package="StochKit2R"),
#'          "single_output.txt",0,10)
#'}
ssaSingle <- function(modelFile,outputFile,startTime,endTime,seed=NULL) {
  # can set seed in R with set.seed()
  
  #checks on modelFile  
  if (!file.exists(modelFile)) {
    stop("ERROR: Model file does not exist")
  }
  #simple checks
  if (startTime>=endTime) {
    stop("startTime must be before endTime")
  }
  
  #checks on outputDir
  if (outputFile=="" | is.null(outputFile)) {
    stop("An output file must be specified")
  }
  #expand path
  outputFile <- tryCatch(suppressWarnings(normalizePath(outputFile)), error = function(e) {stop("Invalid or missing outputFile path, terminating StochKit2R")}, finally = NULL)
  if (file.exists(outputFile)) {
      stop("Output file already exists. Delete existing file or specify a unique file name")
  }
    
  if (is.null(seed)) {
    seed=floor(runif(1,-.Machine$integer.max,.Machine$integer.max))
  }
  
  # process the xml model file, put in R Lists
  StochKit2Rmodel <- buildStochKit2Rmodel(modelFile)
  
  # everything is set up, ready to run the simulation
  ssaSingleStochKit2RInterface(StochKit2Rmodel,outputFile,startTime,endTime,seed)
}