% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/in.window.R
\name{in.window}
\alias{in.window}
\title{Irregular windowing}
\usage{
in.window(x, w, xout = unique(x), b = "[]", warn = 100, ...)
}
\arguments{
\item{x}{the position values to be regrouped in windows}

\item{w}{the window length (top to bottom)}

\item{xout}{the center of each window, defaults to x}

\item{b}{the boundary rule at the top and bottom of the window: "][" means
that neither the top nor bottom are taken in, "[]" means that top and bottom
are taken in, "]]" and "[[" mean that only the top or the bottom are taken
in, respectively. Also accepts: "[)", "(]", "()", "open", "closed",
"left-open", "right-open", "left-closed" and "right-closed": see
\code{\link{rebound}} for more information}

\item{warn}{an integer of the amount of iterations after which a warning is
issued: this could mean that there are too many data points in a window, and
that the computation will become very inefficient. This is up to the user to
see. If you want to remove the warning, set this parameter to Inf}

\item{...}{intensity values corresponding to each x position, making
time-series. They will be provided window by window in the output.}
}
\value{
a list made of the center values of the windows ($xout), a matrix of
the index of the original x values in each corresponding window ($x.index;
the rows correspond to each $xout value), a matrix of the x values in
each corresponding window ($x; the rows correspond to each $xout value),
the amount of points in each window ($n.size), and additional matrices of
additional intensity values provided in \code{...} (names correspond to the
names provided in \code{...}; the rows correspond to each $xout value)
}
\description{
Find the index of points in time-series that fall into a
specific window, even with irregular sampling rate. The iterations needed
in this function are equal to the maximum amount of points found in the
windows, therefore it should be reasonably efficient for short windows at
least.
}
\examples{
# Visual example ----

set.seed(42)

n <- 600
t <- seq_len(n)

p1 <- 30
p2 <- 240

xy <- (1 + 0.6 * sin(t*2*pi/p2)) * sin(t*2*pi/p1)  + 2 * sin(t*2*pi/p2) +
  rnorm(n, sd = 0.5) + t * 0.01

inter_dt <- round(runif(length(xy), min = 0.5, max = 1.5),1)

dt <- cumsum(inter_dt)

keep <- runif(length(dt)) < 0.5

xy <- xy[keep]
dt <- dt[keep]

window <- in.window(dt, w = 30, 1:590, xy = xy)

par(mfrow = c(1,2))

plot(xy, dt, type = "o", pch = 19,
     ylim = c(0,600), main = "Moving Average")

lines(rowMeans(window$xy, na.rm = TRUE), window$xout,
      col = "red", lwd = 2)

plot(window$n.size, window$xout,  pch = 19,
     ylim = c(0,600), xlim = c(0,20), ylab = "dt",
     main = "Amount of Points in Average")

# Test the boundary rule ----

x <- c(1,1,2,3,4,6,8,10,15,16)
xout <- -6:22

output <- in.window(x = x, w = 10, xout = xout, b = "]]")

test <- output$x - output$xout
see  <- cbind(output$xout, output$x)

colnames(see) <- c("xout", paste0("x", seq_len(ncol(see)-1)))

test # difference between x and xout: it is contained in ]-5,5]
see

}
