% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/distance_calculations.R
\name{diagram_distance}
\alias{diagram_distance}
\title{Calculate distance between a pair of persistence diagrams.}
\usage{
diagram_distance(
  D1,
  D2,
  dim = 0,
  p = 2,
  distance = "wasserstein",
  sigma = NULL,
  rho = NULL
)
}
\arguments{
\item{D1}{the first persistence diagram.}

\item{D2}{the second persistence diagram.}

\item{dim}{the non-negative integer homological dimension in which the distance is to be computed, default 0.}

\item{p}{a number representing the wasserstein power parameter, at least 1 and default 2.}

\item{distance}{a string which determines which type of distance calculation to carry out, either "wasserstein" (default) or "fisher".}

\item{sigma}{either NULL (default) or a positive number representing the bandwidth for the Fisher information metric.}

\item{rho}{either NULL (default) or a positive number. If NULL then the exact calculation of the Fisher information metric is returned and otherwise a fast approximation, see details.}
}
\value{
the numeric value of the distance calculation.
}
\description{
Calculates the distance between a pair of persistence diagrams, either the output from a \code{\link{diagram_to_df}} function call
or from a persistent homology calculation like ripsDiag/\code{\link[TDAstats]{calculate_homology}}/\code{\link{PyH}},
in a particular homological dimension.
}
\details{
The most common distance calculations between persistence diagrams
are the wasserstein and bottleneck distances, both of which "match" points between
their two input diagrams and compute the "loss" of the optimal matching 
(see \url{http://www.geometrie.tugraz.at/kerber/kerber_papers/kmn-ghtcpd_journal.pdf} for details). Another 
method for computing distances, the Fisher information metric, 
converts the two diagrams into distributions
defined on the plane, and calculates a distance between the resulting two distributions
(\url{https://proceedings.neurips.cc/paper/2018/file/959ab9a0695c467e7caf75431a872e5c-Paper.pdf}).
If the `distance` parameter is "fisher" then `sigma` must not be NULL. As noted in the Persistence Fisher paper,
there is a fast speed-up approximation which has been implemented from \url{https://github.com/vmorariu/figtree} 
and can be accessed by setting the `rho` parameter. Smaller
values of `rho` will result in tighter approximations at the expense of longer runtime, and vice versa.
}
\examples{

if(require("TDAstats"))
{
  # create two diagrams
  D1 <- TDAstats::calculate_homology(TDAstats::circle2d[sample(1:100,size = 20),],
                      dim = 1,threshold = 2)
  D2 <- TDAstats::calculate_homology(TDAstats::circle2d[sample(1:100,size = 20),],
                      dim = 1,threshold = 2)

  # calculate 2-wasserstein distance between D1 and D2 in dimension 1
  diagram_distance(D1,D2,dim = 1,p = 2,distance = "wasserstein")

  # calculate bottleneck distance between D1 and D2 in dimension 0
  diagram_distance(D1,D2,dim = 0,p = Inf,distance = "wasserstein")

  # Fisher information metric calculation between D1 and D2 for sigma = 1 in dimension 1
  diagram_distance(D1,D2,dim = 1,distance = "fisher",sigma = 1)
  
  # repeat but with fast approximation
  \dontrun{
  diagram_distance(D1,D2,dim = 1,distance = "fisher",sigma = 1,rho = 0.001)
  }
}
}
\references{
Kerber M, Morozov D and Nigmetov A (2017). "Geometry Helps to Compare Persistence Diagrams." \url{http://www.geometrie.tugraz.at/kerber/kerber_papers/kmn-ghtcpd_journal.pdf}.

Le T, Yamada M (2018). "Persistence fisher kernel: a riemannian manifold kernel for persistence diagrams." \url{https://proceedings.neurips.cc/paper/2018/file/959ab9a0695c467e7caf75431a872e5c-Paper.pdf}.

Vlad I. Morariu, Balaji Vasan Srinivasan, Vikas C. Raykar, Ramani Duraiswami, and Larry S. Davis. Automatic online tuning for fast Gaussian summation. Advances in Neural Information Processing Systems (NIPS), 2008.
}
\author{
Shael Brown - \email{shaelebrown@gmail.com}
}
