% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/validation.R
\name{oneStepPredict}
\alias{oneStepPredict}
\title{Calculate one-step-ahead (OSA) residuals for a latent variable model.}
\usage{
oneStepPredict(obj, observation.name = NULL, data.term.indicator = NULL,
  method = c("oneStepGaussianOffMode", "fullGaussian", "oneStepGeneric",
  "oneStepGaussian", "cdf"), subset = NULL, conditional = NULL,
  discrete = NULL, discreteSupport = NULL, range = c(-Inf, Inf),
  seed = 123, parallel = FALSE, trace = TRUE, ...)
}
\arguments{
\item{obj}{Output from \code{MakeADFun}.}

\item{observation.name}{Character naming the observation in the template.}

\item{data.term.indicator}{Character naming an indicator data variable in the template (not required by all methods - see details).}

\item{method}{Method to calculate OSA (see details).}

\item{subset}{Index vector of observations that will be added one by one during OSA. By default \code{1:length(observations)} (with \code{conditional} subtracted).}

\item{conditional}{Index vector of observations that are fixed during OSA. By default the empty set.}

\item{discrete}{Are observations discrete? (assumed FALSE by default)}

\item{discreteSupport}{Possible outcomes of discrete distribution (\code{method="oneStepGeneric"} only).}

\item{range}{Possible range of the observations.}

\item{seed}{Randomization seed (discrete case only). If \code{NULL} the RNG seed is untouched by this routine.}

\item{parallel}{Run in parallel using the \code{parallel} package?}

\item{trace}{Trace progress?}

\item{...}{Control parameters for OSA method}
}
\value{
\code{data.frame} with OSA residuals in column \code{residual}.
}
\description{
Calculate one-step-ahead (OSA) residuals for a latent variable model.
}
\details{
Given a TMB latent variable model this function calculates OSA
standardized residuals that can be used for goodness-of-fit
assessment. The approach is based on a factorization of the joint
distribution of the \emph{observations} \eqn{X_1,...,X_n} into
successive conditional distributions.
Denote by
\deqn{F_n(x_n) = P(X_n \leq x_n | X_1 = x_1,...,X_{n-1}=x_{n-1} )}
the one-step-ahead CDF, and by
\deqn{p_n(x_n) = P(X_n = x_n | X_1 = x_1,...,X_{n-1}=x_{n-1} )}
the corresponding point probabilities (zero for continuous distributions).
In case of continuous observations the sequence
\deqn{\Phi^{-1}(F_1(X_1))\:,...,\:\Phi^{-1}(F_n(X_n))}
will be iid standard normal. These are referred to as the OSA residuals.
In case of discrete observations draw (unit) uniform variables
\eqn{U_1,...,U_n} and construct the randomized OSA residuals
\deqn{\Phi^{-1}(F_1(X_1)-U_1 p_1(X_1))\:,...,\:\Phi^{-1}(F_n(X_n)-U_n p_n(X_n))}
These are also iid standard normal.

The user must specify one of the following methods to calculate
the residuals:
\describe{
\item{method="fullGaussian"}{
This method assumes that the joint distribution of data \emph{and}
random effects is Gaussian (or well approximated by a
Gaussian). It does not require any changes to the user
template. However, if used in conjunction with \code{subset}
and/or \code{conditional} a \code{data.term.indicator} is required
- see the next method.
}
\item{method="oneStepGeneric"}{
This method calculates the one-step conditional probability
density as a ratio of Laplace approximations. The approximation is
integrated (and re-normalized for improved accuracy) using 1D
numerical quadrature to obtain the one-step CDF evaluated at each
data point. The method works in the continuous case as well as the
discrete case (\code{discrete=TRUE}).

It requires a specification of a \code{data.term.indicator}
explained in the following. Suppose the template for the
observations given the random effects (\eqn{u}) looks like
\preformatted{
    DATA_VECTOR(x);
    ...
    nll -= dnorm(x(i), u(i), 0.0, true);
    ...
}

Then this template can be augmented with a
\code{data.term.indicator = "keep"} by changing the template to
\preformatted{
    DATA_VECTOR(x);
    DATA_VECTOR_INDICATOR(keep, x);
    ...
    nll -= keep(i) * dnorm(x(i), u(i), 0.0, true);
    ...
}

The new data vector (\code{keep}) need not be passed from \R. It
automatically becomes a copy of \code{x} filled with ones.
}
\item{method="oneStepGaussian"}{
This is a special case of the generic method where the one step
conditional distribution is approximated by a Gaussian (and can
therefore be handled more efficiently).
}
\item{method="oneStepGaussianOffMode"}{
This is an approximation of the "oneStepGaussian" method that
avoids locating the mode of the one-step conditional density.
}
\item{method="cdf"}{
The generic method can be slow due to the many function
evaluations used during the 1D integration (or summation in the
discrete case). The present method can speed up this process but
requires more changes to the user template. The above template
must be expanded with information about how to calculate the
negative log of the lower and upper CDF:
\preformatted{
    DATA_VECTOR(x);
    DATA_VECTOR_INDICATOR(keep, x);
    ...
    nll -= keep(i) * dnorm(x(i), u(i), 0.0, true);
    nll -= keep.cdf_lower(i) * log( pnorm(x(i), u(i), 0.0) );
    nll -= keep.cdf_upper(i) * log( 1.0 - pnorm(x(i), u(i), 0.0) );
    ...
}

The specialized members \code{keep.cdf_lower} and
\code{keep.cdf_upper} automatically become copies of \code{x}
filled with zeros.
}
}
}
\examples{
######################## Gaussian case
runExample("simple")
osa.simple <- oneStepPredict(obj, observation.name = "x", method="fullGaussian")
qqnorm(osa.simple$residual); abline(0,1)

\dontrun{
######################## Poisson case (First 100 observations)
runExample("ar1xar1")
osa.ar1xar1 <- oneStepPredict(obj, "N", "keep", method="cdf", discrete=TRUE, subset=1:100)
qqnorm(osa.ar1xar1$residual); abline(0,1)
}
}

