\name{TRAMP}
\alias{TRAMP}

\title{TRFLP Analysis and Matching Program}

\description{Determine if TRFLP profiles may match those in a database
  of knowns.  The resulting object can be used to produce a
  presence/absence matrix of known profiles in environmental samples.

  The \code{TRAMPR} package contains a vignette, which includes a worked
  example; type \code{vignette("TRAMPRdemo")} to view it.
}

\usage{
TRAMP(samples, knowns, accept.error=1.5, min.comb=4, method="maximum")
}

\arguments{
  \item{samples}{A \code{\link{TRAMPsamples}} object, containing
    unidentified samples.}
  \item{knowns}{A \code{\link{TRAMPknowns}} object, containing
    identified TRFLP patterns.}
  \item{accept.error}{The largest acceptable difference (in base pairs)
    between any peak in the sample data and the knowns database (see
    Details; interpretation will depend on the value of \code{method}).}
  \item{min.comb}{Minimum number of enzyme/primer combinations required
    before presence will be tested.  The default (4) should be
    reasonable in most cases.  Setting \code{min.comb} to \code{NA} will
    require that all enzyme/primer combinations in the knowns database
    are present in the samples.}
  \item{method}{Method used in calculating the difference between
    samples and knowns; may be one of \code{"maximum"},
    \code{"euclidian"} or \code{"manhattan"} (or any unambiguous
    abbreviation).}
}

\details{
  \code{TRAMP} attempts to determine which species in the
  \sQuote{knowns} database \emph{may} be present in a collection of
  samples.

  A sample matches a known if it has a peak that is \dQuote{close
    enough} to every peak in the known for every enzyme/primer
  combination that they share.  The default is to accept matches where
  the largest distance between a peak in the knowns database and the
  sample is less than \code{accept.error} base pairs (default 2), and
  where at least \code{min.comb} enzyme/primer combinations are shared
  between a sample and a known (default 4).

  The three-dimensional matrix of match errors is generated by
  \code{\link{create.diffsmatrix}}.  In the resulting array,
  \code{m[i,j,k]} is the difference (in base pairs) between the
  \code{i}th sample and the \code{j}th known for the \code{k}th
  enzyme/primer combination.

  If \eqn{p_k} and \eqn{q_k} are the sizes of peaks for the \eqn{k}th
  enzyme/primer combination for a sample and known (respectively), then
  maximum distance is defined as

  \deqn{\max(|p_k - q_k|)}{max(|p_k - q_k|)}

  Euclidian distance is defined as
  \deqn{\frac{1}{n}\sqrt{\sum (p_k - q_k)^2}}{%
    sqrt(sum (p_k - q_k)^2)/n}

  and Manhattan distance is defined as
  \deqn{\frac{1}{n}\sum{|p_k - q_k|}}{%
    sum(|p_k - q_k|)/n}

  where \eqn{n} is the number of shared enzyme/primer combinations,
  since this may vary across sample/known combinations.  For Euclidian
  and Manhattan distances, \code{accept.error} then becomes the
  \emph{mean} distance, rather than the total distance.
}

\value{
  A \code{TRAMP} object, with elements:
  \item{presence}{Presence/absence matrix.  Rows are different samples
    (with rownames from \code{labels(samples)}) and columns are
    different knowns (with colnames from \code{labels(knowns)}).  Do not
    access the presence/absence matrix directly, but use
    \code{\link{summary.TRAMP}}, which provides options for labelling
    knowns, grouping knowns, and excluding \dQuote{ignored} matches.}
  \item{error}{Matrix of distances between the samples and known,
    calculated by one of the methods described above.  Rows correspond
    to different samples, and columns correspond to different knowns.
    The matrix dimension names are set to the values \code{sample.pk}
    and \code{knowns.pk} for the samples and knowns, respectively.}
  \item{n}{A two-dimensional matrix (same dimensions as \code{error}),
    recording the number of enzyme/primer combinations present for each
    combination of samples and knowns.}
  \item{diffsmatrix}{Three-dimensional array of output from
    \code{\link{create.diffsmatrix}}.}
  \item{enzyme.primer}{Different enzyme/primer combinations present in
    the data, in the order of the third dimension of \code{diffsmatrix}
    (see \code{\link{create.diffsmatrix}} for details).}
  \item{samples, knowns, accept.error, min.comb, method}{The input data
    objects and arguments, unmodified.}

  In addition, an element \code{presence.ign} is included to allow
  matches to be ignored.  However, this interface is experimental and
  its current format should not be relied on - use
  \code{\link{remove.TRAMP.match}} rather than interacting directly with
  \code{presence.ign}.

  Matching is based only on peak size (in base pairs), and does not
  consider peak heights.
}

\seealso{
  See \code{\link{create.diffsmatrix}} for discussion of how differences
  between sample and known profiles are generated.

  \code{\link{plot.TRAMP}}, which displays TRAMP fits graphically.

  \code{\link{summary.TRAMP}}, which creates a presence/absence matrix.

  \code{\link{remove.TRAMP.match}}, which marks TRAMP matches as
  ignored.
}

\examples{
data(demo.knowns)
data(demo.samples)

res <- TRAMP(demo.samples, demo.knowns)

## The resulting object can be interrogated with methods:

## The goodness of fit of the sample with sample.pk=101 (see
## ?\link{plot.TRAMP}).
plot(res, 101)

\dontrun{
## To see all plots (this produces many figures), one after another.
op <- par(ask=TRUE)
plot(res)
par(op)
}

## Produce a presence/absence matrix (see ?\link{summary.TRAMP}).
m <- summary(res)
head(m)
}

\keyword{multivariate}
\keyword{cluster}
