% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TPR.fit.R
\name{TPR.fit}
\alias{TPR.fit}
\alias{TPR}
\title{Tensor predictor regression}
\usage{
TPR.fit(x, y, u, method=c('standard', 'FG', '1D', 'ECD', 'PLS'), Gamma_init = NULL)
}
\arguments{
\item{x}{The predictor tensor instance of dimension \eqn{p_1\times p_2\times\cdots\times p_m \times n}, where \eqn{n} is the sample size. Array with the same dimensions and matrix with dimension \eqn{p\times n} are acceptable. If \code{y} is missing, \code{x} should be a list or an environment consisting of predictor and response datasets.}

\item{y}{The response matrix of dimension \eqn{r \times n}, where \eqn{n} is the sample size. Vector of length \eqn{n} is acceptable.}

\item{u}{The dimension of envelope subspace. \eqn{u=(u_1,\cdots, u_m)}. Used for methods "FG", "1D", "ECD" and "PLS". User can use \code{\link{TPRdim}} to select dimension.}

\item{method}{The method used for estimation of tensor response regression. There are four possible choices.
\itemize{
  \item{\code{"standard"}}: The standard OLS type estimation.
  \item{\code{"FG"}}: Envelope estimation with full Grassmannian (FG) algorithm.
  \item{\code{"1D"}}: Envelope estimation with one dimensional optimization approaches by 1D algorithm.
  \item{\code{"ECD"}}: Envelope estimation with one dimensional optimization approaches by ECD algorithm.
  \item{\code{"PLS"}}: The SIMPLS-type estimation without manifold optimization.
}}

\item{Gamma_init}{A list specifying the initial envelope subspace basis for "FG" method. By default, the estimators given by "1D" algorithm is used.}
}
\value{
\item{x}{The original predictor dataset.}
  \item{y}{The original response dataset.}
  \item{call}{The matched call.}
  \item{method}{The implemented method.}
  \item{coefficients}{The estimation of regression coefficient tensor.}
  \item{Gamma}{The estimation of envelope subspace basis.}
  \item{Sigma}{A lists of estimated covariance matrices at each mode for the tensor predictors.}
  \item{fitted.values}{The fitted response matrix.}
  \item{residuals}{The residuals matrix.}
}
\description{
This function is used for estimation of tensor predictor regression. The available method including standard OLS type estimation, PLS type of estimation as well as envelope estimation with FG, 1D and ECD approaches.
}
\details{
Please refer to \strong{Details} part of \code{\link{TPRsim}} for the description of the tensor predictor regression model.
}
\examples{
rm(list = ls())
# The dimension of predictor
p <- c(10, 10, 10)
# The envelope dimensions u.
u <- c(1, 1, 1)
# The dimension of response
r <- 5
# The sample size
n <- 200

# Simulate the data with TPRsim.
dat <- TPRsim(p = p, r = r, u = u, n = n)
x <- dat$x
y <- dat$y
B <- dat$coefficients

fit_std <- TPR.fit(x, y, method="standard")
fit_FG <- TPR.fit(x, y, u, method="FG")
fit_pls <- TPR.fit(x, y, u, method="PLS")

rTensor::fnorm(B-stats::coef(fit_std))
rTensor::fnorm(B-stats::coef(fit_FG))
rTensor::fnorm(B-stats::coef(fit_pls))

## ----------- Pass a list or an environment to x also works ------------- ##
# Pass a list to x
l <- dat[c("x", "y")]
fit_std_l <- TPR.fit(l, method="standard")

# Pass an environment to x
e <- new.env()
e$x <- dat$x
e$y <- dat$y
fit_std_e <- TPR.fit(e, method="standard")

## ----------- Use dataset "square" included in the package ------------- ##
data("square")
x <- square$x
y <- square$y
fit_std <- TPR.fit(x, y, method="standard")

}
\references{
Zhang, X. and Li, L., 2017. Tensor envelope partial least-squares regression. Technometrics, 59(4), pp.426-436.
}
\seealso{
\code{\link{summary.Tenv}} for summaries, calculating mean squared error from the prediction.

\code{\link{plot.Tenv}}(via \code{graphics::image}) for drawing the two-dimensional coefficient plot.

\code{\link{predict.Tenv}} for prediction.

The generic functions \code{\link{coef}, \link{residuals}, \link{fitted}}.

\code{\link{TPRdim}} for selecting the dimension of envelope by cross-validation.

\code{\link{TPRsim}} for generating the simulated data used in tensor prediction regression.

The simulated data \code{\link{square}} used in tensor predictor regression.
}
