% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Comparison.R
\name{compParaSurv}
\alias{compParaSurv}
\title{Compare Parametric Survival Distribution}
\usage{
compParaSurv(
  time,
  status = NULL,
  arm,
  tau = NULL,
  dist1 = "weibull",
  dist0 = NULL,
  init1 = NULL,
  init0 = NULL,
  boot = NULL,
  perm = NULL,
  sig = 0.05,
  eps = 1e-06,
  maxit = 10,
  report = F
)
}
\arguments{
\item{time}{Observation time.}

\item{status}{Status indicator, coded as 1 if an event was observed, 0 if
censored.}

\item{arm}{Treatment indicator, coded as 1 for the target group, 0 for the
reference group.}

\item{tau}{Optional truncation times for calculating RMST.}

\item{dist1}{Distribution to fit for the target group. Selected from among:
exp, gamma, gengamma, log-normal, and weibull.}

\item{dist0}{Distribution to fit for the reference group. Same choices as for
the target group. If omitted, defaults to the distribution specified for
the target group.}

\item{init1}{Initial parameter values for the target group.}

\item{init0}{Initial parameter values for the reference group.}

\item{boot}{If provided, integer number of bootstraps for constructing CIs.}

\item{perm}{If provided, integer number of permutations for calculating p-values.}

\item{sig}{Significance level, for CIs.}

\item{eps}{Tolerance for Newton-Raphson iterations.}

\item{maxit}{Maximum number of NR iterations.}

\item{report}{Report fitting progress?}
}
\value{
An object of class \code{contrast} containing the following:
\describe{
  \item{Model1}{The fitted model for the target group.}
  \item{Model0}{The fitted model for the reference group.}
  \item{Contrast}{Contrasts of means and medians.}
}
}
\description{
Compares the means and medians of parametric survival distributions fit to
each of two treatment arms. Available distributions include: exponential,
gamma, generalized gamma, log-normal, and Weibull.
}
\details{
Status is encoded 0 for censored and 1 for observed. Arm is encoded 0 for 
reference, 1 for target. Tau is an optional numeric vector of truncation times 
for calculating restricted mean survival time, which is the area under the
survival curve up to the specified truncation point.
}
\examples{
\dontrun{
set.seed(100);
# Weibull and Weibull, different means and medians
n = 1e3;
## Target group
d1 = genData(n=n,dist="weibull",theta=c(1,1),p=0.2);
d1$arm = 1;
## Reference group
d0 = genData(n=n,dist="weibull",theta=c(1,2),p=0.2);
d0$arm = 0;
## Overall data set
data = rbind(d1,d0);
## Comparison
comp = compParaSurv(time=data$time,status=data$status,arm=data$arm,tau=0.5,dist1="weibull");

# Gamma and Weibull, same mean, different medians, bootstrap CIs
## Target group
d1 = genData(n=n,dist="gamma",theta=c(2,2),p=0.2);
d1$arm = 1;
## Reference group
d0 = genData(n=n,dist="weibull",theta=c(2,sqrt(pi)/2),p=0.2);
d0$arm = 0;
## Overall data set
data = rbind(d1,d0);
## Comparison
comp = compParaSurv(time=data$time,status=data$status,arm=data$arm,
                    tau=0.5,dist1="gamma",dist0="weibull",boot=2e3);

# Weibull and Log-normal, different means, same median, permutation p.values
## Target group
d1 = genData(n=n,dist="log-normal",theta=c(0,2),p=0.2);
d1$arm = 1;
## Reference group
d0 = genData(n=n,dist="weibull",theta=c(2,sqrt(log(2))),p=0.2);
d0$arm = 0;
## Overall data set
data = rbind(d1,d0);
## Comparison
comp = compParaSurv(time=data$time,status=data$status,arm=data$arm,
                    tau=0.5,dist1="log-normal",dist0="weibull",perm=2e3);
}
}
\seealso{
\itemize{
  \item{Fitting function for parametric survival distributions \code{\link{fitParaSurv}}}
}
}
