#' @title Calculate V-DeltaT Curve
#' @description Calculating the temperature-change-voltage curve for a
#'     particular set of Resistor and Thermistor Values.
#' @param x a vector containing the values of R1 R2 R3 R4 Rb1 Beta1 Rb2 Beta2
#' @param Tdata a vector of temperature-change values
#'
#' @return the voltage values at Point B
#' @export
#'
#' @examples
#' Tdata <- seq(-40, 85, by=5)
#' R_values <- c(1100, 4300, 560, 1100, 220, 3680, 1000, 3560)
#' tempCompCurve(x = R_values, Tdata = Tdata)
tempCompCurve <- function(x, Tdata){
  ## Input voltage
  Vin = 1.1

  ## Thermistor Calculations
  # Values in x: R1 R2 R3 R4 RTH1(T_25degc) Beta1 RTH2(T_25degc) Beta2
  # Thermistors are represented by:
  #   Room temperature is 25degc: T_25
  #   Standard value is at 25degc: RTHx_25
  #   RTHx is the thermistor resistance at various temperature
  #      RTH(T) = RTH(T_25degc) / exp (Beta * (T-T_25)/(T*T_25)))
  T_25 = 298.15;
  T_off = 273.15;
  Beta1 = x[6];
  Beta2 = x[8];
  RTH1 = x[5]/exp(Beta1*((Tdata+T_off)-T_25)/((Tdata+T_off)*T_25));
  RTH2 = x[7]/exp(Beta2*((Tdata+T_off)-T_25)/((Tdata+T_off)*T_25));

  ## Define equivalent circuits for parallel R's and RTH's
  R1_eq = x[1]*RTH1/(x[1]+RTH1);
  R3_eq = x[3]*RTH2/(x[3]+RTH2);

  ## Calculate voltages at Point B
  VB = Vin*(R3_eq+x[4])/(R1_eq+x[2]+R3_eq+x[4]);

  return(VB)
}
