% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/uncertain.interval.R
\name{uncertain.interval}
\alias{uncertain.interval}
\title{Function for the determination of an inconclusive interval for continuous test scores}
\usage{
uncertain.interval(ref, test, sens.ui = 0.55, spec.ui = 0.55,
  intersection = NULL, return.first = T, select = c("nearest", "limited"))
}
\arguments{
\item{ref}{The reference standard. A column in a data frame or a vector indicating the classification by the reference test. The reference standard must be coded either as 0 (absence of the condition) or 1 (presence of the condition)}

\item{test}{The index test or test under evaluation. A column in a dataset or vector indicating the test results in a continuous scale.}

\item{sens.ui}{(default = .55). The sensitivity of the test scores within the uncertain interval is either limited to this value or is the nearest to this value. A value below .5 or larger than .6 is not recommended.}

\item{spec.ui}{(default = .55). The specificity of the test scores within the uncertain interval is either limited to this value or is the nearest to this value. A value below .5 or larger than .6 is not recommended.}

\item{intersection}{(default = NULL) When NULL, the intersection is calculated with \code{get.intersection}, which uses the kernel density method to obtain the intersection. When another value is assigned to this parameter, this value is used instead.}

\item{return.first}{(default = TRUE) Return only the widest possible interval, given the restrictions. When FALSE all calculated intervals with their sensitivity and specificity are returned. NOTE: This function does not always find a suitable interval and can return a vector of NULL values.}

\item{select}{(default = 'nearest') If 'nearest', sensitivity and specificity of the uncertain interval are nearest sens.ui and spec.ui respectively. When 'limited' the solutions have an uncertain interval with a sensitivity and specificity limited by sens.ui and spec.ui respectively.}
}
\value{
{A \code{data.frame} of
 \describe{
 \item{cp.l}{ Lower bound of the Uncertain interval.}
 \item{cp.h}{ Upper bound of the Uncertain interval.}
 \item{FN}{ Count of false negatives within the Uncertain interval.}
 \item{TP}{ Count of true positives within the Uncertain interval.}
 \item{TN}{ Count of true negatives within the Uncertain interval.}
 \item{FP}{ Count of false positives within the Uncertain interval.}
 \item{sensitivity}{ Sensitivity of the test scores within the Uncertain interval.}
 \item{specificity}{ Specificity of the test scores within the Uncertain interval.}
}
Only a single row is returned when parameter \code{return.first} = TRUE (default).}
}
\description{
This function determines an interval around the intersection of the two distributions of individuals without (0) and with (1) the targeted condition. The interval is restricted both by a maximum sensitivity of the test scores within the uncertain interval (sens.ui) and by a maximum specificity of the test scores within the uncertain interval (spec.ui).
}
\details{
{
This essentially non-parametric function finds the best possible solution for the sample.
This function can be used for test with continuous scores or for test with about twenty or more ordened test scores.
The Uncertain interval is defined as an interval below and above the intersection, with a sensitivity and specificity nearby or below a desired value (default .55).

In its core, the \code{uncertain.interval} function is non-parametric, but it uses the gaussian kernel for estimating the intersection between the two distributions. Always check whether your results are within reason. If the results are unsatisfactory, first check on the intersection. The \code{density} function allows for other approximations. Another estimate can be obtained by using a more suitable kernel in the \code{density} function. The parameter \code{intersection} can be used to assign the new estimate to the \code{uncertain.interval} method.

Furthermore, only a single intersection is assumed (or an second intersection where the overlap is negligible). If another intersection exists and the overlap around this intersection is considerable, a second uncertain interval may be determined by using the parameter \code{intersection}. It should be noted that in most cases, a test with more than one intersection with non-negligible overlap is problematic and difficult to apply.

The Uncertain interval method is developed for continuous distributions, although it can be applied to tests with distinguishable categorical distributions. When a test is used with less than 20 discernible values, a warning is issued. The method may work satisfactorily, but results should always be checked carefully.

In general, when estimating decision thresholds, a sample of sufficient size should be used. It is recommended to use at least a sample of 100 patients with the targeted condition, and a 'healthy' sample (without the targeted condition) of the same size or larger.

The Uncertain interval method is not always capable to deliver results, especially when select == 'limited'. Clearly, when there is no overlap between the two distributions, there cannot be an uncertain interval. A very small interval of overlap can also limit the possibilities to find a solution. When there is no solution found, a vector of NA values is returned.

Lastly, it should be noted that the Uncertain interval method has been developed recently, and future research may provide more satisfactory answers.
}
}
\examples{
# A simple test model
set.seed(1)
ref=c(rep(0,500), rep(1,500))
test=c(rnorm(500,0,1), rnorm(500,1,1))
uncertain.interval(ref, test, select='limited')

ref = c(rep(0,20), rep(1,20))
test= c(rnorm(20), rnorm(20, mean=1))
uncertain.interval(ref, test)
}
\references{
Landsheer, J. A. (2016). Interval of Uncertainty: An Alternative Approach for the Determination of Decision Thresholds, with an Illustrative Application for the Prediction of Prostate Cancer. PloS One, 11(11), e0166007.
}

