
% --- Source file: dvg.Rd ---
\name{vgFit}
\alias{vgFit}
\alias{print.vgFit}
\alias{plot.vgFit}

\title{Fit the Variance Gamma to Data}

\description{
  Fits a variance gamma distribution to data. Displays the histogram,
  log-histogram (both with fitted densities), Q-Q plot and P-P plot for the
  fit which has the maximum likelihood.
}

\usage{
  vgFit(x, freq = NULL, breaks = NULL, paramStart = NULL,
        startMethod = "Nelder-Mead", startValues = "SL",
        method = "Nelder-Mead", hessian = FALSE,
        plots = FALSE, printOut = FALSE,
        controlBFGS = list(maxit = 200),
        controlNM = list(maxit = 1000), maxitNLM = 1500, \dots)


  \method{print}{vgFit}(x, digits = max(3, getOption("digits") - 3), \dots)

  \method{plot}{vgFit}(x, which = 1:4,
       plotTitles = paste(c("Histogram of ","Log-Histogram of ",
                            "Q-Q Plot of ","P-P Plot of "), x$obsName,
                          sep = ""),
       ask = prod(par("mfcol")) < length(which) && dev.interactive(),
          \dots)
}

\arguments{
  \item{x}{Data vector for \code{vgFit}. Object of class
    \code{"vgFit"} for \code{print.vgFit} and \code{plot.vgFit}.}
  \item{freq}{A vector of weights with length equal to \code{length(x)}.}
  \item{breaks}{Breaks for histogram, defaults to those generated by
    \code{hist(x, right = FALSE, plot = FALSE)}.}
  \item{paramStart}{A user specified starting parameter vector param taking
    the form \code{c(vgC,sigma,theta,nu)}.}
  \item{startMethod}{Method used by \code{vgFitStart} in calls to
    \code{\link{optim}}, default is "Nelder-Mead". See \bold{Details}.}
  \item{startValues}{Code giving the method of determining starting
    values for finding the maximum likelihood estimate of param, default method
    is "SL". See \bold{Details}.}
  \item{method}{Different optimisation methods to consider, default is
    "Nelder-Mead". See \bold{Details}.}
  \item{hessian}{Logical. If \code{TRUE} the value of the hessian is
    returned.}
  \item{plots}{Logical. If \code{FALSE} suppresses printing of the
    histogram, log-histogram, Q-Q plot and P-P plot.}
  \item{printOut}{Logical. If \code{FALSE} suppresses printing of
    results of fitting.}
  \item{controlBFGS}{A list of control parameters for \code{optim} when using
    the \code{"BFGS"} optimisation.}
  \item{controlNM}{A list of control parameters for \code{optim}
    when using the \code{"Nelder-Mead"} optimisation.}
  \item{maxitNLM}{A positive integer specifying the maximum number of
    iterations when using the \code{"nlm"} optimisation.}
  \item{digits}{Desired number of digits when the object is printed.}
   \item{which}{If a subset of the plots is required, specify a subset of
    the numbers \code{1:4}.}
  \item{plotTitles}{Titles to appear above the plots.}
  \item{ask}{Logical. If \code{TRUE}, the user is \emph{ask}ed before
    each plot, see \code{\link{par}(ask = .)}.}
  \item{\dots}{Passes arguments to \code{par}, \code{hist},
		\code{logHist}, \code{qqhyperb} and \code{pphyperb}.}
}
\details{
  \code{startMethod} can be either \code{"BFGS"} or
	\code{"Nelder-Mead"}.

  \code{startValues} can be one of the following:
  \itemize{
    \item{\code{"US"}}{User-supplied.}
    \item{\code{"SL"}}{Based on a fitted skew-Laplace distribution.}
    \item{\code{"MoM"}}{Method of moments.}
  }
  For the details concerning the use of \code{paramStart},
  \code{startMethod}, and \code{startValues}, see
  \code{\link{vgFitStart}}.

  The three optimisation methods currently available are:
  \itemize{
  \item{\code{"BFGS"}}{Uses the quasi-Newton method \code{"BFGS"} as
    documented in \code{\link{optim}}.}
  \item{\code{"Nelder-Mead"}}{Uses an implementation of the Nelder and
    Mead method as documented in \code{\link{optim}}.}
  \item{\code{"nlm"}}{Uses the \code{\link{nlm}} function in R.}
  }

  For details of how to pass control information for optimisation using
  \code{\link{optim}} and \code{\link{nlm}}, see \code{\link{optim}} and
  \code{\link{nlm}.}

  When \code{method = "Nelder-Mead"} is used, very rarely, it would return an
  error message of "error in optim(paramStart,...)", use \code{method = "BFGS"}
  or \code{method = "nlm"} instead in that case.

  When \code{method = "nlm"} is used, warnings may be produced. These do
	not appear to be a problem.
}
\value{
  A list with components:
    \item{param}{A vector giving the maximum likelihood estimate of
      param, as \code{(c,sigma,theta,nu)}.}
    \item{maxLik}{The value of the maximised log-likelihood.}
    \item{hessian}{If \code{hessian} was set to \code{TRUE}, the value
      of the hessian. Not present otherwise.}
    \item{method}{Optimisation method used.}
    \item{conv}{Convergence code. See the relevant documentation (either
      \code{\link{optim}} or \code{\link{nlm}}) for details on
      convergence.}
    \item{iter}{Number of iterations of optimisation routine.}
    \item{obs}{The data used to fit the hyperbolic distribution.}
    \item{obsName}{A character string with the actual \code{obs} argument
      name.}
    \item{paramStart}{Starting value of param returned by call to
      \code{\link{vgFitStart}}.}
    \item{svName}{Descriptive name for the method finding start values.}
    \item{startValues}{Acronym for the method of finding start values.}
    \item{breaks}{The cell boundaries found by a call to
      \code{\link{hist}}.}
    \item{midpoints}{The cell midpoints found by a call to
      \code{\link{hist}}.}
    \item{empDens}{The estimated density found by a call to
      \code{\link{hist}}.}

  }

\references{
  Seneta, E. (2004). Fitting the variance-gamma model to financial data.
  \emph{J. Appl. Prob.}, 41A:177--187.
}


\author{David Scott \email{d.scott@auckland.ac.nz}, Christine Yang Dong}


\seealso{
  \code{\link{optim}}, \code{\link{nlm}}, \code{\link{par}},
  \code{\link{hist}}, \code{\link[HyperbolicDist]{logHist}},
  \code{\link{qqvg}}, \code{\link{ppvg}},
  \code{\link[HyperbolicDist]{dskewlap}} and
  \code{\link{vgFitStart}}.

}

\examples{
param <- c(0,0.5,0,0.5)
dataVector <- rvg(500, param = param)
## See how well vgFit works
vgFit(dataVector)
vgFit(dataVector, plots = TRUE)
fit <- vgFit(dataVector)
par(mfrow = c(1,2))
plot(fit, which = c(1,3))

## Use nlm instead of default
param <- c(0,0.5,0,0.5)
dataVector <- rvg(500, param = param)
vgFit(dataVector, method = "nlm", hessian = TRUE)


## Use BFGS instead of deault
param <- c(0,0.5,0,0.5)
dataVector <- rvg(500, param = param)
vgFit(dataVector, method = "BFGS", hessian = TRUE)
}