% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_date_vars.R
\name{derive_vars_dtm}
\alias{derive_vars_dtm}
\title{Derive/Impute a Datetime from a Date Character Vector}
\usage{
derive_vars_dtm(
  dataset,
  new_vars_prefix,
  dtc,
  date_imputation = NULL,
  time_imputation = "00:00:00",
  flag_imputation = "auto",
  min_dates = NULL,
  max_dates = NULL,
  preserve = FALSE,
  ignore_seconds_flag = FALSE
)
}
\arguments{
\item{dataset}{Input dataset

The date character vector (\code{dtc}) must be present.}

\item{new_vars_prefix}{Prefix used for the output variable(s).

a character is expected: e.g. \code{new_vars_prefix = "AST"}.}

\item{dtc}{The \code{'--DTC'} date to impute

A character date is expected in a format like \code{yyyy-mm-dd} or
\code{yyyy-mm-ddThh:mm:ss}. If the year part is not recorded (missing date), no
imputation is performed.}

\item{date_imputation}{The value to impute the day/month when a datepart is
missing.

If \code{NULL}: no date imputation is performed and partial dates are returned as
missing.

Otherwise, a character value is expected, either as a
\itemize{
\item format with month and day specified as \code{"mm-dd"}: e.g. \code{"06-15"} for the 15th
of June,
\item or as a keyword: \code{"FIRST"}, \code{"MID"}, \code{"LAST"} to impute to the first/mid/last
day/month.
}

Default is \code{NULL}.}

\item{time_imputation}{The value to impute the time when a timepart is
missing.

A character value is expected, either as a
\itemize{
\item format with hour, min and sec specified as \code{"hh:mm:ss"}: e.g. \code{"00:00:00"}
for the start of the day,
\item or as a keyword: \code{"FIRST"},\code{"LAST"} to impute to the start/end of a day.
}

Default is \code{"00:00:00"}.}

\item{flag_imputation}{Whether the date/time imputation flag(s) must also be derived.

One of "auto", "date" or "both"

Default: "auto"}

\item{min_dates}{Minimum dates

A list of dates is expected. It is ensured that the imputed date is not
before any of the specified dates, e.g., that the imputed adverse event start
date is not before the first treatment date. Only dates which are in the
range of possible dates of the \code{dtc} value are considered. The possible dates
are defined by the missing parts of the \code{dtc} date (see example below). This
ensures that the non-missing parts of the \code{dtc} date are not changed. For
example\preformatted{impute_dtc(
  "2020-11",
  min_dates = list(
    ymd_hms("2020-12-06T12:12:12"),
    ymd_hms("2020-11-11T11:11:11")
   ),
  date_imputation = "first"
)
}

returns \code{"2020-11-11T11:11:11"} because the possible dates for \code{"2020-11"}
range from \code{"2020-11-01T00:00:00"} to \code{"2020-11-30T23:59:59"}. Therefore
\code{"2020-12-06T12:12:12"} is ignored. Returning \code{"2020-12-06T12:12:12"} would
have changed the month although it is not missing (in the \code{dtc} date).}

\item{max_dates}{Maximum dates

A list of dates is expected. It is ensured that the imputed date is not after
any of the specified dates, e.g., that the imputed date is not after the data
cut off date. Only dates which are in the range of possible dates are
considered.}

\item{preserve}{Preserve day if month is missing and day is present

For example \code{"2019---07"} would return \verb{"2019-06-07} if \code{preserve = TRUE}
(and \code{date_imputation = "MID"}).

Permitted Values: \code{TRUE}, \code{FALSE}

Default: \code{FALSE}}

\item{ignore_seconds_flag}{ADaM IG states that given SDTM (\code{'--DTC'}) variable,
if only hours and minutes are ever collected, and seconds are imputed in
(\code{'--DTM'}) as 00, then it is not necessary to set (\code{'--TMF'}) to \code{'S'}. A user can set this
to \code{TRUE} so the \code{'S'} Flag is dropped from (\code{'--TMF'}).

A logical value

Default: \code{FALSE}}
}
\value{
The input dataset with the datetime \code{'--DTM'} (and the date/time imputation
flag \code{'--DTF'}, \code{'--TMF'}) added.
}
\description{
Derive a datetime object (\code{'--DTM'}) from a date character vector (\code{'--DTC'}).
The date and time can be imputed (see \code{date_imputation}/\code{time_imputation} parameters)
and the date/time imputation flag (\code{'--DTF'}, \code{'--TMF'}) can be added.
}
\details{
The presence of a \code{'--DTF'} variable is checked and the variable is not derived
if it already exists in the input dataset. However, if \code{'--TMF'} already exists
in the input dataset, a warning is issued and \code{'--TMF'} will be overwritten.
}
\examples{
library(lubridate)

mhdt <- tibble::tribble(
  ~MHSTDTC,
  "2019-07-18T15:25:40",
  "2019-07-18T15:25",
  "2019-07-18",
  "2019-02",
  "2019",
  "2019---07",
  ""
)

derive_vars_dtm(
  mhdt,
  new_vars_prefix = "AST",
  dtc = MHSTDTC,
  date_imputation = "FIRST",
  time_imputation = "FIRST"
)

# Impute AE end date to the last date and ensure that the imputed date is not
# after the death or data cut off date
adae <- tibble::tribble(
  ~AEENDTC, ~DTHDT, ~DCUTDT,
  "2020-12", ymd("2020-12-06"), ymd("2020-12-24"),
  "2020-11", ymd("2020-12-06"), ymd("2020-12-24")
)

derive_vars_dtm(
  adae,
  dtc = AEENDTC,
  new_vars_prefix = "AEN",
  date_imputation = "last",
  time_imputation = "last",
  max_dates = vars(DTHDT, DCUTDT)
)

# Seconds has been removed from the input dataset.  Function now uses
# ignore_seconds_flag to remove the 'S' from the --TMF variable.
mhdt <- tibble::tribble(
  ~MHSTDTC,
  "2019-07-18T15:25",
  "2019-07-18T15:25",
  "2019-07-18",
  "2019-02",
  "2019",
  "2019---07",
  ""
)

derive_vars_dtm(
  mhdt,
  new_vars_prefix = "AST",
  dtc = MHSTDTC,
  date_imputation = "FIRST",
  time_imputation = "FIRST",
  ignore_seconds_flag = TRUE
)

# A user imputing dates as middle month/day, i.e. date_imputation = "MID" can
# use preserve argument to "preserve" partial dates.  For example, "2019---07",
# will be displayed as "2019-06-07" rather than 2019-06-15 with preserve = TRUE

derive_vars_dtm(
  mhdt,
  new_vars_prefix = "AST",
  dtc = MHSTDTC,
  date_imputation = "MID",
  preserve = TRUE
)
}
\author{
Samia Kabi
}
\keyword{adam}
\keyword{derivation}
\keyword{timing}
