% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_param_tte.R
\name{derive_param_tte}
\alias{derive_param_tte}
\title{Derive a Time-to-Event Parameter}
\usage{
derive_param_tte(
  dataset = NULL,
  dataset_adsl,
  source_datasets,
  by_vars = NULL,
  start_date = TRTSDT,
  event_conditions,
  censor_conditions,
  create_datetime = FALSE,
  set_values_to,
  subject_keys = get_admiral_option("subject_keys")
)
}
\arguments{
\item{dataset}{Input dataset
\code{PARAMCD} is expected.}

\item{dataset_adsl}{ADSL input dataset

The variables specified for \code{start_date}, and
\code{subject_keys} are expected.}

\item{source_datasets}{Source datasets

A named list of datasets is expected. The \code{dataset_name} field of
\code{tte_source()} refers to the dataset provided in the list.}

\item{by_vars}{By variables

If the parameter is specified, separate time to event parameters are
derived for each by group.

The by variables must be in at least one of the source datasets. Each
source dataset must contain either all by variables or none of the by
variables.

The by variables are not included in the output dataset.

\emph{Permitted Values}: list of variables created by \code{exprs()}
e.g. \code{exprs(USUBJID, VISIT)}}

\item{start_date}{Time to event origin date

The variable \code{STARTDT} is set to the specified date. The value is taken
from the ADSL dataset.

If the event or censoring date is before the origin date, \code{ADT} is set to
the origin date.}

\item{event_conditions}{Sources and conditions defining events

A list of \code{event_source()} objects is expected.}

\item{censor_conditions}{Sources and conditions defining censorings

A list of \code{censor_source()} objects is expected.}

\item{create_datetime}{Create datetime variables?

If set to \code{TRUE}, variables \code{ADTM} and \code{STARTDTM} are created. Otherwise,
variables \code{ADT} and \code{STARTDT} are created.}

\item{set_values_to}{Variables to set

A named list returned by \code{exprs()} defining the variables to be set for the
new parameter, e.g. \code{exprs(PARAMCD = "OS", PARAM = "Overall Survival")} is
expected. The values must be symbols, character strings, numeric values,
expressions, or \code{NA}.}

\item{subject_keys}{Variables to uniquely identify a subject

A list of symbols created using \code{exprs()} is expected.}
}
\value{
The input dataset with the new parameter added
}
\description{
Add a time-to-event parameter to the input dataset.
}
\details{
The following steps are performed to create the observations of the
new parameter:

\strong{Deriving the events:}

\enumerate{ \item For each event source dataset the observations as
specified by the \code{filter} element are selected. Then for each patient the
first observation (with respect to \code{date}) is selected.

\item The \code{ADT} variable is set to the variable specified by the
\code{date} element. If the date variable is a datetime variable, only
the datepart is copied.

\item The \code{CNSR} variable is added and set to the \code{censor} element.

\item The variables specified by the \code{set_values_to} element are
added.

\item The selected observations of all event source datasets are combined into a
single dataset.

\item For each patient the first observation (with respect to the \code{ADT}
variable) from the single dataset is selected. }

\strong{Deriving the censoring observations:}

\enumerate{ \item For each censoring source dataset the observations as
specified by the \code{filter} element are selected. Then for each patient the
last observation (with respect to \code{date}) is selected.

\item The \code{ADT} variable is set to the variable specified by the
\code{date} element. If the date variable is a datetime variable, only
the datepart is copied.

\item The \code{CNSR} variable is added and set to the \code{censor} element.

\item The variables specified by the \code{set_values_to} element are
added.

\item The selected observations of all censoring source datasets are
combined into a single dataset.

\item For each patient the last observation (with respect to the \code{ADT}
variable) from the single dataset is selected. }

For each subject (as defined by the \code{subject_keys} parameter) an
observation is selected. If an event is available, the event observation is
selected. Otherwise the censoring observation is selected.

Finally:
\enumerate{
\item The variable specified for \code{start_date} is joined from the
ADSL dataset. Only subjects in both datasets are kept,
i.e., subjects with both an event or censoring and an observation in
\code{dataset_adsl}.
\item The variables as defined by the \code{set_values_to} parameter are added.
\item The \code{ADT}/\code{ADTM} variable is set to the maximum of \code{ADT}/\code{ADTM} and
\code{STARTDT}/\code{STARTDTM} (depending on the \code{create_datetime} parameter).
\item The new observations are added to the output dataset.
}
}
\examples{
library(tibble)
library(dplyr, warn.conflicts = FALSE)
library(lubridate)
data("admiral_adsl")

adsl <- admiral_adsl

# derive overall survival parameter
death <- event_source(
  dataset_name = "adsl",
  filter = DTHFL == "Y",
  date = DTHDT,
  set_values_to = exprs(
    EVNTDESC = "DEATH",
    SRCDOM = "ADSL",
    SRCVAR = "DTHDT"
  )
)

last_alive_dt <- censor_source(
  dataset_name = "adsl",
  date = LSTALVDT,
  set_values_to = exprs(
    EVNTDESC = "LAST DATE KNOWN ALIVE",
    SRCDOM = "ADSL",
    SRCVAR = "LSTALVDT"
  )
)

derive_param_tte(
  dataset_adsl = adsl,
  event_conditions = list(death),
  censor_conditions = list(last_alive_dt),
  source_datasets = list(adsl = adsl),
  set_values_to = exprs(
    PARAMCD = "OS",
    PARAM = "Overall Survival"
  )
) \%>\%
  select(-STUDYID) \%>\%
  filter(row_number() \%in\% 20:30)

# derive duration of response
# only observations for subjects in dataset_adsl are created
adsl <- tribble(
  ~USUBJID, ~DTHFL, ~DTHDT,            ~RSPDT,
  "01",     "Y",    ymd("2021-06-12"), ymd("2021-03-04"),
  "02",     "N",    NA,                NA,
  "03",     "Y",    ymd("2021-08-21"), NA,
  "04",     "N",    NA,                ymd("2021-04-14")
) \%>\%
  mutate(STUDYID = "AB42")

adrs <- tribble(
  ~USUBJID, ~AVALC, ~ADT,              ~ASEQ,
  "01",     "SD",   ymd("2021-01-03"), 1,
  "01",     "PR",   ymd("2021-03-04"), 2,
  "01",     "PD",   ymd("2021-05-05"), 3,
  "02",     "PD",   ymd("2021-02-03"), 1,
  "04",     "SD",   ymd("2021-02-13"), 1,
  "04",     "PR",   ymd("2021-04-14"), 2,
  "04",     "CR",   ymd("2021-05-15"), 3
) \%>\%
  mutate(STUDYID = "AB42", PARAMCD = "OVR")

pd <- event_source(
  dataset_name = "adrs",
  filter = AVALC == "PD",
  date = ADT,
  set_values_to = exprs(
    EVENTDESC = "PD",
    SRCDOM = "ADRS",
    SRCVAR = "ADTM",
    SRCSEQ = ASEQ
  )
)

death <- event_source(
  dataset_name = "adsl",
  filter = DTHFL == "Y",
  date = DTHDT,
  set_values_to = exprs(
    EVENTDESC = "DEATH",
    SRCDOM = "ADSL",
    SRCVAR = "DTHDT"
  )
)

lastvisit <- censor_source(
  dataset_name = "adrs",
  date = ADT,
  censor = 1,
  set_values_to = exprs(
    EVENTDESC = "LAST TUMOR ASSESSMENT",
    SRCDOM = "ADRS",
    SRCVAR = "ADTM",
    SRCSEQ = ASEQ
  )
)

first_response <- censor_source(
  dataset_name = "adsl",
  date = RSPDT,
  censor = 1,
  set_values_to = exprs(
    EVENTDESC = "FIRST RESPONSE",
    SRCDOM = "ADSL",
    SRCVAR = "RSPDT"
  )
)

derive_param_tte(
  dataset_adsl = filter(adsl, !is.na(RSPDT)),
  start_date = RSPDT,
  event_conditions = list(pd, death),
  censor_conditions = list(lastvisit, first_response),
  source_datasets = list(adsl = adsl, adrs = adrs),
  set_values_to = exprs(
    PARAMCD = "DURRSP",
    PARAM = "Duration of Response"
  )
)

# derive time to adverse event for each preferred term
adsl <- tribble(
  ~USUBJID, ~TRTSDT,           ~EOSDT,
  "01",     ymd("2020-12-06"), ymd("2021-03-06"),
  "02",     ymd("2021-01-16"), ymd("2021-02-03")
) \%>\%
  mutate(STUDYID = "AB42")

ae <- tribble(
  ~USUBJID, ~AESTDTC,           ~AESEQ, ~AEDECOD,
  "01",     "2021-01-03T10:56", 1,      "Flu",
  "01",     "2021-03-04",       2,      "Cough",
  "01",     "2021",             3,      "Flu"
) \%>\%
  mutate(STUDYID = "AB42")

ae_ext <- derive_vars_dt(
  ae,
  dtc = AESTDTC,
  new_vars_prefix = "AEST",
  highest_imputation = "M",
  flag_imputation = "none"
)

ttae <- event_source(
  dataset_name = "ae",
  date = AESTDT,
  set_values_to = exprs(
    EVNTDESC = "AE",
    SRCDOM = "AE",
    SRCVAR = "AESTDTC",
    SRCSEQ = AESEQ
  )
)

eos <- censor_source(
  dataset_name = "adsl",
  date = EOSDT,
  set_values_to = exprs(
    EVNTDESC = "END OF STUDY",
    SRCDOM = "ADSL",
    SRCVAR = "EOSDT"
  )
)

derive_param_tte(
  dataset_adsl = adsl,
  by_vars = exprs(AEDECOD),
  start_date = TRTSDT,
  event_conditions = list(ttae),
  censor_conditions = list(eos),
  source_datasets = list(adsl = adsl, ae = ae_ext),
  set_values_to = exprs(
    PARAMCD = paste0("TTAE", as.numeric(as.factor(AEDECOD))),
    PARAM = paste("Time to First", AEDECOD, "Adverse Event"),
    PARCAT1 = "TTAE",
    PARCAT2 = AEDECOD
  )
) \%>\%
  select(USUBJID, STARTDT, PARAMCD, PARAM, ADT, CNSR, SRCSEQ)
}
\seealso{
\code{\link[=event_source]{event_source()}}, \code{\link[=censor_source]{censor_source()}}
}
\concept{der_prm_tte}
\keyword{der_prm_tte}
