% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stats.R
\name{create_p_funs}
\alias{create_p_funs}
\title{Create a p-value functions object that can be used to generate p-values}
\usage{
create_p_funs(
  data,
  model,
  ...,
  parallel = parallel::detectCores(logical = FALSE) - 1,
  model_packages = as.character(NA),
  random_model_call_string = NULL,
  random_model_call_string_vars = character(),
  y_col = NULL,
  pred_fun = function(object, newdata, type = pred_type) {
     stats::predict(object =
    object, newdata = newdata, type = type)
 },
  pred_type = "response",
  rand_it = 1000,
  silent = FALSE,
  .testing_mode = FALSE
)
}
\arguments{
\item{data}{See documentation for \code{\link[=ale]{ale()}}}

\item{model}{See documentation for \code{\link[=ale]{ale()}}}

\item{...}{not used. Inserted to require explicit naming of subsequent arguments.}

\item{parallel}{See documentation for \code{\link[=ale]{ale()}}}

\item{model_packages}{See documentation for \code{\link[=ale]{ale()}}}

\item{random_model_call_string}{character string. If NULL, \code{create_p_funs()} tries to
automatically detect and construct the call for p-values. If it cannot, the
function will fail early. In that case, a character string of the full call
for the model must be provided that includes the random variable. See details.}

\item{random_model_call_string_vars}{See documentation for \code{model_call_string_vars}
in \code{\link[=model_bootstrap]{model_bootstrap()}}; the operation is very similar.}

\item{y_col}{See documentation for \code{\link[=ale]{ale()}}}

\item{pred_fun, pred_type}{See documentation for \code{\link[=ale]{ale()}}.}

\item{rand_it}{non-negative integer length 1. Number of times that the model
should be retrained with a new random variable. The default of 1000 should
give reasonably stable p-values. It can be reduced as low as 100 for faster
test runs.}

\item{silent}{See documentation for \code{\link[=ale]{ale()}}}

\item{.testing_mode}{logical. Internal use only.}
}
\value{
The return value is a list of class \verb{c('p_funs', 'ale', 'list'}) with an
\code{ale_version} attribute whose value is the version of the \code{ale} package used to
create the object. See examples for an illustration of how to inspect this list.
Its elements are:
\itemize{
\item \code{value_to_p}: a list of functions named for each each available ALE statistic.
Each function signature is \verb{function(x)} where x is a numeric. The function returns
the p-value (minimum 0; maximum 1) for the respective statistic based on the random variable analysis.
For an input x that returns p, its interpretation is that p\% of random variables
obtained the same or higher statistic value. For example, to get the p-value
of a NALED of 4.2, enter \code{p_funs$value_to_p(4.2)}. A return value of 0.03 means
that only 3\% of random variables obtained a NALED greater than or equal to 4.2.
\item \code{p_to_random_value}: a list of functions named for each each available ALE statistic.
These are the inverse functions of \code{value_to_p}. The signature is \verb{function(p)}
where p is a numeric from 0 to 1. The function returns the numeric value of the
random variable statistic that would yield the provided p-value.
For an input p that returns x, its interpretation is that p\% of random variables
obtained the same or higher statistic value. For example, to get the random
variable ALED for the 0.05 p-value, enter \code{p_funs$p_to_random_value(0.05)}.
A return value of 102 means that only 5\% of random variables obtained an ALED
greater than or equal to 102.
\item \code{rand_stats}: a tibble whose rows are each of the \code{rand_it} iterations of the
random variable analysis and whose columns are the ALE statistics obtained for
each random variable.
\item \code{residuals}: the actual \code{y_col} values from \code{data} minus the predicted
values from the \code{model} (without random variables) on the \code{data}.
\code{residual_distribution}: the closest estimated distribution for the \code{residuals}
as determined by \code{\link[univariateML:MaximumLikelihoodDistribution]{univariateML::rml()}}. This is the distribution used to generate
all the random variables.
}
}
\description{
Calculating p-values is not trivial for ALE statistics because ALE is
non-parametric and model-agnostic. Because ALE is non-parametric (that is,
it does not assume any particular distribution of data), the \code{ale} package
generates p-values by calculating ALE for many random variables; this makes the
procedure somewhat slow. For this reason, they are not calculated by default;
they must be explicitly requested. Because the \code{ale} package is model-agnostic (that is, it
works with any kind of R model), the \code{\link[=ale]{ale()}} function cannot always automatically
manipulate the model object to create the p-values. It can only do so for
models that follow the standard R statistical modelling conventions, which
includes almost all built-in R algorithms (like \code{\link[stats:lm]{stats::lm()}} and \code{\link[stats:glm]{stats::glm()}}) and many widely
used statistics packages (like \code{mgcv} and \code{survival}), but which excludes most
machine learning algorithms (like \code{tidymodels} and \code{caret}). For non-standard
algorithms, the user needs to do a little work to help the ale function
correctly manipulate its model object:
\itemize{
\item The full model call must be passed as a character string in the argument
'random_model_call_string', with two slight modifications as follows.
\item In the formula that specifies the model, you must add a variable named
'random_variable'. This corresponds to the random variables that \code{\link[=create_p_funs]{create_p_funs()}}
will use to estimate p-values.
\item The dataset on which the model is trained must be named 'rand_data'. This
corresponds to the modified datasets that will be used to train the random
variables.
}

See the example below for how this is implemented.
}
\section{Approach to calculating p-values}{

The \code{ale} package takes a literal frequentist approach to the calculation of
p-values. That is, it literally retrains the model 1000 times, each time
modifying it by adding a distinct random variable to the model.
(The number of iterations is customizable
with the \code{rand_it} argument.) The ALEs and ALE statistics are calculated for
each random variable. The percentiles of the distribution of these
random-variable ALEs are then used to determine p-values for non-random variables.
Thus, p-values are interpreted as the frequency of random variable ALE statistics
that exceed the value of ALE statistic of the actual variable in question.
The specific steps are as follows:
\itemize{
\item The residuals of the original model trained on the training data are calculated
(residuals are the actual y target value minus the predicted values).
\item The closest distribution of the residuals is detected with
\code{univariateML::model_select()}.
\item 1000 new models are trained by generating a random variable each time with
\code{univariateML::rml()} and then training a new model with that random variable
added.
\item The ALEs and ALE statistics are calculated for each random variable.
\item For each ALE statistic, the empirical cumulative distribution function
(from \code{stats::ecdf()}) is used to create a function to determine p-values
according to the distribution of the random variables' ALE statistics.
}
}

\examples{
\donttest{
# Sample 1000 rows from the ggplot2::diamonds dataset (for a simple example)
set.seed(0)
diamonds_sample <- ggplot2::diamonds[sample(nrow(ggplot2::diamonds), 1000), ]

# Create a GAM model with flexible curves to predict diamond price
# Smooth all numeric variables and include all other variables
gam_diamonds <- mgcv::gam(
  price ~ s(carat) + s(depth) + s(table) + s(x) + s(y) + s(z) +
    cut + color + clarity,
  data = diamonds_sample
)
summary(gam_diamonds)

# Create p-value functions
pf_diamonds <- create_p_funs(
  diamonds_sample,
  gam_diamonds,
  # only 100 iterations for a quick demo; but usually should remain at 1000
  rand_it = 100,
)

# Examine the structure of the returned object
str(pf_diamonds)
# In RStudio: View(pf_diamonds)

# Calculate ALEs with p-values
ale_gam_diamonds <- ale(
  diamonds_sample,
  gam_diamonds,
  p_values = pf_diamonds
)

# Plot the ALE data. The horizontal bands in the plots use the p-values.
ale_gam_diamonds$plots |>
  patchwork::wrap_plots()


# For non-standard models that give errors with the default settings,
# you can use 'random_model_call_string' to specify a model for the estimation
# of p-values from random variables as in this example.
# See details above for an explanation.
pf_diamonds <- create_p_funs(
  diamonds_sample,
  gam_diamonds,
  random_model_call_string = 'mgcv::gam(
    price ~ s(carat) + s(depth) + s(table) + s(x) + s(y) + s(z) +
        cut + color + clarity + random_variable,
    data = rand_data
  )',
  # only 100 iterations for a quick demo; but usually should remain at 1000
  rand_it = 100,
)

}


}
\references{
Okoli, Chitu. 2023.
“Statistical Inference Using Machine Learning and Classical Techniques Based
on Accumulated Local Effects (ALE).” arXiv. \url{https://arxiv.org/abs/2310.09877}.
}
