% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/uhc_plots.R
\name{prep_uhc}
\alias{prep_uhc}
\alias{prep_uhc.glm}
\alias{prep_uhc.fit_logit}
\alias{prep_uhc.fit_clogit}
\title{Prepare Data for UHC Plots for a Fitted Model}
\usage{
prep_uhc(object, test_dat, n_samp = 1000, n_dens = 512, verbose = TRUE)

\method{prep_uhc}{glm}(object, test_dat, n_samp = 1000, n_dens = 512, verbose = TRUE)

\method{prep_uhc}{fit_logit}(object, test_dat, n_samp = 1000, n_dens = 512, verbose = TRUE)

\method{prep_uhc}{fit_clogit}(object, test_dat, n_samp = 1000, n_dens = 512, verbose = TRUE)
}
\arguments{
\item{object}{\verb{[glm, fit_logit, fit_clogit]} \cr A fitted RSF or (i)SSF model.
Should be fit to \emph{training} dataset separate from the testing data.}

\item{test_dat}{\verb{[data.frame]} \cr A \code{data.frame} with \emph{testing} data from
which to sample test points. Should be separate from the data used to train
the model passed to \code{object}.}

\item{n_samp}{\verb{[numeric = 1000]} A \code{vector} of length 1 giving the number of
samples to use to characterize the used habitat distribution under the model.}

\item{n_dens}{\verb{[numeric = 512]} A \code{numeric} vector of length 1 giving the
number of equally spaced points at which density (used, available, and
sampled) is estimated. Passed to \code{stats::density.default()}, which indicates
that \code{n} should usually be specified as a power of 2.}

\item{verbose}{\verb{[logical]} Should messages be displayed (\code{TRUE}) or not
(\code{FALSE})?}
}
\value{
Returns a \code{list} of class \code{uhc_data} with elements:
\itemize{
\item \code{orig}: List of \code{data.frame}s, one per variable (see \code{vars}). Each
\code{data.frame} contains the density plot data (\code{x} and \code{y}) for the original
used (\code{dist == "U"}) and available (\code{dist == "A"}) data.
\item \code{samp}: List of \code{data.frame}s, one per variable (see \code{vars}). Each
\code{data.frame} contains the density plot data (\code{x} and \code{y}) for each iteration
of bootstrap resampling (\code{iter}).
\item \code{vars}: Character vector with names of the habitat variables for which to
create UHC plots.
\item \code{type}: Named character vector with the type for each of \code{vars} (either
\code{"numeric"} or \code{"factor"}).
\item \code{resp}: Character vector of length 1 with the name of the response
variable.
}
}
\description{
Creates data used to make used-habitat calibration plots
}
\details{
This function performs the heavy lifting of creating UHC plots.
It creates the data used later by the \code{plot()} method, which actually
draws the UHC plots. This function (1) creates density plots of the used
and available locations from the \emph{test} data, and (2) resamples the (a)
fitted coefficients and (b) test data (weighted by the exponential habitat
selection function) to create the distribution of used habitat under the
model.

Note that \code{test_dat} should contain at least all of the variables that
appear in the model \code{object}. Any further habitat variables in \code{test_dat}
will also have UHC plots generated, treating these variables as possible
candidate variables that are simply not included in this particular model.
}
\examples{

\donttest{
# Load packages
library(amt)
library(dplyr)
library(terra)
library(sf)

# HSF ----------------------------------------------
# Load data
data(uhc_hsf_locs)
data(uhc_hab)
hab <- rast(uhc_hab, type = "xyz", crs = "epsg:32612")
# Convert "cover" layer to factor
levels(hab[[4]]) <- data.frame(id = 1:3,
                               cover = c("grass", "forest", "wetland"))

# Split into train (80\%) and test (20\%)
set.seed(1)
uhc_hsf_locs$train <- rbinom(n = nrow(uhc_hsf_locs),
                             size = 1, prob = 0.8)
train <- uhc_hsf_locs[uhc_hsf_locs$train == 1, ]
test <- uhc_hsf_locs[uhc_hsf_locs$train == 0, ]

# Available locations
avail_train <- random_points(st_as_sf(st_as_sfc(st_bbox(hab))),
                             n = nrow(train) * 10)

avail_test <- random_points(st_as_sf(st_as_sfc(st_bbox(hab))),
                            n = nrow(test) * 10)

# Combine with used
train_dat <- train |>
  make_track(x, y, crs = 32612) |>
  mutate(case_ = TRUE) |>
  bind_rows(avail_train) |>
  # Attach covariates
  extract_covariates(hab) |>
  # Assign large weights to available
  mutate(weight = case_when(
    case_ ~ 1,
    !case_ ~ 5000
  ))

test_dat <- test |>
  make_track(x, y, crs = 32612) |>
  mutate(case_ = TRUE) |>
  bind_rows(avail_test) |>
  # Attach covariates
  extract_covariates(hab) |>
  # Assign large weights to available
  mutate(weight = case_when(
    case_ ~ 1,
    !case_ ~ 5000
  ))

# Fit (correct) HSF
hsf1 <- glm(case_ ~ forage + temp + I(temp^2) + pred + cover,
            data = train_dat, family = binomial(), weights = weight)

# Drop weights from 'test_dat'
test_dat$weight <- NULL

# Prep UHC plots
uhc_dat <- prep_uhc(object = hsf1, test_dat = test_dat,
                    n_samp = 500, verbose = TRUE)

# Plot all variables
plot(uhc_dat)

# Plot only first variable
plot(uhc_dat[1])

# Plot only "cover" variable
plot(uhc_dat["cover"])

# Coerce to data.frame
df <- as.data.frame(uhc_dat)

# Simplify sampled lines to confidence envelopes
conf <- conf_envelope(df)

# Default plot for the envelopes version
plot(conf)
}

}
\references{
Fieberg, J.R., Forester, J.D., Street, G.M., Johnson, D.H., ArchMiller, A.A.,
and Matthiopoulos, J. 2018. Used-habitat calibration plots: A new procedure
for validating species distribution, resource selection, and step-selection
models. \emph{Ecography} 41:737–752.
}
\seealso{
See Fieberg \emph{et al.} 2018 for details about UHC plots.

Default plotting method available: \code{\link{plot.uhc_data}()}

Coercion to \code{data.frame}: \code{\link{as.data.frame.uhc_data}()}

Subsetting method: \code{\link{Extract.uhc_data}}
}
\author{
Brian J. Smith
}
