% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/orsf_vi.R
\name{orsf_vi}
\alias{orsf_vi}
\alias{orsf_vi_negate}
\alias{orsf_vi_permute}
\alias{orsf_vi_anova}
\title{ORSF variable importance}
\usage{
orsf_vi(object, group_factors = TRUE, importance = NULL, oobag_fun = NULL, ...)

orsf_vi_negate(object, group_factors = TRUE, oobag_fun = NULL, ...)

orsf_vi_permute(object, group_factors = TRUE, oobag_fun = NULL, ...)

orsf_vi_anova(object, group_factors = TRUE, ...)
}
\arguments{
\item{object}{(\emph{orsf_fit}) a trained oblique random survival forest
(see \link{orsf}).}

\item{group_factors}{(\emph{logical}) if \code{TRUE}, the importance of factor
variables will be reported overall by aggregating the importance
of individual levels of the factor. If \code{FALSE}, the importance of
individual factor levels will be returned.}

\item{importance}{(\emph{character}) Indicate method for variable importance:
\itemize{
\item 'anova': compute analysis of variance (ANOVA) importance
\item 'negate': compute negation importance
\item 'permute': compute permutation importance
}}

\item{oobag_fun}{(\emph{function}) to be used for evaluating out-of-bag prediction accuracy after negating coefficients
(if importance = 'negate') or permuting the values of a predictor
(if importance = 'permute')
\itemize{
\item When \code{oobag_fun = NULL} (the default), Harrell's C-statistic (1982) is used to evaluate accuracy.
\item if you use your own \code{oobag_fun} note the following:
\itemize{
\item \code{oobag_fun} should have two inputs: \code{y_mat} and \code{s_vec}
\item \code{y_mat} is a two column matrix with first column named 'time', second named 'status'
\item \code{s_vec} is a numeric vector containing predicted survival probabilities.
\item \code{oobag_fun} should return a numeric output of length 1
\item the same \code{oobag_fun} should have been used when you created \code{object}
so that the initial value of out-of-bag prediction accuracy is
consistent with the values that will be computed while variable
importance is estimated.
}
}

For more details, see the out-of-bag
\href{https://docs.ropensci.org/aorsf/articles/oobag.html}{vignette}.}

\item{...}{Further arguments passed to or from other methods (not currently used).}
}
\value{
\code{orsf_vi} functions return a named numeric vector.
\itemize{
\item Names of the vector are the predictor variables used by \code{object}
\item Values of the vector are the estimated importance of the given predictor.
}

The returned vector is sorted from highest to lowest value, with higher
values indicating higher importance.
}
\description{
Estimate the importance of individual variables using oblique random
survival forests.
}
\details{
When an \code{orsf_fit} object is fitted with importance = 'anova', 'negate', or
'permute', the output will have a vector of importance values based on
the requested type of importance. However, you may still want to call
\code{orsf_vi()} on this output if you want to group factor levels into one
overall importance value.

\code{orsf_vi()} is a general purpose function to extract or compute variable
importance estimates from an \code{'orsf_fit'} object (see \link{orsf}).
\code{orsf_vi_negate()}, \code{orsf_vi_permute()}, and \code{orsf_vi_anova()} are wrappers
for \code{orsf_vi()}. The way these functions work depends on whether the
\code{object} they are given already has variable importance estimates in it
or not (see examples).
}
\section{Variable importance methods}{


\strong{negation importance}: Each variable is assessed separately by multiplying the variable's coefficients by -1 and then determining how much the model's performance changes. The worse the model's performance after negating coefficients for a given variable, the more important the variable. This technique is promising b/c it does not require permutation and it emphasizes variables with larger coefficients in linear combinations, but it is also relatively new and hasn't been studied as much as permutation importance. See \href{https://arxiv.org/abs/2208.01129}{Jaeger, 2022} for more details on this technique.

\strong{permutation importance}: Each variable is assessed separately by randomly permuting the variable's values and then determining how much the model's performance changes. The worse the model's performance after permuting the values of a given variable, the more important the variable. This technique is flexible, intuitive, and frequently used. It also has several \href{https://christophm.github.io/interpretable-ml-book/feature-importance.html#disadvantages-9}{known limitations}

\strong{analysis of variance (ANOVA) importance}: A p-value is computed for each coefficient in each linear combination of variables in each decision tree. Importance for an individual predictor variable is the proportion of times a p-value for its coefficient is < 0.01. This technique is very efficient computationally, but may not be as effective as permutation or negation in terms of selecting signal over noise variables. See \href{https://link.springer.com/chapter/10.1007/978-3-642-23783-6_29}{Menze, 2011} for more details on this technique.
}

\section{Examples}{
\subsection{ANOVA importance}{

The default variable importance technique, ANOVA, is calculated while
you fit an ORSF ensemble.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit <- orsf(pbc_orsf,
                    Surv(time, status) ~ . - id)

fit
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## ---------- Oblique random survival forest
## 
##      Linear combinations: Accelerated
##           N observations: 276
##                 N events: 111
##                  N trees: 500
##       N predictors total: 17
##    N predictors per node: 5
##  Average leaves per tree: 24
## Min observations in leaf: 5
##       Min events in leaf: 1
##           OOB stat value: 0.84
##            OOB stat type: Harrell's C-statistic
##      Variable importance: anova
## 
## -----------------------------------------
}\if{html}{\out{</div>}}

ANOVA is the default because it is fast, but it may not be as decisive
as the permutation and negation techniques for variable selection.
}

\subsection{Raw VI values}{

the ‘raw’ variable importance values are stored in the fit object

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit$importance
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##     edema_1   ascites_1        bili      copper         age 
##  0.39603233  0.35528942  0.27837977  0.19605331  0.18822292 
##     albumin     protime        chol   edema_0.5       stage 
##  0.17033964  0.15320911  0.14883599  0.14599194  0.13627743 
##   spiders_1         ast    hepato_1       sex_f        trig 
##  0.13495783  0.12754159  0.12057626  0.10669014  0.09723320 
##    alk.phos    platelet trt_placebo 
##  0.09183673  0.07599581  0.06846999
}\if{html}{\out{</div>}}

these are ‘raw’ because values for factors have not been aggregated into
a single value. Currently there is one value for k-1 levels of a k level
factor. For example, you can see edema_1 and edema_0.5 in the importance
values above because edema is a factor variable with levels of 0, 0.5,
and 1.
}

\subsection{Collapse VI across factor levels}{

To get aggregated values across all levels of each factor, use
\code{orsf_vi()} with group_factors set to \code{TRUE} (the default)

\if{html}{\out{<div class="sourceCode r">}}\preformatted{orsf_vi(fit)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##    ascites       bili      edema     copper        age    albumin 
## 0.35528942 0.27837977 0.24719876 0.19605331 0.18822292 0.17033964 
##    protime       chol      stage    spiders        ast     hepato 
## 0.15320911 0.14883599 0.13627743 0.13495783 0.12754159 0.12057626 
##        sex       trig   alk.phos   platelet        trt 
## 0.10669014 0.09723320 0.09183673 0.07599581 0.06846999
}\if{html}{\out{</div>}}
}

\subsection{Add VI to an ORSF}{

You can fit an ORSF without VI, then add VI later

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_no_vi <- orsf(pbc_orsf,
                  Surv(time, status) ~ . - id,
                  importance = 'none')

# Note: you can't call orsf_vi_anova() on fit_no_vi because anova
# VI can only be computed while the forest is being grown.

orsf_vi_negate(fit_no_vi)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##          bili           age        copper       protime 
##  0.0164096687  0.0119295687  0.0105230256  0.0095853303 
##       ascites         edema           sex       spiders 
##  0.0053136070  0.0036428720  0.0029693686  0.0027609919 
##      alk.phos        hepato         stage       albumin 
##  0.0025005209  0.0022400500  0.0018753907  0.0009897895 
##           ast          trig           trt      platelet 
##  0.0005730360 -0.0013544488 -0.0017191081 -0.0031256512 
##          chol 
## -0.0040633465
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{orsf_vi_permute(fit_no_vi)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##          bili           age       protime        copper 
##  0.0152115024  0.0098458012  0.0057824547  0.0039070640 
##         stage       ascites        hepato       albumin 
##  0.0037507814  0.0033340279  0.0026047093  0.0024484268 
##         edema      alk.phos          trig          chol 
##  0.0020949305  0.0014586372  0.0011981663  0.0004688477 
##       spiders           trt           sex      platelet 
##  0.0002083767  0.0000000000 -0.0004167535 -0.0014586372
}\if{html}{\out{</div>}}
}

\subsection{ORSF and VI all at once}{

fit an ORSF and compute vi at the same time

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_permute_vi <- orsf(pbc_orsf,
                        Surv(time, status) ~ . - id,
                        importance = 'permute')

# get the vi instantly (i.e., it doesn't need to be computed again)
orsf_vi_permute(fit_permute_vi)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##          bili           age       albumin         stage 
##  0.0102625547  0.0082308814  0.0047926651  0.0044280058 
##       ascites        copper         edema           sex 
##  0.0034903105  0.0030214628  0.0021730718  0.0019795791 
##          chol           ast       protime      platelet 
##  0.0019274849  0.0013023547  0.0010939779  0.0004167535 
##       spiders           trt          trig      alk.phos 
##  0.0003125651 -0.0001041884 -0.0002083767 -0.0009897895 
##        hepato 
## -0.0015628256
}\if{html}{\out{</div>}}

You can still get negation VI from this fit, but it needs to be computed

\if{html}{\out{<div class="sourceCode r">}}\preformatted{orsf_vi_negate(fit_permute_vi)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##          bili           age       ascites       protime 
##  0.0152635966  0.0103146489  0.0056782663  0.0051052303 
##           sex         stage        copper         edema 
##  0.0044801000  0.0039591582  0.0031777454  0.0027361851 
##       spiders       albumin          chol           ast 
##  0.0008335070 -0.0003125651 -0.0005209419 -0.0006251302 
##        hepato          trig      alk.phos           trt 
## -0.0011981663 -0.0029172744 -0.0032819337 -0.0048447593 
##      platelet 
## -0.0054698896
}\if{html}{\out{</div>}}
}
}

\references{
Harrell FE, Califf RM, Pryor DB, Lee KL, Rosati RA. Evaluating the Yield of Medical Tests. \emph{JAMA} 1982; 247(18):2543-2546. DOI: 10.1001/jama.1982.03320430047030

Breiman L. Random forests. \emph{Machine learning} 2001 Oct; 45(1):5-32. DOI: 10.1023/A:1010933404324

Menze BH, Kelm BM, Splitthoff DN, Koethe U, Hamprecht FA. On oblique random forests. \emph{Joint European Conference on Machine Learning and Knowledge Discovery in Databases} 2011 Sep 4; pp. 453-469. DOI: 10.1007/978-3-642-23783-6_29

Jaeger BC, Welden S, Lenoir K, Speiser JL, Segar MW, Pandey A, Pajewski NM. Accelerated and interpretable oblique random survival forests. \emph{arXiv e-prints} 2022 Aug; arXiv-2208. URL: https://arxiv.org/abs/2208.01129
}
