#' Lists variables names and definitions used in a model
#' 
#' Returns a list containing the names and definitions of variables in apollo_probabilities, apollo_randCoeff and apollo_lcPars
#' 
#' It looks for variables in apollo_beta, apollo_randCoeff, draws, and 
#' apollo_probabilities. It returns them in a list ordered by origin.
#' 
#' @param apollo_probabilities Likelihood function of the whole model.
#' @param apollo_inputs List of arguments and settings generated by \link{apollo_validateInputs}.
#' 
#' @return A list of expressions containing all definitions in apollo_randCoeff and apollo_probabilities
#' @export
#' @importFrom utils lsf.str
apollo_varList <- function(apollo_probabilities, apollo_inputs){
  
  #### Useful functions ####
  is.val <- function(e) if(is.symbol(e) || is.numeric(e) || is.character(e) || is.logical(e) ) return(TRUE) else return(FALSE)
  #containsAssignment <- function(e){
  #  if(is.val(e)) return(FALSE)
  #  test <- length(e)==3 && is.symbol(e[[1]]) && (as.character(e[[1]]) %in% c('=', '<-'))
  #  if(test) return(TRUE)
  #  tmp <- rep(FALSE, length(e))
  #  for(i in 1:length(e)) if(!is.null(e[[i]])) tmp[i] <- containsAssignment(e[[i]])
  #  return( any(tmp) )
  #}
  # Returns the names of the indices (as a character vector) used inside "for" loops in "e", if any
  getForIndices <- function(e){
    if(is.function(e)) e <- body(e)
    ans <- c()
    # Case 1: Is a symbol or a value
    if(is.symbol(e) || is.numeric(e) || is.character(e) || is.logical(e)) return(ans)
    # Case 2: is a 'for' loop
    test1 <- is.call(e) || is.expression(e)
    test2 <- test1 && !is.null(e[[1]]) && as.character(e[[1]])=='for'
    if(test2) ans <- c(ans, as.character(e[[2]]), getForIndices(e[[4]]))
    # Case 3: is a call, but not a 'for' loop
    if(test1 && !test2) for(i in 1:length(e)) if(!is.null(e[[i]])) ans <- c(ans, getForIndices(e[[i]]))
    # Return
    return(ans)
  }
  # Identifies if an expression is a call to a list element
  isListElem <- function(e){
    test <- (is.call(e) || is.expression(e)) && length(e)>1 && is.symbol(e[[1]])
    test <- test && as.character(e[[1]]) %in% c('[[', '$')
    return(test)
  }
  # If argument is a function definition, returns only the body of the function. e.g. function(x) x^2 --> x^2
  getBody <- function(e){
    isFunDef <- (is.expression(e) || is.call(e)) && length(e)>=1
    isFunDef <- isFunDef && is.symbol(e[[1]]) && as.character(e[[1]])=='function'
    if(isFunDef) return(e[[3]]) else return(e)
  }
  # Read names and definitions of variables inside apollo_probabilities
  extractDef <- function(e, includeListName=TRUE){
    # If e is a function, extract its body
    if(is.function(e)) e <- body(e)
    # If it's a value, return immediately
    if(is.val(e)) return(NULL)
    # Check if it is an assignment
    isAssignment <- length(e)==3 && is.symbol(e[[1]]) && (e[[1]]=="=" || e[[1]]=="<-")
    lSingleVar   <- isAssignment && length(e[[2]])==1 && is.symbol(e[[2]])
    lListElem    <- isAssignment && length(e[[2]])==3 && is.symbol(e[[2]][[1]]) && (as.character(e[[2]][[1]]) %in% c('[[', '$'))
    rSingleVar   <- isAssignment && length(e[[3]])==1 && is.symbol(e[[3]])
    rExpression  <- isAssignment && (is.expression(e[[3]]) || is.call(e[[3]])) && length(e[[3]])>=1
    rList        <- rExpression  && is.symbol(e[[3]][[1]]) && as.character(e[[3]][[1]])=='list'
    rFunCall     <- rExpression  && is.symbol(e[[3]][[1]]) && (as.character(e[[3]][[1]]) %in% as.vector(lsf.str('package:apollo')))
    #  as.character(e[[3]][[1]]) %in% paste0('apollo_', c('avgIntraDraws', 'panelProd', 'avgInterDraws', 
    #                                                     'prepareProb', 'combineModels'))
    ans <- list()
    # Variable or expression into a single variable
    if(lSingleVar && !rList && !rFunCall){
      ans[[as.character(e[[2]])]] <- getBody(e[[3]])
      return(ans)
    }
    # List into a single variable: expand into one entry per element L$elem
    if(lSingleVar && rList && !rFunCall && length(e[[3]])>1){
      if(includeListName) tmp <- paste0(as.character(e[[2]]), '$') else tmp <- ''
      for(i in 2:length(e[[3]])) ans[[ paste0(tmp, names(e[[3]])[i]) ]] <- getBody(e[[3]][[i]])
      return(ans)
    }
    # Variable or expression into list element
    if(lListElem && !rList && !rFunCall && is.val(e[[2]][[3]])){
      if(includeListName){
        test <- isListElem(e[[2]][[2]]) # upper level list is itself a list element
        if( test) tmp <- paste0(paste0(as.character(e[[2]][[2]])[-1], collapse='$'), '$')
        if(!test) tmp <- paste0(as.character(e[[2]][[2]]), '$') 
      } else tmp <- ''
      ans[[paste0(tmp, as.character(e[[2]][[3]]))]] <- getBody(e[[3]])
      return(ans)
    }
    # List into list element
    if(lListElem && rList && !rFunCall && is.val(e[[2]][[3]])){
      tmp <- paste0(as.character(e[[2]][[3]]), '$')
      #if(includeListName) tmp <- paste0(as.character(e[[2]][[2]]), '$', tmp)
      if(includeListName){
        test <- isListElem(e[[2]][[2]]) # upper level list is itself a list element
        if( test) tmp <- paste0(paste0(as.character(e[[2]][[2]])[-1], collapse='$'), '$', tmp)
        if(!test) tmp <- paste0(as.character(e[[2]][[2]]), '$', tmp) 
      }
      if(!is.null(names(e[[3]]))) tmp2 <- names(e[[3]]) else tmp2 <- 0:(length(e[[3]])-1)
      if(length(e[[3]])>=2) for(i in 2:length(e[[3]])) ans[[ paste0(tmp, tmp2[i]) ]] <- getBody(e[[3]][[i]])
      return(ans)
    }
    # Check if it is a return with a list inside defined in place. e.g: return(list(a=1, b=2))
    isReturn <- !isAssignment && !is.symbol(e) && length(e)==2 && is.symbol(e[[1]]) && as.character(e[[1]])=='return'
    retList  <- isReturn && is.call(e[[2]]) && length(e[[2]])>1 && is.symbol(e[[2]][[1]]) && as.character(e[[2]][[1]])=='list'
    defInPlace <- retList && length(e[[2]])>1
    if(defInPlace){
      for(i in 2:length(e[[2]])) ans[[ names(e[[2]])[i] ]] <- getBody(e[[2]][[i]])
      return(ans)
    }
    # If it's an expression or call but not an assignment
    if(!isAssignment && !isReturn && (is.expression(e) || is.call(e))){
      for(i in 1:length(e)) if(!is.null(e[[i]])) ans <- c(ans, extractDef(e[[i]], includeListName))
      return(ans)
    }
    # In any other case
    return(NULL)
  }
  replaceByDef <- function(e, defs){
    # Case 1: x
    test1 <- is.symbol(e) && (as.character(e) %in% names(defs))
    if(test1) e <- defs[[as.character(e)]]
    # Case 2: L$x or L[['x']] or L[["x"]]
    test2 <- !test1 && is.call(e) && length(e)==3 && is.symbol(e[[1]]) && (as.character(e[[1]]) %in% c('$','[['))
    test2 <- test2 && is.val(e[[3]]) && is.symbol(e[[2]])
    if(test2) tmp  <- paste0(as.character(e[[2]]), '$', as.character(e[[3]]))
    test2 <- test2 && (tmp %in% names(defs))
    if(test2) e <- defs[[tmp]]
    # Case 3: expression
    if( !test1 && !test2 && (is.call(e) || is.expression(e)) ){
      for(i in 1:length(e)) if(!is.null(e[[i]])) e[[i]] <- replaceByDef(e[[i]], defs)
    } 
    # Return
    return(e)
  }
  
  # Load definitions inside apollo_randCoeff, apollo_lcPars and apollo_probabilities, except "for" loop indices
  defs <- list()
  if(apollo_inputs$apollo_control$mixing && is.function(apollo_inputs$apollo_randCoeff)){
    defs <- c(defs, extractDef(apollo_inputs$apollo_randCoeff, includeListName=TRUE) ) # necessary for recursive LVs
    defs <- c(defs, extractDef(apollo_inputs$apollo_randCoeff, includeListName=FALSE))
    tmp  <- which( names(defs) %in% getForIndices(apollo_inputs$apollo_randCoeff) )
    if(any(tmp)) defs <- defs[-tmp]
  }
  if(is.function(apollo_inputs$apollo_lcPars)){
    defs2 <- extractDef(apollo_inputs$apollo_lcPars, includeListName=TRUE) # necessary for recursive defs
    defs2 <- c(defs2, extractDef(apollo_inputs$apollo_lcPars, includeListName=FALSE))
    tmp   <- which( names(defs2) %in% getForIndices(apollo_inputs$apollo_lcPars) )
    if(any(tmp)) defs2 <- defs2[-tmp]
    defs  <- c(defs, defs2)
    rm(defs2)
  }
  defs <- c(defs, extractDef(apollo_probabilities))
  tmp   <- which( names(defs) %in% getForIndices(apollo_probabilities) )
  if(any(tmp)) defs <- defs[-tmp]
  
  # Replace recursive defs
  if(length(defs)>1){
    toDrop <- c()
    for(i in 2:length(defs)){
      defs[[i]] <- replaceByDef(defs[[i]], defs[1:(i-1)])
      tmp <- which(names(defs)[1:(i-1)] == names(defs)[i])
      if(length(tmp)>0){
        names(defs)[tmp] <- gsub('/', '', tempfile())
        toDrop <- c(toDrop, tmp)
      }
    }
    if(length(toDrop)>0) defs <- defs[-toDrop]
  }
  
  return(defs)
}