% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quickSPC.R
\name{quickSPC}
\alias{quickSPC}
\title{Quickly Assemble a SoilProfileCollection}
\usage{
quickSPC(
  x,
  id = "id",
  d = "depths",
  n = "name",
  m = "soil_color",
  interval = 10
)
}
\arguments{
\item{x}{either a \code{list} or \code{character} vector, see Details and Examples}

\item{id}{character, specified when \code{x} is a \code{list}, name of ID list element}

\item{d}{character, specified when \code{x} is a \code{list}, name of depths list element}

\item{n}{character, specified when \code{x} is a \code{list}, name of horizon name list element}

\item{m}{character, specified when \code{x} is a \code{list}, name of list element containing Munsell color notation}

\item{interval, }{numeric, typically an integer and only specified when using character templates in mode 2. See Details.}
}
\value{
\code{SoilProfileCollection} object
}
\description{
Quickly assemble a single-profile, \code{SoilProfileCollection} object from two possible templates. This function is a useful shortcut for creating theoretical \code{SoilProfileCollection} objects for testing or demonstrative purposes.
}
\details{
The list template for a single SPC allows for full specification of ID, horizon designation, bottom depths, and an arbitrary number of horizon-level attributes. A compact notation is used for profile ID (single value) and horizon depths (bottom depths, assuming datum of 0). Horizon designation and additional data (e.g. clay content) are specified as vectors all of equal length, matching the number of horizons in the profile.

The character template can be provided in one of several formats:
\enumerate{
\item 'A-Bt1-Bt2-Bt3-Cr-R'
\item 'ApAp|AA|E|BhsBhs|Bw1Bw1|CCCCC'
}

Format 1 is interpreted as a horizon sequence delimited by '-' or newline character (\\n). Random integer thickness are assigned to horizons, and profile ID created via \code{digest::digest(..., algo = 'xxhash32')}. Iteration over templates in this format is automatic when \code{x} is a character vector of \code{length > 1}.

Format 2 is interpreted as a horizon sequence delimited by '|'. Horizon thickness is proportional to replication of horizon designation and scaled by the \code{interval} argument. Profile ID is created via \code{digest::digest(..., algo = 'xxhash32')}. Iteration over templates in this format is automatic when \code{x} is a character vector of \code{length > 1}.

Explicit naming of profile IDs can be accomplished by specifying an ID via prefix, as in "ID:A-Bt1-Bt2-Cr-R" or "ID:ApAp|AA|E|BhsBhs|Bw1Bw1|CCCCC". Labels specified before a ":" will be interpreted as a profile ID. These labels are optional but if specified must be unique within \code{x}.

Single-horizon profile templates must include a trailing horizon delimiter: '-', '\\n', or '|' depending on the format.
}
\examples{

# list-based template
x <- list(
id = 'P1',
depths = c(25, 33, 100, 150),
name = c('A', 'Bw', 'Bt', 'Cr'),
clay = c(12, 15, 22, 25),
soil_color = c('10YR 3/3', '10YR 4/4', '10YR 4/6', '5G 6/2')
)

s <- quickSPC(x)
plotSPC(s, name.style = 'center-center', cex.names = 1)

# character template, mode 1
# horizon thickness is generated at random (uniform [5,20])
x <- 'A-Bt1-Bt2-Bt3-Cr-R'

s <- quickSPC(x)
plotSPC(s, name.style = 'center-center', cex.names = 1)


# multiple templates
x <- c(
'A-Bt1-Bt2-Bt3-Cr-R', 
'A-C1-C2-C3-C4-Ab', 
'Ap-A-A/E-E-Bhs-Cr'
)

# this interface is vectorized 
s <- quickSPC(x)
plotSPC(s, name.style = 'center-center', cex.names = 1)


# optionally specify profile IDs using "ID:" prefix
x <- c(
'P1:A-Bt1-Bt2-Bt3-Cr-R',
'P2:A-C1-C2-C3-C4-Ab',
'P3:Ap-A-A/E-E-Bhs-Cr'
)

s <- quickSPC(x)
plotSPC(s, name.style = 'center-center', cex.names = 1)


# optionally specify:
# horizon bottom depths in cm
# soil color in Munsell notation
x <- c(
'1. simple:Oe-A-E-Bhs',
'2. full:Oe,10,10YR 2/2-A,20,10YR 3/3-E,30,2.5Y 8/2-Bhs,60,7.5YR 4/6'
)

s <- quickSPC(x)
plotSPC(s, name.style = 'center-center', cex.names = 1)

# use newline character as delimiter, more compact
x <- 'Oe,10,10YR 2/2
A,20,10YR 3/3
E,30,2.5Y 8/2
Bhs,60,7.5YR 4/6
BC,125,7.5YR 6/4
C,150,10YR 6/2'

plotSPC(quickSPC(x), name.style = 'center-center', cex.names = 1)


# character template, mode 2
# horizon thickness is proportional to replication of 
# horizon designation and scaled by 'interval' argument
# default of 10 depth units
# e.g. A horizon is 3 * 10 = 30 depth units thick.
x <- c(
  'AAA|BwBwBwBw|CCCCCCC|CdCdCdCd',
  'ApAp|AA|E|BhsBhs|Bw1Bw1|CCCCC',
  'A|Bt1Bt1Bt1|Bt2Bt2Bt2|Bt3|Cr|RRRRR'
  )

# each horizon label is '10' depth-units (default)
s <- quickSPC(x)
plotSPC(s, name.style = 'center-center', 
        cex.names = 1, depth.axis = FALSE, 
        hz.depths = TRUE
)

# each horizon label is '5' depth-units
s <- quickSPC(x, interval = 5)
plotSPC(s, name.style = 'center-center', 
        cex.names = 1, depth.axis = FALSE, 
        hz.depths = TRUE
)

# optionally specify some / all profile IDs with "ID:" prefix
x <- c(
  'P1:AAA|BwBwBwBw|CCCCCCC|CdCdCdCd',
  'P2:ApAp|AA|E|BhsBhs|Bw1Bw1|CCCCC',
  'A|Bt1Bt1Bt1|Bt2Bt2Bt2|Bt3|Cr|RRRRR'
  )

s <- quickSPC(x)
plotSPC(s, name.style = 'center-center', 
        cex.names = 1, depth.axis = FALSE, 
        hz.depths = TRUE
)


# make a NODATA profile, with a random hash ID
#  note the use of trailing horizon delimiter
#  note the use of NA soil color field
x <- 'NODATA,150,NA-'
s <- quickSPC(x)
plotSPC(s, name.style = 'center-center', 
        cex.names = 1, depth.axis = FALSE, 
        hz.depths = TRUE)


}
