context("K-Partitions")

test_that("Integer partitions in k parts - npartitions", {
    expect_equal(npartitions(20, 10), 42)
    expect_equal(npartitions(20, 21), 0)
    expect_equal(sum(sapply(1:20, function(k) npartitions(20, k))), 627)
    expect_equal(npartitions(100, 10), 2977866)
    expect_error(npartitions(200, 50), "integer overflow")
    expect_equal(npartitions(20, 10, bigz = TRUE), 42)
    expect_equal(npartitions(100, 10, bigz = TRUE), 2977866)
    expect_equal(npartitions(200, 50, bigz = TRUE), gmp::as.bigz("39403290219"))
    expect_equal(npartitions(20, 0), 0)
    expect_equal(npartitions(0, 0), 1)
    expect_error(npartitions(20, -1), "expect integer")
    expect_error(npartitions(20, 1.5), "expect integer")
})

test_that("Integer k-partitions - ascending partitions", {
    part <- partitions(20, 10)
    expect_equal(nrow(part), 42)
    expect_equal(ncol(part), 10)
    expect_equal(part[1, ], c(rep(1, 9), 11))
    expect_equal(part[2, ], c(rep(1, 8), 2, 10))
    expect_equal(part[42, ], rep(2, 10))
    expect_true(all(apply(part, 1, sum) == 20))

    part <- partitions(20, 10, layout = "row")
    expect_equal(nrow(part), 42)
    expect_equal(ncol(part), 10)
    expect_equal(part[1, ], c(rep(1, 9), 11))
    expect_equal(part[2, ], c(rep(1, 8), 2, 10))
    expect_equal(part[42, ], rep(2, 10))
    expect_true(all(apply(part, 1, sum) == 20))

    part <- partitions(20, 10, layout = "column")
    expect_equal(nrow(part), 10)
    expect_equal(ncol(part), 42)
    expect_equal(part[, 1], c(rep(1, 9), 11))
    expect_equal(part[, 2], c(rep(1, 8), 2, 10))
    expect_equal(part[, 42], rep(2, 10))
    expect_true(all(apply(part, 2, sum) == 20))

    part <- partitions(20, 10, layout = "list")
    expect_equal(length(part), 42)
    expect_equal(part[[1]], c(rep(1, 9), 11))
    expect_equal(part[[2]], c(rep(1, 8), 2, 10))
    expect_equal(part[[42]], rep(2, 10))
    expect_true(all(sapply(part, sum) == 20))

    expect_error(partitions(200, 50), "too many results")
    expect_equal(dim(partitions(20, 21)), c(0, 21))
    expect_equal(dim(partitions(20, 21, layout = "column")), c(21, 0))
    expect_equal(length(partitions(20, 21, layout = "list")), 0)
    expect_error(partitions(20, -1), "expect integer")
    expect_error(partitions(20, 1.5), "expect integer")
})

test_that("Integer k-partitions - descending partitions", {
    part <- partitions(20, 10, descending = TRUE)
    expect_equal(nrow(part), 42)
    expect_equal(ncol(part), 10)
    expect_equal(part[1, ], c(11, rep(1, 9)))
    expect_equal(part[2, ], c(10, 2, rep(1, 8)))
    expect_equal(part[42, ], rep(2, 10))
    expect_true(all(apply(part, 1, sum) == 20))

    part <- partitions(20, 10, descending = TRUE, layout = "row")
    expect_equal(nrow(part), 42)
    expect_equal(ncol(part), 10)
    expect_equal(part[1, ], c(11, rep(1, 9)))
    expect_equal(part[2, ], c(10, 2, rep(1, 8)))
    expect_equal(part[42, ], rep(2, 10))
    expect_true(all(apply(part, 1, sum) == 20))

    part <- partitions(20, 10, descending = TRUE, layout = "column")
    expect_equal(nrow(part), 10)
    expect_equal(ncol(part), 42)
    expect_equal(part[, 1], c(11, rep(1, 9)))
    expect_equal(part[, 2], c(10, 2, rep(1, 8)))
    expect_equal(part[, 42], rep(2, 10))
    expect_true(all(apply(part, 2, sum) == 20))

    part <- partitions(20, 10, descending = TRUE, layout = "list")
    expect_equal(length(part), 42)
    expect_equal(part[[1]], c(11, rep(1, 9)))
    expect_equal(part[[2]], c(10, 2, rep(1, 8)))
    expect_equal(part[[42]], rep(2, 10))
    expect_true(all(sapply(part, sum) == 20))

    expect_error(partitions(200, 50, descending = TRUE), "too many results")
    expect_equal(dim(partitions(20, 21, descending = TRUE)), c(0, 21))
    expect_equal(dim(partitions(20, 21, descending = TRUE, layout = "column")), c(21, 0))
    expect_equal(length(partitions(20, 21, descending = TRUE, layout = "list")), 0)
    expect_error(partitions(20, -1, descending = TRUE), "expect integer")
    expect_error(partitions(20, 1.5, descending = TRUE), "expect integer")
})

test_that("Integer k-partitions - ascending ipartitions", {
    part <- partitions(20, 10)
    ipart <- ipartitions(20, 10)
    expect_is(ipart, "Partitions")
    expect_equal(ipart$collect(), part)
    expect_equal(ipart$getnext(), part[1, ])
    expect_equal(ipart$getnext(), part[2, ])
    expect_equal(ipart$getnext(2), part[3:4, ])
    ipart$getnext(30)
    expect_equal(nrow(ipart$getnext(30)), 8)
    expect_equal(ipart$getnext(), NULL)

    part <- partitions(20, 10, layout = "row")
    ipart$reset()
    expect_equal(ipart$collect(layout = "row"), part)
    expect_equal(ipart$getnext(layout = "row"), part[1, , drop = FALSE])
    expect_equal(ipart$getnext(layout = "row"), part[2, , drop = FALSE])
    expect_equal(ipart$getnext(2, layout = "row"), part[3:4, ])
    ipart$getnext(30, layout = "row")
    expect_equal(nrow(ipart$getnext(30, layout = "row")), 8)
    expect_equal(ipart$getnext(layout = "row"), NULL)

    part <- partitions(20, 10, layout = "column")
    ipart$reset()
    expect_equal(ipart$collect(layout = "column"), part)
    expect_equal(ipart$getnext(layout = "column"), part[, 1, drop = FALSE])
    expect_equal(ipart$getnext(layout = "column"), part[, 2, drop = FALSE])
    expect_equal(ipart$getnext(2, layout = "column"), part[, 3:4])
    ipart$getnext(30, layout = "column")
    expect_equal(ncol(ipart$getnext(30, layout = "column")), 8)
    expect_equal(ipart$getnext(layout = "column"), NULL)

    part <- partitions(20, 10, layout = "list")
    ipart$reset()
    expect_equal(ipart$collect(layout = "list"), part)
    expect_equal(ipart$getnext(layout = "list"), part[1])
    expect_equal(ipart$getnext(layout = "list"), part[2])
    expect_equal(ipart$getnext(2, layout = "list"), part[3:4])
    ipart$getnext(30, layout = "list")
    expect_equal(length(ipart$getnext(30, layout = "list")), 8)
    expect_equal(ipart$getnext(layout = "list"), NULL)

    ipart <- ipartitions(20, 21)
    expect_equal(dim(ipart$collect()), c(0, 21))
    expect_equal(ipart$getnext(), NULL)
    expect_error(ipartitions(20, -1), "expect integer")
    expect_error(ipartitions(20, 1.5), "expect integer")
})

test_that("Integer partitions - descending ipartitions", {
    part <- partitions(20, 10, descending = TRUE)
    ipart <- ipartitions(20, 10, descending = TRUE)
    expect_is(ipart, "Partitions")
    expect_equal(ipart$collect(), part)
    expect_equal(ipart$getnext(), part[1, ])
    expect_equal(ipart$getnext(), part[2, ])
    expect_equal(ipart$getnext(2), part[3:4, ])
    ipart$getnext(30)
    expect_equal(nrow(ipart$getnext(30)), 8)
    expect_equal(ipart$getnext(30), NULL)

    part <- partitions(20, 10, descending = TRUE, layout = "row")
    ipart$reset()
    expect_equal(ipart$collect(layout = "row"), part)
    expect_equal(ipart$getnext(layout = "row"), part[1, , drop = FALSE])
    expect_equal(ipart$getnext(layout = "row"), part[2, , drop = FALSE])
    expect_equal(ipart$getnext(2, layout = "row"), part[3:4, ])
    ipart$getnext(30, layout = "row")
    expect_equal(nrow(ipart$getnext(30, layout = "row")), 8)
    expect_equal(ipart$getnext(layout = "row"), NULL)

    part <- partitions(20, 10, descending = TRUE, layout = "column")
    ipart$reset()
    expect_equal(ipart$collect(layout = "column"), part)
    expect_equal(ipart$getnext(layout = "column"), part[, 1, drop = FALSE])
    expect_equal(ipart$getnext(layout = "column"), part[, 2, drop = FALSE])
    expect_equal(ipart$getnext(2, layout = "column"), part[, 3:4])
    ipart$getnext(30, layout = "column")
    expect_equal(ncol(ipart$getnext(30, layout = "column")), 8)
    expect_equal(ipart$getnext(layout = "column"), NULL)

    part <- partitions(20, 10, descending = TRUE, layout = "list")
    ipart$reset()
    expect_equal(ipart$collect(layout = "list"), part)
    expect_equal(ipart$getnext(layout = "list"), part[1])
    expect_equal(ipart$getnext(layout = "list"), part[2])
    expect_equal(ipart$getnext(2, layout = "list"), part[3:4])
    ipart$getnext(30, layout = "list")
    expect_equal(length(ipart$getnext(30, layout = "list")), 8)
    expect_equal(ipart$getnext(layout = "list"), NULL)

    ipart <- ipartitions(20, 21, descending = TRUE)
    expect_equal(dim(ipart$collect()), c(0, 21))
    expect_equal(ipart$getnext(), NULL)
    expect_error(ipartitions(20, -1, descending = TRUE), "expect integer")
    expect_error(ipartitions(20, 1.5, descending = TRUE), "expect integer")
})
