##*******************************************************
## Function ruleInduction
##
## This is a poor man's implementation which mines all rules and
## then filters the rules which do not stam from the itemsets

setMethod("ruleInduction",  signature(x = "itemsets"),
    function(x, transactions, confidence = 0.8, control = NULL) {

        ## control args are: method, reduce and verbose
        method  <- if(is.null(control$m))   "ptree" else control$m
        reduce  <- if(is.null(control$r))   FALSE else control$r
        verbose <- if(is.null(control$v))   FALSE else control$v

        methods <- c("apriori", "ptree", "tidlists")
    
        method <-  methods[pmatch(method , methods)]
        if(is.na(method)) error("unknown method")


        pt1 <- proc.time()
        ## check data
        nItems <- nitems(transactions)
        if (nItems != nitems(items(x)))
        stop("Dimensions of x and transactions do not match!")

        if(reduce && method != "ptree") {
            ifreq <- itemFrequency(items(x), type = "abs") 
            items.involved <- which(ifreq > 0)

            if (verbose) cat("reducing data from", nitems(items(x)), 
                "items to", length(items.involved) ,"items\n")
            x.sub <- x
            x.sub@items <- x@items[, items.involved]
            transactions.sub <- transactions[, items.involved]    

        }else{
            x.sub <- x
            transactions.sub <- transactions
        }
        
        pt2 <- proc.time()
        if(verbose) cat("preprocessing done [", pt2[1]-pt1[1], "s].\n", 
            sep = "")
        
        if(verbose) cat("using method:", method, "\n")

        ## find rules
        rules <- 
        if(method == "ptree") ruleInduction.ptree(x, transactions, 
            confidence, reduce, verbose)
        else if(method == "tidlists") ruleInduction.tidlists(x.sub, 
            transactions.sub, confidence, verbose)
        else ruleInduction.apriori(x.sub, transactions.sub, 
            confidence, verbose)
        
        pt3 <-  proc.time()
        if(verbose) cat("searching done [", pt3[1]-pt2[1], "s].\n", sep = "")


        if(reduce && method != "ptree"){
            ## expand items back to full space
            ## -1 since indices in ngCMatix start with 0 
            items.index <- items.involved - as.integer(1)

            ## fix dim
            rules@lhs@data@Dim[1] <- nItems
            rules@rhs@data@Dim[1] <- nItems

            ## fix column indices
            ## +1 since indices in ngCMatix start with 0
            rules@lhs@data@i <- items.index[(rules@lhs@data@i + as.integer(1))]
            rules@rhs@data@i <- items.index[(rules@rhs@data@i + as.integer(1))]

            ## fix item labels
            rules@lhs@itemInfo <- itemInfo(transactions)
            rules@rhs@itemInfo <- itemInfo(transactions)
        }
        
        pt4 <-  proc.time()
        if(verbose) cat("postprocessing done [", pt4[1]-pt3[1], "s].\n", 
            sep = "")

        ## return found rules
        rules 

    })


ruleInduction.apriori <- function(x, transactions, confidence = 0.8,
        verbose = FALSE) {

        ## itemset sizes
        isetsSize <-  size(x)

        ## find minimal support and mine all rules
        ## Note: minSupport is reduced by epsilon so we get the rules
        ##	with support == min support in x
        minSupport <-  min(quality(x)$support) - 1/length(transactions)

        rules <- apriori(transactions,  parameter = list(
                support = minSupport,
                confidence = confidence, 
                target = "rules", 
                minlen = min(isetsSize), 
                maxlen = max(isetsSize)),
            control = list(verbose = verbose))

        ## find rules which were generated by the itemsets
        if (verbose) cat(paste("starting to filter", length(rules), "rules.\n"))
        take <- !is.na(match(items(rules), items(x)))
        if (verbose) cat("filtering done.\n")

        rules <- rules[take]
        if (verbose) cat("left with", length(rules), "rules.\n")

        ## return found rules
        rules 
    }

ruleInduction.tidlists <- function(x, transactions, confidence = 0.8, verbose = FALSE) {
    tid <- as(transactions, "tidLists")
    data <- .Call("R_tid_rules" ,tid@data, x@items@data)
    names(data) <- c("support", "confidence",
        "lhs_i", "lhs_p", "rhs_i", "rhs_p", "Dim")

    quality <- data.frame(support = data$support, confidence = data$confidence)

    lhs <- new("ngCMatrix", i = data$lhs_i, p = data$lhs_p, Dim = data$Dim)
    rhs <- new("ngCMatrix", i = data$rhs_i, p = data$rhs_p, Dim = data$Dim)

    lhs <- new("itemMatrix", data = lhs, itemInfo = x@items@itemInfo)
    rhs <- new("itemMatrix", data = rhs, itemInfo = x@items@itemInfo)

    rules <- new("rules", lhs = lhs, rhs = rhs, quality = quality)
    
    rules <- rules[quality(rules)$confidence > confidence]
    rules
}

## this framework is inefficient as we do the support counting twice

ruleInduction.ptree <- 
function(x, transactions, confidence = 0.8, reduce = FALSE, verbose = FALSE) {
    r <- .Call("R_pncount", x@items@data, transactions@data, FALSE, reduce, verbose)
    names(r) <- c("data.lhs","data.rhs","support","confidence","lift")
    new("rules",
        lhs     = new("itemMatrix", data     = r$data.lhs, 
                                    itemInfo = transactions@itemInfo),
        rhs     = new("itemMatrix", data     = r$data.rhs, 
                                    itemInfo = transactions@itemInfo),
        quality = data.frame(r[3:5]))[r$confidence > confidence]
}

