\name{species_table}
\alias{species_table}
\title{Species occurrence matrices for AusPlots
}
\description{This function takes a data frame of individual raw point-intercept hits from AusPlots generated using the \code{\link{get_ausplots}} function, and generates species occurrence matrices as desired based on presence/absence, cover, frequency or IVI.
}
\usage{
species_table(veg.PI, m_kind=c("PA", "percent_cover", "freq", "IVI"),
	cover_type=c("PFC", "OCC"), species_name=c("SN","HD","GS"), 
	strip_bryophytes=FALSE)
}
\arguments{
  \item{veg.PI}{The input raw point-intercept data generated by \code{\link{get_ausplots}}
}
  \item{m_kind}{The desired species scoring method: PA: binary (presence/absence), percent cover, frequency (based on occurrences on the 10 point-intercept transects within plots) or IVI (combination of cover and frequency - see Details).
}
  \item{cover_type}{A choice between PFC: 'projected foliage cover' and OCC: 'opaque canopy cover' and only applies to percent_cover and IVI. If 'PFC' is selected (the default), hits scored as 'in canopy sky' are removed, whereas they are retained as cover for that species for 'OCC'. 'PFC' may be more suitable for some applications such as comparison to remotely sensed cover products.
 }
 \item{species_name}{Set to "SN" to use "standardised_names" (the default), "HD" to use "herbarium_determination", and "GS" to use "genus_species" See details for an explanation of each setting.
}
 \item{strip_bryophytes}{Logical. Switch to \code{TRUE} to exclude bryophyte records and only return vascular plants in Angiosperm (flowering plants), Gymnosperm (conifers) and Pteridophyte (ferns and allies) groups. Although there are a number of bryophyte records in Ausplots, they are not surveyed systematically.
}
}

\details{Species occurrence matrices are the basis for a wide range of downstream analyses in community ecology, such as classification, ordination, alpha and beta diversity, abundance distributions and many more. Many of these analyses can be performed in the vegan R package using the returned matrix.

IVI (Importance Value Index; Ellenberg & Mueller-Dombois 1974) is calculated as the sum of relative abundance (percent cover) and relative frequency (percent frequency of occurrence on the 10 transects used to gather the point-intercept data). It has a theoretical range of 0 - 200 and is calculated as follows:

\deqn{(cover / total cover) * 100 + (frequency / total frequency) * 100}

\code{species_name} provides multiple options for calculating plant cover, presence or frequency. If \code{species_name="HD"}, the function will use the "herbarium_determination" provided by state herbaria to identify unique species. "herbarium_determination" are made to the lowest possible taxonomic rank, which may be the family, genus, species, subspecies or variety. If \code{species_name="SN"}, the default, the function will use the the "standardised_name" to identify unique species. Values are based on the \code{veg.PI$herbarium_determination} but have been matched to the accepted scientific name according to the Australian Plant Census (APC: https://www.anbg.gov.au/cpbr/program/hc/hc-APC.html) and the Australian Plant Name Index (APNI: https://www.anbg.gov.au/cpbr/program/hc/hc-APNI.html). The standardised name field therefore helps to ensure taxonomic consistency. It excludes any records that don't match the APC. If \code{species_name="GS"}, only the standardised genus species rank (i.e., "genus_species" in the veg.PI input table) is used to identify species. \code{species_name="GS"} maximises consistency in species comparisons between plots, but reduces the taxonomic detail.

The returned occurrence matrix (species against sites) can be converted to 'long' format (with rows as species records) using:

\code{my.matrix <- species_table(my.veg.PI, m_kind="PA", cover_type="PFC")}
\cr
\code{library(reshape2)}
\cr
\code{my.matrix <- as.data.frame(melt(as.matrix(my.matrix)))}
\cr
\code{my.matrix <- my.matrix[-which(my.matrix $value == 0),]}
\cr
\cr

The \code{\link{round}} function can be used on the returned matrix to estimate percent covers to the nearest percent if desired (i.e., with digits=0)
}


\value{Returns a data frame with species as columns and sites as rows. Values are determined by argument \code{m_kind}.
}

\author{Greg R. Guerin, Samantha E.M. Munroe
}

\references{
Ellenberg, D. & Mueller-Dombois, D. (1974) Aims and methods of vegetation ecology. Wiley, New York, NY.
}

\seealso{
\code{\link{get_ausplots}}
}
\examples{
\dontrun{
#get veg.PI data for selected plots:
my.data <- get_ausplots(my.Plot_IDs=c("SATFLB0004", "QDAMGD0022", "NTASTU0002"),
	veg.vouchers=FALSE, site_info=FALSE)

#generate a percent cover occurrence matrix using "genus_species" species identification:
species_table(my.data$veg.PI, m_kind="percent_cover", cover_type="PFC", 
species_name="GS")
}
}
