% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stsm_estimate.R
\name{stsm_estimate}
\alias{stsm_estimate}
\title{Trend cycle seasonal decomposition using the Kalman filter.}
\usage{
stsm_estimate(
  y,
  exo_obs = NULL,
  exo_state = NULL,
  state_eqns = NULL,
  freq = NULL,
  decomp = NULL,
  trend = NULL,
  unconstrained = FALSE,
  saturating_growth = FALSE,
  multiplicative = NULL,
  par = NULL,
  seasons = NULL,
  cycle = NULL,
  arma = c(p = NA, q = NA),
  interpolate = NA,
  interpolate_method = NA,
  det_obs = FALSE,
  det_trend = NULL,
  det_seas = FALSE,
  det_drift = FALSE,
  det_cycle = FALSE,
  sig_level = NULL,
  sig_level_seas = NULL,
  sig_level_cycle = NULL,
  sig_level_trend = NULL,
  optim_methods = c("BFGS", "NM", "CG", "SANN"),
  maxit = 10000,
  verbose = FALSE,
  cores = NULL
)
}
\arguments{
\item{y}{Univariate time series of data values. May also be a 2 column data frame containing a date column.}

\item{exo_obs}{Matrix of exogenous variables to be used in the observation equation.}

\item{exo_state}{Matrix of exogenous variables to be used in the state matrix.}

\item{state_eqns}{Character vector of equations to apply exo_state to the unobserved components. If left as the default, then all variables in
exo_state will be applied to all the unobserved components. The equations should look like:
"trend ~ var - 1", "drift ~ var - 1", "cycle ~ var - 1", "seasonal ~ var - 1".
If only some equations are specified, it will be assumed that the exogenous data will be applied to only those specified equations.}

\item{freq}{Frequency of the data (1 (yearly), 4 (quarterly), 12 (monthly), 365.25/7 (weekly), 365.25 (daily)), default is NULL and will be automatically detected}

\item{decomp}{Decomposition model ("tend-cycle-seasonal", "trend-seasonal", "trend-cycle", "trend-noise")}

\item{trend}{Trend specification ("random-walk", "random-walk-drift", "double-random-walk", "random-walk2"). The default is NULL which will choose the best of all specifications based on the maximum likelihood.
"random-walk" is the random walk trend.
"random-walk-drift" is the random walk with constant drift trend.
"double-random-walk" is the random walk with random walk drift trend.
"random-walk2" is a 2nd order random walk trend as in the Hodrick-Prescott filter.
If trend is "random-walk", the trend model is T_t = T_{t-1} + e_t, 
e_t ~ N(0, sig_t^2)
If trend is "random-walk-drift", the trend model is T_t = T_{t-1} + D_{t-1} + e_t, 
e_t ~ N(0, sig_t^2) with
D_t = d + phi_d*D_{t-1} + n_t, n_t ~ N(0, sig_d^2)
If trend is "double-random-walk", the trend model is T_t = M_{t-1} + T_{t-1} + e_t, 
e_t ~ N(0, sig_t^2) with
M_t = M_{t-1} + n_t, n_t ~ N(0, sig_d^2)
If trend is "random-walk2", the trend model is T_t = 2T_{t-1} - T_{t-2} + e_t, 
e_t ~ N(0, sig_t^2)}

\item{unconstrained}{Logical whether to remove inequality constraints on the trend during estimation}

\item{saturating_growth}{Force the growth rate to converge to 0 in the long term}

\item{multiplicative}{If data should be logged to create a multiplicative model.
If multiplicative = TRUE, then the data is logged and the original model becomes multiplicative 
(Y_t = T_t * C_t * S_t * BX_t * e_t)}

\item{par}{Initial parameters, default is NULL and will auto-select them}

\item{seasons}{The seasonal periods: i.e. c(365.25, 7 if yearly and weekly seasonality). Default is NULL and will be estimated via wavelet analysis. 
Can set to FALSE if want no seasonality}

\item{cycle, }{The period for the longer-term cycle. Default is NULL and will be estimated via wavelet analysis.
Can set to FALSE if want no cycle, "trig" for trigonometric specification only, or "arma" for ARMA(p,q) specification only.}

\item{arma}{Named vector with values for p and q corresponding to the ARMA(p,q) specification if
cycle is set to 'arma'. If NA, then will auto-select the order.}

\item{interpolate}{Character string giving frequency to interpolate to: i.e. "quarterly", "monthly", "weekly", "daily"}

\item{interpolate_method}{Character string giving the interpolation method: 
i.e. "eop" for end of period, "avg" for period average, or "sum" for period sum.}

\item{det_obs}{Set the observation equation error variance to 0 (deterministic observation equation)
If det_obs = TRUE then the error variance of the observation equation (sig_e) is set to 0}

\item{det_trend}{Set the trend error variance to 0 (deterministic trend)
If det_trend = TRUE then the error variance of the trend equation (sig_t) is set to 0 and 
is referred to as a smooth trend}

\item{det_seas}{Set the seasonality error variances to 0 (deterministic seasonality)
If det_seas = TRUE then the error variance all seasonality frequency j equations (sig_s) 
are set to 0 and is referred to as deterministic seasonality}

\item{det_drift}{Set the drift error variance to 0 (deterministic drift)
If det_drift = TRUE then the error variance of the drift equation (sig_d) is set to 0 and 
is refereed to as a deterministic drift}

\item{det_cycle}{Set the cycle error variance to 0 (deterministic cycle)
If det_cycle = TRUE then the error variance of the cycle equation (sig_c) is set to 0 and 
is referred to as a deterministic cycle}

\item{sig_level}{Significance level to determine statistically significance for all tests. Default is 0.01}

\item{sig_level_seas}{Significance level to determine statistically significant seasonal frequencies. Default is 0.01}

\item{sig_level_cycle}{Significance level to determine a statistically significant cycle frequency. Default is 0.01}

\item{sig_level_trend}{Significance level to determine statistically significant order of integration. Default is 0.01}

\item{optim_methods}{Vector of 1 to 3 optimization methods in order of preference ("NR", "BFGS", "CG", "BHHH", or "SANN")}

\item{maxit}{Maximum number of iterations for the optimization}

\item{verbose}{Logical whether to print messages or not}

\item{cores}{Number of cores to use for seasonality and cycle detection}
}
\value{
List of estimation values including a data table with coefficients, convergence code, frequency, decomposition, seasonality, cyclicality, and trend specification
as well as the a data table with the original data with dates. Any exogenous data given is also returned.
}
\description{
Estimates a structural time series model using the Kalman filter and maximum likelihood.
The seasonal and cycle components are assumed to be of a trigonometric form.
The function checks three trend specifications to decompose a univariate time series
into trend, cycle, and/or seasonal components plus noise. The function automatically
detects the frequency and checks for a seasonal and cycle component if the user does not specify
the frequency or decomposition model. This can be turned off by setting freq or specifying decomp.
State space model for decomposition follows
Yt = T_t + C_t + S_t + B*X_t + e_t, e_t ~ N(0, sig_e^2)
Y is the data
T is the trend component
C is the cycle component
S is the seasonal component
X is the exogenous data with parameter vector B
e is the observation error
}
\examples{
\dontrun{
#GDP Not seasonally adjusted
library(autostsm)
data("NA000334Q", package = "autostsm") #From FRED
NA000334Q = data.table(NA000334Q, keep.rownames = TRUE)
colnames(NA000334Q) = c("date", "y")
NA000334Q[, "date" := as.Date(date)]
NA000334Q[, "y" := as.numeric(y)]
NA000334Q = NA000334Q[date >= "1990-01-01", ]
stsm = stsm_estimate(NA000334Q)
}
}
