% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spectrum_blur_ratio.R
\name{spectrum_blur_ratio}
\alias{spectrum_blur_ratio}
\title{Measure blur ratio in the frequency domain}
\usage{
spectrum_blur_ratio(
  X,
  cores = getOption("mc.cores", 1),
  pb = getOption("pb", TRUE),
  spec.smooth = getOption("spec.smooth", 5),
  spectra = FALSE,
  res = 150,
  hop.size = getOption("hop.size", 11.6),
  wl = getOption("wl", NULL),
  ovlp = getOption("ovlp", 70),
  path = getOption("sound.files.path", "."),
  n.bins = 100
)
}
\arguments{
\item{X}{The output of \code{\link{set_reference_sounds}} which is an object of class 'data.frame', 'selection_table' or 'extended_selection_table' (the last 2 classes are created by the function \code{\link[warbleR]{selection_table}} from the warbleR package) with the reference to the test sounds . Must contain the following columns: 1) "sound.files": name of the .wav files, 2) "selec": unique selection identifier (within a sound file), 3) "start": start time and 4) "end": end time of selections, 5) "bottom.freq": low frequency for bandpass, 6) "top.freq": high frequency for bandpass, 7) "sound.id": ID of sounds used to identify counterparts across distances and 8) "reference": identity of sounds to be used as reference for each test sound (row). See \code{\link{set_reference_sounds}} for more details on the structure of 'X'.}

\item{cores}{Numeric vector of length 1. Controls whether parallel computing is applied by specifying the number of cores to be used. Default is 1 (i.e. no parallel computing).}

\item{pb}{Logical argument to control if progress bar is shown. Default is \code{TRUE}.}

\item{spec.smooth}{Numeric vector of length 1 determining the length of the sliding window used for a sum smooth for power spectrum calculation (in kHz). Default is 5.}

\item{spectra}{Logical to control if power spectra are returned (as attributes). Default is \code{FALSE}.}

\item{res}{Numeric argument of length 1. Controls image resolution. Default is 150 (faster) although 300 - 400 is recommended for publication/presentation quality.}

\item{hop.size}{A numeric vector of length 1 specifying the time window duration (in ms). Default is 11.6 ms, which is equivalent to 512 wl for a 44.1 kHz sampling rate. Ignored if 'wl' is supplied.}

\item{wl}{A numeric vector of length 1 specifying the window length of the spectrogram, default
is NULL. If supplied, 'hop.size' is ignored. Applied to both spectra and spectrograms on image files.}

\item{ovlp}{Numeric vector of length 1 specifying the percentage of overlap between two
consecutive windows, as in \code{\link[seewave]{spectro}}. Default is 70. Applied to both spectra and spectrograms on image files.}

\item{path}{Character string containing the directory path where the sound files are found. Only needed when 'X' is not an extended selection table. If not supplied the current working directory is used.}

\item{n.bins}{Numeric vector of length 1 specifying the number of frequency bins to use for representing power spectra. Default is 100. If null the raw power spectrum is used (note that this can result in high RAM memory usage for large data sets). Power spectrum values are interpolated using \code{\link[stats]{approx}}.}
}
\value{
Object 'X' with an additional column, 'spectrum.blur.ratio', containing the computed spectrum blur ratio values. If \code{spectra = TRUE} the output would include power spectra for all sounds as attributes ('attributes(X)$spectra').
}
\description{
\code{spectrum_blur_ratio} measures blur ratio of frequency spectra from sounds referenced in an extended selection table.
}
\details{
Spectral blur ratio measures the degradation of sound as a function of the change in sound power in the frequency domain, analogous to the blur ratio proposed by Dabelsteen et al (1993) for the time domain (and implemented in \code{\link{blur_ratio}}). Low values indicate low degradation of sounds. The function measures the blur ratio of spectra from sounds in which a reference playback has been re-recorded at different distances. Spectral blur ratio is measured as the mismatch between power spectra (expressed as probability density functions) of the reference sound and the re-recorded sound. The function compares each sound type to the corresponding reference sound. The 'sound.id' column must be used to tell the function to only compare sounds belonging to the same category (e.g. song-types). Two methods for setting the experimental design are provided. All wave objects in the extended selection table must have the same sampling rate so the length of spectra is comparable. The function uses \code{\link[seewave]{spec}} internally to compute power spectra. NA is returned if at least one the power spectra cannot be computed.
}
\examples{
{
  # load example data
  data("test_sounds_est")

# add reference to X
X <- set_reference_sounds(X = test_sounds_est)

  # get spetrum blur ratio
  spectrum_blur_ratio(X = X)

  # using method 2
   X <- set_reference_sounds(X = test_sounds_est, method = 2)
  spectrum_blur_ratio(X = X)

  # get power spectra
  sbr <- spectrum_blur_ratio(X = X, spectra = TRUE)
  spctr <- attributes(sbr)$spectra

  # make distance a factor for plotting
  spctr$distance <- as.factor(spctr$distance)

  
  # plot
  rlang::check_installed("ggplot2")
  library(ggplot2)
  
  ggplot(spctr[spctr$freq > 0.3, ], aes(y = amp, x = freq,
  col = distance)) +
  geom_line() +
  facet_wrap(~sound.id) +
  scale_color_viridis_d(alpha = 0.7) +
  labs(x = "Frequency (kHz)", y = "Amplitude (PMF)") +
  coord_flip() +
  theme_classic()
}

}
\references{
{
Dabelsteen, T., Larsen, O. N., & Pedersen, S. B. (1993). Habitat-induced degradation of sound signals: Quantifying the effects of communication sounds and bird location on blur ratio, excess attenuation, and signal-to-noise ratio in blackbird song. The Journal of the Acoustical Society of America, 93(4), 2206.

Araya-Salas M., E. Grabarczyk, M. Quiroz-Oliva, A. Garcia-Rodriguez, A. Rico-Guevara. (2023), baRulho: an R package to quantify degradation in animal acoustic signals .bioRxiv 2023.11.22.568305.
}
}
\seealso{
\code{\link{blur_ratio}}

Other quantify degradation: 
\code{\link{blur_ratio}()},
\code{\link{detection_distance}()},
\code{\link{envelope_correlation}()},
\code{\link{plot_blur_ratio}()},
\code{\link{plot_degradation}()},
\code{\link{set_reference_sounds}()},
\code{\link{signal_to_noise_ratio}()},
\code{\link{spcc}()},
\code{\link{spectrum_correlation}()},
\code{\link{tail_to_signal_ratio}()}
}
\author{
Marcelo Araya-Salas (\email{marcelo.araya@ucr.ac.cr})
}
\concept{quantify degradation}
