basictabler
================

[![Build Status](https://travis-ci.org/cbailiss/basictabler.svg?branch=master)](https://travis-ci.org/cbailiss/basictabler) [![CRAN\_Status\_Badge](http://www.r-pkg.org/badges/version/basictabler)](https://cran.r-project.org/package=basictabler)

The `basictabler` package enables rich tables to be created and rendered/exported with just a few lines of R.

The `basictabler` package:

-   Provides an easy way of creating basic tables, especially from data frames and matrices.
-   Provides flexibility so that the structure/content of the table can be easily built/modified.
-   Provides formatting options to simplify rendering/exporting data.
-   Provides styling options so the tables can be themed/branded as needed.

The tables are rendered as htmlwidgets or plain text. The HTML/text can be exported for use outside of R.

The tables can also be exported to Excel, including the styling/formatting. The formatting/styling is specified once and can then be used when rendering to both HTML and Excel - i.e. it is not necessary to specify the formatting/styling separately for each output format.

`basictabler` is a companion package to the `pivottabler` package. `pivottabler` is focussed on generating pivot tables and can aggregate data. `basictabler` does not aggregate data but offers more control of table structure.

### Installation

You can install:

-   the latest released version from CRAN with

``` r
install.packages("basictabler")
```

-   the latest development version from github with

``` r
devtools::install_github("cbailiss/basictabler", build_vignettes = TRUE)
```

### Example

#### Trivial Example

Creating a tiny HTML table from a data frame and immediately rendering it as a htmlwidget:

``` r
library(basictabler)
qhtbl(data.frame(a=1:2, b=3:4))
```

#### Another Example

Creating a table from a data frame, specifying column names and value formats:

``` r
# aggregate the sample data to make a small data frame
library(basictabler)
library(dplyr)
tocsummary <- bhmsummary %>%
  group_by(TOC) %>%
  summarise(OnTimeArrivals=sum(OnTimeArrivals),
            OnTimeDepartures=sum(OnTimeDepartures),
            TotalTrains=sum(TrainCount)) %>%
  ungroup() %>%
  mutate(OnTimeArrivalPercent=OnTimeArrivals/TotalTrains*100,
         OnTimeDeparturePercent=OnTimeDepartures/TotalTrains*100) %>%
  arrange(TOC)

# To specify formatting, a list is created which contains one element for each column in 
# the data frame, i.e. tocsummary contains six columns so the columnFormats list has six elements.
# The values in the first column in the data frame won't be formatted since NULL has been specified.
# The values in the 2nd, 3rd and 4th columns will be formatted using format(value, big.mark=",")
# The values in the 5th and 6th columns will be formatted using sprintf(value, "%.1f")
columnFormats=list(NULL, list(big.mark=","), list(big.mark=","), list(big.mark=","), "%.1f", "%.1f")

# render the table directly as a html widget
qhtbl(tocsummary, firstColumnAsRowHeaders=TRUE,
            explicitColumnHeaders=c("TOC", "On-Time Arrivals", "On-Time Departures",
                                    "Total Trains", "On-Time Arrival %", "On-Time Departure %"),
            columnFormats=columnFormats)
```

![<http://cbailiss.me.uk/basictablerreadmeimgs/example1.png>](http://cbailiss.me.uk/basictablerreadmeimgs/example1.png)

#### Excel Output

The same styling/formatting used for the HTML output is also used when outputting to Excel - greatly reducing the amount of script that needs to be written to create Excel output. The only additional formatting that typically needs applying is the Excel cell format strings.

``` r
# aggregate the sample data to make a small data frame
library(basictabler)
library(dplyr)
tocsummary <- bhmsummary %>%
  group_by(TOC) %>%
  summarise(OnTimeArrivals=sum(OnTimeArrivals),
            OnTimeDepartures=sum(OnTimeDepartures),
            TotalTrains=sum(TrainCount)) %>%
  ungroup() %>%
  mutate(OnTimeArrivalPercent=OnTimeArrivals/TotalTrains*100,
         OnTimeDeparturePercent=OnTimeDepartures/TotalTrains*100) %>%
  arrange(TOC)

columnFormats=list(NULL, list(big.mark=","), list(big.mark=","), list(big.mark=","), "%.1f", "%.1f")

# create the table
tbl <- qtbl(tocsummary, firstColumnAsRowHeaders=TRUE,
            explicitColumnHeaders=c("TOC", "On-Time Arrivals", "On-Time Departures",
                                    "Total Trains", "On-Time Arrival %", "On-Time Departure %"),
            columnFormats=columnFormats)

# style setting function
setStyle <- function(cell, baseStyleName, declarations) {
  if(is.null(cell$style)) 
    cell$style <- tbl$createInlineStyle(baseStyleName=baseStyleName, declarations=declarations)
  else cell$style$setPropertyValues(declarations=declarations)
}
# set the number formatting on the count cells
cells <- tbl$findCells(rowNumbers=2:5, columnNumbers=2:4)
invisible(lapply(cells, setStyle, baseStyleName="Cell", declarations=list("xl-value-format"="#,##0")))
# set the number formatting on the percentage cells
cells <- tbl$findCells(rowNumbers=2:5, columnNumbers=5:6)
invisible(lapply(cells, setStyle, baseStyleName="Cell", declarations=list("xl-value-format"="##0.0")))

# render the table to an Excel workbook
library(openxlsx)
wb <- createWorkbook(creator = Sys.getenv("USERNAME"))
addWorksheet(wb, "Data")
tbl$writeToExcelWorksheet(wb=wb, wsName="Data", 
                          topRowNumber=2, leftMostColumnNumber=2, applyStyles=TRUE)
saveWorkbook(wb, file="C:\\test.xlsx", overwrite = TRUE)
```

![<http://cbailiss.me.uk/basictablerreadmeimgs/example2.png>](http://cbailiss.me.uk/basictablerreadmeimgs/example2.png)

In the screenshot above, Gridlines have been made invisible to make the styling easier to see (by clearing the checkbox on the 'View' ribbon). Columns were also auto-sized - though the widths of columns could also be manually specified from R. See the Excel Export vignette for more details.

### More Information

Tables can be further manipulated once created, including adding/removing cells/rows/columns and specifying styling and formatting.

It is also possible to create tables row-by-row, column-by-column and/or cell-by-cell.

See the package vignettes for more information:

``` r
# to see a list of available package vignettes:
vignette(package="basictabler")
# to open a specific vignette
vignette(topic="v01-introduction", package="basictabler")
```

The vignettes can also be read on CRAN at: <https://cran.r-project.org/package=basictabler>
