% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/beam.R
\name{beam_profile_overlap}
\alias{beam_profile_overlap}
\title{Calculate overlap between a vertical profile ('vp') of biological scatterers
and the vertical radiation profile emitted by the radar}
\usage{
beam_profile_overlap(
  vp,
  elev,
  distance,
  antenna,
  zlim = c(0, 4000),
  noise_floor = -Inf,
  noise_floor_ref_range = 1,
  steps = 500,
  quantity = "dens",
  normalize = T,
  beam_angle = 1,
  k = 4/3,
  lat,
  re = 6378,
  rp = 6357
)
}
\arguments{
\item{vp}{a vertical profile of class vp}

\item{elev}{numeric vector. Beam elevation(s) in degrees.}

\item{distance}{the distance(s) from the radar along sea level (down range)
for which to calculate the overlap in m.}

\item{antenna}{radar antenna height. If missing taken from \code{vp}}

\item{zlim}{altitude range in meter, given as a numeric vector of length two.}

\item{noise_floor}{The system noise floor in dBZ. The total system noise
expressed as the reflectivity factor it would represent at a distance
\code{noise_floor_ref_range} from the radar. NOT YET IMPLEMENTED}

\item{noise_floor_ref_range}{the reference distance from the radar at which
\code{noise_floor} is expressed. NOT YET IMPLEMENTED}

\item{steps}{number of integration steps over altitude range zlim, defining
altitude grid size used for numeric integrations}

\item{quantity}{profile quantity to use for the altitude distribution, one of
'dens' or 'eta'.}

\item{normalize}{Whether to normalize the radiation coverage pattern over the
altitude range specified by zlim}

\item{beam_angle}{numeric. Beam opening angle in degrees, typically the
angle between the half-power (-3 dB) points of the main lobe}

\item{k}{Standard refraction coefficient.}

\item{lat}{radar latitude. If missing taken from \code{vp}}

\item{re}{Earth equatorial radius in km.}

\item{rp}{Earth polar radius in km.}
}
\value{
A data.frame with columns distance and overlap.
}
\description{
Calculates the distribution overlap between a vertical profile ('vp') and the
vertical radiation profile of a set of emitted radar beams at various
elevation angles as given by \link{beam_profile}.
}
\details{
This function also calculates the \code{overlap} quantity in the output of
\link{integrate_to_ppi}.

Overlap is calculated as the \href{https://en.wikipedia.org/wiki/Bhattacharyya_distance}{Bhattacharyya coefficient} (i.e.
distribution overlap) between the (normalized) vertical profile vp and the
(normalized) radiation coverage pattern as calculated by
\link{beam_profile}.

The current implementation does not (yet) take into account the system
noise floor when calculating the overlap.

In the ODIM data model the attribute \code{/how/NEZ} or \code{/how/NEZH}
specifies the system noise floor (the Noise Equivalent Z or noise
equivalent reflectivity factor. the H refers to the horizontal channel of a
dual-polarization radar). In addition, the attribute \code{/how/LOG} gives
"security distance above mean noise level (dB) threshold value". This is
equivalent to the log receiver signal-to-noise ratio, i.e. the dB above the
noise floor for the signal processor to report a valid reflectivity value.
We recommend using \code{NEZH}+\code{LOG} for \code{noise_floor}, as this
is the effective noise floor of the system below which no data will be
reported by the radar signal processor.

Typical values are \code{NEZH} = -45 to -50 dBZ at 1 km from the radar.
\code{LOG} is typically around 1 dB.

Need to evaluate beam by beam the returned signal relative to a uniform
beam filling of at least NEZH + LOG If returned signal is lower, the gate
is below noise level.
}
\examples{
# locate example volume file:
pvolfile <- system.file("extdata", "volume.h5", package = "bioRad")

# load the example polar volume file:
pvol <- read_pvolfile(pvolfile)

# let us use this example vertical profile:
data(example_vp)
example_vp

# calculate overlap between vertical profile of birds
# and the vertical radiation profile emitted by the radar:
bpo <- beam_profile_overlap(example_vp, get_elevation_angles(pvol), seq(0, 100000, 1000))

# plot the calculated overlap:
plot(bpo)
}
