#' Download a basemap for `map(ppi)`
#'
#' Downloads a basemap for [map.ppi()] from Stamen Maps or Google Maps using
#' [ggmap::get_map()]. To use Google Maps as `source`, you will have to register
#' with Google, enable billing and provide an API key to ggmap. See the [ggmap
#' README](https://github.com/dkahle/ggmap#attention) for details.
#'
#' @param x A `ppi` object.
#' @param verbose Logical. When `TRUE`, prints information to console.
#' @param zoom Integer. Optional zoom level from 3 (continent) to 21 (building),
#'   see [ggmap::get_map()]. When undefined, the zoom level will be the one
#'   matching the `ppi` extent.
#' @param alpha Numeric. Transparency of the basemap, value between 0 and 1.
#' @param source Character. Map service to be used: `stamen` or `google`.
#' @param maptype Character. Type of basemap to plot. For Stamen Maps:
#'   `terrain`, `terrain-background`, `terrain-labels`, `terrain-lines`,
#'   `toner`, `toner-2010`, `toner-2011`, `toner-background`, `toner-hybrid`,
#'   `toner-labels`, `toner-lines`, `toner-lite` or `watercolor`. For Google
#'   Maps: `terrain`, `satellite`, `roadmap` or `hybrid`.
#' @param ... Arguments to pass to [ggmap::get_map()].
#'
#' @export
#' @returns A ggmap object for `map(ppi)`
#' @seealso
#' * [map.ppi()]
#'
#' @examples
#' # Project a scan as a ppi
#' ppi <- project_as_ppi(example_scan)
#' \donttest{
#' # Create a basemap that matches the extent of the ppi
#' basemap <- download_basemap(ppi)
#'
#' # Map the radial velocity of the ppi onto the basemap
#' map(ppi, map = basemap, param = "VRADH")
#'
#' # Increase the transparency of the basemap
#' basemap <- download_basemap(ppi, alpha = 0.3)
#' map(ppi, map = basemap, param = "VRADH")
#'
#' # Download a different type of basemap, e.g. a gray-scale image.
#' # See get_map() in ggmap library for full documentation of the options.
#' basemap <- download_basemap(ppi, maptype = "toner-lite")
#' map(ppi, map = basemap, param = "VRADH")
#' }
download_basemap <- function(x, verbose = TRUE, zoom, alpha = 1,
                             source = "stamen", maptype = "terrain", ...) {
  stopifnot(inherits(x, "ppi"))
  rlang::check_installed("ggmap",'to run `download_basemap`', version = '3.0.0')
  if(utils::packageVersion("ggmap") < numeric_version("3.0.0.903")){
    # not throw a true warning to pass CRAN checks
    message("Warning message:\n ggmap not up-to-date (version < 3.0.0.903), upgrade is required using devtools::install_github(\"dkahle/ggmap\")")
  }

  if (is.na(raster::crs(x$data))) {
    stop("Not a projected ppi, download_basemap() expects a ppi generated by project_as_ppi() with argument project=TRUE")
  }

  if (source != "google") {
    location <- c(left = x$geo$bbox["lon", "min"], bottom = x$geo$bbox["lat", "min"], right = x$geo$bbox["lon", "max"], top = x$geo$bbox["lat", "max"])
  } else {
    location <- c(lon = mean(x$geo$bbox["lon", ]), lat = mean(x$geo$bbox["lat", ]))
  }

  if (!missing(zoom)) {
    if (!is.numeric(zoom)) {
      stop("zoom should be a numeric integer")
    }
  }
  # check size of ppi and determine zoom
  if (missing(zoom)) {
    use_zoom <- ggmap::calc_zoom(x$geo$bbox["lon", ], x$geo$bbox["lat", ])
  } else {
    use_zoom <- zoom
  }

  if (verbose) {
    message(paste("Downloading zoom =", use_zoom, "..."))
  }
  map <- ggmap::get_map(
    location = location,
    zoom = use_zoom,
    source = source,
    maptype = maptype,
    ...
  )
  bboxmap <- attributes(map)$bb

  if ((x$geo$bbox["lon", "max"] - x$geo$bbox["lon", "min"] >
    bboxmap$ur.lon - bboxmap$ll.lon) ||
    (x$geo$bbox["lat", "max"] - x$geo$bbox["lat", "min"] >
      bboxmap$ur.lat - bboxmap$ll.lat)) {
    if (missing(zoom)) {
      if (verbose) {
        message(paste("Map too small, downloading zoom =", use_zoom - 1, "..."))
      }
      map <- ggmap::get_map(
        location = location,
        zoom = use_zoom - 1,
        source = source,
        maptype = maptype,
        ...
      )
      bboxmap <- attributes(map)$bb
      if ((x$geo$bbox["lon", "max"] - x$geo$bbox["lon", "min"] >
        bboxmap$ur.lon - bboxmap$ll.lon) ||
        (x$geo$bbox["lat", "max"] - x$geo$bbox["lat", "min"] >
          bboxmap$ur.lat - bboxmap$ll.lat)) {
        if (verbose) {
          message(paste("Map still too small, downloading zoom =", use_zoom - 2, "..."))
        }
        map <- ggmap::get_map(
          location = location,
          zoom = use_zoom - 2,
          source = source,
          maptype = maptype,
          ...
        )
      }
    } else {
      warning("Map is smaller than ppi bounding box.")
    }
  }
  attributes(map)$geo <- x$geo
  attributes(map)$ppi <- TRUE
  # add transparency
  add_color_transparency(map, alpha = alpha)
}
