% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/composite_ppi.R
\name{composite_ppi}
\alias{composite_ppi}
\title{Create a composite of multiple plan position indicators (\code{ppi})}
\usage{
composite_ppi(
  x,
  param = "all",
  nx = 100,
  ny = 100,
  xlim,
  ylim,
  res,
  crs,
  raster = NA,
  method = "max",
  idp = 2,
  idw_max_distance = NA,
  coverage = FALSE
)
}
\arguments{
\item{x}{A list of \code{ppi} objects.}

\item{param}{Character (vector). One or more parameter name(s) to composite.
To composite all available scan parameters use \code{all} (default).}

\item{nx}{number of raster pixels in the x (longitude) dimension}

\item{ny}{number of raster pixels in the y (latitude) dimension}

\item{xlim}{x (longitude) range}

\item{ylim}{y (latitude) range}

\item{res}{numeric vector of length 1 or 2 to set the resolution of the raster (see \link[raster:resolution]{res}).
If this argument is used, arguments \code{nx} and \code{ny} are ignored. Unit is identical to \code{xlim} and \code{ylim}.}

\item{crs}{character or object of class CRS. PROJ.4 type description of a Coordinate Reference System (map projection).
When 'NA' (default), an azimuthal equidistant projection with origin at the radar location is used.
To use a WSG84 (lat,lon) projection, use crs="+proj=longlat +datum=WGS84"}

\item{raster}{(optional) RasterLayer with a CRS. When specified this raster topology is used for the output, and nx, ny, res
arguments are ignored.}

\item{method}{Character (vector). Compositing method(s), either \code{mean},
\code{min}, \code{max} or \code{idw}. To apply different methods for each of the
parameters, provide a vector with the same length as \code{param}.}

\item{idp}{Numeric. Inverse distance weighting power.}

\item{idw_max_distance}{Numeric. Maximum distance from the radar to consider
in inverse distance weighting. Measurements beyond this distance will have
a weighting factor of zero.}

\item{coverage}{Logical. When \code{TRUE}, adds an additional \code{coverage} parameter
to the \code{ppi} indicating the number of \code{ppi}s covering a single composite
pixel.}
}
\value{
A \code{ppi} object.
}
\description{
Combines multiple plan position indicators (\code{ppi}) into a single
\code{ppi}. Can be used to make a composite of \code{ppi}'s from multiple
radars.
}
\details{
The function can combine multiple \code{ppi}s of different scan elevations of the
same radar or \code{ppi}s of different radars. The coordinates of the returned
\code{ppi} object are in the WGS84 datum, unless a different \code{crs} is provided. If
only \code{res} is provided, but no \code{crs} is set, \code{res} is in meters and the
origin of the composite \code{ppi} is set to the mean(lat, lon) location.

The \code{method} parameter determines how values of different \code{ppi}s at the same
geographic location are combined:
\itemize{
\item \code{mean}: Compute the average value.
\item \code{max}: Compute the maximum value. If \code{ppi}s are of the same radar and the
same polar volume, this computes a max product, showing the maximum detected
signal at that geographic location.
\item \code{min}: Compute the minimum value.
\item \code{idw}: This option is useful primarily when compositing \code{ppi}s of multiple
radars. Performs an inverse distance weighting, where values are weighted
according to 1/(distance from the radar)^\code{idp}.
}

Argument \code{method} determines how values of different ppi's at the same
geographic location are combined.
\itemize{
\item \code{mean}: Compute the average value
\item \code{max}: Compute the maximum value. If ppi's are of the same radar and the
same polar volume, this computes a max product, showing the maximum
detected signal at that geographic location.
\item \code{min}: Compute the minimum value
\item \code{idw}: This option is useful primarily when compositing ppi's of multiple
radars. Performs an inverse distance weighting, where values are weighted
according to 1/(distance from the radar)^\code{idp}
}

The coordinates system of the returned \code{ppi} is a WGS84
(lat, lon) datum, unless a different \code{crs} is provided. If only
\code{res} is provided, but no \code{crs} is set, \code{res} is in
meter units and the origin of the composite \code{ppi} is set to the
mean (lat, lon) location.

This function is a prototype and under active development
}
\examples{
# Locate and read the polar volume example file
\donttest{
pvolfile <- system.file("extdata", "volume.h5", package = "bioRad")
pvol <- read_pvolfile(pvolfile)

# Calculate a ppi for each elevation scan, 1000m grid resolution
ppis <- lapply(pvol$scans, project_as_ppi, grid_size=1000)

# Overlay the ppis, calculating the maximum value observed
# across the available scans at each geographic location
composite <- composite_ppi(ppis, method = "max", res=1000)

# Download basemap
bm <- download_basemap(composite)

# Plot the calculated max product on the basemap
map(composite, bm)
}
}
