% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/des_info.R
\name{des_info}
\alias{des_info}
\title{Produce a graph of design layout, skeletal ANOVA table and data frame with complete design}
\usage{
des_info(
  design.obj,
  nrows,
  ncols,
  brows = NA,
  bcols = NA,
  byrow = TRUE,
  fac.names = NULL,
  fac.sep = c("", " "),
  buffer = NULL,
  plot = TRUE,
  rotation = 0,
  size = 4,
  margin = FALSE,
  save = FALSE,
  savename = paste0(design.obj$parameters$design, "_design"),
  plottype = "pdf",
  return.seed = TRUE,
  quiet = FALSE,
  ...
)
}
\arguments{
\item{design.obj}{An \code{agricolae} design object.}

\item{nrows}{The number of rows in the design.}

\item{ncols}{The number of columns in the design.}

\item{brows}{For RCBD only. The number of rows in a block.}

\item{bcols}{For RCBD only. The number of columns in a block.}

\item{byrow}{For split-plot only. Logical (default: \code{TRUE}). Provides a way to arrange plots within whole-plots when there are multiple possible arrangements.}

\item{fac.names}{Allows renaming of the \code{A} level of factorial designs (i.e. those using \code{\link[agricolae:design.ab]{agricolae::design.ab()}}) by passing (optionally named) vectors of new labels to be applied to the factors within a list. See examples and details for more information.}

\item{fac.sep}{The separator used by \code{fac.names}. Used to combine factorial design levels. If a vector of 2 levels is supplied, the first separates factor levels and label, and the second separates the different factors.}

\item{buffer}{The type of buffer. One of edge, row, column, double row, double column, or block (coming soon).}

\item{plot}{Logical (default \code{TRUE}). If \code{TRUE}, display a plot of the generated design. A plot can always be produced later using \code{\link[=autoplot]{autoplot()}}.}

\item{rotation}{Rotate the text output as Treatments within the plot. Allows for easier reading of long treatment labels. Takes positive and negative values being number of degrees of rotation from horizontal.}

\item{size}{Increase or decrease the text size within the plot for treatment labels. Numeric with default value of 4.}

\item{margin}{Logical (default FALSE). Setting to \code{TRUE} will add a margin (white space) between plot and axes.}

\item{save}{One of \code{FALSE} (default)/\code{"none"}, \code{TRUE}/\code{"both"}, \code{"plot"} or \code{"workbook"}. Specifies which output to save.}

\item{savename}{A filename for the design to be saved to. Default is the type of the design combined with "_design".}

\item{plottype}{The type of file to save the plot as. Usually one of \code{"pdf"}, \code{"png"}, or \code{"jpg"}. See \code{\link[ggplot2:ggsave]{ggplot2::ggsave()}} for all possible options.}

\item{return.seed}{Logical (default TRUE). Output the seed used in the design?}

\item{quiet}{Logical (default FALSE). Return the objects without printing output.}

\item{...}{Additional parameters passed to \code{\link[ggplot2:ggsave]{ggplot2::ggsave()}} for saving the plot.}
}
\value{
A list containing a data frame with the complete design, a ggplot object with plot layout, the seed (if \code{return.seed = TRUE}), and the \code{satab} object, allowing repeat output of the \code{satab} table via \code{cat(output$satab)}.
}
\description{
Produce a graph of design layout, skeletal ANOVA table and data frame with complete design
}
\details{
If \code{save = TRUE} (or \code{"both"}), both the plot and the workbook will be saved to the current working directory, with filename given by \code{savename}. If one of either \code{"plot"} or \code{"workbook"} is specified, only that output is saved. If \code{save = FALSE} (the default, or equivalently \code{"none"}), nothing will be output.

\code{fac.names} can be supplied to provide more intuitive names for factors and their levels in factorial designs. They should be specified in a list format, for example \code{fac.names = list(A_names = c("a", "b", "c"), B_names = c("x", "y", "z"))}. This will result a design output with a column named \code{A_names} with levels \verb{a, b, c} and another named \code{B_names} with levels \verb{x, y, z}. Only the first two elements of the list will be used.

If \code{fac.sep} is a single element (e.g. "\emph{"), this is used to separate all factor labels (e.g. A_1_B_1). If it is two elements (e.g. c("", "}")), the first element separates the factor names and their levels, and the second level separates the two factors (e.g. A1_B1).

\code{...} allows extra arguments to be passed to ggsave for output of the plot. The details of possible arguments can be found in  \code{\link[ggplot2:ggsave]{ggplot2::ggsave()}}.
}
\examples{
library(agricolae)

# Completely Randomised Design
trt <- c(1, 5, 10, 20)
rep <- 5
outdesign <- design.crd(trt = trt, r = rep, seed = 42)
des.out <- des_info(design.obj = outdesign, nrows = 4, ncols = 5)

# Randomised Complete Block Design
trt <- LETTERS[1:11]
rep <- 4
outdesign <- design.rcbd(trt = trt, r = rep, seed = 42)
des.out <- des_info(
  design.obj = outdesign, nrows = 11,
  ncols = 4, brows = 11, bcols = 1
)

# Latin Square Design
trt <- c("S1", "S2", "S3", "S4")
outdesign <- design.lsd(trt)
des.out <- des_info(design.obj = outdesign, nrows = 4, ncols = 4)

# Factorial Design (Crossed, Completely Randomised)
trt <- c(3, 2) # Factorial 3 x 2
rep <- 3
outdesign <- design.ab(trt, r = rep, design = "crd")
des.out <- des_info(design.obj = outdesign, nrows = 6, ncols = 3)

# Factorial Design (Crossed, Completely Randomised), renaming factors
trt <- c(3, 2) # Factorial 3 x 2
rep <- 3
outdesign <- design.ab(trt, r = rep, design = "crd")
des.out <- des_info(design.obj = outdesign, nrows = 6, ncols = 3,
                    fac.names = list(N = c(50, 100, 150),
                                     Water = c("Irrigated", "Rain-fed")))

# Factorial Design (Nested, Latin Square)
trt <- c("A1", "A2", "A3", "A4", "B1", "B2", "B3")
outdesign <- design.lsd(trt)
des.out <- des_info(design.obj = outdesign, nrows = 7, ncols = 7)

# Split plot design
trt1 <- c("A", "B")
trt2 <- 1:4
rep <- 4
outdesign <- design.split(trt1, trt2, r = rep)
des.out <- des_info(design.obj = outdesign, nrows = 8, ncols = 4, brows = 4, bcols = 2)
}
