% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/partition_metrics.R
\name{partition_metrics}
\alias{partition_metrics}
\title{Calculate metrics for one or several partitions}
\usage{
partition_metrics(
  cluster_object,
  dissimilarity = NULL,
  dissimilarity_index = NULL,
  net = NULL,
  site_col = 1,
  species_col = 2,
  eval_metric = c("pc_distance", "anosim", "avg_endemism", "tot_endemism")
)
}
\arguments{
\item{cluster_object}{a \code{bioregion.clusters} object}

\item{dissimilarity}{a \code{dist} object or a \code{bioregion.pairwise.metric} object (output
from \code{\link[=similarity_to_dissimilarity]{similarity_to_dissimilarity()}}). Necessary if \code{eval_metric}
includes \code{pc_distance} and \code{tree} is not a
\code{bioregion.hierar.tree} object}

\item{dissimilarity_index}{a character string indicating the dissimilarity
(beta-diversity) index to be used in case \code{dist} is a \code{data.frame} with
multiple dissimilarity indices}

\item{net}{the species-site network (i.e., bipartite network). Should be
provided if \code{eval_metric} includes \code{"avg_endemism"} or \code{"tot_endemism"}}

\item{site_col}{name or number for the column of site nodes (i.e. primary
nodes). Should be provided if \code{eval_metric} includes \code{"avg_endemism"} or
\code{"tot_endemism"}}

\item{species_col}{name or number for the column of species nodes (i.e.
feature nodes). Should be provided if \code{eval_metric} includes
\code{"avg_endemism"} or \code{"tot_endemism"}}

\item{eval_metric}{character string or vector of character strings indicating
metric(s) to be calculated to investigate the effect of different number
of clusters. Available options: \code{"pc_distance"}, \code{"anosim"},
\code{"avg_endemism"} and \code{"tot_endemism"}}
}
\value{
a \code{list} of class \code{bioregion.partition.metrics} with two to three elements:
\itemize{
\item{\code{args}: input arguments
}
\item{\code{evaluation_df}: the data.frame containing \code{eval_metric}
for all explored numbers of clusters
}
\item{\code{endemism_results}: if endemism calculations were requested, a list
with the endemism results for each partition
}
}
}
\description{
This function aims at calculating metrics for one or several partitions,
usually on outputs from \code{netclu_}, \code{hclu_} or \code{nhclu_} functions. Metrics
may require the users to provide either a similarity or dissimilarity
matrix, or to provide the initial species-site table.
}
\details{
\loadmathjax

\strong{Evaluation metrics:}
\itemize{
\item{\code{pc_distance}: this metric is the method used by
\insertCite{Holt2013}{bioregion}. It is a ratio of the between-cluster sum of
dissimilarity (beta-diversity) versus the total sum of dissimilarity
(beta-diversity) for the full dissimilarity matrix. In other words, it is
calculated on the basis of two elements. First, the total sum of
dissimilarity is calculated by summing the entire dissimilarity matrix
(\code{dist}). Second, the between-cluster sum of dissimilarity is calculated as
follows: for a given number of cluster, the dissimilarity is only summed
between clusters, not within clusters. To do that efficiently, all pairs of
sites within the same clusters have their dissimilarity set to zero in
the dissimilarity matrix, and then the dissimilarity matrix is summed. The
\code{pc_distance} ratio is obtained by dividing the between-cluster sum of
dissimilarity by the total sum of dissimilarity.}

\item{\code{anosim}: This metric is the statistic used in Analysis of
Similarities, as suggested in \insertCite{Castro-Insua2018}{bioregion} (see
\link[vegan:anosim]{vegan::anosim()}). It compares the between-cluster
dissimilarities to the within-cluster dissimilarities. It is based based on
the difference of mean ranks between groups and within groups with the
following formula:
\mjeqn{R = (r_B - r_W)/(N (N-1) / 4)}{R = (r_B - r_W)/(N (N-1) / 4)},
where \mjeqn{r_B}{r_B} and \mjeqn{r_W}{r_W} are the average ranks
between and within clusters respectively, and \mjeqn{N}{N} is the total
number of sites.
Note that the function does not estimate the significance here, it only
computes the statistic - for significance testing see
\link[vegan:anosim]{vegan::anosim()}}.

\item{\code{avg_endemism}: this metric is the average percentage of
endemism in clusters as
recommended by \insertCite{Kreft2010}{bioregion}. Calculated as follows:
\mjeqn{End_{mean} = \frac{\sum_{i=1}^K E_i / S_i}{K}}{Pc_endemism_mean = sum(Ei / Si) / K}
where \mjeqn{E_i}{Ei} is the number of endemic species in cluster i,
\mjeqn{S_i}{Si} is the number of
species in cluster i, and K the maximum number of clusters.
}

\item{\code{tot_endemism}: this metric is the total endemism across all clusters,
as recommended by \insertCite{Kreft2010}{bioregion}. Calculated as follows:
\mjeqn{End_{tot} = \frac{E}{C}}{Endemism_total = E/C}

where \mjeqn{E}{E} is total the number of endemics (i.e., species found in
only one cluster) and \mjeqn{C}{C} is the number of non-endemic species.
}
}
}
\examples{
comat <- matrix(sample(0:1000, size = 500, replace = TRUE, prob = 1/1:1001),
20, 25)
rownames(comat) <- paste0("Site",1:20)
colnames(comat) <- paste0("Species",1:25)

comnet <- mat_to_net(comat)

dissim <- dissimilarity(comat, metric = "all")

# User-defined number of clusters
tree1 <- hclu_hierarclust(dissim, n_clust = 2:20, index = "Simpson")
tree1

a <- partition_metrics(tree1, dissimilarity = dissim, net = comnet,
                       site_col = "Node1", species_col = "Node2",
                       eval_metric = c("tot_endemism", "avg_endemism",
                                      "pc_distance", "anosim"))
a

}
\references{
\insertRef{Castro-Insua2018}{bioregion}

\insertRef{Ficetola2017}{bioregion}

\insertRef{Holt2013}{bioregion}

\insertRef{Kreft2010}{bioregion}

\insertRef{Langfelder2008}{bioregion}
}
\seealso{
\link{compare_partitions}
}
\author{
Boris Leroy (\email{leroy.boris@gmail.com}),
Maxime Lenormand (\email{maxime.lenormand@inrae.fr}) and
Pierre Denelle (\email{pierre.denelle@gmail.com})
}
