% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coxlps.R
\name{coxlps}
\alias{coxlps}
\title{Fit a Cox proportional hazards regression model with Laplace-P-splines.}
\usage{
coxlps(formula, data, K = 30, penorder = 2, tmax = NULL)
}
\arguments{
\item{formula}{A formula object where the ~ operator separates the response
from the covariates. In a Cox model, it takes the form
\emph{response ~ covariates}, where \emph{response} is a survival object
returned by the \code{\link[survival]{Surv}} function of the \emph{survival}
package.}

\item{data}{Optional. A data frame to match the variable names provided in
\code{formula}.}

\item{K}{A positive integer specifying the number of cubic B-spline
functions in the basis. Default is \emph{K = 30} and allowed values
are \code{10 <= K <= 60}.}

\item{penorder}{The penalty order used on finite differences of the
coefficients of contiguous B-splines. Can be either 2 for a second-order
penalty (the default) or 3 for a third-order penalty.}

\item{tmax}{A user-specified value for the upper bound of the B-spline basis.
The default is NULL, so that the B-spline basis is specified in the interval
\emph{[0, tup]}, where \emph{tup} is the  upper bound of the follow-up
times. It is required that \emph{tmax} > \emph{tup}.}
}
\value{
An object of class \code{coxlps} containing various components from
  the fit. Details can be found in coxlps.object. Plot of
  estimated smooth hazard and survival curves can be obtained using
  \code{\link{plot.coxlps}}. If required, estimated baseline quantities
  on specific time values can be obtained with coxlps.baseline.
}
\description{
Fits a Cox proportional hazards regression model for right censored data by
combining Bayesian P-splines and Laplace approximations.
}
\details{
The log-baseline hazard is modeled as a linear combination of
  \code{K} cubic B-splines as obtained from \code{\link{cubicbs}}. The
  B-spline basis is specified in the interval \emph{[0, tup]}, where
  \emph{tup} is the upper bound of the follow-up times,
  i.e. the largest observed follow-up time. Following Jullion
  and Lambert (2007), a robust Gamma prior is imposed on the roughness
  penalty parameter. A grid-based approach is used to explore the posterior
  penalty space and the resulting quadrature points serve to compute the
  approximate (joint) marginal posterior of the latent field vector. Point
  and set estimates of latent field elements are obtained from a finite
  mixture of Gaussian densities. The routine centers the columns of the
  covariate matrix around their mean value for numerical stability.
}
\examples{

### Example 1 (Simulated survival data)

set.seed(3)

# Simulate survival data  with simsurvdata
betas <- c(0.13, 0.52, 0.30)
simul <- simsurvdata(a = 3.8, b = 2.2, n = 250, betas = betas , censperc = 20)
simul
simdat <- simul$survdata
plot(simul) # Plot survival data

# Estimation with coxlps
fit <- coxlps(Surv(time, delta) ~ x1 + x2 + x3, data = simdat, K = 15)
# Compare coxlps and coxph
fit
summary(coxph(Surv(time, delta) ~ x1 + x2 + x3, data = simdat))

# Fitted baseline survival vs target
plot(fit, h0 = FALSE, cred.int = 0.95, overlay.km = TRUE)
domt <- seq(0, 4, length = 100)
lines(domt, simul$S0(domt), type = "l", col = "red")
legend("topright", col=c("black", "blue", "red"), lty = rep(1,3),
      c("Bayesian LPS", "Kaplan-Meier", "Target"), cex = 0.8, bty = "n")

### Example 2 (Kidney transplant data)

data(kidneytran)
Surv.obj <- Surv(kidneytran$time, kidneytran$delta)
fit <- coxlps(Surv.obj ~ age + gender + race, data = kidneytran)
coxphfit <- coxph(Surv.obj ~ age + gender + race, data = kidneytran)
## Compare coxph and coxlps results
summary(coxphfit)
fit
## Plot Kaplan-Meier curve vs Laplace-P-spline fit
plot(fit, h0 = FALSE, overlay.km = TRUE, plot.cred = FALSE)

### Example 3 (Laryngeal cancer data)

data(laryngeal)
fit <- coxlps(Surv(time, delta) ~ age + diagyr + as.factor(stage),
               data = laryngeal)
coxphfit <- coxph(Surv(time, delta) ~ age + diagyr + as.factor(stage),
                  data = laryngeal)
## Compare coxph and coxlps results
summary(coxphfit)
fit
## Plot Kaplan-Meier curve vs Laplace-P-spline fit
plot(fit, h0 = FALSE, overlay.km = TRUE, plot.cred = FALSE)


}
\references{
Cox, D.R. (1972). Regression models and life-tables.
  \emph{Journal of the Royal Statistical Society: Series B (Methodological)}
  \strong{34}(2): 187-202.

Gressani, O. and Lambert, P. (2018). Fast Bayesian inference
  using Laplace approximations in a flexible promotion time cure model based
  on P-splines. \emph{Computational Statistical & Data Analysis} \strong{124}:
  151-167.

Jullion, A. and Lambert, P. (2007). Robust specification of the
  roughness penalty prior distribution in spatially adaptive Bayesian
  P-splines models. \emph{Computational Statistical & Data Analysis}
  \strong{51}(5): 2542-2558.
}
\seealso{
\code{\link{Surv}}, \code{\link{coxph}}, \code{\link{simsurvdata}}
}
\author{
Oswaldo Gressani \email{oswaldo_gressani@hotmail.fr}.
}
