% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/curelps_extract.R
\name{curelps.extract}
\alias{curelps.extract}
\title{Extract estimates of survival functions and cure probability for the
promotion time cure model.}
\usage{
curelps.extract(object, time = NULL, curvetype = c("baseline", "population", "probacure"),
                covar.profile, compute.cred = TRUE, cred.int = 0.95, verbose = TRUE)
}
\arguments{
\item{object}{An object of class \code{curelps}.}

\item{time}{A vector of time values on which to compute the estimates.
Each component of \code{time} must be between 0 and the largest observed
follow-up time.}

\item{curvetype}{The curve on which estimates are computed ; \code{baseline}
(the default) is for the baseline survival, \code{population} is for the
population survival function for a profile of covariates given in
\code{covar.profile}, and \code{probacure} is for the probability to be
cured (for a profile of covariates given in \code{covar.profile}) given
that the subject has survived until time t.}

\item{covar.profile}{A numeric vector of the same length as the number
of covariates in the model. This corresponds to the profile of covariates
for which to compute the population survival function  and cure probability
estimates. The order of the covariates in \code{covar.profile} is the same
as the order specified in \code{formula} of the \code{curelps} routine.
Each component of \code{covar.profile} should be in the range of the
observed values for the corresponding covariate. If \code{covar.profile}
is left unspecified by the user, the default will be to take the median
covariate values.}

\item{compute.cred}{Should credible intervals be computed? Default is TRUE.}

\item{cred.int}{The level for an approximate pointwise credible interval.
Default is 0.95.}

\item{verbose}{Should estimates be printed to console?}
}
\value{
A list with the following components:

\item{fit.time}{Estimates on the time values provided in \code{time}.}

\item{cred.int}{The chosen level to construct approximate pointwise
  credible intervals.}

\item{covar.profile}{The chosen profile of covariates.}
}
\description{
The routine takes as input an object of class \code{curelps} and computes
estimates of the baseline survival curve, the population survival
curve and the cure probability on a specified time vector. Approximate
pointwise credible intervals are available.
}
\examples{

# Example on phase III clinical trial e1684 on melanoma data

data(ecog1684)

# Kaplan-Meier curve
plot(survfit(Surv(time, status) ~ 1, data = ecog1684), mark.time = TRUE)
fit <- curelps(Surv(time, status) ~ lt(age + trt+ sex) +
             st(age + trt + sex), data = ecog1684, K = 20, penorder = 2)
fit
profile1 <- c(0, 1, 1, 0, 1, 1) # Mean age, trt = IFN, sex = Female.
profile2 <- c(0, 0, 1, 0, 0, 1) # Mean age, trt = control, sex = Female.

# Extract cure probabilities
curelps.extract(fit, time = c(0, 1, 2, 3), curvetype = "probacure",
                covar.profile = profile1, cred.int = 0.90)
curelps.extract(fit, time = c(0, 1, 2, 3), curvetype = "probacure",
                covar.profile = profile2, cred.int = 0.90)
}
\seealso{
\code{\link{curelps}}, \code{\link{curelps.object}},
 \code{\link{plot.curelps}}, \code{\link{print.curelps}}.
}
\author{
Oswaldo Gressani \email{oswaldo_gressani@hotmail.fr}.
}
