% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gibbs_bspline.R
\name{gibbs_bspline}
\alias{gibbs_bspline}
\title{Metropolis-within-Gibbs sampler for spectral inference of a stationary time series using a B-spline prior}
\usage{
gibbs_bspline(data, Ntotal, burnin, thin = 1, k.theta = 0.01, MG = 1,
  G0.alpha = 1, G0.beta = 1, LG = 20, MH = 1, H0.alpha = 1,
  H0.beta = 1, LH = 20, tau.alpha = 0.001, tau.beta = 0.001,
  kmax = 100, k1 = 20)
}
\arguments{
\item{data}{numeric vector}

\item{Ntotal}{total number of iterations to run the Markov chain}

\item{burnin}{number of initial iterations to be discarded}

\item{thin}{thinning number (post-processing)}

\item{k.theta}{prior parameter for number of B-spline densities k (proportional to exp(-k.theta*k^2)) in mixture}

\item{MG}{Dirichlet process base measure constant for weights of B-spline densities in mixture (> 0)}

\item{G0.alpha, G0.beta}{parameters of Beta base measure of Dirichlet process for weights of B-spline densities in mixture (default is Uniform[0, 1])}

\item{LG}{truncation parameter of Dirichlet process in stick breaking representation for weights of B-spline densities}

\item{MH}{Dirichlet process base measure constant for knot placements of B-spline densities (> 0)}

\item{H0.alpha, H0.beta}{parameters of Beta base measure of Dirichlet process for knot placements of B-spline densities (default is Uniform[0, 1])}

\item{LH}{truncation parameter of Dirichlet process in stick breaking representation for knot placements of B-spline densities}

\item{tau.alpha, tau.beta}{prior parameters for tau (Inverse Gamma)}

\item{kmax}{upper bound for number of B-spline densities in mixture}

\item{k1}{starting value for k.  If k1 = NA then a random starting value between 5 and kmax is selected.}
}
\value{
A list containing the following components:
   \item{psd.median,psd.mean}{psd estimates: (pointwise) posterior median and mean}
   \item{psd.p05,psd.p95}{pointwise credibility interval}
   \item{psd.u05,psd.u95}{uniform credibility interval}
   \item{k,tau,V,Z,U,X}{posterior traces of model parameters}
   \item{knots.trace}{trace of knot placements}
   \item{ll.trace}{trace of log likelihood}
}
\description{
This function updates the (cubic) B-spline prior using the Whittle likelihood and obtains samples from the pseudo-posterior to infer the spectral density of a stationary time series.
}
\details{
The function \code{gibbs_bspline} is an implementation of the (serial version of the) MCMC algorithm presented in Edwards et al. (2018).  This algorithm uses a nonparametric B-spline prior to estimate the spectral density of a stationary time series and can be considered a generalisation of the algorithm of Choudhuri et al. (2004), which used the Bernstein polynomial prior.  A Dirichlet process prior is used to find the weights for the B-spline densities used in the finite mixture and a seperate and independent Dirichlet process prior used to place knots.  The algorithm therefore allows for a data-driven choice of the number of knots/mixtures and their locations.
}
\examples{
\dontrun{

# Generate AR(1) data with rho = 0.9
n = 128
data = arima.sim(n, model = list(ar = 0.9))
data = data - mean(data)

# Run MCMC (may take some time)
mcmc = gibbs_bspline(data, Ntotal = 4000, burnin = 2000, thin = 1)

# Compare estimate with true PSD
require(beyondWhittle)  # For psd_arma() function
freq = 2 * pi / n * (1:(n / 2 + 1) - 1)[-c(1, n / 2 + 1)]
psd.true <- psd_arma(freq, ar = 0.9, ma = numeric(0), sigma2 = 1)
plot(x = freq, y = psd.true, col = 2, type = "l", xlab = "Frequency", ylab = "PSD")
lines(x = freq, y = mcmc$psd.median, type = "l")
lines(x = freq, y = mcmc$psd.p05, type = "l", lty = 2)
lines(x = freq, y = mcmc$psd.p95, type = "l", lty = 2)
legend(x = "topright", legend = c("true psd", "pointwise median", "pointwise CI"), 
lty = c(1, 1, 2), col = c(2, 1, 1))
}
}
\references{
Edwards, M. C., Meyer, R., and Christensen, N. (2018), Bayesian nonparametric spectral density estimation using B-spline priors, \emph{Statistics and Computing}, https://doi.org/10.1007/s11222-017-9796-9.

Choudhuri, N., Ghosal, S., and Roy, A. (2004), Bayesian estimation of the spectral density of a time series, \emph{Journal of the American Statistical Association}, 99(468):1050--1059.
}
