% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/irf.R, R/plot.bvarirf.R
\name{irf}
\alias{irf}
\alias{plot.bvarirf}
\title{Impulse Response Function}
\usage{
irf(object, impulse = NULL, response = NULL, n.ahead = 5,
  ci = 0.95, type = "feir", cumulative = FALSE)

\method{plot}{bvarirf}(x, ...)
}
\arguments{
\item{object}{an object of class \code{"bvar"}, usually, a result of a call to
\code{\link{bvar}} or \code{\link{bvec_to_bvar}}.}

\item{impulse}{name of the impulse variable.}

\item{response}{name of the response variable.}

\item{n.ahead}{number of steps ahead.}

\item{ci}{a numeric between 0 and 1 specifying the probability mass covered by the
credible intervals. Defaults to 0.95.}

\item{type}{type of the impulse resoponse. Possible choices are forecast error \code{"feir"}
(default), orthogonalised \code{"oir"}, structural \code{"sir"}, generalised \code{"gir"},
and structural generalised \code{"sgir"} impulse responses.}

\item{cumulative}{logical specifying whether a cumulative IRF should be calculated.}

\item{x}{an object of class "bvarirf", usually, a result of a call to \code{\link{irf}}.}

\item{...}{further graphical parameters.}
}
\value{
A time-series object of class \code{"bvarirf"}.
}
\description{
Computes the impulse response coefficients of an object of class \code{"bvar"} for
\code{n.ahead} steps.

A plot function for objects of class "bvarirf".
}
\details{
The function produces different types of impulse responses for the VAR model
\deqn{y_t = \sum_{i = 1}^{p} A_{i} y_{t-i} + A_0^{-1} u_t,}
with \eqn{u_t \sim N(0, \Sigma)}.

Forecast error impulse responses \eqn{\Phi_i} are obtained by recursions
\deqn{\Phi_i = \sum_{j = 1}^{i} \Phi_{i-j} A_j,   i = 1, 2,...,h}
with \eqn{\Phi_0 = I_K}.

Orthogonalised impulse responses \eqn{\Theta^o_i} are calculated as \eqn{\Theta^o_i = \Phi_i P},
where P is the lower triangular Choleski decomposition of \eqn{\Sigma}. \eqn{A_0} is assumed to
be an identity matrix.

Structural impulse responses \eqn{\Theta^s_i} are calculated as \eqn{\Theta^s_i = \Phi_i A_0^{-1}}.

(Structural) Generalised impulse responses for variable \eqn{j}, i.e. \eqn{\Theta^g_ji} are calculated as
\eqn{\Theta^g_{ji} = \sigma_{jj}^{-1/2} \Phi_i A_0^{-1} \Sigma e_j}, where \eqn{\sigma_{jj}} is the variance
of the \eqn{j^{th}} diagonal element of \eqn{\Sigma} and \eqn{e_i} is a selection vector containing
one in its \eqn{j^{th}} element and zero otherwise. If the \code{"bvar"} object does not contain draws
of \eqn{A_0}, it is assumed to be an identity matrix.
}
\examples{
data("e1")
e1 <- diff(log(e1))

data <- gen_var(e1, p = 2, deterministic = "const")

y <- data$Y[, 1:73]
x <- data$Z[, 1:73]

set.seed(1234567)

iter <- 500 # Number of iterations of the Gibbs sampler
# Chosen number of iterations should be much higher, e.g. 30000.

burnin <- 100 # Number of burn-in draws
store <- iter - burnin

t <- ncol(y) # Number of observations
k <- nrow(y) # Number of endogenous variables
m <- k * nrow(x) # Number of estimated coefficients

# Set (uninformative) priors
a_mu_prior <- matrix(0, m) # Vector of prior parameter means
a_v_i_prior <- diag(0, m) # Inverse of the prior covariance matrix

u_sigma_df_prior <- 0 # Prior degrees of freedom
u_sigma_scale_prior <- diag(0, k) # Prior covariance matrix
u_sigma_df_post <- t + u_sigma_df_prior # Posterior degrees of freedom

# Initial values
u_sigma_i <- diag(.00001, k)
u_sigma <- solve(u_sigma_i)

# Data containers for posterior draws
draws_a <- matrix(NA, m, store)
draws_sigma <- matrix(NA, k^2, store)

# Start Gibbs sampler
for (draw in 1:iter) {
  # Draw conditional mean parameters
  a <- post_normal(y, x, u_sigma_i, a_mu_prior, a_v_i_prior)

# Draw variance-covariance matrix
u <- y - matrix(a, k) \%*\% x # Obtain residuals
u_sigma_scale_post <- solve(u_sigma_scale_prior + tcrossprod(u))
u_sigma_i <- matrix(rWishart(1, u_sigma_df_post, u_sigma_scale_post)[,, 1], k)
u_sigma <- solve(u_sigma_i) # Invert Sigma_i to obtain Sigma

# Store draws
if (draw > burnin) {
  draws_a[, draw - burnin] <- a
  draws_sigma[, draw - burnin] <- u_sigma
  }
}

# Generate bvar object
bvar_est <- bvar(y = y, x = x, A = draws_a[1:18,],
                 C = draws_a[19:21, ], Sigma = draws_sigma)

# Generate impulse response
IR <- irf(bvar_est, impulse = "income", response = "cons", n.ahead = 8)

# Plot
plot(IR, main = "Forecast Error Impulse Response", xlab = "Period", ylab = "Response")

data("e1")
e1 <- diff(log(e1))

data <- gen_var(e1, p = 2, deterministic = "const")

y <- data$Y[, 1:73]
x <- data$Z[, 1:73]

set.seed(1234567)

iter <- 500 # Number of iterations of the Gibbs sampler
# Chosen number of iterations should be much higher, e.g. 30000.

burnin <- 100 # Number of burn-in draws
store <- iter - burnin

t <- ncol(y) # Number of observations
k <- nrow(y) # Number of endogenous variables
m <- k * nrow(x) # Number of estimated coefficients

# Set (uninformative) priors
a_mu_prior <- matrix(0, m) # Vector of prior parameter means
a_v_i_prior <- diag(0, m) # Inverse of the prior covariance matrix

u_sigma_df_prior <- 0 # Prior degrees of freedom
u_sigma_scale_prior <- diag(0, k) # Prior covariance matrix
u_sigma_df_post <- t + u_sigma_df_prior # Posterior degrees of freedom

# Initial values
u_sigma_i <- diag(.00001, k)
u_sigma <- solve(u_sigma_i)

# Data containers for posterior draws
draws_a <- matrix(NA, m, store)
draws_sigma <- matrix(NA, k^2, store)

# Start Gibbs sampler
for (draw in 1:iter) {
  # Draw conditional mean parameters
  a <- post_normal(y, x, u_sigma_i, a_mu_prior, a_v_i_prior)

# Draw variance-covariance matrix
u <- y - matrix(a, k) \%*\% x # Obtain residuals
u_sigma_scale_post <- solve(u_sigma_scale_prior + tcrossprod(u))
u_sigma_i <- matrix(rWishart(1, u_sigma_df_post, u_sigma_scale_post)[,, 1], k)
u_sigma <- solve(u_sigma_i) # Invert Sigma_i to obtain Sigma

# Store draws
if (draw > burnin) {
  draws_a[, draw - burnin] <- a
  draws_sigma[, draw - burnin] <- u_sigma
  }
}

# Generate bvar object
bvar_est <- bvar(y = y, x = x, A = draws_a[1:18,],
                 C = draws_a[19:21, ], Sigma = draws_sigma)

# Generate impulse response
IR <- irf(bvar_est, impulse = "income", response = "cons", n.ahead = 8)

# Plot
plot(IR, main = "Forecast Error Impulse Response", xlab = "Period", ylab = "Response")

}
\references{
Lütkepohl, H. (2007). \emph{New introduction to multiple time series analysis} (2nd ed.). Berlin: Springer.

Pesaran, H. H., Shin, Y. (1998). Generalized impulse response analysis in linear multivariate models. \emph{Economics Letters, 58}, 17-29.
}
