% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/csem_resample.R
\name{resamplecSEMResults}
\alias{resamplecSEMResults}
\title{Resample cSEMResults}
\usage{
resamplecSEMResults(
 .object                = NULL,
 .resample_method       = c("bootstrap", "jackknife"), 
 .resample_method2      = c("none", "bootstrap", "jackknife"), 
 .R                     = 499,
 .R2                    = 199,
 .handle_inadmissibles  = c("drop", "ignore", "replace"),
 .user_funs             = NULL,
 .eval_plan             = c("sequential", "multicore", "multisession"),
 .force                 = FALSE,
 .seed                  = NULL,
 .sign_change_option    = c("none","individual","individual_reestimate",
                            "construct_reestimate"),
 ...
)
}
\arguments{
\item{.object}{An R object of class \link{cSEMResults} resulting from a call to \code{\link[=csem]{csem()}}.}

\item{.resample_method}{Character string. The resampling method to use. One of:
"\emph{bootstrap}" or "\emph{jackknife}". Defaults to "\emph{bootstrap}".}

\item{.resample_method2}{Character string. The resampling method to use when resampling
from a resample. One of: "\emph{none}", "\emph{bootstrap}" or "\emph{jackknife}". For
"\emph{bootstrap}" the number of draws is provided via \code{.R2}. Currently,
resampling from each resample is only required for the studentized confidence
interval ("\emph{CI_t_interval}") computed by the \code{\link[=infer]{infer()}} function. Defaults to "\emph{none}".}

\item{.R}{Integer. The number of bootstrap replications. Defaults to \code{499}.}

\item{.R2}{Integer. The number of bootstrap replications to use when
resampling from a resample. Defaults to \code{199}.}

\item{.handle_inadmissibles}{Character string. How should inadmissible results
be treated? One of "\emph{drop}", "\emph{ignore}", or "\emph{replace}". If "\emph{drop}", all
replications/resamples yielding an inadmissible result will be dropped
(i.e. the number of results returned will potentially be less than \code{.R}).
For "\emph{ignore}" all results are returned even if all or some of the replications
yielded inadmissible results (i.e. number of results returned is equal to \code{.R}).
For "\emph{replace}" resampling continues until there are exactly \code{.R} admissible solutions.
Depending on the frequency of inadmissible solutions this may significantly increase
computing time. Defaults to "\emph{drop}".}

\item{.user_funs}{A function or a (named) list of functions to apply to every
resample. The functions must take \code{.object} as its first argument (e.g.,
\verb{myFun <- function(.object, ...) \{body-of-the-function\}}).
Function output should preferably be a (named)
vector but matrices are also accepted. However, the output will be
vectorized (columnwise) in this case. See the examples section for details.}

\item{.eval_plan}{Character string. The evaluation plan to use. One of
"\emph{sequential}", "\emph{multicore}", or "\emph{multisession}". In the two latter cases
all available cores will be used. Defaults to "\emph{sequential}".}

\item{.force}{Logical. Should .object be resampled even if it contains resamples
already?. Defaults to \code{FALSE}.}

\item{.seed}{Integer or \code{NULL}. The random seed to use. Defaults to \code{NULL} in which
case an arbitrary seed is chosen. Note that the scope of the seed is limited
to the body of the function it is used in. Hence, the global seed will
not be altered!}

\item{.sign_change_option}{Character string. Which sign change option should
be used to handle flipping signs when resampling? One of "\emph{none}","\emph{individual}",
"\emph{individual_reestimate}", "\emph{construct_reestimate}". Defaults to "\emph{none}".}

\item{...}{Further arguments passed to functions supplied to \code{.user_funs}.}
}
\value{
The core structure is the same structure as that of \code{.object} with
the following elements added:
\itemize{
\item{ \verb{$Estimates_resamples}: A list containing the \code{.R} resamples and
the original estimates for each of the resampled quantities (Path_estimates,
Loading_estimates, Weight_estimates, user defined functions).
Each list element is a list containing elements
\verb{$Resamples} and \verb{$Original}. \verb{$Resamples} is a \verb{(.R x K)} matrix with each
row representing one resample for each of the \code{K} parameters/statistics.
\verb{$Original} contains the original estimates (vectorized by column if the output of
the user provided function is a matrix.}
\item {\verb{$Information_resamples}: A list containing additional information.}
}
Use \verb{str(<.object>, list.len = 3)} on the resulting object for an overview.
}
\description{
Resample a \link{cSEMResults} object using bootstrap or jackknife resampling.
The function is called by \code{\link[=csem]{csem()}} if the user sets
\code{csem(..., .resample_method = "bootstrap")} or
\code{csem(..., .resample_method = "jackknife")} but may also be called directly.
}
\details{
Given \code{M} resamples (for bootstrap \code{M = .R} and for jackknife \code{M = N}, where
\code{N} is the number of observations) based on the data used to compute the
\link{cSEMResults} object provided via \code{.object}, \code{resamplecSEMResults()} essentially calls
\code{\link[=csem]{csem()}} on each resample using the arguments of the original call (ignoring any arguments
related to resampling) and returns estimates for each of a subset of
practically useful resampled parameters/statistics computed by \code{\link[=csem]{csem()}}.
Currently, the following estimates are computed and returned by default based
on each resample: Path estimates, Loading estimates, Weight estimates.

In practical application users may need to resample a specific statistic (e.g,
the heterotrait-monotrait ratio of correlations (HTMT) or differences between path
coefficients such as beta_1 - beta_2).
Such statistics may be provided by a function \code{fun(.object, ...)} or a list of
such functions via the \code{.user_funs} argument. The first argument of
these functions must always be \code{.object}.
Internally, the function will be applied on each
resample to produce the desired statistic. Hence, arbitrary complicated statistics
may be resampled as long as the body of the function draws on elements contained
in the \link{cSEMResults} object only. Output of \code{fun(.object, ...)} should preferably
be a (named) vector but matrices are also accepted.
However, the output will be vectorized (columnwise) in this case.
See the examples section for details.

Both resampling the original \link{cSEMResults} object (call it "first resample")
and resampling based on a resampled \link{cSEMResults} object (call it "second resample")
are supported. Choices for the former
are "\emph{bootstrap}" and "\emph{jackknife}". Resampling based on a resample is turned off
by default (\code{.resample_method2 = "none"}) as this significantly
increases computation time (there are now \code{M * M2} resamples to compute, where
\code{M2} is \code{.R2} or \code{N}).
Resamples of a resample are required, e.g., for the studentized confidence
interval computed by the \code{\link[=infer]{infer()}} function. Typically, bootstrap resamples
are used in this case \insertCite{Davison1997}{cSEM}.

As \code{\link[=csem]{csem()}} accepts a single data set, a list of data sets as well as data sets
that contain a column name used to split the data into groups,
the \link{cSEMResults} object may contain multiple data sets.
In this case, resampling is done by data set or group. Note that depending
on the number of data sets/groups, the computation may be considerably
slower as resampling will be repeated for each data set/group. However, apart
from speed considerations users don not need to worry about the type of
input used to compute the \link{cSEMResults} object as \code{resamplecSEMResults()}
is able to deal with each case.

The number of bootstrap runs for the first and second run are given by \code{.R} and \code{.R2}.
The default is \code{499} for the first and \code{199} for the second run
but should be increased in real applications. See e.g.,
\insertCite{Hesterberg2015;textual}{cSEM}, p.380,
\insertCite{Davison1997;textual}{cSEM}, and
\insertCite{Efron2016;textual}{cSEM} for recommendations.
For jackknife \code{.R} are \code{.R2} are ignored.

Resampling may produce inadmissible results (as checked by \code{\link[=verify]{verify()}}).
By default these results are dropped however users may choose to \code{"ignore"}
or \code{"replace"} inadmissible results in which resampling continuous until
the necessary number of admissible results is reached.

The \pkg{cSEM} package supports (multi)processing via the \href{https://github.com/HenrikBengtsson/future/}{future}
framework \insertCite{Bengtsson2018}{cSEM}. Users may simply choose an evaluation plan
via \code{.eval_plan} and the package takes care of all the complicated backend
issues. Currently, users may choose between standard single-core/single-session
evaluation (\code{"sequential"}) and multiprocessing (\code{"multisession"} or \code{"multicore"}). The future package
provides other options (e.g., \code{"cluster"} or \code{"remote"}), however, they probably
will not be needed in the context of the \pkg{cSEM} package as simulations usually
do not require high-performance clusters. Depending on the operating system, the future
package will manage to distribute tasks to multiple R sessions (Windows)
or multiple cores. Note that multiprocessing is not necessary always faster
when only a "small" number of replications is required as the overhead of
initializing new sessions or distributing tasks to different cores
will not immediately be compensated by the availability of multiple sessions/cores.

Random number generation (RNG) uses the L'Ecuyer-CRMR RGN stream as implemented in the
\href{https://github.com/HenrikBengtsson/future.apply/}{future.apply package} \insertCite{Bengtsson2018a}{cSEM}.
It is independent of the evaluation plan. Hence, setting e.g., \code{.seed = 123} will
generate the same random number and replicates
for both \code{.eval_plan = "sequential"}, \code{.eval_plan = "multisession"}, and \code{.eval_plan = "multicore"}.
See \link[future.apply:future_lapply]{?future_lapply} for details.
}
\examples{
\dontrun{
# Note: example not run as resampling is time consuming
# ===========================================================================
# Basic usage
# ===========================================================================
model <- "
# Structural model
QUAL ~ EXPE
EXPE ~ IMAG
SAT  ~ IMAG + EXPE + QUAL + VAL
LOY  ~ IMAG + SAT
VAL  ~ EXPE + QUAL

# Measurement model
EXPE =~ expe1 + expe2 + expe3 + expe4 + expe5
IMAG =~ imag1 + imag2 + imag3 + imag4 + imag5
LOY  =~ loy1  + loy2  + loy3  + loy4
QUAL =~ qual1 + qual2 + qual3 + qual4 + qual5
SAT  =~ sat1  + sat2  + sat3  + sat4
VAL  =~ val1  + val2  + val3  + val4
"

## Estimate the model without resampling 
a <- csem(satisfaction, model)

## Bootstrap and jackknife estimation
boot <- resamplecSEMResults(a)
jack <- resamplecSEMResults(a, .resample_method = "jackknife") 

## Alternatively use .resample_method in csem()
boot_csem <- csem(satisfaction, model, .resample_method = "bootstrap")
jack_csem <- csem(satisfaction, model, .resample_method = "jackknife")

# ===========================================================================
# Extended usage
# ===========================================================================
### Double resampling  ------------------------------------------------------
# The confidence intervals (e.g. the bias-corrected and accelearated CI) 
# require double resampling. Use .resample_method2 for this.

boot1 <- resamplecSEMResults(
  .object = a, 
  .resample_method  = "bootstrap", 
  .R                = 50,
  .resample_method2 = "bootstrap", 
  .R2               = 20,
  .seed             = 1303
  )

## Again, this is identical to using csem 
boot1_csem <- csem(
  .data             = satisfaction, 
  .model            = model, 
  .resample_method  = "bootstrap",
  .R                = 50,
  .resample_method2 = "bootstrap",
  .R2               = 20,
  .seed             = 1303
  )

identical(boot1, boot1_csem) # only true if .seed was set

### Inference ---------------------------------------------------------------
# To get inferencial quanitites such as the estimated standard error or
# the percentile confidence intervall for each resampled quantity use 
# postestimation function infer()

inference <- infer(boot1)
inference$Path_estimates$sd
inference$Path_estimates$CI_percentile

# As usual summarize() can be called directly
summarize(boot1)

# In the example above .R x .R2 = 50 x 20 = 1000. Multiprocessing will be
# faster on most systems here and is therefore recommended. Note that multiprocessing
# does not affect the random number generation

boot2 <- resamplecSEMResults(
  .object           = a, 
  .resample_method  = "bootstrap", 
  .R                = 50,
  .resample_method2 = "bootstrap", 
  .R2               = 20,
  .eval_plan        = "multisession", 
  .seed             = 1303
  )

identical(boot1, boot2)}
}
\references{
\insertAllCited{}
}
\seealso{
\link{csem}, \code{\link[=summarize]{summarize()}}, \code{\link[=infer]{infer()}}, \link{cSEMResults}
}
