% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/integral.R
\name{integral}
\alias{integral}
\title{Numerical Integration}
\usage{
integral(
  f,
  bounds,
  params = list(),
  coordinates = "cartesian",
  relTol = 0.001,
  absTol = 1e-12,
  method = NULL,
  vectorize = NULL,
  drop = TRUE,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{f}{array of \code{characters} or a \code{function} returning a \code{numeric} array.}

\item{bounds}{\code{list} containing the lower and upper bounds for each variable. If the two bounds coincide, or if a single number is specified, the corresponding variable is not integrated and its value is fixed.}

\item{params}{\code{list} of additional parameters passed to \code{f}.}

\item{coordinates}{coordinate system to use. One of: \code{cartesian}, \code{polar}, \code{spherical}, \code{cylindrical}, \code{parabolic}, \code{parabolic-cylindrical} or a character vector of scale factors for each variable.}

\item{relTol}{the maximum relative tolerance.}

\item{absTol}{the absolute tolerance.}

\item{method}{the method to use. One of \code{"mc"}, \code{"hcubature"}, \code{"pcubature"}, \code{"cuhre"}, \code{"divonne"}, \code{"suave"} or \code{"vegas"}. Methods other than \code{"mc"} (naive Monte Carlo) require the \pkg{cubature} package to be installed (efficient integration in C). The defaul uses \code{"hcubature"} if \pkg{cubature} is installed or \code{"mc"} otherwise.}

\item{vectorize}{\code{logical}. Use vectorization? If \code{TRUE}, it can significantly boost performance but \code{f} needs to handle the vector of inputs appropriately. The default uses \code{FALSE} if \code{f} is a \code{function}, \code{TRUE} otherwise.}

\item{drop}{if \code{TRUE}, return the integral as a vector and not as an \code{array} for vector-valued functions.}

\item{verbose}{\code{logical}. Print on progress?}

\item{...}{additional arguments passed to \code{\link[cubature]{cubintegrate}}, when method \code{"hcubature"}, \code{"pcubature"}, \code{"cuhre"}, \code{"divonne"}, \code{"suave"} or \code{"vegas"} is used.}
}
\value{
list with components
\describe{
 \item{value}{the final estimate of the integral.}
 \item{error}{estimate of the modulus of the absolute error.}
 \item{cuba}{\pkg{cubature} output when method \code{"hcubature"}, \code{"pcubature"}, \code{"cuhre"}, \code{"divonne"}, \code{"suave"} or \code{"vegas"} is used.}
}
}
\description{
Computes the integrals of \code{functions} or \code{characters} in arbitrary 
\href{https://en.wikipedia.org/wiki/Orthogonal_coordinates#Table_of_orthogonal_coordinates}{orthogonal coordinate systems}.
}
\details{
The function integrates seamlessly with \pkg{cubature} for efficient 
numerical integration in \proglang{C}. If the package \pkg{cubature} is not 
installed, the function implements a naive Monte Carlo integration by default.
For arbitrary orthogonal coordinates \eqn{q_1\dots q_n} the integral is computed as:

\deqn{\int J\cdot f(q_1\dots q_n) dq_1\dots dq_n}

where \eqn{J=\prod_i h_i} is the Jacobian determinant of the transformation 
and is equal to the product of the scale factors \eqn{h_1\dots h_n}.
}
\examples{
### unidimensional integral
i <- integral("sin(x)", bounds = list(x = c(0,pi)))
i$value

### multidimensional integral
f <- function(x,y) x*y
i <- integral(f, bounds = list(x = c(0,1), y = c(0,1)))
i$value

### vector-valued integrals
f <- function(x,y) c(x, y, x*y)
i <- integral(f, bounds = list(x = c(0,1), y = c(0,1)))
i$value

### tensor-valued integrals
f <- function(x,y) array(c(x^2, x*y, x*y, y^2), dim = c(2,2))
i <- integral(f, bounds = list(x = c(0,1), y = c(0,1)))
i$value

### area of a circle
i <- integral(1, 
              bounds = list(r = c(0,1), theta = c(0,2*pi)), 
              coordinates = "polar")
i$value

### surface of a sphere
i <- integral(1, 
              bounds = list(r = 1, theta = c(0,pi), phi = c(0,2*pi)), 
              coordinates = "spherical")
i$value

### volume of a sphere
i <- integral(1, 
         bounds = list(r = c(0,1), theta = c(0,pi), phi = c(0,2*pi)), 
         coordinates = "spherical")
i$value

}
\references{
Guidotti, E. (2020). "calculus: High dimensional numerical and symbolic calculus in R". \url{https://arxiv.org/abs/2101.00086}
}
\seealso{
Other integrals: 
\code{\link{ode}()}
}
\concept{integrals}
