#' Error propagation
#' 
#' This function performs error propagation on 
#' coefficients and standard errors.
#' 
#' @param markers The marker names being tested
#' @param beta_m The main-effects coefficient matrix for 
#' the pairwise regression of the given pair.
#' @param se The standard errors for the marker pair.
#' @param beta_cov The model covariance matrix from the 
#' pairwise regression
#' 
#' @return Returns the error propagated coefficients and 
#' standard errors for m12 and m21
#' 
#' @import corpcor
#' @import propagate
#' 
calc_delta_errors <- function(markers,beta_m,se,beta_cov) {
  # print(markers)		
  beta_main <- beta_m[,1:2]
  beta_inter <- beta_m[,3]
  n_rows <- dim(beta_main)[1] #No of ETs
  n_cols <- dim(beta_main)[2]
  
  se_main <- se[,1:2]
  se_inter <- se[,3]
  
  if(n_rows == n_cols){
    act_delta <- solve(beta_main)%*%beta_inter
  }else{
    act_delta <- try(pseudoinverse(beta_main) %*% beta_inter, silent = TRUE)
    if(class(act_delta)[1] == "try-error"){
      act_delta <- c(NA, NA)
    }
  }
  
  m12 = act_delta[1]/(1+act_delta[2])
  m21 = act_delta[2]/(1+act_delta[1])
  
  
  ##### Code for error prop
  col_hdrs <- NULL		
  full_matrix <- matrix(0,n_rows,3*n_rows)
  coeffs <- NULL
  se <- NULL
  
  for(i in 1:n_rows) {
    numr <- c("1","2","12")
    col_hdrs <- append(col_hdrs,paste(letters[i],numr,sep=""))
  }
  
  for(i in 1:n_rows){
    coeffs <- append(coeffs,append(beta_main[i,],beta_inter[i]))
    se <- append(se,append(se_main[i,],se_inter[i]))
  }
  
  full_matrix	 <- rbind(coeffs,se)
  rownames(full_matrix) <- c("coeff","se")
  colnames(full_matrix) <- col_hdrs
  
  colnames(beta_cov) <- colnames(full_matrix)
  rownames(beta_cov) <- colnames(full_matrix)
  
  if (n_rows == 2) {
    #m12
    EXPR1 <- expression((b2*a12-a2*b12)/(a1*b2-a2*b1+(-b1*a12+a1*b12)))
    #m21
    EXPR2 <- expression((-b1*a12+a1*b12)/((a1+a12)*b2-a2*(b1+b12)))
  } else if (n_rows == 3) {
    #m12
    EXPR1 <- expression((a2^2*(b1*b12 + c1*c12) + (b2*c1 - b1*c2)*(b2*c12 - b12*c2) - a2*(a12*b1*b2 + a1*b12*b2 + a12*c1*c2 + a1*c12*c2) + a1*a12*(b2^2 + c2^2))/(a1^2*b12*b2 + a1^2*b2^2 + b12*b2*c1^2 + b2^2*c1^2 + a2^2*(b1^2 + c1^2) - b1*b2*c1*c12 - b1*b12*c1*c2 - 2*b1*b2*c1*c2 + a1^2*c12*c2 + b1^2*c12*c2 + a1^2*c2^2 + b1^2*c2^2 - a1*a2*(b1*b12 + 2*b1*b2 + c1*c12 + 2*c1*c2) + a12*(a2*(b1^2 + c1^2) - a1*(b1*b2 + c1*c2))))
    #m21
    EXPR2 <- expression((-(a1*a2*(b1*b12 + c1*c12)) + (b12*c1 - b1*c12)*(b2*c1 - b1*c2) + a1^2*(b12*b2 + c12*c2) + a12*(a2*(b1^2 + c1^2) - a1*(b1*b2 + c1*c2)))/(a2^2*(b1^2 + b1*b12 + c1*(c1 + c12)) + (b2*c1 - b1*c2)*(b2*(c1 + c12) - (b1 + b12)*c2) + a1^2*(b2^2 + c2^2) + a1*a12*(b2^2 + c2^2) - a2*(a12*(b1*b2 + c1*c2) + a1*(2*b1*b2 + b12*b2 + 2*c1*c2 + c12*c2))))
  } else if (n_rows == 4) {
    #m12
    EXPR1 <- expression((b2^2*c1*c12 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1*b12*c2^2 + b2^2*d1*d12 + c2^2*d1*d12 + a2^2*(b1*b12 + c1*c12 + d1*d12) - b12*b2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1*b12*d2^2 + c1*c12*d2^2 + a1*a12*(b2^2 + c2^2 + d2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2) + a1*(b12*b2 + c12*c2 + d12*d2)))/(a1^2*b12*b2 + a1^2*b2^2 + b12*b2*c1^2 + b2^2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 - 2*b1*b2*c1*c2 + a1^2*c12*c2 + b1^2*c12*c2 + a1^2*c2^2 + b1^2*c2^2 + b12*b2*d1^2 + b2^2*d1^2 + c12*c2*d1^2 + c2^2*d1^2 + a2^2*(b1^2 + c1^2 + d1^2) - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - 2*b1*b2*d1*d2 - c1*c12*d1*d2 - 2*c1*c2*d1*d2 + a1^2*d12*d2 + b1^2*d12*d2 + c1^2*d12*d2 + a1^2*d2^2 + b1^2*d2^2 + c1^2*d2^2 - a1*a2*(b1*b12 + 2*b1*b2 + c1*c12 + 2*c1*c2 + d1*d12 + 2*d1*d2) + a12*(a2*(b1^2 + c1^2 + d1^2) - a1*(b1*b2 + c1*c2 + d1*d2))))
    #m21
    EXPR2 <- expression((b12*b2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 + b1^2*c12*c2 + b12*b2*d1^2 + c12*c2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - a1*a2*(b1*b12 + c1*c12 + d1*d12) - b1*b12*d1*d2 - c1*c12*d1*d2 + b1^2*d12*d2 + c1^2*d12*d2 + a1^2*(b12*b2 + c12*c2 + d12*d2) + a12*(a2*(b1^2 + c1^2 + d1^2) - a1*(b1*b2 + c1*c2 + d1*d2)))/(b2^2*c1^2 + b2^2*c1*c12 - 2*b1*b2*c1*c2 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1^2*c2^2 + b1*b12*c2^2 + b2^2*d1^2 + c2^2*d1^2 + b2^2*d1*d12 + c2^2*d1*d12 + a2^2*(b1^2 + b1*b12 + c1^2 + c1*c12 + d1*(d1 + d12)) - 2*b1*b2*d1*d2 - b12*b2*d1*d2 - 2*c1*c2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1^2*d2^2 + b1*b12*d2^2 + c1^2*d2^2 + c1*c12*d2^2 + a1^2*(b2^2 + c2^2 + d2^2) + a1*a12*(b2^2 + c2^2 + d2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2) + a1*(2*b1*b2 + b12*b2 + 2*c1*c2 + c12*c2 + 2*d1*d2 + d12*d2))))
  } else if (n_rows == 5) {
    #m12
    EXPR1 <- expression((b2^2*c1*c12 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1*b12*c2^2 + b2^2*d1*d12 + c2^2*d1*d12 - b12*b2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1*b12*d2^2 + c1*c12*d2^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 + a2^2*(b1*b12 + c1*c12 + d1*d12 + e1*e12) - b12*b2*e1*e2 - c12*c2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1*b12*e2^2 + c1*c12*e2^2 + d1*d12*e2^2 + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2) + a1*(b12*b2 + c12*c2 + d12*d2 + e12*e2)))/(a1^2*b12*b2 + a1^2*b2^2 + b12*b2*c1^2 + b2^2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 - 2*b1*b2*c1*c2 + a1^2*c12*c2 + b1^2*c12*c2 + a1^2*c2^2 + b1^2*c2^2 + b12*b2*d1^2 + b2^2*d1^2 + c12*c2*d1^2 + c2^2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - 2*b1*b2*d1*d2 - c1*c12*d1*d2 - 2*c1*c2*d1*d2 + a1^2*d12*d2 + b1^2*d12*d2 + c1^2*d12*d2 + a1^2*d2^2 + b1^2*d2^2 + c1^2*d2^2 + b12*b2*e1^2 + b2^2*e1^2 + c12*c2*e1^2 + c2^2*e1^2 + d12*d2*e1^2 + d2^2*e1^2 + a2^2*(b1^2 + c1^2 + d1^2 + e1^2) - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - 2*b1*b2*e1*e2 - c1*c12*e1*e2 - 2*c1*c2*e1*e2 - d1*d12*e1*e2 - 2*d1*d2*e1*e2 + a1^2*e12*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + a1^2*e2^2 + b1^2*e2^2 + c1^2*e2^2 + d1^2*e2^2 - a1*a2*(b1*b12 + 2*b1*b2 + c1*c12 + 2*c1*c2 + d1*d12 + 2*d1*d2 + e1*e12 + 2*e1*e2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2))))
    #m21
    EXPR2 <- expression((b12*b2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 + b1^2*c12*c2 + b12*b2*d1^2 + c12*c2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - c1*c12*d1*d2 + b1^2*d12*d2 + c1^2*d12*d2 + b12*b2*e1^2 + c12*c2*e1^2 + d12*d2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - a1*a2*(b1*b12 + c1*c12 + d1*d12 + e1*e12) - b1*b12*e1*e2 - c1*c12*e1*e2 - d1*d12*e1*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + a1^2*(b12*b2 + c12*c2 + d12*d2 + e12*e2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2)))/(b2^2*c1^2 + b2^2*c1*c12 - 2*b1*b2*c1*c2 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1^2*c2^2 + b1*b12*c2^2 + b2^2*d1^2 + c2^2*d1^2 + b2^2*d1*d12 + c2^2*d1*d12 - 2*b1*b2*d1*d2 - b12*b2*d1*d2 - 2*c1*c2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1^2*d2^2 + b1*b12*d2^2 + c1^2*d2^2 + c1*c12*d2^2 + b2^2*e1^2 + c2^2*e1^2 + d2^2*e1^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 + a2^2*(b1^2 + b1*b12 + c1^2 + c1*c12 + d1^2 + d1*d12 + e1^2 + e1*e12) - 2*b1*b2*e1*e2 - b12*b2*e1*e2 - 2*c1*c2*e1*e2 - c12*c2*e1*e2 - 2*d1*d2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1^2*e2^2 + b1*b12*e2^2 + c1^2*e2^2 + c1*c12*e2^2 + d1^2*e2^2 + d1*d12*e2^2 + a1^2*(b2^2 + c2^2 + d2^2 + e2^2) + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2) + a1*(2*b1*b2 + b12*b2 + 2*c1*c2 + c12*c2 + 2*d1*d2 + d12*d2 + 2*e1*e2 + e12*e2))))
  } else if (n_rows == 6) {
    #m12
    EXPR1 <- expression((b2^2*c1*c12 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1*b12*c2^2 + b2^2*d1*d12 + c2^2*d1*d12 - b12*b2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1*b12*d2^2 + c1*c12*d2^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 - b12*b2*e1*e2 - c12*c2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1*b12*e2^2 + c1*c12*e2^2 + d1*d12*e2^2 + b2^2*f1*f12 + c2^2*f1*f12 + d2^2*f1*f12 + e2^2*f1*f12 + a2^2*(b1*b12 + c1*c12 + d1*d12 + e1*e12 + f1*f12) - b12*b2*f1*f2 - c12*c2*f1*f2 - d12*d2*f1*f2 - e12*e2*f1*f2 - b1*b2*f12*f2 - c1*c2*f12*f2 - d1*d2*f12*f2 - e1*e2*f12*f2 + b1*b12*f2^2 + c1*c12*f2^2 + d1*d12*f2^2 + e1*e12*f2^2 + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2) + a1*(b12*b2 + c12*c2 + d12*d2 + e12*e2 + f12*f2)))/(a1^2*b12*b2 + a1^2*b2^2 + b12*b2*c1^2 + b2^2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 - 2*b1*b2*c1*c2 + a1^2*c12*c2 + b1^2*c12*c2 + a1^2*c2^2 + b1^2*c2^2 + b12*b2*d1^2 + b2^2*d1^2 + c12*c2*d1^2 + c2^2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - 2*b1*b2*d1*d2 - c1*c12*d1*d2 - 2*c1*c2*d1*d2 + a1^2*d12*d2 + b1^2*d12*d2 + c1^2*d12*d2 + a1^2*d2^2 + b1^2*d2^2 + c1^2*d2^2 + b12*b2*e1^2 + b2^2*e1^2 + c12*c2*e1^2 + c2^2*e1^2 + d12*d2*e1^2 + d2^2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - 2*b1*b2*e1*e2 - c1*c12*e1*e2 - 2*c1*c2*e1*e2 - d1*d12*e1*e2 - 2*d1*d2*e1*e2 + a1^2*e12*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + a1^2*e2^2 + b1^2*e2^2 + c1^2*e2^2 + d1^2*e2^2 + b12*b2*f1^2 + b2^2*f1^2 + c12*c2*f1^2 + c2^2*f1^2 + d12*d2*f1^2 + d2^2*f1^2 + e12*e2*f1^2 + e2^2*f1^2 + a2^2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2) - b1*b2*f1*f12 - c1*c2*f1*f12 - d1*d2*f1*f12 - e1*e2*f1*f12 - b1*b12*f1*f2 - 2*b1*b2*f1*f2 - c1*c12*f1*f2 - 2*c1*c2*f1*f2 - d1*d12*f1*f2 - 2*d1*d2*f1*f2 - e1*e12*f1*f2 - 2*e1*e2*f1*f2 + a1^2*f12*f2 + b1^2*f12*f2 + c1^2*f12*f2 + d1^2*f12*f2 + e1^2*f12*f2 + a1^2*f2^2 + b1^2*f2^2 + c1^2*f2^2 + d1^2*f2^2 + e1^2*f2^2 - a1*a2*(b1*b12 + 2*b1*b2 + c1*c12 + 2*c1*c2 + d1*d12 + 2*d1*d2 + e1*e12 + 2*e1*e2 + f1*f12 + 2*f1*f2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2))))
    #m21
    EXPR2 <- expression((b12*b2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 + b1^2*c12*c2 + b12*b2*d1^2 + c12*c2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - c1*c12*d1*d2 + b1^2*d12*d2 + c1^2*d12*d2 + b12*b2*e1^2 + c12*c2*e1^2 + d12*d2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - c1*c12*e1*e2 - d1*d12*e1*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + b12*b2*f1^2 + c12*c2*f1^2 + d12*d2*f1^2 + e12*e2*f1^2 - b1*b2*f1*f12 - c1*c2*f1*f12 - d1*d2*f1*f12 - e1*e2*f1*f12 - a1*a2*(b1*b12 + c1*c12 + d1*d12 + e1*e12 + f1*f12) - b1*b12*f1*f2 - c1*c12*f1*f2 - d1*d12*f1*f2 - e1*e12*f1*f2 + b1^2*f12*f2 + c1^2*f12*f2 + d1^2*f12*f2 + e1^2*f12*f2 + a1^2*(b12*b2 + c12*c2 + d12*d2 + e12*e2 + f12*f2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2)))/(b2^2*c1^2 + b2^2*c1*c12 - 2*b1*b2*c1*c2 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1^2*c2^2 + b1*b12*c2^2 + b2^2*d1^2 + c2^2*d1^2 + b2^2*d1*d12 + c2^2*d1*d12 - 2*b1*b2*d1*d2 - b12*b2*d1*d2 - 2*c1*c2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1^2*d2^2 + b1*b12*d2^2 + c1^2*d2^2 + c1*c12*d2^2 + b2^2*e1^2 + c2^2*e1^2 + d2^2*e1^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 - 2*b1*b2*e1*e2 - b12*b2*e1*e2 - 2*c1*c2*e1*e2 - c12*c2*e1*e2 - 2*d1*d2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1^2*e2^2 + b1*b12*e2^2 + c1^2*e2^2 + c1*c12*e2^2 + d1^2*e2^2 + d1*d12*e2^2 + b2^2*f1^2 + c2^2*f1^2 + d2^2*f1^2 + e2^2*f1^2 + b2^2*f1*f12 + c2^2*f1*f12 + d2^2*f1*f12 + e2^2*f1*f12 + a2^2*(b1^2 + b1*b12 + c1^2 + c1*c12 + d1^2 + d1*d12 + e1^2 + e1*e12 + f1^2 + f1*f12) - 2*b1*b2*f1*f2 - b12*b2*f1*f2 - 2*c1*c2*f1*f2 - c12*c2*f1*f2 - 2*d1*d2*f1*f2 - d12*d2*f1*f2 - 2*e1*e2*f1*f2 - e12*e2*f1*f2 - b1*b2*f12*f2 - c1*c2*f12*f2 - d1*d2*f12*f2 - e1*e2*f12*f2 + b1^2*f2^2 + b1*b12*f2^2 + c1^2*f2^2 + c1*c12*f2^2 + d1^2*f2^2 + d1*d12*f2^2 + e1^2*f2^2 + e1*e12*f2^2 + a1^2*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2) + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2) + a1*(2*b1*b2 + b12*b2 + 2*c1*c2 + c12*c2 + 2*d1*d2 + d12*d2 + 2*e1*e2 + e12*e2 + 2*f1*f2 + f12*f2))))
  } else if (n_rows == 7) {
    ##m12
    EXPR1 <- expression((b2^2*c1*c12 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1*b12*c2^2 + b2^2*d1*d12 + c2^2*d1*d12 - b12*b2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1*b12*d2^2 + c1*c12*d2^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 - b12*b2*e1*e2 - c12*c2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1*b12*e2^2 + c1*c12*e2^2 + d1*d12*e2^2 + b2^2*f1*f12 + c2^2*f1*f12 + d2^2*f1*f12 + e2^2*f1*f12 - b12*b2*f1*f2 - c12*c2*f1*f2 - d12*d2*f1*f2 - e12*e2*f1*f2 - b1*b2*f12*f2 - c1*c2*f12*f2 - d1*d2*f12*f2 - e1*e2*f12*f2 + b1*b12*f2^2 + c1*c12*f2^2 + d1*d12*f2^2 + e1*e12*f2^2 + b2^2*g1*g12 + c2^2*g1*g12 + d2^2*g1*g12 + e2^2*g1*g12 + f2^2*g1*g12 + a2^2*(b1*b12 + c1*c12 + d1*d12 + e1*e12 + f1*f12 + g1*g12) - b12*b2*g1*g2 - c12*c2*g1*g2 - d12*d2*g1*g2 - e12*e2*g1*g2 - f12*f2*g1*g2 - b1*b2*g12*g2 - c1*c2*g12*g2 - d1*d2*g12*g2 - e1*e2*g12*g2 - f1*f2*g12*g2 + b1*b12*g2^2 + c1*c12*g2^2 + d1*d12*g2^2 + e1*e12*g2^2 + f1*f12*g2^2 + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2) + a1*(b12*b2 + c12*c2 + d12*d2 + e12*e2 + f12*f2 + g12*g2)))/(a1^2*b12*b2 + a1^2*b2^2 + b12*b2*c1^2 + b2^2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 - 2*b1*b2*c1*c2 + a1^2*c12*c2 + b1^2*c12*c2 + a1^2*c2^2 + b1^2*c2^2 + b12*b2*d1^2 + b2^2*d1^2 + c12*c2*d1^2 + c2^2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - 2*b1*b2*d1*d2 - c1*c12*d1*d2 - 2*c1*c2*d1*d2 + a1^2*d12*d2 + b1^2*d12*d2 + c1^2*d12*d2 + a1^2*d2^2 + b1^2*d2^2 + c1^2*d2^2 + b12*b2*e1^2 + b2^2*e1^2 + c12*c2*e1^2 + c2^2*e1^2 + d12*d2*e1^2 + d2^2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - 2*b1*b2*e1*e2 - c1*c12*e1*e2 - 2*c1*c2*e1*e2 - d1*d12*e1*e2 - 2*d1*d2*e1*e2 + a1^2*e12*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + a1^2*e2^2 + b1^2*e2^2 + c1^2*e2^2 + d1^2*e2^2 + b12*b2*f1^2 + b2^2*f1^2 + c12*c2*f1^2 + c2^2*f1^2 + d12*d2*f1^2 + d2^2*f1^2 + e12*e2*f1^2 + e2^2*f1^2 - b1*b2*f1*f12 - c1*c2*f1*f12 - d1*d2*f1*f12 - e1*e2*f1*f12 - b1*b12*f1*f2 - 2*b1*b2*f1*f2 - c1*c12*f1*f2 - 2*c1*c2*f1*f2 - d1*d12*f1*f2 - 2*d1*d2*f1*f2 - e1*e12*f1*f2 - 2*e1*e2*f1*f2 + a1^2*f12*f2 + b1^2*f12*f2 + c1^2*f12*f2 + d1^2*f12*f2 + e1^2*f12*f2 + a1^2*f2^2 + b1^2*f2^2 + c1^2*f2^2 + d1^2*f2^2 + e1^2*f2^2 + b12*b2*g1^2 + b2^2*g1^2 + c12*c2*g1^2 + c2^2*g1^2 + d12*d2*g1^2 + d2^2*g1^2 + e12*e2*g1^2 + e2^2*g1^2 + f12*f2*g1^2 + f2^2*g1^2 + a2^2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2) - b1*b2*g1*g12 - c1*c2*g1*g12 - d1*d2*g1*g12 - e1*e2*g1*g12 - f1*f2*g1*g12 - b1*b12*g1*g2 - 2*b1*b2*g1*g2 - c1*c12*g1*g2 - 2*c1*c2*g1*g2 - d1*d12*g1*g2 - 2*d1*d2*g1*g2 - e1*e12*g1*g2 - 2*e1*e2*g1*g2 - f1*f12*g1*g2 - 2*f1*f2*g1*g2 + a1^2*g12*g2 + b1^2*g12*g2 + c1^2*g12*g2 + d1^2*g12*g2 + e1^2*g12*g2 + f1^2*g12*g2 + a1^2*g2^2 + b1^2*g2^2 + c1^2*g2^2 + d1^2*g2^2 + e1^2*g2^2 + f1^2*g2^2 - a1*a2*(b1*b12 + 2*b1*b2 + c1*c12 + 2*c1*c2 + d1*d12 + 2*d1*d2 + e1*e12 + 2*e1*e2 + f1*f12 + 2*f1*f2 + g1*g12 + 2*g1*g2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2))))
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       ##m21
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     EXPR2 <- expression((b12*b2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 + b1^2*c12*c2 + b12*b2*d1^2 + c12*c2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - c1*c12*d1*d2 + b1^2*d12*d2 + c1^2*d12*d2 + b12*b2*e1^2 + c12*c2*e1^2 + d12*d2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - c1*c12*e1*e2 - d1*d12*e1*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + b12*b2*f1^2 + c12*c2*f1^2 + d12*d2*f1^2 + e12*e2*f1^2 - b1*b2*f1*f12 - c1*c2*f1*f12 - d1*d2*f1*f12 - e1*e2*f1*f12 - b1*b12*f1*f2 - c1*c12*f1*f2 - d1*d12*f1*f2 - e1*e12*f1*f2 + b1^2*f12*f2 + c1^2*f12*f2 + d1^2*f12*f2 + e1^2*f12*f2 + b12*b2*g1^2 + c12*c2*g1^2 + d12*d2*g1^2 + e12*e2*g1^2 + f12*f2*g1^2 - b1*b2*g1*g12 - c1*c2*g1*g12 - d1*d2*g1*g12 - e1*e2*g1*g12 - f1*f2*g1*g12 - a1*a2*(b1*b12 + c1*c12 + d1*d12 + e1*e12 + f1*f12 + g1*g12) - b1*b12*g1*g2 - c1*c12*g1*g2 - d1*d12*g1*g2 - e1*e12*g1*g2 - f1*f12*g1*g2 + b1^2*g12*g2 + c1^2*g12*g2 + d1^2*g12*g2 + e1^2*g12*g2 + f1^2*g12*g2 + a1^2*(b12*b2 + c12*c2 + d12*d2 + e12*e2 + f12*f2 + g12*g2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2)))/(b2^2*c1^2 + b2^2*c1*c12 - 2*b1*b2*c1*c2 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1^2*c2^2 + b1*b12*c2^2 + b2^2*d1^2 + c2^2*d1^2 + b2^2*d1*d12 + c2^2*d1*d12 - 2*b1*b2*d1*d2 - b12*b2*d1*d2 - 2*c1*c2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1^2*d2^2 + b1*b12*d2^2 + c1^2*d2^2 + c1*c12*d2^2 + b2^2*e1^2 + c2^2*e1^2 + d2^2*e1^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 - 2*b1*b2*e1*e2 - b12*b2*e1*e2 - 2*c1*c2*e1*e2 - c12*c2*e1*e2 - 2*d1*d2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1^2*e2^2 + b1*b12*e2^2 + c1^2*e2^2 + c1*c12*e2^2 + d1^2*e2^2 + d1*d12*e2^2 + b2^2*f1^2 + c2^2*f1^2 + d2^2*f1^2 + e2^2*f1^2 + b2^2*f1*f12 + c2^2*f1*f12 + d2^2*f1*f12 + e2^2*f1*f12 - 2*b1*b2*f1*f2 - b12*b2*f1*f2 - 2*c1*c2*f1*f2 - c12*c2*f1*f2 - 2*d1*d2*f1*f2 - d12*d2*f1*f2 - 2*e1*e2*f1*f2 - e12*e2*f1*f2 - b1*b2*f12*f2 - c1*c2*f12*f2 - d1*d2*f12*f2 - e1*e2*f12*f2 + b1^2*f2^2 + b1*b12*f2^2 + c1^2*f2^2 + c1*c12*f2^2 + d1^2*f2^2 + d1*d12*f2^2 + e1^2*f2^2 + e1*e12*f2^2 + b2^2*g1^2 + c2^2*g1^2 + d2^2*g1^2 + e2^2*g1^2 + f2^2*g1^2 + b2^2*g1*g12 + c2^2*g1*g12 + d2^2*g1*g12 + e2^2*g1*g12 + f2^2*g1*g12 + a2^2*(b1^2 + b1*b12 + c1^2 + c1*c12 + d1^2 + d1*d12 + e1^2 + e1*e12 + f1^2 + f1*f12 + g1^2 + g1*g12) - 2*b1*b2*g1*g2 - b12*b2*g1*g2 - 2*c1*c2*g1*g2 - c12*c2*g1*g2 - 2*d1*d2*g1*g2 - d12*d2*g1*g2 - 2*e1*e2*g1*g2 - e12*e2*g1*g2 - 2*f1*f2*g1*g2 - f12*f2*g1*g2 - b1*b2*g12*g2 - c1*c2*g12*g2 - d1*d2*g12*g2 - e1*e2*g12*g2 - f1*f2*g12*g2 + b1^2*g2^2 + b1*b12*g2^2 + c1^2*g2^2 + c1*c12*g2^2 + d1^2*g2^2 + d1*d12*g2^2 + e1^2*g2^2 + e1*e12*g2^2 + f1^2*g2^2 + f1*f12*g2^2 + a1^2*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2) + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2) + a1*(2*b1*b2 + b12*b2 + 2*c1*c2 + c12*c2 + 2*d1*d2 + d12*d2 + 2*e1*e2 + e12*e2 + 2*f1*f2 + f12*f2 + 2*g1*g2 + g12*g2))))
  } else if (n_rows == 8) {
    #m12
    EXPR1 <- expression((b2^2*c1*c12 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1*b12*c2^2 + b2^2*d1*d12 + c2^2*d1*d12 - b12*b2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1*b12*d2^2 + c1*c12*d2^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 - b12*b2*e1*e2 - c12*c2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1*b12*e2^2 + c1*c12*e2^2 + d1*d12*e2^2 + b2^2*f1*f12 + c2^2*f1*f12 + d2^2*f1*f12 + e2^2*f1*f12 - b12*b2*f1*f2 - c12*c2*f1*f2 - d12*d2*f1*f2 - e12*e2*f1*f2 - b1*b2*f12*f2 - c1*c2*f12*f2 - d1*d2*f12*f2 - e1*e2*f12*f2 + b1*b12*f2^2 + c1*c12*f2^2 + d1*d12*f2^2 + e1*e12*f2^2 + b2^2*g1*g12 + c2^2*g1*g12 + d2^2*g1*g12 + e2^2*g1*g12 + f2^2*g1*g12 - b12*b2*g1*g2 - c12*c2*g1*g2 - d12*d2*g1*g2 - e12*e2*g1*g2 - f12*f2*g1*g2 - b1*b2*g12*g2 - c1*c2*g12*g2 - d1*d2*g12*g2 - e1*e2*g12*g2 - f1*f2*g12*g2 + b1*b12*g2^2 + c1*c12*g2^2 + d1*d12*g2^2 + e1*e12*g2^2 + f1*f12*g2^2 + b2^2*h1*h12 + c2^2*h1*h12 + d2^2*h1*h12 + e2^2*h1*h12 + f2^2*h1*h12 + g2^2*h1*h12 + a2^2*(b1*b12 + c1*c12 + d1*d12 + e1*e12 + f1*f12 + g1*g12 + h1*h12) - b12*b2*h1*h2 - c12*c2*h1*h2 - d12*d2*h1*h2 - e12*e2*h1*h2 - f12*f2*h1*h2 - g12*g2*h1*h2 - b1*b2*h12*h2 - c1*c2*h12*h2 - d1*d2*h12*h2 - e1*e2*h12*h2 - f1*f2*h12*h2 - g1*g2*h12*h2 + b1*b12*h2^2 + c1*c12*h2^2 + d1*d12*h2^2 + e1*e12*h2^2 + f1*f12*h2^2 + g1*g12*h2^2 + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2) + a1*(b12*b2 + c12*c2 + d12*d2 + e12*e2 + f12*f2 + g12*g2 + h12*h2)))/(a1^2*b12*b2 + a1^2*b2^2 + b12*b2*c1^2 + b2^2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 - 2*b1*b2*c1*c2 + a1^2*c12*c2 + b1^2*c12*c2 + a1^2*c2^2 + b1^2*c2^2 + b12*b2*d1^2 + b2^2*d1^2 + c12*c2*d1^2 + c2^2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - 2*b1*b2*d1*d2 - c1*c12*d1*d2 - 2*c1*c2*d1*d2 + a1^2*d12*d2 + b1^2*d12*d2 + c1^2*d12*d2 + a1^2*d2^2 + b1^2*d2^2 + c1^2*d2^2 + b12*b2*e1^2 + b2^2*e1^2 + c12*c2*e1^2 + c2^2*e1^2 + d12*d2*e1^2 + d2^2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - 2*b1*b2*e1*e2 - c1*c12*e1*e2 - 2*c1*c2*e1*e2 - d1*d12*e1*e2 - 2*d1*d2*e1*e2 + a1^2*e12*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + a1^2*e2^2 + b1^2*e2^2 + c1^2*e2^2 + d1^2*e2^2 + b12*b2*f1^2 + b2^2*f1^2 + c12*c2*f1^2 + c2^2*f1^2 + d12*d2*f1^2 + d2^2*f1^2 + e12*e2*f1^2 + e2^2*f1^2 - b1*b2*f1*f12 - c1*c2*f1*f12 - d1*d2*f1*f12 - e1*e2*f1*f12 - b1*b12*f1*f2 - 2*b1*b2*f1*f2 - c1*c12*f1*f2 - 2*c1*c2*f1*f2 - d1*d12*f1*f2 - 2*d1*d2*f1*f2 - e1*e12*f1*f2 - 2*e1*e2*f1*f2 + a1^2*f12*f2 + b1^2*f12*f2 + c1^2*f12*f2 + d1^2*f12*f2 + e1^2*f12*f2 + a1^2*f2^2 + b1^2*f2^2 + c1^2*f2^2 + d1^2*f2^2 + e1^2*f2^2 + b12*b2*g1^2 + b2^2*g1^2 + c12*c2*g1^2 + c2^2*g1^2 + d12*d2*g1^2 + d2^2*g1^2 + e12*e2*g1^2 + e2^2*g1^2 + f12*f2*g1^2 + f2^2*g1^2 - b1*b2*g1*g12 - c1*c2*g1*g12 - d1*d2*g1*g12 - e1*e2*g1*g12 - f1*f2*g1*g12 - b1*b12*g1*g2 - 2*b1*b2*g1*g2 - c1*c12*g1*g2 - 2*c1*c2*g1*g2 - d1*d12*g1*g2 - 2*d1*d2*g1*g2 - e1*e12*g1*g2 - 2*e1*e2*g1*g2 - f1*f12*g1*g2 - 2*f1*f2*g1*g2 + a1^2*g12*g2 + b1^2*g12*g2 + c1^2*g12*g2 + d1^2*g12*g2 + e1^2*g12*g2 + f1^2*g12*g2 + a1^2*g2^2 + b1^2*g2^2 + c1^2*g2^2 + d1^2*g2^2 + e1^2*g2^2 + f1^2*g2^2 + b12*b2*h1^2 + b2^2*h1^2 + c12*c2*h1^2 + c2^2*h1^2 + d12*d2*h1^2 + d2^2*h1^2 + e12*e2*h1^2 + e2^2*h1^2 + f12*f2*h1^2 + f2^2*h1^2 + g12*g2*h1^2 + g2^2*h1^2 + a2^2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2) - b1*b2*h1*h12 - c1*c2*h1*h12 - d1*d2*h1*h12 - e1*e2*h1*h12 - f1*f2*h1*h12 - g1*g2*h1*h12 - b1*b12*h1*h2 - 2*b1*b2*h1*h2 - c1*c12*h1*h2 - 2*c1*c2*h1*h2 - d1*d12*h1*h2 - 2*d1*d2*h1*h2 - e1*e12*h1*h2 - 2*e1*e2*h1*h2 - f1*f12*h1*h2 - 2*f1*f2*h1*h2 - g1*g12*h1*h2 - 2*g1*g2*h1*h2 + a1^2*h12*h2 + b1^2*h12*h2 + c1^2*h12*h2 + d1^2*h12*h2 + e1^2*h12*h2 + f1^2*h12*h2 + g1^2*h12*h2 + a1^2*h2^2 + b1^2*h2^2 + c1^2*h2^2 + d1^2*h2^2 + e1^2*h2^2 + f1^2*h2^2 + g1^2*h2^2 - a1*a2*(b1*b12 + 2*b1*b2 + c1*c12 + 2*c1*c2 + d1*d12 + 2*d1*d2 + e1*e12 + 2*e1*e2 + f1*f12 + 2*f1*f2 + g1*g12 + 2*g1*g2 + h1*h12 + 2*h1*h2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2))))
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  #m21
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  EXPR2 <- expression((b12*b2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 + b1^2*c12*c2 + b12*b2*d1^2 + c12*c2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - c1*c12*d1*d2 + b1^2*d12*d2 + c1^2*d12*d2 + b12*b2*e1^2 + c12*c2*e1^2 + d12*d2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - c1*c12*e1*e2 - d1*d12*e1*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + b12*b2*f1^2 + c12*c2*f1^2 + d12*d2*f1^2 + e12*e2*f1^2 - b1*b2*f1*f12 - c1*c2*f1*f12 - d1*d2*f1*f12 - e1*e2*f1*f12 - b1*b12*f1*f2 - c1*c12*f1*f2 - d1*d12*f1*f2 - e1*e12*f1*f2 + b1^2*f12*f2 + c1^2*f12*f2 + d1^2*f12*f2 + e1^2*f12*f2 + b12*b2*g1^2 + c12*c2*g1^2 + d12*d2*g1^2 + e12*e2*g1^2 + f12*f2*g1^2 - b1*b2*g1*g12 - c1*c2*g1*g12 - d1*d2*g1*g12 - e1*e2*g1*g12 - f1*f2*g1*g12 - b1*b12*g1*g2 - c1*c12*g1*g2 - d1*d12*g1*g2 - e1*e12*g1*g2 - f1*f12*g1*g2 + b1^2*g12*g2 + c1^2*g12*g2 + d1^2*g12*g2 + e1^2*g12*g2 + f1^2*g12*g2 + b12*b2*h1^2 + c12*c2*h1^2 + d12*d2*h1^2 + e12*e2*h1^2 + f12*f2*h1^2 + g12*g2*h1^2 - b1*b2*h1*h12 - c1*c2*h1*h12 - d1*d2*h1*h12 - e1*e2*h1*h12 - f1*f2*h1*h12 - g1*g2*h1*h12 - a1*a2*(b1*b12 + c1*c12 + d1*d12 + e1*e12 + f1*f12 + g1*g12 + h1*h12) - b1*b12*h1*h2 - c1*c12*h1*h2 - d1*d12*h1*h2 - e1*e12*h1*h2 - f1*f12*h1*h2 - g1*g12*h1*h2 + b1^2*h12*h2 + c1^2*h12*h2 + d1^2*h12*h2 + e1^2*h12*h2 + f1^2*h12*h2 + g1^2*h12*h2 + a1^2*(b12*b2 + c12*c2 + d12*d2 + e12*e2 + f12*f2 + g12*g2 + h12*h2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2)))/(b2^2*c1^2 + b2^2*c1*c12 - 2*b1*b2*c1*c2 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1^2*c2^2 + b1*b12*c2^2 + b2^2*d1^2 + c2^2*d1^2 + b2^2*d1*d12 + c2^2*d1*d12 - 2*b1*b2*d1*d2 - b12*b2*d1*d2 - 2*c1*c2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1^2*d2^2 + b1*b12*d2^2 + c1^2*d2^2 + c1*c12*d2^2 + b2^2*e1^2 + c2^2*e1^2 + d2^2*e1^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 - 2*b1*b2*e1*e2 - b12*b2*e1*e2 - 2*c1*c2*e1*e2 - c12*c2*e1*e2 - 2*d1*d2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1^2*e2^2 + b1*b12*e2^2 + c1^2*e2^2 + c1*c12*e2^2 + d1^2*e2^2 + d1*d12*e2^2 + b2^2*f1^2 + c2^2*f1^2 + d2^2*f1^2 + e2^2*f1^2 + b2^2*f1*f12 + c2^2*f1*f12 + d2^2*f1*f12 + e2^2*f1*f12 - 2*b1*b2*f1*f2 - b12*b2*f1*f2 - 2*c1*c2*f1*f2 - c12*c2*f1*f2 - 2*d1*d2*f1*f2 - d12*d2*f1*f2 - 2*e1*e2*f1*f2 - e12*e2*f1*f2 - b1*b2*f12*f2 - c1*c2*f12*f2 - d1*d2*f12*f2 - e1*e2*f12*f2 + b1^2*f2^2 + b1*b12*f2^2 + c1^2*f2^2 + c1*c12*f2^2 + d1^2*f2^2 + d1*d12*f2^2 + e1^2*f2^2 + e1*e12*f2^2 + b2^2*g1^2 + c2^2*g1^2 + d2^2*g1^2 + e2^2*g1^2 + f2^2*g1^2 + b2^2*g1*g12 + c2^2*g1*g12 + d2^2*g1*g12 + e2^2*g1*g12 + f2^2*g1*g12 - 2*b1*b2*g1*g2 - b12*b2*g1*g2 - 2*c1*c2*g1*g2 - c12*c2*g1*g2 - 2*d1*d2*g1*g2 - d12*d2*g1*g2 - 2*e1*e2*g1*g2 - e12*e2*g1*g2 - 2*f1*f2*g1*g2 - f12*f2*g1*g2 - b1*b2*g12*g2 - c1*c2*g12*g2 - d1*d2*g12*g2 - e1*e2*g12*g2 - f1*f2*g12*g2 + b1^2*g2^2 + b1*b12*g2^2 + c1^2*g2^2 + c1*c12*g2^2 + d1^2*g2^2 + d1*d12*g2^2 + e1^2*g2^2 + e1*e12*g2^2 + f1^2*g2^2 + f1*f12*g2^2 + b2^2*h1^2 + c2^2*h1^2 + d2^2*h1^2 + e2^2*h1^2 + f2^2*h1^2 + g2^2*h1^2 + b2^2*h1*h12 + c2^2*h1*h12 + d2^2*h1*h12 + e2^2*h1*h12 + f2^2*h1*h12 + g2^2*h1*h12 + a2^2*(b1^2 + b1*b12 + c1^2 + c1*c12 + d1^2 + d1*d12 + e1^2 + e1*e12 + f1^2 + f1*f12 + g1^2 + g1*g12 + h1^2 + h1*h12) - 2*b1*b2*h1*h2 - b12*b2*h1*h2 - 2*c1*c2*h1*h2 - c12*c2*h1*h2 - 2*d1*d2*h1*h2 - d12*d2*h1*h2 - 2*e1*e2*h1*h2 - e12*e2*h1*h2 - 2*f1*f2*h1*h2 - f12*f2*h1*h2 - 2*g1*g2*h1*h2 - g12*g2*h1*h2 - b1*b2*h12*h2 - c1*c2*h12*h2 - d1*d2*h12*h2 - e1*e2*h12*h2 - f1*f2*h12*h2 - g1*g2*h12*h2 + b1^2*h2^2 + b1*b12*h2^2 + c1^2*h2^2 + c1*c12*h2^2 + d1^2*h2^2 + d1*d12*h2^2 + e1^2*h2^2 + e1*e12*h2^2 + f1^2*h2^2 + f1*f12*h2^2 + g1^2*h2^2 + g1*g12*h2^2 + a1^2*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2) + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2) + a1*(2*b1*b2 + b12*b2 + 2*c1*c2 + c12*c2 + 2*d1*d2 + d12*d2 + 2*e1*e2 + e12*e2 + 2*f1*f2 + f12*f2 + 2*g1*g2 + g12*g2 + 2*h1*h2 + h12*h2))))
  } else if (n_rows == 9) {
    #m12
    EXPR1 <- expression((b2^2*c1*c12 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1*b12*c2^2 + b2^2*d1*d12 + c2^2*d1*d12 - b12*b2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1*b12*d2^2 + c1*c12*d2^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 - b12*b2*e1*e2 - c12*c2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1*b12*e2^2 + c1*c12*e2^2 + d1*d12*e2^2 + b2^2*f1*f12 + c2^2*f1*f12 + d2^2*f1*f12 + e2^2*f1*f12 - b12*b2*f1*f2 - c12*c2*f1*f2 - d12*d2*f1*f2 - e12*e2*f1*f2 - b1*b2*f12*f2 - c1*c2*f12*f2 - d1*d2*f12*f2 - e1*e2*f12*f2 + b1*b12*f2^2 + c1*c12*f2^2 + d1*d12*f2^2 + e1*e12*f2^2 + b2^2*g1*g12 + c2^2*g1*g12 + d2^2*g1*g12 + e2^2*g1*g12 + f2^2*g1*g12 - b12*b2*g1*g2 - c12*c2*g1*g2 - d12*d2*g1*g2 - e12*e2*g1*g2 - f12*f2*g1*g2 - b1*b2*g12*g2 - c1*c2*g12*g2 - d1*d2*g12*g2 - e1*e2*g12*g2 - f1*f2*g12*g2 + b1*b12*g2^2 + c1*c12*g2^2 + d1*d12*g2^2 + e1*e12*g2^2 + f1*f12*g2^2 + b2^2*h1*h12 + c2^2*h1*h12 + d2^2*h1*h12 + e2^2*h1*h12 + f2^2*h1*h12 + g2^2*h1*h12 - b12*b2*h1*h2 - c12*c2*h1*h2 - d12*d2*h1*h2 - e12*e2*h1*h2 - f12*f2*h1*h2 - g12*g2*h1*h2 - b1*b2*h12*h2 - c1*c2*h12*h2 - d1*d2*h12*h2 - e1*e2*h12*h2 - f1*f2*h12*h2 - g1*g2*h12*h2 + b1*b12*h2^2 + c1*c12*h2^2 + d1*d12*h2^2 + e1*e12*h2^2 + f1*f12*h2^2 + g1*g12*h2^2 + b2^2*i1*i12 + c2^2*i1*i12 + d2^2*i1*i12 + e2^2*i1*i12 + f2^2*i1*i12 + g2^2*i1*i12 + h2^2*i1*i12 + a2^2*(b1*b12 + c1*c12 + d1*d12 + e1*e12 + f1*f12 + g1*g12 + h1*h12 + i1*i12) - b12*b2*i1*i2 - c12*c2*i1*i2 - d12*d2*i1*i2 - e12*e2*i1*i2 - f12*f2*i1*i2 - g12*g2*i1*i2 - h12*h2*i1*i2 - b1*b2*i12*i2 - c1*c2*i12*i2 - d1*d2*i12*i2 - e1*e2*i12*i2 - f1*f2*i12*i2 - g1*g2*i12*i2 - h1*h2*i12*i2 + b1*b12*i2^2 + c1*c12*i2^2 + d1*d12*i2^2 + e1*e12*i2^2 + f1*f12*i2^2 + g1*g12*i2^2 + h1*h12*i2^2 + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2 + i2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2) + a1*(b12*b2 + c12*c2 + d12*d2 + e12*e2 + f12*f2 + g12*g2 + h12*h2 + i12*i2)))/(a1^2*b12*b2 + a1^2*b2^2 + b12*b2*c1^2 + b2^2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 - 2*b1*b2*c1*c2 + a1^2*c12*c2 + b1^2*c12*c2 + a1^2*c2^2 + b1^2*c2^2 + b12*b2*d1^2 + b2^2*d1^2 + c12*c2*d1^2 + c2^2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - 2*b1*b2*d1*d2 - c1*c12*d1*d2 - 2*c1*c2*d1*d2 + a1^2*d12*d2 + b1^2*d12*d2 + c1^2*d12*d2 + a1^2*d2^2 + b1^2*d2^2 + c1^2*d2^2 + b12*b2*e1^2 + b2^2*e1^2 + c12*c2*e1^2 + c2^2*e1^2 + d12*d2*e1^2 + d2^2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - 2*b1*b2*e1*e2 - c1*c12*e1*e2 - 2*c1*c2*e1*e2 - d1*d12*e1*e2 - 2*d1*d2*e1*e2 + a1^2*e12*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + a1^2*e2^2 + b1^2*e2^2 + c1^2*e2^2 + d1^2*e2^2 + b12*b2*f1^2 + b2^2*f1^2 + c12*c2*f1^2 + c2^2*f1^2 + d12*d2*f1^2 + d2^2*f1^2 + e12*e2*f1^2 + e2^2*f1^2 - b1*b2*f1*f12 - c1*c2*f1*f12 - d1*d2*f1*f12 - e1*e2*f1*f12 - b1*b12*f1*f2 - 2*b1*b2*f1*f2 - c1*c12*f1*f2 - 2*c1*c2*f1*f2 - d1*d12*f1*f2 - 2*d1*d2*f1*f2 - e1*e12*f1*f2 - 2*e1*e2*f1*f2 + a1^2*f12*f2 + b1^2*f12*f2 + c1^2*f12*f2 + d1^2*f12*f2 + e1^2*f12*f2 + a1^2*f2^2 + b1^2*f2^2 + c1^2*f2^2 + d1^2*f2^2 + e1^2*f2^2 + b12*b2*g1^2 + b2^2*g1^2 + c12*c2*g1^2 + c2^2*g1^2 + d12*d2*g1^2 + d2^2*g1^2 + e12*e2*g1^2 + e2^2*g1^2 + f12*f2*g1^2 + f2^2*g1^2 - b1*b2*g1*g12 - c1*c2*g1*g12 - d1*d2*g1*g12 - e1*e2*g1*g12 - f1*f2*g1*g12 - b1*b12*g1*g2 - 2*b1*b2*g1*g2 - c1*c12*g1*g2 - 2*c1*c2*g1*g2 - d1*d12*g1*g2 - 2*d1*d2*g1*g2 - e1*e12*g1*g2 - 2*e1*e2*g1*g2 - f1*f12*g1*g2 - 2*f1*f2*g1*g2 + a1^2*g12*g2 + b1^2*g12*g2 + c1^2*g12*g2 + d1^2*g12*g2 + e1^2*g12*g2 + f1^2*g12*g2 + a1^2*g2^2 + b1^2*g2^2 + c1^2*g2^2 + d1^2*g2^2 + e1^2*g2^2 + f1^2*g2^2 + b12*b2*h1^2 + b2^2*h1^2 + c12*c2*h1^2 + c2^2*h1^2 + d12*d2*h1^2 + d2^2*h1^2 + e12*e2*h1^2 + e2^2*h1^2 + f12*f2*h1^2 + f2^2*h1^2 + g12*g2*h1^2 + g2^2*h1^2 - b1*b2*h1*h12 - c1*c2*h1*h12 - d1*d2*h1*h12 - e1*e2*h1*h12 - f1*f2*h1*h12 - g1*g2*h1*h12 - b1*b12*h1*h2 - 2*b1*b2*h1*h2 - c1*c12*h1*h2 - 2*c1*c2*h1*h2 - d1*d12*h1*h2 - 2*d1*d2*h1*h2 - e1*e12*h1*h2 - 2*e1*e2*h1*h2 - f1*f12*h1*h2 - 2*f1*f2*h1*h2 - g1*g12*h1*h2 - 2*g1*g2*h1*h2 + a1^2*h12*h2 + b1^2*h12*h2 + c1^2*h12*h2 + d1^2*h12*h2 + e1^2*h12*h2 + f1^2*h12*h2 + g1^2*h12*h2 + a1^2*h2^2 + b1^2*h2^2 + c1^2*h2^2 + d1^2*h2^2 + e1^2*h2^2 + f1^2*h2^2 + g1^2*h2^2 + b12*b2*i1^2 + b2^2*i1^2 + c12*c2*i1^2 + c2^2*i1^2 + d12*d2*i1^2 + d2^2*i1^2 + e12*e2*i1^2 + e2^2*i1^2 + f12*f2*i1^2 + f2^2*i1^2 + g12*g2*i1^2 + g2^2*i1^2 + h12*h2*i1^2 + h2^2*i1^2 + a2^2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2 + i1^2) - b1*b2*i1*i12 - c1*c2*i1*i12 - d1*d2*i1*i12 - e1*e2*i1*i12 - f1*f2*i1*i12 - g1*g2*i1*i12 - h1*h2*i1*i12 - b1*b12*i1*i2 - 2*b1*b2*i1*i2 - c1*c12*i1*i2 - 2*c1*c2*i1*i2 - d1*d12*i1*i2 - 2*d1*d2*i1*i2 - e1*e12*i1*i2 - 2*e1*e2*i1*i2 - f1*f12*i1*i2 - 2*f1*f2*i1*i2 - g1*g12*i1*i2 - 2*g1*g2*i1*i2 - h1*h12*i1*i2 - 2*h1*h2*i1*i2 + a1^2*i12*i2 + b1^2*i12*i2 + c1^2*i12*i2 + d1^2*i12*i2 + e1^2*i12*i2 + f1^2*i12*i2 + g1^2*i12*i2 + h1^2*i12*i2 + a1^2*i2^2 + b1^2*i2^2 + c1^2*i2^2 + d1^2*i2^2 + e1^2*i2^2 + f1^2*i2^2 + g1^2*i2^2 + h1^2*i2^2 - a1*a2*(b1*b12 + 2*b1*b2 + c1*c12 + 2*c1*c2 + d1*d12 + 2*d1*d2 + e1*e12 + 2*e1*e2 + f1*f12 + 2*f1*f2 + g1*g12 + 2*g1*g2 + h1*h12 + 2*h1*h2 + i1*i12 + 2*i1*i2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2 + i1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2))))
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           #m21
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         EXPR2 <- expression((b12*b2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 + b1^2*c12*c2 + b12*b2*d1^2 + c12*c2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - c1*c12*d1*d2 + b1^2*d12*d2 + c1^2*d12*d2 + b12*b2*e1^2 + c12*c2*e1^2 + d12*d2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - c1*c12*e1*e2 - d1*d12*e1*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + b12*b2*f1^2 + c12*c2*f1^2 + d12*d2*f1^2 + e12*e2*f1^2 - b1*b2*f1*f12 - c1*c2*f1*f12 - d1*d2*f1*f12 - e1*e2*f1*f12 - b1*b12*f1*f2 - c1*c12*f1*f2 - d1*d12*f1*f2 - e1*e12*f1*f2 + b1^2*f12*f2 + c1^2*f12*f2 + d1^2*f12*f2 + e1^2*f12*f2 + b12*b2*g1^2 + c12*c2*g1^2 + d12*d2*g1^2 + e12*e2*g1^2 + f12*f2*g1^2 - b1*b2*g1*g12 - c1*c2*g1*g12 - d1*d2*g1*g12 - e1*e2*g1*g12 - f1*f2*g1*g12 - b1*b12*g1*g2 - c1*c12*g1*g2 - d1*d12*g1*g2 - e1*e12*g1*g2 - f1*f12*g1*g2 + b1^2*g12*g2 + c1^2*g12*g2 + d1^2*g12*g2 + e1^2*g12*g2 + f1^2*g12*g2 + b12*b2*h1^2 + c12*c2*h1^2 + d12*d2*h1^2 + e12*e2*h1^2 + f12*f2*h1^2 + g12*g2*h1^2 - b1*b2*h1*h12 - c1*c2*h1*h12 - d1*d2*h1*h12 - e1*e2*h1*h12 - f1*f2*h1*h12 - g1*g2*h1*h12 - b1*b12*h1*h2 - c1*c12*h1*h2 - d1*d12*h1*h2 - e1*e12*h1*h2 - f1*f12*h1*h2 - g1*g12*h1*h2 + b1^2*h12*h2 + c1^2*h12*h2 + d1^2*h12*h2 + e1^2*h12*h2 + f1^2*h12*h2 + g1^2*h12*h2 + b12*b2*i1^2 + c12*c2*i1^2 + d12*d2*i1^2 + e12*e2*i1^2 + f12*f2*i1^2 + g12*g2*i1^2 + h12*h2*i1^2 - b1*b2*i1*i12 - c1*c2*i1*i12 - d1*d2*i1*i12 - e1*e2*i1*i12 - f1*f2*i1*i12 - g1*g2*i1*i12 - h1*h2*i1*i12 - a1*a2*(b1*b12 + c1*c12 + d1*d12 + e1*e12 + f1*f12 + g1*g12 + h1*h12 + i1*i12) - b1*b12*i1*i2 - c1*c12*i1*i2 - d1*d12*i1*i2 - e1*e12*i1*i2 - f1*f12*i1*i2 - g1*g12*i1*i2 - h1*h12*i1*i2 + b1^2*i12*i2 + c1^2*i12*i2 + d1^2*i12*i2 + e1^2*i12*i2 + f1^2*i12*i2 + g1^2*i12*i2 + h1^2*i12*i2 + a1^2*(b12*b2 + c12*c2 + d12*d2 + e12*e2 + f12*f2 + g12*g2 + h12*h2 + i12*i2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2 + i1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2)))/(b2^2*c1^2 + b2^2*c1*c12 - 2*b1*b2*c1*c2 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1^2*c2^2 + b1*b12*c2^2 + b2^2*d1^2 + c2^2*d1^2 + b2^2*d1*d12 + c2^2*d1*d12 - 2*b1*b2*d1*d2 - b12*b2*d1*d2 - 2*c1*c2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1^2*d2^2 + b1*b12*d2^2 + c1^2*d2^2 + c1*c12*d2^2 + b2^2*e1^2 + c2^2*e1^2 + d2^2*e1^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 - 2*b1*b2*e1*e2 - b12*b2*e1*e2 - 2*c1*c2*e1*e2 - c12*c2*e1*e2 - 2*d1*d2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1^2*e2^2 + b1*b12*e2^2 + c1^2*e2^2 + c1*c12*e2^2 + d1^2*e2^2 + d1*d12*e2^2 + b2^2*f1^2 + c2^2*f1^2 + d2^2*f1^2 + e2^2*f1^2 + b2^2*f1*f12 + c2^2*f1*f12 + d2^2*f1*f12 + e2^2*f1*f12 - 2*b1*b2*f1*f2 - b12*b2*f1*f2 - 2*c1*c2*f1*f2 - c12*c2*f1*f2 - 2*d1*d2*f1*f2 - d12*d2*f1*f2 - 2*e1*e2*f1*f2 - e12*e2*f1*f2 - b1*b2*f12*f2 - c1*c2*f12*f2 - d1*d2*f12*f2 - e1*e2*f12*f2 + b1^2*f2^2 + b1*b12*f2^2 + c1^2*f2^2 + c1*c12*f2^2 + d1^2*f2^2 + d1*d12*f2^2 + e1^2*f2^2 + e1*e12*f2^2 + b2^2*g1^2 + c2^2*g1^2 + d2^2*g1^2 + e2^2*g1^2 + f2^2*g1^2 + b2^2*g1*g12 + c2^2*g1*g12 + d2^2*g1*g12 + e2^2*g1*g12 + f2^2*g1*g12 - 2*b1*b2*g1*g2 - b12*b2*g1*g2 - 2*c1*c2*g1*g2 - c12*c2*g1*g2 - 2*d1*d2*g1*g2 - d12*d2*g1*g2 - 2*e1*e2*g1*g2 - e12*e2*g1*g2 - 2*f1*f2*g1*g2 - f12*f2*g1*g2 - b1*b2*g12*g2 - c1*c2*g12*g2 - d1*d2*g12*g2 - e1*e2*g12*g2 - f1*f2*g12*g2 + b1^2*g2^2 + b1*b12*g2^2 + c1^2*g2^2 + c1*c12*g2^2 + d1^2*g2^2 + d1*d12*g2^2 + e1^2*g2^2 + e1*e12*g2^2 + f1^2*g2^2 + f1*f12*g2^2 + b2^2*h1^2 + c2^2*h1^2 + d2^2*h1^2 + e2^2*h1^2 + f2^2*h1^2 + g2^2*h1^2 + b2^2*h1*h12 + c2^2*h1*h12 + d2^2*h1*h12 + e2^2*h1*h12 + f2^2*h1*h12 + g2^2*h1*h12 - 2*b1*b2*h1*h2 - b12*b2*h1*h2 - 2*c1*c2*h1*h2 - c12*c2*h1*h2 - 2*d1*d2*h1*h2 - d12*d2*h1*h2 - 2*e1*e2*h1*h2 - e12*e2*h1*h2 - 2*f1*f2*h1*h2 - f12*f2*h1*h2 - 2*g1*g2*h1*h2 - g12*g2*h1*h2 - b1*b2*h12*h2 - c1*c2*h12*h2 - d1*d2*h12*h2 - e1*e2*h12*h2 - f1*f2*h12*h2 - g1*g2*h12*h2 + b1^2*h2^2 + b1*b12*h2^2 + c1^2*h2^2 + c1*c12*h2^2 + d1^2*h2^2 + d1*d12*h2^2 + e1^2*h2^2 + e1*e12*h2^2 + f1^2*h2^2 + f1*f12*h2^2 + g1^2*h2^2 + g1*g12*h2^2 + b2^2*i1^2 + c2^2*i1^2 + d2^2*i1^2 + e2^2*i1^2 + f2^2*i1^2 + g2^2*i1^2 + h2^2*i1^2 + b2^2*i1*i12 + c2^2*i1*i12 + d2^2*i1*i12 + e2^2*i1*i12 + f2^2*i1*i12 + g2^2*i1*i12 + h2^2*i1*i12 + a2^2*(b1^2 + b1*b12 + c1^2 + c1*c12 + d1^2 + d1*d12 + e1^2 + e1*e12 + f1^2 + f1*f12 + g1^2 + g1*g12 + h1^2 + h1*h12 + i1^2 + i1*i12) - 2*b1*b2*i1*i2 - b12*b2*i1*i2 - 2*c1*c2*i1*i2 - c12*c2*i1*i2 - 2*d1*d2*i1*i2 - d12*d2*i1*i2 - 2*e1*e2*i1*i2 - e12*e2*i1*i2 - 2*f1*f2*i1*i2 - f12*f2*i1*i2 - 2*g1*g2*i1*i2 - g12*g2*i1*i2 - 2*h1*h2*i1*i2 - h12*h2*i1*i2 - b1*b2*i12*i2 - c1*c2*i12*i2 - d1*d2*i12*i2 - e1*e2*i12*i2 - f1*f2*i12*i2 - g1*g2*i12*i2 - h1*h2*i12*i2 + b1^2*i2^2 + b1*b12*i2^2 + c1^2*i2^2 + c1*c12*i2^2 + d1^2*i2^2 + d1*d12*i2^2 + e1^2*i2^2 + e1*e12*i2^2 + f1^2*i2^2 + f1*f12*i2^2 + g1^2*i2^2 + g1*g12*i2^2 + h1^2*i2^2 + h1*h12*i2^2 + a1^2*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2 + i2^2) + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2 + i2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2) + a1*(2*b1*b2 + b12*b2 + 2*c1*c2 + c12*c2 + 2*d1*d2 + d12*d2 + 2*e1*e2 + e12*e2 + 2*f1*f2 + f12*f2 + 2*g1*g2 + g12*g2 + 2*h1*h2 + h12*h2 + 2*i1*i2 + i12*i2))))
  } else if (n_rows == 10) {
    #m12
    EXPR1 <- expression((b2^2*c1*c12 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1*b12*c2^2 + b2^2*d1*d12 + c2^2*d1*d12 - b12*b2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1*b12*d2^2 + c1*c12*d2^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 - b12*b2*e1*e2 - c12*c2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1*b12*e2^2 + c1*c12*e2^2 + d1*d12*e2^2 + b2^2*f1*f12 + c2^2*f1*f12 + d2^2*f1*f12 + e2^2*f1*f12 - b12*b2*f1*f2 - c12*c2*f1*f2 - d12*d2*f1*f2 - e12*e2*f1*f2 - b1*b2*f12*f2 - c1*c2*f12*f2 - d1*d2*f12*f2 - e1*e2*f12*f2 + b1*b12*f2^2 + c1*c12*f2^2 + d1*d12*f2^2 + e1*e12*f2^2 + b2^2*g1*g12 + c2^2*g1*g12 + d2^2*g1*g12 + e2^2*g1*g12 + f2^2*g1*g12 - b12*b2*g1*g2 - c12*c2*g1*g2 - d12*d2*g1*g2 - e12*e2*g1*g2 - f12*f2*g1*g2 - b1*b2*g12*g2 - c1*c2*g12*g2 - d1*d2*g12*g2 - e1*e2*g12*g2 - f1*f2*g12*g2 + b1*b12*g2^2 + c1*c12*g2^2 + d1*d12*g2^2 + e1*e12*g2^2 + f1*f12*g2^2 + b2^2*h1*h12 + c2^2*h1*h12 + d2^2*h1*h12 + e2^2*h1*h12 + f2^2*h1*h12 + g2^2*h1*h12 - b12*b2*h1*h2 - c12*c2*h1*h2 - d12*d2*h1*h2 - e12*e2*h1*h2 - f12*f2*h1*h2 - g12*g2*h1*h2 - b1*b2*h12*h2 - c1*c2*h12*h2 - d1*d2*h12*h2 - e1*e2*h12*h2 - f1*f2*h12*h2 - g1*g2*h12*h2 + b1*b12*h2^2 + c1*c12*h2^2 + d1*d12*h2^2 + e1*e12*h2^2 + f1*f12*h2^2 + g1*g12*h2^2 + b2^2*i1*i12 + c2^2*i1*i12 + d2^2*i1*i12 + e2^2*i1*i12 + f2^2*i1*i12 + g2^2*i1*i12 + h2^2*i1*i12 - b12*b2*i1*i2 - c12*c2*i1*i2 - d12*d2*i1*i2 - e12*e2*i1*i2 - f12*f2*i1*i2 - g12*g2*i1*i2 - h12*h2*i1*i2 - b1*b2*i12*i2 - c1*c2*i12*i2 - d1*d2*i12*i2 - e1*e2*i12*i2 - f1*f2*i12*i2 - g1*g2*i12*i2 - h1*h2*i12*i2 + b1*b12*i2^2 + c1*c12*i2^2 + d1*d12*i2^2 + e1*e12*i2^2 + f1*f12*i2^2 + g1*g12*i2^2 + h1*h12*i2^2 + b2^2*j1*j12 + c2^2*j1*j12 + d2^2*j1*j12 + e2^2*j1*j12 + f2^2*j1*j12 + g2^2*j1*j12 + h2^2*j1*j12 + i2^2*j1*j12 + a2^2*(b1*b12 + c1*c12 + d1*d12 + e1*e12 + f1*f12 + g1*g12 + h1*h12 + i1*i12 + j1*j12) - b12*b2*j1*j2 - c12*c2*j1*j2 - d12*d2*j1*j2 - e12*e2*j1*j2 - f12*f2*j1*j2 - g12*g2*j1*j2 - h12*h2*j1*j2 - i12*i2*j1*j2 - b1*b2*j12*j2 - c1*c2*j12*j2 - d1*d2*j12*j2 - e1*e2*j12*j2 - f1*f2*j12*j2 - g1*g2*j12*j2 - h1*h2*j12*j2 - i1*i2*j12*j2 + b1*b12*j2^2 + c1*c12*j2^2 + d1*d12*j2^2 + e1*e12*j2^2 + f1*f12*j2^2 + g1*g12*j2^2 + h1*h12*j2^2 + i1*i12*j2^2 + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2 + i2^2 + j2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2 + j1*j2) + a1*(b12*b2 + c12*c2 + d12*d2 + e12*e2 + f12*f2 + g12*g2 + h12*h2 + i12*i2 + j12*j2)))/(a1^2*b12*b2 + a1^2*b2^2 + b12*b2*c1^2 + b2^2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 - 2*b1*b2*c1*c2 + a1^2*c12*c2 + b1^2*c12*c2 + a1^2*c2^2 + b1^2*c2^2 + b12*b2*d1^2 + b2^2*d1^2 + c12*c2*d1^2 + c2^2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - 2*b1*b2*d1*d2 - c1*c12*d1*d2 - 2*c1*c2*d1*d2 + a1^2*d12*d2 + b1^2*d12*d2 + c1^2*d12*d2 + a1^2*d2^2 + b1^2*d2^2 + c1^2*d2^2 + b12*b2*e1^2 + b2^2*e1^2 + c12*c2*e1^2 + c2^2*e1^2 + d12*d2*e1^2 + d2^2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - 2*b1*b2*e1*e2 - c1*c12*e1*e2 - 2*c1*c2*e1*e2 - d1*d12*e1*e2 - 2*d1*d2*e1*e2 + a1^2*e12*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + a1^2*e2^2 + b1^2*e2^2 + c1^2*e2^2 + d1^2*e2^2 + b12*b2*f1^2 + b2^2*f1^2 + c12*c2*f1^2 + c2^2*f1^2 + d12*d2*f1^2 + d2^2*f1^2 + e12*e2*f1^2 + e2^2*f1^2 - b1*b2*f1*f12 - c1*c2*f1*f12 - d1*d2*f1*f12 - e1*e2*f1*f12 - b1*b12*f1*f2 - 2*b1*b2*f1*f2 - c1*c12*f1*f2 - 2*c1*c2*f1*f2 - d1*d12*f1*f2 - 2*d1*d2*f1*f2 - e1*e12*f1*f2 - 2*e1*e2*f1*f2 + a1^2*f12*f2 + b1^2*f12*f2 + c1^2*f12*f2 + d1^2*f12*f2 + e1^2*f12*f2 + a1^2*f2^2 + b1^2*f2^2 + c1^2*f2^2 + d1^2*f2^2 + e1^2*f2^2 + b12*b2*g1^2 + b2^2*g1^2 + c12*c2*g1^2 + c2^2*g1^2 + d12*d2*g1^2 + d2^2*g1^2 + e12*e2*g1^2 + e2^2*g1^2 + f12*f2*g1^2 + f2^2*g1^2 - b1*b2*g1*g12 - c1*c2*g1*g12 - d1*d2*g1*g12 - e1*e2*g1*g12 - f1*f2*g1*g12 - b1*b12*g1*g2 - 2*b1*b2*g1*g2 - c1*c12*g1*g2 - 2*c1*c2*g1*g2 - d1*d12*g1*g2 - 2*d1*d2*g1*g2 - e1*e12*g1*g2 - 2*e1*e2*g1*g2 - f1*f12*g1*g2 - 2*f1*f2*g1*g2 + a1^2*g12*g2 + b1^2*g12*g2 + c1^2*g12*g2 + d1^2*g12*g2 + e1^2*g12*g2 + f1^2*g12*g2 + a1^2*g2^2 + b1^2*g2^2 + c1^2*g2^2 + d1^2*g2^2 + e1^2*g2^2 + f1^2*g2^2 + b12*b2*h1^2 + b2^2*h1^2 + c12*c2*h1^2 + c2^2*h1^2 + d12*d2*h1^2 + d2^2*h1^2 + e12*e2*h1^2 + e2^2*h1^2 + f12*f2*h1^2 + f2^2*h1^2 + g12*g2*h1^2 + g2^2*h1^2 - b1*b2*h1*h12 - c1*c2*h1*h12 - d1*d2*h1*h12 - e1*e2*h1*h12 - f1*f2*h1*h12 - g1*g2*h1*h12 - b1*b12*h1*h2 - 2*b1*b2*h1*h2 - c1*c12*h1*h2 - 2*c1*c2*h1*h2 - d1*d12*h1*h2 - 2*d1*d2*h1*h2 - e1*e12*h1*h2 - 2*e1*e2*h1*h2 - f1*f12*h1*h2 - 2*f1*f2*h1*h2 - g1*g12*h1*h2 - 2*g1*g2*h1*h2 + a1^2*h12*h2 + b1^2*h12*h2 + c1^2*h12*h2 + d1^2*h12*h2 + e1^2*h12*h2 + f1^2*h12*h2 + g1^2*h12*h2 + a1^2*h2^2 + b1^2*h2^2 + c1^2*h2^2 + d1^2*h2^2 + e1^2*h2^2 + f1^2*h2^2 + g1^2*h2^2 + b12*b2*i1^2 + b2^2*i1^2 + c12*c2*i1^2 + c2^2*i1^2 + d12*d2*i1^2 + d2^2*i1^2 + e12*e2*i1^2 + e2^2*i1^2 + f12*f2*i1^2 + f2^2*i1^2 + g12*g2*i1^2 + g2^2*i1^2 + h12*h2*i1^2 + h2^2*i1^2 - b1*b2*i1*i12 - c1*c2*i1*i12 - d1*d2*i1*i12 - e1*e2*i1*i12 - f1*f2*i1*i12 - g1*g2*i1*i12 - h1*h2*i1*i12 - b1*b12*i1*i2 - 2*b1*b2*i1*i2 - c1*c12*i1*i2 - 2*c1*c2*i1*i2 - d1*d12*i1*i2 - 2*d1*d2*i1*i2 - e1*e12*i1*i2 - 2*e1*e2*i1*i2 - f1*f12*i1*i2 - 2*f1*f2*i1*i2 - g1*g12*i1*i2 - 2*g1*g2*i1*i2 - h1*h12*i1*i2 - 2*h1*h2*i1*i2 + a1^2*i12*i2 + b1^2*i12*i2 + c1^2*i12*i2 + d1^2*i12*i2 + e1^2*i12*i2 + f1^2*i12*i2 + g1^2*i12*i2 + h1^2*i12*i2 + a1^2*i2^2 + b1^2*i2^2 + c1^2*i2^2 + d1^2*i2^2 + e1^2*i2^2 + f1^2*i2^2 + g1^2*i2^2 + h1^2*i2^2 + b12*b2*j1^2 + b2^2*j1^2 + c12*c2*j1^2 + c2^2*j1^2 + d12*d2*j1^2 + d2^2*j1^2 + e12*e2*j1^2 + e2^2*j1^2 + f12*f2*j1^2 + f2^2*j1^2 + g12*g2*j1^2 + g2^2*j1^2 + h12*h2*j1^2 + h2^2*j1^2 + i12*i2*j1^2 + i2^2*j1^2 + a2^2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2 + i1^2 + j1^2) - b1*b2*j1*j12 - c1*c2*j1*j12 - d1*d2*j1*j12 - e1*e2*j1*j12 - f1*f2*j1*j12 - g1*g2*j1*j12 - h1*h2*j1*j12 - i1*i2*j1*j12 - b1*b12*j1*j2 - 2*b1*b2*j1*j2 - c1*c12*j1*j2 - 2*c1*c2*j1*j2 - d1*d12*j1*j2 - 2*d1*d2*j1*j2 - e1*e12*j1*j2 - 2*e1*e2*j1*j2 - f1*f12*j1*j2 - 2*f1*f2*j1*j2 - g1*g12*j1*j2 - 2*g1*g2*j1*j2 - h1*h12*j1*j2 - 2*h1*h2*j1*j2 - i1*i12*j1*j2 - 2*i1*i2*j1*j2 + a1^2*j12*j2 + b1^2*j12*j2 + c1^2*j12*j2 + d1^2*j12*j2 + e1^2*j12*j2 + f1^2*j12*j2 + g1^2*j12*j2 + h1^2*j12*j2 + i1^2*j12*j2 + a1^2*j2^2 + b1^2*j2^2 + c1^2*j2^2 + d1^2*j2^2 + e1^2*j2^2 + f1^2*j2^2 + g1^2*j2^2 + h1^2*j2^2 + i1^2*j2^2 - a1*a2*(b1*b12 + 2*b1*b2 + c1*c12 + 2*c1*c2 + d1*d12 + 2*d1*d2 + e1*e12 + 2*e1*e2 + f1*f12 + 2*f1*f2 + g1*g12 + 2*g1*g2 + h1*h12 + 2*h1*h2 + i1*i12 + 2*i1*i2 + j1*j12 + 2*j1*j2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2 + i1^2 + j1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2 + j1*j2))))
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          #m21
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          EXPR2 <- expression((b12*b2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 + b1^2*c12*c2 + b12*b2*d1^2 + c12*c2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - c1*c12*d1*d2 + b1^2*d12*d2 + c1^2*d12*d2 + b12*b2*e1^2 + c12*c2*e1^2 + d12*d2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - c1*c12*e1*e2 - d1*d12*e1*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + b12*b2*f1^2 + c12*c2*f1^2 + d12*d2*f1^2 + e12*e2*f1^2 - b1*b2*f1*f12 - c1*c2*f1*f12 - d1*d2*f1*f12 - e1*e2*f1*f12 - b1*b12*f1*f2 - c1*c12*f1*f2 - d1*d12*f1*f2 - e1*e12*f1*f2 + b1^2*f12*f2 + c1^2*f12*f2 + d1^2*f12*f2 + e1^2*f12*f2 + b12*b2*g1^2 + c12*c2*g1^2 + d12*d2*g1^2 + e12*e2*g1^2 + f12*f2*g1^2 - b1*b2*g1*g12 - c1*c2*g1*g12 - d1*d2*g1*g12 - e1*e2*g1*g12 - f1*f2*g1*g12 - b1*b12*g1*g2 - c1*c12*g1*g2 - d1*d12*g1*g2 - e1*e12*g1*g2 - f1*f12*g1*g2 + b1^2*g12*g2 + c1^2*g12*g2 + d1^2*g12*g2 + e1^2*g12*g2 + f1^2*g12*g2 + b12*b2*h1^2 + c12*c2*h1^2 + d12*d2*h1^2 + e12*e2*h1^2 + f12*f2*h1^2 + g12*g2*h1^2 - b1*b2*h1*h12 - c1*c2*h1*h12 - d1*d2*h1*h12 - e1*e2*h1*h12 - f1*f2*h1*h12 - g1*g2*h1*h12 - b1*b12*h1*h2 - c1*c12*h1*h2 - d1*d12*h1*h2 - e1*e12*h1*h2 - f1*f12*h1*h2 - g1*g12*h1*h2 + b1^2*h12*h2 + c1^2*h12*h2 + d1^2*h12*h2 + e1^2*h12*h2 + f1^2*h12*h2 + g1^2*h12*h2 + b12*b2*i1^2 + c12*c2*i1^2 + d12*d2*i1^2 + e12*e2*i1^2 + f12*f2*i1^2 + g12*g2*i1^2 + h12*h2*i1^2 - b1*b2*i1*i12 - c1*c2*i1*i12 - d1*d2*i1*i12 - e1*e2*i1*i12 - f1*f2*i1*i12 - g1*g2*i1*i12 - h1*h2*i1*i12 - b1*b12*i1*i2 - c1*c12*i1*i2 - d1*d12*i1*i2 - e1*e12*i1*i2 - f1*f12*i1*i2 - g1*g12*i1*i2 - h1*h12*i1*i2 + b1^2*i12*i2 + c1^2*i12*i2 + d1^2*i12*i2 + e1^2*i12*i2 + f1^2*i12*i2 + g1^2*i12*i2 + h1^2*i12*i2 + b12*b2*j1^2 + c12*c2*j1^2 + d12*d2*j1^2 + e12*e2*j1^2 + f12*f2*j1^2 + g12*g2*j1^2 + h12*h2*j1^2 + i12*i2*j1^2 - b1*b2*j1*j12 - c1*c2*j1*j12 - d1*d2*j1*j12 - e1*e2*j1*j12 - f1*f2*j1*j12 - g1*g2*j1*j12 - h1*h2*j1*j12 - i1*i2*j1*j12 - a1*a2*(b1*b12 + c1*c12 + d1*d12 + e1*e12 + f1*f12 + g1*g12 + h1*h12 + i1*i12 + j1*j12) - b1*b12*j1*j2 - c1*c12*j1*j2 - d1*d12*j1*j2 - e1*e12*j1*j2 - f1*f12*j1*j2 - g1*g12*j1*j2 - h1*h12*j1*j2 - i1*i12*j1*j2 + b1^2*j12*j2 + c1^2*j12*j2 + d1^2*j12*j2 + e1^2*j12*j2 + f1^2*j12*j2 + g1^2*j12*j2 + h1^2*j12*j2 + i1^2*j12*j2 + a1^2*(b12*b2 + c12*c2 + d12*d2 + e12*e2 + f12*f2 + g12*g2 + h12*h2 + i12*i2 + j12*j2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2 + i1^2 + j1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2 + j1*j2)))/(b2^2*c1^2 + b2^2*c1*c12 - 2*b1*b2*c1*c2 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1^2*c2^2 + b1*b12*c2^2 + b2^2*d1^2 + c2^2*d1^2 + b2^2*d1*d12 + c2^2*d1*d12 - 2*b1*b2*d1*d2 - b12*b2*d1*d2 - 2*c1*c2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1^2*d2^2 + b1*b12*d2^2 + c1^2*d2^2 + c1*c12*d2^2 + b2^2*e1^2 + c2^2*e1^2 + d2^2*e1^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 - 2*b1*b2*e1*e2 - b12*b2*e1*e2 - 2*c1*c2*e1*e2 - c12*c2*e1*e2 - 2*d1*d2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1^2*e2^2 + b1*b12*e2^2 + c1^2*e2^2 + c1*c12*e2^2 + d1^2*e2^2 + d1*d12*e2^2 + b2^2*f1^2 + c2^2*f1^2 + d2^2*f1^2 + e2^2*f1^2 + b2^2*f1*f12 + c2^2*f1*f12 + d2^2*f1*f12 + e2^2*f1*f12 - 2*b1*b2*f1*f2 - b12*b2*f1*f2 - 2*c1*c2*f1*f2 - c12*c2*f1*f2 - 2*d1*d2*f1*f2 - d12*d2*f1*f2 - 2*e1*e2*f1*f2 - e12*e2*f1*f2 - b1*b2*f12*f2 - c1*c2*f12*f2 - d1*d2*f12*f2 - e1*e2*f12*f2 + b1^2*f2^2 + b1*b12*f2^2 + c1^2*f2^2 + c1*c12*f2^2 + d1^2*f2^2 + d1*d12*f2^2 + e1^2*f2^2 + e1*e12*f2^2 + b2^2*g1^2 + c2^2*g1^2 + d2^2*g1^2 + e2^2*g1^2 + f2^2*g1^2 + b2^2*g1*g12 + c2^2*g1*g12 + d2^2*g1*g12 + e2^2*g1*g12 + f2^2*g1*g12 - 2*b1*b2*g1*g2 - b12*b2*g1*g2 - 2*c1*c2*g1*g2 - c12*c2*g1*g2 - 2*d1*d2*g1*g2 - d12*d2*g1*g2 - 2*e1*e2*g1*g2 - e12*e2*g1*g2 - 2*f1*f2*g1*g2 - f12*f2*g1*g2 - b1*b2*g12*g2 - c1*c2*g12*g2 - d1*d2*g12*g2 - e1*e2*g12*g2 - f1*f2*g12*g2 + b1^2*g2^2 + b1*b12*g2^2 + c1^2*g2^2 + c1*c12*g2^2 + d1^2*g2^2 + d1*d12*g2^2 + e1^2*g2^2 + e1*e12*g2^2 + f1^2*g2^2 + f1*f12*g2^2 + b2^2*h1^2 + c2^2*h1^2 + d2^2*h1^2 + e2^2*h1^2 + f2^2*h1^2 + g2^2*h1^2 + b2^2*h1*h12 + c2^2*h1*h12 + d2^2*h1*h12 + e2^2*h1*h12 + f2^2*h1*h12 + g2^2*h1*h12 - 2*b1*b2*h1*h2 - b12*b2*h1*h2 - 2*c1*c2*h1*h2 - c12*c2*h1*h2 - 2*d1*d2*h1*h2 - d12*d2*h1*h2 - 2*e1*e2*h1*h2 - e12*e2*h1*h2 - 2*f1*f2*h1*h2 - f12*f2*h1*h2 - 2*g1*g2*h1*h2 - g12*g2*h1*h2 - b1*b2*h12*h2 - c1*c2*h12*h2 - d1*d2*h12*h2 - e1*e2*h12*h2 - f1*f2*h12*h2 - g1*g2*h12*h2 + b1^2*h2^2 + b1*b12*h2^2 + c1^2*h2^2 + c1*c12*h2^2 + d1^2*h2^2 + d1*d12*h2^2 + e1^2*h2^2 + e1*e12*h2^2 + f1^2*h2^2 + f1*f12*h2^2 + g1^2*h2^2 + g1*g12*h2^2 + b2^2*i1^2 + c2^2*i1^2 + d2^2*i1^2 + e2^2*i1^2 + f2^2*i1^2 + g2^2*i1^2 + h2^2*i1^2 + b2^2*i1*i12 + c2^2*i1*i12 + d2^2*i1*i12 + e2^2*i1*i12 + f2^2*i1*i12 + g2^2*i1*i12 + h2^2*i1*i12 - 2*b1*b2*i1*i2 - b12*b2*i1*i2 - 2*c1*c2*i1*i2 - c12*c2*i1*i2 - 2*d1*d2*i1*i2 - d12*d2*i1*i2 - 2*e1*e2*i1*i2 - e12*e2*i1*i2 - 2*f1*f2*i1*i2 - f12*f2*i1*i2 - 2*g1*g2*i1*i2 - g12*g2*i1*i2 - 2*h1*h2*i1*i2 - h12*h2*i1*i2 - b1*b2*i12*i2 - c1*c2*i12*i2 - d1*d2*i12*i2 - e1*e2*i12*i2 - f1*f2*i12*i2 - g1*g2*i12*i2 - h1*h2*i12*i2 + b1^2*i2^2 + b1*b12*i2^2 + c1^2*i2^2 + c1*c12*i2^2 + d1^2*i2^2 + d1*d12*i2^2 + e1^2*i2^2 + e1*e12*i2^2 + f1^2*i2^2 + f1*f12*i2^2 + g1^2*i2^2 + g1*g12*i2^2 + h1^2*i2^2 + h1*h12*i2^2 + b2^2*j1^2 + c2^2*j1^2 + d2^2*j1^2 + e2^2*j1^2 + f2^2*j1^2 + g2^2*j1^2 + h2^2*j1^2 + i2^2*j1^2 + b2^2*j1*j12 + c2^2*j1*j12 + d2^2*j1*j12 + e2^2*j1*j12 + f2^2*j1*j12 + g2^2*j1*j12 + h2^2*j1*j12 + i2^2*j1*j12 + a2^2*(b1^2 + b1*b12 + c1^2 + c1*c12 + d1^2 + d1*d12 + e1^2 + e1*e12 + f1^2 + f1*f12 + g1^2 + g1*g12 + h1^2 + h1*h12 + i1^2 + i1*i12 + j1^2 + j1*j12) - 2*b1*b2*j1*j2 - b12*b2*j1*j2 - 2*c1*c2*j1*j2 - c12*c2*j1*j2 - 2*d1*d2*j1*j2 - d12*d2*j1*j2 - 2*e1*e2*j1*j2 - e12*e2*j1*j2 - 2*f1*f2*j1*j2 - f12*f2*j1*j2 - 2*g1*g2*j1*j2 - g12*g2*j1*j2 - 2*h1*h2*j1*j2 - h12*h2*j1*j2 - 2*i1*i2*j1*j2 - i12*i2*j1*j2 - b1*b2*j12*j2 - c1*c2*j12*j2 - d1*d2*j12*j2 - e1*e2*j12*j2 - f1*f2*j12*j2 - g1*g2*j12*j2 - h1*h2*j12*j2 - i1*i2*j12*j2 + b1^2*j2^2 + b1*b12*j2^2 + c1^2*j2^2 + c1*c12*j2^2 + d1^2*j2^2 + d1*d12*j2^2 + e1^2*j2^2 + e1*e12*j2^2 + f1^2*j2^2 + f1*f12*j2^2 + g1^2*j2^2 + g1*g12*j2^2 + h1^2*j2^2 + h1*h12*j2^2 + i1^2*j2^2 + i1*i12*j2^2 + a1^2*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2 + i2^2 + j2^2) + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2 + i2^2 + j2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2 + j1*j2) + a1*(2*b1*b2 + b12*b2 + 2*c1*c2 + c12*c2 + 2*d1*d2 + d12*d2 + 2*e1*e2 + e12*e2 + 2*f1*f2 + f12*f2 + 2*g1*g2 + g12*g2 + 2*h1*h2 + h12*h2 + 2*i1*i2 + i12*i2 + 2*j1*j2 + j12*j2))))
  } else if (n_rows == 11) {
    #m12
    EXPR1 <- expression((b2^2*c1*c12 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1*b12*c2^2 + b2^2*d1*d12 + c2^2*d1*d12 - b12*b2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1*b12*d2^2 + c1*c12*d2^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 - b12*b2*e1*e2 - c12*c2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1*b12*e2^2 + c1*c12*e2^2 + d1*d12*e2^2 + b2^2*f1*f12 + c2^2*f1*f12 + d2^2*f1*f12 + e2^2*f1*f12 - b12*b2*f1*f2 - c12*c2*f1*f2 - d12*d2*f1*f2 - e12*e2*f1*f2 - b1*b2*f12*f2 - c1*c2*f12*f2 - d1*d2*f12*f2 - e1*e2*f12*f2 + b1*b12*f2^2 + c1*c12*f2^2 + d1*d12*f2^2 + e1*e12*f2^2 + b2^2*g1*g12 + c2^2*g1*g12 + d2^2*g1*g12 + e2^2*g1*g12 + f2^2*g1*g12 - b12*b2*g1*g2 - c12*c2*g1*g2 - d12*d2*g1*g2 - e12*e2*g1*g2 - f12*f2*g1*g2 - b1*b2*g12*g2 - c1*c2*g12*g2 - d1*d2*g12*g2 - e1*e2*g12*g2 - f1*f2*g12*g2 + b1*b12*g2^2 + c1*c12*g2^2 + d1*d12*g2^2 + e1*e12*g2^2 + f1*f12*g2^2 + b2^2*h1*h12 + c2^2*h1*h12 + d2^2*h1*h12 + e2^2*h1*h12 + f2^2*h1*h12 + g2^2*h1*h12 - b12*b2*h1*h2 - c12*c2*h1*h2 - d12*d2*h1*h2 - e12*e2*h1*h2 - f12*f2*h1*h2 - g12*g2*h1*h2 - b1*b2*h12*h2 - c1*c2*h12*h2 - d1*d2*h12*h2 - e1*e2*h12*h2 - f1*f2*h12*h2 - g1*g2*h12*h2 + b1*b12*h2^2 + c1*c12*h2^2 + d1*d12*h2^2 + e1*e12*h2^2 + f1*f12*h2^2 + g1*g12*h2^2 + b2^2*i1*i12 + c2^2*i1*i12 + d2^2*i1*i12 + e2^2*i1*i12 + f2^2*i1*i12 + g2^2*i1*i12 + h2^2*i1*i12 - b12*b2*i1*i2 - c12*c2*i1*i2 - d12*d2*i1*i2 - e12*e2*i1*i2 - f12*f2*i1*i2 - g12*g2*i1*i2 - h12*h2*i1*i2 - b1*b2*i12*i2 - c1*c2*i12*i2 - d1*d2*i12*i2 - e1*e2*i12*i2 - f1*f2*i12*i2 - g1*g2*i12*i2 - h1*h2*i12*i2 + b1*b12*i2^2 + c1*c12*i2^2 + d1*d12*i2^2 + e1*e12*i2^2 + f1*f12*i2^2 + g1*g12*i2^2 + h1*h12*i2^2 + b2^2*j1*j12 + c2^2*j1*j12 + d2^2*j1*j12 + e2^2*j1*j12 + f2^2*j1*j12 + g2^2*j1*j12 + h2^2*j1*j12 + i2^2*j1*j12 - b12*b2*j1*j2 - c12*c2*j1*j2 - d12*d2*j1*j2 - e12*e2*j1*j2 - f12*f2*j1*j2 - g12*g2*j1*j2 - h12*h2*j1*j2 - i12*i2*j1*j2 - b1*b2*j12*j2 - c1*c2*j12*j2 - d1*d2*j12*j2 - e1*e2*j12*j2 - f1*f2*j12*j2 - g1*g2*j12*j2 - h1*h2*j12*j2 - i1*i2*j12*j2 + b1*b12*j2^2 + c1*c12*j2^2 + d1*d12*j2^2 + e1*e12*j2^2 + f1*f12*j2^2 + g1*g12*j2^2 + h1*h12*j2^2 + i1*i12*j2^2 + b2^2*k1*k12 + c2^2*k1*k12 + d2^2*k1*k12 + e2^2*k1*k12 + f2^2*k1*k12 + g2^2*k1*k12 + h2^2*k1*k12 + i2^2*k1*k12 + j2^2*k1*k12 + a2^2*(b1*b12 + c1*c12 + d1*d12 + e1*e12 + f1*f12 + g1*g12 + h1*h12 + i1*i12 + j1*j12 + k1*k12) - b12*b2*k1*k2 - c12*c2*k1*k2 - d12*d2*k1*k2 - e12*e2*k1*k2 - f12*f2*k1*k2 - g12*g2*k1*k2 - h12*h2*k1*k2 - i12*i2*k1*k2 - j12*j2*k1*k2 - b1*b2*k12*k2 - c1*c2*k12*k2 - d1*d2*k12*k2 - e1*e2*k12*k2 - f1*f2*k12*k2 - g1*g2*k12*k2 - h1*h2*k12*k2 - i1*i2*k12*k2 - j1*j2*k12*k2 + b1*b12*k2^2 + c1*c12*k2^2 + d1*d12*k2^2 + e1*e12*k2^2 + f1*f12*k2^2 + g1*g12*k2^2 + h1*h12*k2^2 + i1*i12*k2^2 + j1*j12*k2^2 + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2 + i2^2 + j2^2 + k2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2 + j1*j2 + k1*k2) + a1*(b12*b2 + c12*c2 + d12*d2 + e12*e2 + f12*f2 + g12*g2 + h12*h2 + i12*i2 + j12*j2 + k12*k2)))/(a1^2*b12*b2 + a1^2*b2^2 + b12*b2*c1^2 + b2^2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 - 2*b1*b2*c1*c2 + a1^2*c12*c2 + b1^2*c12*c2 + a1^2*c2^2 + b1^2*c2^2 + b12*b2*d1^2 + b2^2*d1^2 + c12*c2*d1^2 + c2^2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - 2*b1*b2*d1*d2 - c1*c12*d1*d2 - 2*c1*c2*d1*d2 + a1^2*d12*d2 + b1^2*d12*d2 + c1^2*d12*d2 + a1^2*d2^2 + b1^2*d2^2 + c1^2*d2^2 + b12*b2*e1^2 + b2^2*e1^2 + c12*c2*e1^2 + c2^2*e1^2 + d12*d2*e1^2 + d2^2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - 2*b1*b2*e1*e2 - c1*c12*e1*e2 - 2*c1*c2*e1*e2 - d1*d12*e1*e2 - 2*d1*d2*e1*e2 + a1^2*e12*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + a1^2*e2^2 + b1^2*e2^2 + c1^2*e2^2 + d1^2*e2^2 + b12*b2*f1^2 + b2^2*f1^2 + c12*c2*f1^2 + c2^2*f1^2 + d12*d2*f1^2 + d2^2*f1^2 + e12*e2*f1^2 + e2^2*f1^2 - b1*b2*f1*f12 - c1*c2*f1*f12 - d1*d2*f1*f12 - e1*e2*f1*f12 - b1*b12*f1*f2 - 2*b1*b2*f1*f2 - c1*c12*f1*f2 - 2*c1*c2*f1*f2 - d1*d12*f1*f2 - 2*d1*d2*f1*f2 - e1*e12*f1*f2 - 2*e1*e2*f1*f2 + a1^2*f12*f2 + b1^2*f12*f2 + c1^2*f12*f2 + d1^2*f12*f2 + e1^2*f12*f2 + a1^2*f2^2 + b1^2*f2^2 + c1^2*f2^2 + d1^2*f2^2 + e1^2*f2^2 + b12*b2*g1^2 + b2^2*g1^2 + c12*c2*g1^2 + c2^2*g1^2 + d12*d2*g1^2 + d2^2*g1^2 + e12*e2*g1^2 + e2^2*g1^2 + f12*f2*g1^2 + f2^2*g1^2 - b1*b2*g1*g12 - c1*c2*g1*g12 - d1*d2*g1*g12 - e1*e2*g1*g12 - f1*f2*g1*g12 - b1*b12*g1*g2 - 2*b1*b2*g1*g2 - c1*c12*g1*g2 - 2*c1*c2*g1*g2 - d1*d12*g1*g2 - 2*d1*d2*g1*g2 - e1*e12*g1*g2 - 2*e1*e2*g1*g2 - f1*f12*g1*g2 - 2*f1*f2*g1*g2 + a1^2*g12*g2 + b1^2*g12*g2 + c1^2*g12*g2 + d1^2*g12*g2 + e1^2*g12*g2 + f1^2*g12*g2 + a1^2*g2^2 + b1^2*g2^2 + c1^2*g2^2 + d1^2*g2^2 + e1^2*g2^2 + f1^2*g2^2 + b12*b2*h1^2 + b2^2*h1^2 + c12*c2*h1^2 + c2^2*h1^2 + d12*d2*h1^2 + d2^2*h1^2 + e12*e2*h1^2 + e2^2*h1^2 + f12*f2*h1^2 + f2^2*h1^2 + g12*g2*h1^2 + g2^2*h1^2 - b1*b2*h1*h12 - c1*c2*h1*h12 - d1*d2*h1*h12 - e1*e2*h1*h12 - f1*f2*h1*h12 - g1*g2*h1*h12 - b1*b12*h1*h2 - 2*b1*b2*h1*h2 - c1*c12*h1*h2 - 2*c1*c2*h1*h2 - d1*d12*h1*h2 - 2*d1*d2*h1*h2 - e1*e12*h1*h2 - 2*e1*e2*h1*h2 - f1*f12*h1*h2 - 2*f1*f2*h1*h2 - g1*g12*h1*h2 - 2*g1*g2*h1*h2 + a1^2*h12*h2 + b1^2*h12*h2 + c1^2*h12*h2 + d1^2*h12*h2 + e1^2*h12*h2 + f1^2*h12*h2 + g1^2*h12*h2 + a1^2*h2^2 + b1^2*h2^2 + c1^2*h2^2 + d1^2*h2^2 + e1^2*h2^2 + f1^2*h2^2 + g1^2*h2^2 + b12*b2*i1^2 + b2^2*i1^2 + c12*c2*i1^2 + c2^2*i1^2 + d12*d2*i1^2 + d2^2*i1^2 + e12*e2*i1^2 + e2^2*i1^2 + f12*f2*i1^2 + f2^2*i1^2 + g12*g2*i1^2 + g2^2*i1^2 + h12*h2*i1^2 + h2^2*i1^2 - b1*b2*i1*i12 - c1*c2*i1*i12 - d1*d2*i1*i12 - e1*e2*i1*i12 - f1*f2*i1*i12 - g1*g2*i1*i12 - h1*h2*i1*i12 - b1*b12*i1*i2 - 2*b1*b2*i1*i2 - c1*c12*i1*i2 - 2*c1*c2*i1*i2 - d1*d12*i1*i2 - 2*d1*d2*i1*i2 - e1*e12*i1*i2 - 2*e1*e2*i1*i2 - f1*f12*i1*i2 - 2*f1*f2*i1*i2 - g1*g12*i1*i2 - 2*g1*g2*i1*i2 - h1*h12*i1*i2 - 2*h1*h2*i1*i2 + a1^2*i12*i2 + b1^2*i12*i2 + c1^2*i12*i2 + d1^2*i12*i2 + e1^2*i12*i2 + f1^2*i12*i2 + g1^2*i12*i2 + h1^2*i12*i2 + a1^2*i2^2 + b1^2*i2^2 + c1^2*i2^2 + d1^2*i2^2 + e1^2*i2^2 + f1^2*i2^2 + g1^2*i2^2 + h1^2*i2^2 + b12*b2*j1^2 + b2^2*j1^2 + c12*c2*j1^2 + c2^2*j1^2 + d12*d2*j1^2 + d2^2*j1^2 + e12*e2*j1^2 + e2^2*j1^2 + f12*f2*j1^2 + f2^2*j1^2 + g12*g2*j1^2 + g2^2*j1^2 + h12*h2*j1^2 + h2^2*j1^2 + i12*i2*j1^2 + i2^2*j1^2 - b1*b2*j1*j12 - c1*c2*j1*j12 - d1*d2*j1*j12 - e1*e2*j1*j12 - f1*f2*j1*j12 - g1*g2*j1*j12 - h1*h2*j1*j12 - i1*i2*j1*j12 - b1*b12*j1*j2 - 2*b1*b2*j1*j2 - c1*c12*j1*j2 - 2*c1*c2*j1*j2 - d1*d12*j1*j2 - 2*d1*d2*j1*j2 - e1*e12*j1*j2 - 2*e1*e2*j1*j2 - f1*f12*j1*j2 - 2*f1*f2*j1*j2 - g1*g12*j1*j2 - 2*g1*g2*j1*j2 - h1*h12*j1*j2 - 2*h1*h2*j1*j2 - i1*i12*j1*j2 - 2*i1*i2*j1*j2 + a1^2*j12*j2 + b1^2*j12*j2 + c1^2*j12*j2 + d1^2*j12*j2 + e1^2*j12*j2 + f1^2*j12*j2 + g1^2*j12*j2 + h1^2*j12*j2 + i1^2*j12*j2 + a1^2*j2^2 + b1^2*j2^2 + c1^2*j2^2 + d1^2*j2^2 + e1^2*j2^2 + f1^2*j2^2 + g1^2*j2^2 + h1^2*j2^2 + i1^2*j2^2 + b12*b2*k1^2 + b2^2*k1^2 + c12*c2*k1^2 + c2^2*k1^2 + d12*d2*k1^2 + d2^2*k1^2 + e12*e2*k1^2 + e2^2*k1^2 + f12*f2*k1^2 + f2^2*k1^2 + g12*g2*k1^2 + g2^2*k1^2 + h12*h2*k1^2 + h2^2*k1^2 + i12*i2*k1^2 + i2^2*k1^2 + j12*j2*k1^2 + j2^2*k1^2 + a2^2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2 + i1^2 + j1^2 + k1^2) - b1*b2*k1*k12 - c1*c2*k1*k12 - d1*d2*k1*k12 - e1*e2*k1*k12 - f1*f2*k1*k12 - g1*g2*k1*k12 - h1*h2*k1*k12 - i1*i2*k1*k12 - j1*j2*k1*k12 - b1*b12*k1*k2 - 2*b1*b2*k1*k2 - c1*c12*k1*k2 - 2*c1*c2*k1*k2 - d1*d12*k1*k2 - 2*d1*d2*k1*k2 - e1*e12*k1*k2 - 2*e1*e2*k1*k2 - f1*f12*k1*k2 - 2*f1*f2*k1*k2 - g1*g12*k1*k2 - 2*g1*g2*k1*k2 - h1*h12*k1*k2 - 2*h1*h2*k1*k2 - i1*i12*k1*k2 - 2*i1*i2*k1*k2 - j1*j12*k1*k2 - 2*j1*j2*k1*k2 + a1^2*k12*k2 + b1^2*k12*k2 + c1^2*k12*k2 + d1^2*k12*k2 + e1^2*k12*k2 + f1^2*k12*k2 + g1^2*k12*k2 + h1^2*k12*k2 + i1^2*k12*k2 + j1^2*k12*k2 + a1^2*k2^2 + b1^2*k2^2 + c1^2*k2^2 + d1^2*k2^2 + e1^2*k2^2 + f1^2*k2^2 + g1^2*k2^2 + h1^2*k2^2 + i1^2*k2^2 + j1^2*k2^2 - a1*a2*(b1*b12 + 2*b1*b2 + c1*c12 + 2*c1*c2 + d1*d12 + 2*d1*d2 + e1*e12 + 2*e1*e2 + f1*f12 + 2*f1*f2 + g1*g12 + 2*g1*g2 + h1*h12 + 2*h1*h2 + i1*i12 + 2*i1*i2 + j1*j12 + 2*j1*j2 + k1*k12 + 2*k1*k2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2 + i1^2 + j1^2 + k1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2 + j1*j2 + k1*k2))))
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    #m21
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    EXPR2 <- expression((b12*b2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 + b1^2*c12*c2 + b12*b2*d1^2 + c12*c2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - c1*c12*d1*d2 + b1^2*d12*d2 + c1^2*d12*d2 + b12*b2*e1^2 + c12*c2*e1^2 + d12*d2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - c1*c12*e1*e2 - d1*d12*e1*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + b12*b2*f1^2 + c12*c2*f1^2 + d12*d2*f1^2 + e12*e2*f1^2 - b1*b2*f1*f12 - c1*c2*f1*f12 - d1*d2*f1*f12 - e1*e2*f1*f12 - b1*b12*f1*f2 - c1*c12*f1*f2 - d1*d12*f1*f2 - e1*e12*f1*f2 + b1^2*f12*f2 + c1^2*f12*f2 + d1^2*f12*f2 + e1^2*f12*f2 + b12*b2*g1^2 + c12*c2*g1^2 + d12*d2*g1^2 + e12*e2*g1^2 + f12*f2*g1^2 - b1*b2*g1*g12 - c1*c2*g1*g12 - d1*d2*g1*g12 - e1*e2*g1*g12 - f1*f2*g1*g12 - b1*b12*g1*g2 - c1*c12*g1*g2 - d1*d12*g1*g2 - e1*e12*g1*g2 - f1*f12*g1*g2 + b1^2*g12*g2 + c1^2*g12*g2 + d1^2*g12*g2 + e1^2*g12*g2 + f1^2*g12*g2 + b12*b2*h1^2 + c12*c2*h1^2 + d12*d2*h1^2 + e12*e2*h1^2 + f12*f2*h1^2 + g12*g2*h1^2 - b1*b2*h1*h12 - c1*c2*h1*h12 - d1*d2*h1*h12 - e1*e2*h1*h12 - f1*f2*h1*h12 - g1*g2*h1*h12 - b1*b12*h1*h2 - c1*c12*h1*h2 - d1*d12*h1*h2 - e1*e12*h1*h2 - f1*f12*h1*h2 - g1*g12*h1*h2 + b1^2*h12*h2 + c1^2*h12*h2 + d1^2*h12*h2 + e1^2*h12*h2 + f1^2*h12*h2 + g1^2*h12*h2 + b12*b2*i1^2 + c12*c2*i1^2 + d12*d2*i1^2 + e12*e2*i1^2 + f12*f2*i1^2 + g12*g2*i1^2 + h12*h2*i1^2 - b1*b2*i1*i12 - c1*c2*i1*i12 - d1*d2*i1*i12 - e1*e2*i1*i12 - f1*f2*i1*i12 - g1*g2*i1*i12 - h1*h2*i1*i12 - b1*b12*i1*i2 - c1*c12*i1*i2 - d1*d12*i1*i2 - e1*e12*i1*i2 - f1*f12*i1*i2 - g1*g12*i1*i2 - h1*h12*i1*i2 + b1^2*i12*i2 + c1^2*i12*i2 + d1^2*i12*i2 + e1^2*i12*i2 + f1^2*i12*i2 + g1^2*i12*i2 + h1^2*i12*i2 + b12*b2*j1^2 + c12*c2*j1^2 + d12*d2*j1^2 + e12*e2*j1^2 + f12*f2*j1^2 + g12*g2*j1^2 + h12*h2*j1^2 + i12*i2*j1^2 - b1*b2*j1*j12 - c1*c2*j1*j12 - d1*d2*j1*j12 - e1*e2*j1*j12 - f1*f2*j1*j12 - g1*g2*j1*j12 - h1*h2*j1*j12 - i1*i2*j1*j12 - b1*b12*j1*j2 - c1*c12*j1*j2 - d1*d12*j1*j2 - e1*e12*j1*j2 - f1*f12*j1*j2 - g1*g12*j1*j2 - h1*h12*j1*j2 - i1*i12*j1*j2 + b1^2*j12*j2 + c1^2*j12*j2 + d1^2*j12*j2 + e1^2*j12*j2 + f1^2*j12*j2 + g1^2*j12*j2 + h1^2*j12*j2 + i1^2*j12*j2 + b12*b2*k1^2 + c12*c2*k1^2 + d12*d2*k1^2 + e12*e2*k1^2 + f12*f2*k1^2 + g12*g2*k1^2 + h12*h2*k1^2 + i12*i2*k1^2 + j12*j2*k1^2 - b1*b2*k1*k12 - c1*c2*k1*k12 - d1*d2*k1*k12 - e1*e2*k1*k12 - f1*f2*k1*k12 - g1*g2*k1*k12 - h1*h2*k1*k12 - i1*i2*k1*k12 - j1*j2*k1*k12 - a1*a2*(b1*b12 + c1*c12 + d1*d12 + e1*e12 + f1*f12 + g1*g12 + h1*h12 + i1*i12 + j1*j12 + k1*k12) - b1*b12*k1*k2 - c1*c12*k1*k2 - d1*d12*k1*k2 - e1*e12*k1*k2 - f1*f12*k1*k2 - g1*g12*k1*k2 - h1*h12*k1*k2 - i1*i12*k1*k2 - j1*j12*k1*k2 + b1^2*k12*k2 + c1^2*k12*k2 + d1^2*k12*k2 + e1^2*k12*k2 + f1^2*k12*k2 + g1^2*k12*k2 + h1^2*k12*k2 + i1^2*k12*k2 + j1^2*k12*k2 + a1^2*(b12*b2 + c12*c2 + d12*d2 + e12*e2 + f12*f2 + g12*g2 + h12*h2 + i12*i2 + j12*j2 + k12*k2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2 + i1^2 + j1^2 + k1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2 + j1*j2 + k1*k2)))/(b2^2*c1^2 + b2^2*c1*c12 - 2*b1*b2*c1*c2 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1^2*c2^2 + b1*b12*c2^2 + b2^2*d1^2 + c2^2*d1^2 + b2^2*d1*d12 + c2^2*d1*d12 - 2*b1*b2*d1*d2 - b12*b2*d1*d2 - 2*c1*c2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1^2*d2^2 + b1*b12*d2^2 + c1^2*d2^2 + c1*c12*d2^2 + b2^2*e1^2 + c2^2*e1^2 + d2^2*e1^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 - 2*b1*b2*e1*e2 - b12*b2*e1*e2 - 2*c1*c2*e1*e2 - c12*c2*e1*e2 - 2*d1*d2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1^2*e2^2 + b1*b12*e2^2 + c1^2*e2^2 + c1*c12*e2^2 + d1^2*e2^2 + d1*d12*e2^2 + b2^2*f1^2 + c2^2*f1^2 + d2^2*f1^2 + e2^2*f1^2 + b2^2*f1*f12 + c2^2*f1*f12 + d2^2*f1*f12 + e2^2*f1*f12 - 2*b1*b2*f1*f2 - b12*b2*f1*f2 - 2*c1*c2*f1*f2 - c12*c2*f1*f2 - 2*d1*d2*f1*f2 - d12*d2*f1*f2 - 2*e1*e2*f1*f2 - e12*e2*f1*f2 - b1*b2*f12*f2 - c1*c2*f12*f2 - d1*d2*f12*f2 - e1*e2*f12*f2 + b1^2*f2^2 + b1*b12*f2^2 + c1^2*f2^2 + c1*c12*f2^2 + d1^2*f2^2 + d1*d12*f2^2 + e1^2*f2^2 + e1*e12*f2^2 + b2^2*g1^2 + c2^2*g1^2 + d2^2*g1^2 + e2^2*g1^2 + f2^2*g1^2 + b2^2*g1*g12 + c2^2*g1*g12 + d2^2*g1*g12 + e2^2*g1*g12 + f2^2*g1*g12 - 2*b1*b2*g1*g2 - b12*b2*g1*g2 - 2*c1*c2*g1*g2 - c12*c2*g1*g2 - 2*d1*d2*g1*g2 - d12*d2*g1*g2 - 2*e1*e2*g1*g2 - e12*e2*g1*g2 - 2*f1*f2*g1*g2 - f12*f2*g1*g2 - b1*b2*g12*g2 - c1*c2*g12*g2 - d1*d2*g12*g2 - e1*e2*g12*g2 - f1*f2*g12*g2 + b1^2*g2^2 + b1*b12*g2^2 + c1^2*g2^2 + c1*c12*g2^2 + d1^2*g2^2 + d1*d12*g2^2 + e1^2*g2^2 + e1*e12*g2^2 + f1^2*g2^2 + f1*f12*g2^2 + b2^2*h1^2 + c2^2*h1^2 + d2^2*h1^2 + e2^2*h1^2 + f2^2*h1^2 + g2^2*h1^2 + b2^2*h1*h12 + c2^2*h1*h12 + d2^2*h1*h12 + e2^2*h1*h12 + f2^2*h1*h12 + g2^2*h1*h12 - 2*b1*b2*h1*h2 - b12*b2*h1*h2 - 2*c1*c2*h1*h2 - c12*c2*h1*h2 - 2*d1*d2*h1*h2 - d12*d2*h1*h2 - 2*e1*e2*h1*h2 - e12*e2*h1*h2 - 2*f1*f2*h1*h2 - f12*f2*h1*h2 - 2*g1*g2*h1*h2 - g12*g2*h1*h2 - b1*b2*h12*h2 - c1*c2*h12*h2 - d1*d2*h12*h2 - e1*e2*h12*h2 - f1*f2*h12*h2 - g1*g2*h12*h2 + b1^2*h2^2 + b1*b12*h2^2 + c1^2*h2^2 + c1*c12*h2^2 + d1^2*h2^2 + d1*d12*h2^2 + e1^2*h2^2 + e1*e12*h2^2 + f1^2*h2^2 + f1*f12*h2^2 + g1^2*h2^2 + g1*g12*h2^2 + b2^2*i1^2 + c2^2*i1^2 + d2^2*i1^2 + e2^2*i1^2 + f2^2*i1^2 + g2^2*i1^2 + h2^2*i1^2 + b2^2*i1*i12 + c2^2*i1*i12 + d2^2*i1*i12 + e2^2*i1*i12 + f2^2*i1*i12 + g2^2*i1*i12 + h2^2*i1*i12 - 2*b1*b2*i1*i2 - b12*b2*i1*i2 - 2*c1*c2*i1*i2 - c12*c2*i1*i2 - 2*d1*d2*i1*i2 - d12*d2*i1*i2 - 2*e1*e2*i1*i2 - e12*e2*i1*i2 - 2*f1*f2*i1*i2 - f12*f2*i1*i2 - 2*g1*g2*i1*i2 - g12*g2*i1*i2 - 2*h1*h2*i1*i2 - h12*h2*i1*i2 - b1*b2*i12*i2 - c1*c2*i12*i2 - d1*d2*i12*i2 - e1*e2*i12*i2 - f1*f2*i12*i2 - g1*g2*i12*i2 - h1*h2*i12*i2 + b1^2*i2^2 + b1*b12*i2^2 + c1^2*i2^2 + c1*c12*i2^2 + d1^2*i2^2 + d1*d12*i2^2 + e1^2*i2^2 + e1*e12*i2^2 + f1^2*i2^2 + f1*f12*i2^2 + g1^2*i2^2 + g1*g12*i2^2 + h1^2*i2^2 + h1*h12*i2^2 + b2^2*j1^2 + c2^2*j1^2 + d2^2*j1^2 + e2^2*j1^2 + f2^2*j1^2 + g2^2*j1^2 + h2^2*j1^2 + i2^2*j1^2 + b2^2*j1*j12 + c2^2*j1*j12 + d2^2*j1*j12 + e2^2*j1*j12 + f2^2*j1*j12 + g2^2*j1*j12 + h2^2*j1*j12 + i2^2*j1*j12 - 2*b1*b2*j1*j2 - b12*b2*j1*j2 - 2*c1*c2*j1*j2 - c12*c2*j1*j2 - 2*d1*d2*j1*j2 - d12*d2*j1*j2 - 2*e1*e2*j1*j2 - e12*e2*j1*j2 - 2*f1*f2*j1*j2 - f12*f2*j1*j2 - 2*g1*g2*j1*j2 - g12*g2*j1*j2 - 2*h1*h2*j1*j2 - h12*h2*j1*j2 - 2*i1*i2*j1*j2 - i12*i2*j1*j2 - b1*b2*j12*j2 - c1*c2*j12*j2 - d1*d2*j12*j2 - e1*e2*j12*j2 - f1*f2*j12*j2 - g1*g2*j12*j2 - h1*h2*j12*j2 - i1*i2*j12*j2 + b1^2*j2^2 + b1*b12*j2^2 + c1^2*j2^2 + c1*c12*j2^2 + d1^2*j2^2 + d1*d12*j2^2 + e1^2*j2^2 + e1*e12*j2^2 + f1^2*j2^2 + f1*f12*j2^2 + g1^2*j2^2 + g1*g12*j2^2 + h1^2*j2^2 + h1*h12*j2^2 + i1^2*j2^2 + i1*i12*j2^2 + b2^2*k1^2 + c2^2*k1^2 + d2^2*k1^2 + e2^2*k1^2 + f2^2*k1^2 + g2^2*k1^2 + h2^2*k1^2 + i2^2*k1^2 + j2^2*k1^2 + b2^2*k1*k12 + c2^2*k1*k12 + d2^2*k1*k12 + e2^2*k1*k12 + f2^2*k1*k12 + g2^2*k1*k12 + h2^2*k1*k12 + i2^2*k1*k12 + j2^2*k1*k12 + a2^2*(b1^2 + b1*b12 + c1^2 + c1*c12 + d1^2 + d1*d12 + e1^2 + e1*e12 + f1^2 + f1*f12 + g1^2 + g1*g12 + h1^2 + h1*h12 + i1^2 + i1*i12 + j1^2 + j1*j12 + k1^2 + k1*k12) - 2*b1*b2*k1*k2 - b12*b2*k1*k2 - 2*c1*c2*k1*k2 - c12*c2*k1*k2 - 2*d1*d2*k1*k2 - d12*d2*k1*k2 - 2*e1*e2*k1*k2 - e12*e2*k1*k2 - 2*f1*f2*k1*k2 - f12*f2*k1*k2 - 2*g1*g2*k1*k2 - g12*g2*k1*k2 - 2*h1*h2*k1*k2 - h12*h2*k1*k2 - 2*i1*i2*k1*k2 - i12*i2*k1*k2 - 2*j1*j2*k1*k2 - j12*j2*k1*k2 - b1*b2*k12*k2 - c1*c2*k12*k2 - d1*d2*k12*k2 - e1*e2*k12*k2 - f1*f2*k12*k2 - g1*g2*k12*k2 - h1*h2*k12*k2 - i1*i2*k12*k2 - j1*j2*k12*k2 + b1^2*k2^2 + b1*b12*k2^2 + c1^2*k2^2 + c1*c12*k2^2 + d1^2*k2^2 + d1*d12*k2^2 + e1^2*k2^2 + e1*e12*k2^2 + f1^2*k2^2 + f1*f12*k2^2 + g1^2*k2^2 + g1*g12*k2^2 + h1^2*k2^2 + h1*h12*k2^2 + i1^2*k2^2 + i1*i12*k2^2 + j1^2*k2^2 + j1*j12*k2^2 + a1^2*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2 + i2^2 + j2^2 + k2^2) + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2 + i2^2 + j2^2 + k2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2 + j1*j2 + k1*k2) + a1*(2*b1*b2 + b12*b2 + 2*c1*c2 + c12*c2 + 2*d1*d2 + d12*d2 + 2*e1*e2 + e12*e2 + 2*f1*f2 + f12*f2 + 2*g1*g2 + g12*g2 + 2*h1*h2 + h12*h2 + 2*i1*i2 + i12*i2 + 2*j1*j2 + j12*j2 + 2*k1*k2 + k12*k2))))
  } else if (n_rows == 12) {
    #m12
    EXPR1 <- expression((b2^2*c1*c12 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1*b12*c2^2 + b2^2*d1*d12 + c2^2*d1*d12 - b12*b2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1*b12*d2^2 + c1*c12*d2^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 - b12*b2*e1*e2 - c12*c2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1*b12*e2^2 + c1*c12*e2^2 + d1*d12*e2^2 + b2^2*f1*f12 + c2^2*f1*f12 + d2^2*f1*f12 + e2^2*f1*f12 - b12*b2*f1*f2 - c12*c2*f1*f2 - d12*d2*f1*f2 - e12*e2*f1*f2 - b1*b2*f12*f2 - c1*c2*f12*f2 - d1*d2*f12*f2 - e1*e2*f12*f2 + b1*b12*f2^2 + c1*c12*f2^2 + d1*d12*f2^2 + e1*e12*f2^2 + b2^2*g1*g12 + c2^2*g1*g12 + d2^2*g1*g12 + e2^2*g1*g12 + f2^2*g1*g12 - b12*b2*g1*g2 - c12*c2*g1*g2 - d12*d2*g1*g2 - e12*e2*g1*g2 - f12*f2*g1*g2 - b1*b2*g12*g2 - c1*c2*g12*g2 - d1*d2*g12*g2 - e1*e2*g12*g2 - f1*f2*g12*g2 + b1*b12*g2^2 + c1*c12*g2^2 + d1*d12*g2^2 + e1*e12*g2^2 + f1*f12*g2^2 + b2^2*h1*h12 + c2^2*h1*h12 + d2^2*h1*h12 + e2^2*h1*h12 + f2^2*h1*h12 + g2^2*h1*h12 - b12*b2*h1*h2 - c12*c2*h1*h2 - d12*d2*h1*h2 - e12*e2*h1*h2 - f12*f2*h1*h2 - g12*g2*h1*h2 - b1*b2*h12*h2 - c1*c2*h12*h2 - d1*d2*h12*h2 - e1*e2*h12*h2 - f1*f2*h12*h2 - g1*g2*h12*h2 + b1*b12*h2^2 + c1*c12*h2^2 + d1*d12*h2^2 + e1*e12*h2^2 + f1*f12*h2^2 + g1*g12*h2^2 + b2^2*i1*i12 + c2^2*i1*i12 + d2^2*i1*i12 + e2^2*i1*i12 + f2^2*i1*i12 + g2^2*i1*i12 + h2^2*i1*i12 - b12*b2*i1*i2 - c12*c2*i1*i2 - d12*d2*i1*i2 - e12*e2*i1*i2 - f12*f2*i1*i2 - g12*g2*i1*i2 - h12*h2*i1*i2 - b1*b2*i12*i2 - c1*c2*i12*i2 - d1*d2*i12*i2 - e1*e2*i12*i2 - f1*f2*i12*i2 - g1*g2*i12*i2 - h1*h2*i12*i2 + b1*b12*i2^2 + c1*c12*i2^2 + d1*d12*i2^2 + e1*e12*i2^2 + f1*f12*i2^2 + g1*g12*i2^2 + h1*h12*i2^2 + b2^2*j1*j12 + c2^2*j1*j12 + d2^2*j1*j12 + e2^2*j1*j12 + f2^2*j1*j12 + g2^2*j1*j12 + h2^2*j1*j12 + i2^2*j1*j12 - b12*b2*j1*j2 - c12*c2*j1*j2 - d12*d2*j1*j2 - e12*e2*j1*j2 - f12*f2*j1*j2 - g12*g2*j1*j2 - h12*h2*j1*j2 - i12*i2*j1*j2 - b1*b2*j12*j2 - c1*c2*j12*j2 - d1*d2*j12*j2 - e1*e2*j12*j2 - f1*f2*j12*j2 - g1*g2*j12*j2 - h1*h2*j12*j2 - i1*i2*j12*j2 + b1*b12*j2^2 + c1*c12*j2^2 + d1*d12*j2^2 + e1*e12*j2^2 + f1*f12*j2^2 + g1*g12*j2^2 + h1*h12*j2^2 + i1*i12*j2^2 + b2^2*k1*k12 + c2^2*k1*k12 + d2^2*k1*k12 + e2^2*k1*k12 + f2^2*k1*k12 + g2^2*k1*k12 + h2^2*k1*k12 + i2^2*k1*k12 + j2^2*k1*k12 - b12*b2*k1*k2 - c12*c2*k1*k2 - d12*d2*k1*k2 - e12*e2*k1*k2 - f12*f2*k1*k2 - g12*g2*k1*k2 - h12*h2*k1*k2 - i12*i2*k1*k2 - j12*j2*k1*k2 - b1*b2*k12*k2 - c1*c2*k12*k2 - d1*d2*k12*k2 - e1*e2*k12*k2 - f1*f2*k12*k2 - g1*g2*k12*k2 - h1*h2*k12*k2 - i1*i2*k12*k2 - j1*j2*k12*k2 + b1*b12*k2^2 + c1*c12*k2^2 + d1*d12*k2^2 + e1*e12*k2^2 + f1*f12*k2^2 + g1*g12*k2^2 + h1*h12*k2^2 + i1*i12*k2^2 + j1*j12*k2^2 + b2^2*l1*l12 + c2^2*l1*l12 + d2^2*l1*l12 + e2^2*l1*l12 + f2^2*l1*l12 + g2^2*l1*l12 + h2^2*l1*l12 + i2^2*l1*l12 + j2^2*l1*l12 + k2^2*l1*l12 + a2^2*(b1*b12 + c1*c12 + d1*d12 + e1*e12 + f1*f12 + g1*g12 + h1*h12 + i1*i12 + j1*j12 + k1*k12 + l1*l12) - b12*b2*l1*l2 - c12*c2*l1*l2 - d12*d2*l1*l2 - e12*e2*l1*l2 - f12*f2*l1*l2 - g12*g2*l1*l2 - h12*h2*l1*l2 - i12*i2*l1*l2 - j12*j2*l1*l2 - k12*k2*l1*l2 - b1*b2*l12*l2 - c1*c2*l12*l2 - d1*d2*l12*l2 - e1*e2*l12*l2 - f1*f2*l12*l2 - g1*g2*l12*l2 - h1*h2*l12*l2 - i1*i2*l12*l2 - j1*j2*l12*l2 - k1*k2*l12*l2 + b1*b12*l2^2 + c1*c12*l2^2 + d1*d12*l2^2 + e1*e12*l2^2 + f1*f12*l2^2 + g1*g12*l2^2 + h1*h12*l2^2 + i1*i12*l2^2 + j1*j12*l2^2 + k1*k12*l2^2 + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2 + i2^2 + j2^2 + k2^2 + l2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2 + j1*j2 + k1*k2 + l1*l2) + a1*(b12*b2 + c12*c2 + d12*d2 + e12*e2 + f12*f2 + g12*g2 + h12*h2 + i12*i2 + j12*j2 + k12*k2 + l12*l2)))/(a1^2*b12*b2 + a1^2*b2^2 + b12*b2*c1^2 + b2^2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 - 2*b1*b2*c1*c2 + a1^2*c12*c2 + b1^2*c12*c2 + a1^2*c2^2 + b1^2*c2^2 + b12*b2*d1^2 + b2^2*d1^2 + c12*c2*d1^2 + c2^2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - 2*b1*b2*d1*d2 - c1*c12*d1*d2 - 2*c1*c2*d1*d2 + a1^2*d12*d2 + b1^2*d12*d2 + c1^2*d12*d2 + a1^2*d2^2 + b1^2*d2^2 + c1^2*d2^2 + b12*b2*e1^2 + b2^2*e1^2 + c12*c2*e1^2 + c2^2*e1^2 + d12*d2*e1^2 + d2^2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - 2*b1*b2*e1*e2 - c1*c12*e1*e2 - 2*c1*c2*e1*e2 - d1*d12*e1*e2 - 2*d1*d2*e1*e2 + a1^2*e12*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + a1^2*e2^2 + b1^2*e2^2 + c1^2*e2^2 + d1^2*e2^2 + b12*b2*f1^2 + b2^2*f1^2 + c12*c2*f1^2 + c2^2*f1^2 + d12*d2*f1^2 + d2^2*f1^2 + e12*e2*f1^2 + e2^2*f1^2 - b1*b2*f1*f12 - c1*c2*f1*f12 - d1*d2*f1*f12 - e1*e2*f1*f12 - b1*b12*f1*f2 - 2*b1*b2*f1*f2 - c1*c12*f1*f2 - 2*c1*c2*f1*f2 - d1*d12*f1*f2 - 2*d1*d2*f1*f2 - e1*e12*f1*f2 - 2*e1*e2*f1*f2 + a1^2*f12*f2 + b1^2*f12*f2 + c1^2*f12*f2 + d1^2*f12*f2 + e1^2*f12*f2 + a1^2*f2^2 + b1^2*f2^2 + c1^2*f2^2 + d1^2*f2^2 + e1^2*f2^2 + b12*b2*g1^2 + b2^2*g1^2 + c12*c2*g1^2 + c2^2*g1^2 + d12*d2*g1^2 + d2^2*g1^2 + e12*e2*g1^2 + e2^2*g1^2 + f12*f2*g1^2 + f2^2*g1^2 - b1*b2*g1*g12 - c1*c2*g1*g12 - d1*d2*g1*g12 - e1*e2*g1*g12 - f1*f2*g1*g12 - b1*b12*g1*g2 - 2*b1*b2*g1*g2 - c1*c12*g1*g2 - 2*c1*c2*g1*g2 - d1*d12*g1*g2 - 2*d1*d2*g1*g2 - e1*e12*g1*g2 - 2*e1*e2*g1*g2 - f1*f12*g1*g2 - 2*f1*f2*g1*g2 + a1^2*g12*g2 + b1^2*g12*g2 + c1^2*g12*g2 + d1^2*g12*g2 + e1^2*g12*g2 + f1^2*g12*g2 + a1^2*g2^2 + b1^2*g2^2 + c1^2*g2^2 + d1^2*g2^2 + e1^2*g2^2 + f1^2*g2^2 + b12*b2*h1^2 + b2^2*h1^2 + c12*c2*h1^2 + c2^2*h1^2 + d12*d2*h1^2 + d2^2*h1^2 + e12*e2*h1^2 + e2^2*h1^2 + f12*f2*h1^2 + f2^2*h1^2 + g12*g2*h1^2 + g2^2*h1^2 - b1*b2*h1*h12 - c1*c2*h1*h12 - d1*d2*h1*h12 - e1*e2*h1*h12 - f1*f2*h1*h12 - g1*g2*h1*h12 - b1*b12*h1*h2 - 2*b1*b2*h1*h2 - c1*c12*h1*h2 - 2*c1*c2*h1*h2 - d1*d12*h1*h2 - 2*d1*d2*h1*h2 - e1*e12*h1*h2 - 2*e1*e2*h1*h2 - f1*f12*h1*h2 - 2*f1*f2*h1*h2 - g1*g12*h1*h2 - 2*g1*g2*h1*h2 + a1^2*h12*h2 + b1^2*h12*h2 + c1^2*h12*h2 + d1^2*h12*h2 + e1^2*h12*h2 + f1^2*h12*h2 + g1^2*h12*h2 + a1^2*h2^2 + b1^2*h2^2 + c1^2*h2^2 + d1^2*h2^2 + e1^2*h2^2 + f1^2*h2^2 + g1^2*h2^2 + b12*b2*i1^2 + b2^2*i1^2 + c12*c2*i1^2 + c2^2*i1^2 + d12*d2*i1^2 + d2^2*i1^2 + e12*e2*i1^2 + e2^2*i1^2 + f12*f2*i1^2 + f2^2*i1^2 + g12*g2*i1^2 + g2^2*i1^2 + h12*h2*i1^2 + h2^2*i1^2 - b1*b2*i1*i12 - c1*c2*i1*i12 - d1*d2*i1*i12 - e1*e2*i1*i12 - f1*f2*i1*i12 - g1*g2*i1*i12 - h1*h2*i1*i12 - b1*b12*i1*i2 - 2*b1*b2*i1*i2 - c1*c12*i1*i2 - 2*c1*c2*i1*i2 - d1*d12*i1*i2 - 2*d1*d2*i1*i2 - e1*e12*i1*i2 - 2*e1*e2*i1*i2 - f1*f12*i1*i2 - 2*f1*f2*i1*i2 - g1*g12*i1*i2 - 2*g1*g2*i1*i2 - h1*h12*i1*i2 - 2*h1*h2*i1*i2 + a1^2*i12*i2 + b1^2*i12*i2 + c1^2*i12*i2 + d1^2*i12*i2 + e1^2*i12*i2 + f1^2*i12*i2 + g1^2*i12*i2 + h1^2*i12*i2 + a1^2*i2^2 + b1^2*i2^2 + c1^2*i2^2 + d1^2*i2^2 + e1^2*i2^2 + f1^2*i2^2 + g1^2*i2^2 + h1^2*i2^2 + b12*b2*j1^2 + b2^2*j1^2 + c12*c2*j1^2 + c2^2*j1^2 + d12*d2*j1^2 + d2^2*j1^2 + e12*e2*j1^2 + e2^2*j1^2 + f12*f2*j1^2 + f2^2*j1^2 + g12*g2*j1^2 + g2^2*j1^2 + h12*h2*j1^2 + h2^2*j1^2 + i12*i2*j1^2 + i2^2*j1^2 - b1*b2*j1*j12 - c1*c2*j1*j12 - d1*d2*j1*j12 - e1*e2*j1*j12 - f1*f2*j1*j12 - g1*g2*j1*j12 - h1*h2*j1*j12 - i1*i2*j1*j12 - b1*b12*j1*j2 - 2*b1*b2*j1*j2 - c1*c12*j1*j2 - 2*c1*c2*j1*j2 - d1*d12*j1*j2 - 2*d1*d2*j1*j2 - e1*e12*j1*j2 - 2*e1*e2*j1*j2 - f1*f12*j1*j2 - 2*f1*f2*j1*j2 - g1*g12*j1*j2 - 2*g1*g2*j1*j2 - h1*h12*j1*j2 - 2*h1*h2*j1*j2 - i1*i12*j1*j2 - 2*i1*i2*j1*j2 + a1^2*j12*j2 + b1^2*j12*j2 + c1^2*j12*j2 + d1^2*j12*j2 + e1^2*j12*j2 + f1^2*j12*j2 + g1^2*j12*j2 + h1^2*j12*j2 + i1^2*j12*j2 + a1^2*j2^2 + b1^2*j2^2 + c1^2*j2^2 + d1^2*j2^2 + e1^2*j2^2 + f1^2*j2^2 + g1^2*j2^2 + h1^2*j2^2 + i1^2*j2^2 + b12*b2*k1^2 + b2^2*k1^2 + c12*c2*k1^2 + c2^2*k1^2 + d12*d2*k1^2 + d2^2*k1^2 + e12*e2*k1^2 + e2^2*k1^2 + f12*f2*k1^2 + f2^2*k1^2 + g12*g2*k1^2 + g2^2*k1^2 + h12*h2*k1^2 + h2^2*k1^2 + i12*i2*k1^2 + i2^2*k1^2 + j12*j2*k1^2 + j2^2*k1^2 - b1*b2*k1*k12 - c1*c2*k1*k12 - d1*d2*k1*k12 - e1*e2*k1*k12 - f1*f2*k1*k12 - g1*g2*k1*k12 - h1*h2*k1*k12 - i1*i2*k1*k12 - j1*j2*k1*k12 - b1*b12*k1*k2 - 2*b1*b2*k1*k2 - c1*c12*k1*k2 - 2*c1*c2*k1*k2 - d1*d12*k1*k2 - 2*d1*d2*k1*k2 - e1*e12*k1*k2 - 2*e1*e2*k1*k2 - f1*f12*k1*k2 - 2*f1*f2*k1*k2 - g1*g12*k1*k2 - 2*g1*g2*k1*k2 - h1*h12*k1*k2 - 2*h1*h2*k1*k2 - i1*i12*k1*k2 - 2*i1*i2*k1*k2 - j1*j12*k1*k2 - 2*j1*j2*k1*k2 + a1^2*k12*k2 + b1^2*k12*k2 + c1^2*k12*k2 + d1^2*k12*k2 + e1^2*k12*k2 + f1^2*k12*k2 + g1^2*k12*k2 + h1^2*k12*k2 + i1^2*k12*k2 + j1^2*k12*k2 + a1^2*k2^2 + b1^2*k2^2 + c1^2*k2^2 + d1^2*k2^2 + e1^2*k2^2 + f1^2*k2^2 + g1^2*k2^2 + h1^2*k2^2 + i1^2*k2^2 + j1^2*k2^2 + b12*b2*l1^2 + b2^2*l1^2 + c12*c2*l1^2 + c2^2*l1^2 + d12*d2*l1^2 + d2^2*l1^2 + e12*e2*l1^2 + e2^2*l1^2 + f12*f2*l1^2 + f2^2*l1^2 + g12*g2*l1^2 + g2^2*l1^2 + h12*h2*l1^2 + h2^2*l1^2 + i12*i2*l1^2 + i2^2*l1^2 + j12*j2*l1^2 + j2^2*l1^2 + k12*k2*l1^2 + k2^2*l1^2 + a2^2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2 + i1^2 + j1^2 + k1^2 + l1^2) - b1*b2*l1*l12 - c1*c2*l1*l12 - d1*d2*l1*l12 - e1*e2*l1*l12 - f1*f2*l1*l12 - g1*g2*l1*l12 - h1*h2*l1*l12 - i1*i2*l1*l12 - j1*j2*l1*l12 - k1*k2*l1*l12 - b1*b12*l1*l2 - 2*b1*b2*l1*l2 - c1*c12*l1*l2 - 2*c1*c2*l1*l2 - d1*d12*l1*l2 - 2*d1*d2*l1*l2 - e1*e12*l1*l2 - 2*e1*e2*l1*l2 - f1*f12*l1*l2 - 2*f1*f2*l1*l2 - g1*g12*l1*l2 - 2*g1*g2*l1*l2 - h1*h12*l1*l2 - 2*h1*h2*l1*l2 - i1*i12*l1*l2 - 2*i1*i2*l1*l2 - j1*j12*l1*l2 - 2*j1*j2*l1*l2 - k1*k12*l1*l2 - 2*k1*k2*l1*l2 + a1^2*l12*l2 + b1^2*l12*l2 + c1^2*l12*l2 + d1^2*l12*l2 + e1^2*l12*l2 + f1^2*l12*l2 + g1^2*l12*l2 + h1^2*l12*l2 + i1^2*l12*l2 + j1^2*l12*l2 + k1^2*l12*l2 + a1^2*l2^2 + b1^2*l2^2 + c1^2*l2^2 + d1^2*l2^2 + e1^2*l2^2 + f1^2*l2^2 + g1^2*l2^2 + h1^2*l2^2 + i1^2*l2^2 + j1^2*l2^2 + k1^2*l2^2 - a1*a2*(b1*b12 + 2*b1*b2 + c1*c12 + 2*c1*c2 + d1*d12 + 2*d1*d2 + e1*e12 + 2*e1*e2 + f1*f12 + 2*f1*f2 + g1*g12 + 2*g1*g2 + h1*h12 + 2*h1*h2 + i1*i12 + 2*i1*i2 + j1*j12 + 2*j1*j2 + k1*k12 + 2*k1*k2 + l1*l12 + 2*l1*l2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2 + i1^2 + j1^2 + k1^2 + l1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2 + j1*j2 + k1*k2 + l1*l2))))
                           #m21
                         EXPR2 <- expression((b12*b2*c1^2 - b1*b2*c1*c12 - b1*b12*c1*c2 + b1^2*c12*c2 + b12*b2*d1^2 + c12*c2*d1^2 - b1*b2*d1*d12 - c1*c2*d1*d12 - b1*b12*d1*d2 - c1*c12*d1*d2 + b1^2*d12*d2 + c1^2*d12*d2 + b12*b2*e1^2 + c12*c2*e1^2 + d12*d2*e1^2 - b1*b2*e1*e12 - c1*c2*e1*e12 - d1*d2*e1*e12 - b1*b12*e1*e2 - c1*c12*e1*e2 - d1*d12*e1*e2 + b1^2*e12*e2 + c1^2*e12*e2 + d1^2*e12*e2 + b12*b2*f1^2 + c12*c2*f1^2 + d12*d2*f1^2 + e12*e2*f1^2 - b1*b2*f1*f12 - c1*c2*f1*f12 - d1*d2*f1*f12 - e1*e2*f1*f12 - b1*b12*f1*f2 - c1*c12*f1*f2 - d1*d12*f1*f2 - e1*e12*f1*f2 + b1^2*f12*f2 + c1^2*f12*f2 + d1^2*f12*f2 + e1^2*f12*f2 + b12*b2*g1^2 + c12*c2*g1^2 + d12*d2*g1^2 + e12*e2*g1^2 + f12*f2*g1^2 - b1*b2*g1*g12 - c1*c2*g1*g12 - d1*d2*g1*g12 - e1*e2*g1*g12 - f1*f2*g1*g12 - b1*b12*g1*g2 - c1*c12*g1*g2 - d1*d12*g1*g2 - e1*e12*g1*g2 - f1*f12*g1*g2 + b1^2*g12*g2 + c1^2*g12*g2 + d1^2*g12*g2 + e1^2*g12*g2 + f1^2*g12*g2 + b12*b2*h1^2 + c12*c2*h1^2 + d12*d2*h1^2 + e12*e2*h1^2 + f12*f2*h1^2 + g12*g2*h1^2 - b1*b2*h1*h12 - c1*c2*h1*h12 - d1*d2*h1*h12 - e1*e2*h1*h12 - f1*f2*h1*h12 - g1*g2*h1*h12 - b1*b12*h1*h2 - c1*c12*h1*h2 - d1*d12*h1*h2 - e1*e12*h1*h2 - f1*f12*h1*h2 - g1*g12*h1*h2 + b1^2*h12*h2 + c1^2*h12*h2 + d1^2*h12*h2 + e1^2*h12*h2 + f1^2*h12*h2 + g1^2*h12*h2 + b12*b2*i1^2 + c12*c2*i1^2 + d12*d2*i1^2 + e12*e2*i1^2 + f12*f2*i1^2 + g12*g2*i1^2 + h12*h2*i1^2 - b1*b2*i1*i12 - c1*c2*i1*i12 - d1*d2*i1*i12 - e1*e2*i1*i12 - f1*f2*i1*i12 - g1*g2*i1*i12 - h1*h2*i1*i12 - b1*b12*i1*i2 - c1*c12*i1*i2 - d1*d12*i1*i2 - e1*e12*i1*i2 - f1*f12*i1*i2 - g1*g12*i1*i2 - h1*h12*i1*i2 + b1^2*i12*i2 + c1^2*i12*i2 + d1^2*i12*i2 + e1^2*i12*i2 + f1^2*i12*i2 + g1^2*i12*i2 + h1^2*i12*i2 + b12*b2*j1^2 + c12*c2*j1^2 + d12*d2*j1^2 + e12*e2*j1^2 + f12*f2*j1^2 + g12*g2*j1^2 + h12*h2*j1^2 + i12*i2*j1^2 - b1*b2*j1*j12 - c1*c2*j1*j12 - d1*d2*j1*j12 - e1*e2*j1*j12 - f1*f2*j1*j12 - g1*g2*j1*j12 - h1*h2*j1*j12 - i1*i2*j1*j12 - b1*b12*j1*j2 - c1*c12*j1*j2 - d1*d12*j1*j2 - e1*e12*j1*j2 - f1*f12*j1*j2 - g1*g12*j1*j2 - h1*h12*j1*j2 - i1*i12*j1*j2 + b1^2*j12*j2 + c1^2*j12*j2 + d1^2*j12*j2 + e1^2*j12*j2 + f1^2*j12*j2 + g1^2*j12*j2 + h1^2*j12*j2 + i1^2*j12*j2 + b12*b2*k1^2 + c12*c2*k1^2 + d12*d2*k1^2 + e12*e2*k1^2 + f12*f2*k1^2 + g12*g2*k1^2 + h12*h2*k1^2 + i12*i2*k1^2 + j12*j2*k1^2 - b1*b2*k1*k12 - c1*c2*k1*k12 - d1*d2*k1*k12 - e1*e2*k1*k12 - f1*f2*k1*k12 - g1*g2*k1*k12 - h1*h2*k1*k12 - i1*i2*k1*k12 - j1*j2*k1*k12 - b1*b12*k1*k2 - c1*c12*k1*k2 - d1*d12*k1*k2 - e1*e12*k1*k2 - f1*f12*k1*k2 - g1*g12*k1*k2 - h1*h12*k1*k2 - i1*i12*k1*k2 - j1*j12*k1*k2 + b1^2*k12*k2 + c1^2*k12*k2 + d1^2*k12*k2 + e1^2*k12*k2 + f1^2*k12*k2 + g1^2*k12*k2 + h1^2*k12*k2 + i1^2*k12*k2 + j1^2*k12*k2 + b12*b2*l1^2 + c12*c2*l1^2 + d12*d2*l1^2 + e12*e2*l1^2 + f12*f2*l1^2 + g12*g2*l1^2 + h12*h2*l1^2 + i12*i2*l1^2 + j12*j2*l1^2 + k12*k2*l1^2 - b1*b2*l1*l12 - c1*c2*l1*l12 - d1*d2*l1*l12 - e1*e2*l1*l12 - f1*f2*l1*l12 - g1*g2*l1*l12 - h1*h2*l1*l12 - i1*i2*l1*l12 - j1*j2*l1*l12 - k1*k2*l1*l12 - a1*a2*(b1*b12 + c1*c12 + d1*d12 + e1*e12 + f1*f12 + g1*g12 + h1*h12 + i1*i12 + j1*j12 + k1*k12 + l1*l12) - b1*b12*l1*l2 - c1*c12*l1*l2 - d1*d12*l1*l2 - e1*e12*l1*l2 - f1*f12*l1*l2 - g1*g12*l1*l2 - h1*h12*l1*l2 - i1*i12*l1*l2 - j1*j12*l1*l2 - k1*k12*l1*l2 + b1^2*l12*l2 + c1^2*l12*l2 + d1^2*l12*l2 + e1^2*l12*l2 + f1^2*l12*l2 + g1^2*l12*l2 + h1^2*l12*l2 + i1^2*l12*l2 + j1^2*l12*l2 + k1^2*l12*l2 + a1^2*(b12*b2 + c12*c2 + d12*d2 + e12*e2 + f12*f2 + g12*g2 + h12*h2 + i12*i2 + j12*j2 + k12*k2 + l12*l2) + a12*(a2*(b1^2 + c1^2 + d1^2 + e1^2 + f1^2 + g1^2 + h1^2 + i1^2 + j1^2 + k1^2 + l1^2) - a1*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2 + j1*j2 + k1*k2 + l1*l2)))/(b2^2*c1^2 + b2^2*c1*c12 - 2*b1*b2*c1*c2 - b12*b2*c1*c2 - b1*b2*c12*c2 + b1^2*c2^2 + b1*b12*c2^2 + b2^2*d1^2 + c2^2*d1^2 + b2^2*d1*d12 + c2^2*d1*d12 - 2*b1*b2*d1*d2 - b12*b2*d1*d2 - 2*c1*c2*d1*d2 - c12*c2*d1*d2 - b1*b2*d12*d2 - c1*c2*d12*d2 + b1^2*d2^2 + b1*b12*d2^2 + c1^2*d2^2 + c1*c12*d2^2 + b2^2*e1^2 + c2^2*e1^2 + d2^2*e1^2 + b2^2*e1*e12 + c2^2*e1*e12 + d2^2*e1*e12 - 2*b1*b2*e1*e2 - b12*b2*e1*e2 - 2*c1*c2*e1*e2 - c12*c2*e1*e2 - 2*d1*d2*e1*e2 - d12*d2*e1*e2 - b1*b2*e12*e2 - c1*c2*e12*e2 - d1*d2*e12*e2 + b1^2*e2^2 + b1*b12*e2^2 + c1^2*e2^2 + c1*c12*e2^2 + d1^2*e2^2 + d1*d12*e2^2 + b2^2*f1^2 + c2^2*f1^2 + d2^2*f1^2 + e2^2*f1^2 + b2^2*f1*f12 + c2^2*f1*f12 + d2^2*f1*f12 + e2^2*f1*f12 - 2*b1*b2*f1*f2 - b12*b2*f1*f2 - 2*c1*c2*f1*f2 - c12*c2*f1*f2 - 2*d1*d2*f1*f2 - d12*d2*f1*f2 - 2*e1*e2*f1*f2 - e12*e2*f1*f2 - b1*b2*f12*f2 - c1*c2*f12*f2 - d1*d2*f12*f2 - e1*e2*f12*f2 + b1^2*f2^2 + b1*b12*f2^2 + c1^2*f2^2 + c1*c12*f2^2 + d1^2*f2^2 + d1*d12*f2^2 + e1^2*f2^2 + e1*e12*f2^2 + b2^2*g1^2 + c2^2*g1^2 + d2^2*g1^2 + e2^2*g1^2 + f2^2*g1^2 + b2^2*g1*g12 + c2^2*g1*g12 + d2^2*g1*g12 + e2^2*g1*g12 + f2^2*g1*g12 - 2*b1*b2*g1*g2 - b12*b2*g1*g2 - 2*c1*c2*g1*g2 - c12*c2*g1*g2 - 2*d1*d2*g1*g2 - d12*d2*g1*g2 - 2*e1*e2*g1*g2 - e12*e2*g1*g2 - 2*f1*f2*g1*g2 - f12*f2*g1*g2 - b1*b2*g12*g2 - c1*c2*g12*g2 - d1*d2*g12*g2 - e1*e2*g12*g2 - f1*f2*g12*g2 + b1^2*g2^2 + b1*b12*g2^2 + c1^2*g2^2 + c1*c12*g2^2 + d1^2*g2^2 + d1*d12*g2^2 + e1^2*g2^2 + e1*e12*g2^2 + f1^2*g2^2 + f1*f12*g2^2 + b2^2*h1^2 + c2^2*h1^2 + d2^2*h1^2 + e2^2*h1^2 + f2^2*h1^2 + g2^2*h1^2 + b2^2*h1*h12 + c2^2*h1*h12 + d2^2*h1*h12 + e2^2*h1*h12 + f2^2*h1*h12 + g2^2*h1*h12 - 2*b1*b2*h1*h2 - b12*b2*h1*h2 - 2*c1*c2*h1*h2 - c12*c2*h1*h2 - 2*d1*d2*h1*h2 - d12*d2*h1*h2 - 2*e1*e2*h1*h2 - e12*e2*h1*h2 - 2*f1*f2*h1*h2 - f12*f2*h1*h2 - 2*g1*g2*h1*h2 - g12*g2*h1*h2 - b1*b2*h12*h2 - c1*c2*h12*h2 - d1*d2*h12*h2 - e1*e2*h12*h2 - f1*f2*h12*h2 - g1*g2*h12*h2 + b1^2*h2^2 + b1*b12*h2^2 + c1^2*h2^2 + c1*c12*h2^2 + d1^2*h2^2 + d1*d12*h2^2 + e1^2*h2^2 + e1*e12*h2^2 + f1^2*h2^2 + f1*f12*h2^2 + g1^2*h2^2 + g1*g12*h2^2 + b2^2*i1^2 + c2^2*i1^2 + d2^2*i1^2 + e2^2*i1^2 + f2^2*i1^2 + g2^2*i1^2 + h2^2*i1^2 + b2^2*i1*i12 + c2^2*i1*i12 + d2^2*i1*i12 + e2^2*i1*i12 + f2^2*i1*i12 + g2^2*i1*i12 + h2^2*i1*i12 - 2*b1*b2*i1*i2 - b12*b2*i1*i2 - 2*c1*c2*i1*i2 - c12*c2*i1*i2 - 2*d1*d2*i1*i2 - d12*d2*i1*i2 - 2*e1*e2*i1*i2 - e12*e2*i1*i2 - 2*f1*f2*i1*i2 - f12*f2*i1*i2 - 2*g1*g2*i1*i2 - g12*g2*i1*i2 - 2*h1*h2*i1*i2 - h12*h2*i1*i2 - b1*b2*i12*i2 - c1*c2*i12*i2 - d1*d2*i12*i2 - e1*e2*i12*i2 - f1*f2*i12*i2 - g1*g2*i12*i2 - h1*h2*i12*i2 + b1^2*i2^2 + b1*b12*i2^2 + c1^2*i2^2 + c1*c12*i2^2 + d1^2*i2^2 + d1*d12*i2^2 + e1^2*i2^2 + e1*e12*i2^2 + f1^2*i2^2 + f1*f12*i2^2 + g1^2*i2^2 + g1*g12*i2^2 + h1^2*i2^2 + h1*h12*i2^2 + b2^2*j1^2 + c2^2*j1^2 + d2^2*j1^2 + e2^2*j1^2 + f2^2*j1^2 + g2^2*j1^2 + h2^2*j1^2 + i2^2*j1^2 + b2^2*j1*j12 + c2^2*j1*j12 + d2^2*j1*j12 + e2^2*j1*j12 + f2^2*j1*j12 + g2^2*j1*j12 + h2^2*j1*j12 + i2^2*j1*j12 - 2*b1*b2*j1*j2 - b12*b2*j1*j2 - 2*c1*c2*j1*j2 - c12*c2*j1*j2 - 2*d1*d2*j1*j2 - d12*d2*j1*j2 - 2*e1*e2*j1*j2 - e12*e2*j1*j2 - 2*f1*f2*j1*j2 - f12*f2*j1*j2 - 2*g1*g2*j1*j2 - g12*g2*j1*j2 - 2*h1*h2*j1*j2 - h12*h2*j1*j2 - 2*i1*i2*j1*j2 - i12*i2*j1*j2 - b1*b2*j12*j2 - c1*c2*j12*j2 - d1*d2*j12*j2 - e1*e2*j12*j2 - f1*f2*j12*j2 - g1*g2*j12*j2 - h1*h2*j12*j2 - i1*i2*j12*j2 + b1^2*j2^2 + b1*b12*j2^2 + c1^2*j2^2 + c1*c12*j2^2 + d1^2*j2^2 + d1*d12*j2^2 + e1^2*j2^2 + e1*e12*j2^2 + f1^2*j2^2 + f1*f12*j2^2 + g1^2*j2^2 + g1*g12*j2^2 + h1^2*j2^2 + h1*h12*j2^2 + i1^2*j2^2 + i1*i12*j2^2 + b2^2*k1^2 + c2^2*k1^2 + d2^2*k1^2 + e2^2*k1^2 + f2^2*k1^2 + g2^2*k1^2 + h2^2*k1^2 + i2^2*k1^2 + j2^2*k1^2 + b2^2*k1*k12 + c2^2*k1*k12 + d2^2*k1*k12 + e2^2*k1*k12 + f2^2*k1*k12 + g2^2*k1*k12 + h2^2*k1*k12 + i2^2*k1*k12 + j2^2*k1*k12 - 2*b1*b2*k1*k2 - b12*b2*k1*k2 - 2*c1*c2*k1*k2 - c12*c2*k1*k2 - 2*d1*d2*k1*k2 - d12*d2*k1*k2 - 2*e1*e2*k1*k2 - e12*e2*k1*k2 - 2*f1*f2*k1*k2 - f12*f2*k1*k2 - 2*g1*g2*k1*k2 - g12*g2*k1*k2 - 2*h1*h2*k1*k2 - h12*h2*k1*k2 - 2*i1*i2*k1*k2 - i12*i2*k1*k2 - 2*j1*j2*k1*k2 - j12*j2*k1*k2 - b1*b2*k12*k2 - c1*c2*k12*k2 - d1*d2*k12*k2 - e1*e2*k12*k2 - f1*f2*k12*k2 - g1*g2*k12*k2 - h1*h2*k12*k2 - i1*i2*k12*k2 - j1*j2*k12*k2 + b1^2*k2^2 + b1*b12*k2^2 + c1^2*k2^2 + c1*c12*k2^2 + d1^2*k2^2 + d1*d12*k2^2 + e1^2*k2^2 + e1*e12*k2^2 + f1^2*k2^2 + f1*f12*k2^2 + g1^2*k2^2 + g1*g12*k2^2 + h1^2*k2^2 + h1*h12*k2^2 + i1^2*k2^2 + i1*i12*k2^2 + j1^2*k2^2 + j1*j12*k2^2 + b2^2*l1^2 + c2^2*l1^2 + d2^2*l1^2 + e2^2*l1^2 + f2^2*l1^2 + g2^2*l1^2 + h2^2*l1^2 + i2^2*l1^2 + j2^2*l1^2 + k2^2*l1^2 + b2^2*l1*l12 + c2^2*l1*l12 + d2^2*l1*l12 + e2^2*l1*l12 + f2^2*l1*l12 + g2^2*l1*l12 + h2^2*l1*l12 + i2^2*l1*l12 + j2^2*l1*l12 + k2^2*l1*l12 + a2^2*(b1^2 + b1*b12 + c1^2 + c1*c12 + d1^2 + d1*d12 + e1^2 + e1*e12 + f1^2 + f1*f12 + g1^2 + g1*g12 + h1^2 + h1*h12 + i1^2 + i1*i12 + j1^2 + j1*j12 + k1^2 + k1*k12 + l1^2 + l1*l12) - 2*b1*b2*l1*l2 - b12*b2*l1*l2 - 2*c1*c2*l1*l2 - c12*c2*l1*l2 - 2*d1*d2*l1*l2 - d12*d2*l1*l2 - 2*e1*e2*l1*l2 - e12*e2*l1*l2 - 2*f1*f2*l1*l2 - f12*f2*l1*l2 - 2*g1*g2*l1*l2 - g12*g2*l1*l2 - 2*h1*h2*l1*l2 - h12*h2*l1*l2 - 2*i1*i2*l1*l2 - i12*i2*l1*l2 - 2*j1*j2*l1*l2 - j12*j2*l1*l2 - 2*k1*k2*l1*l2 - k12*k2*l1*l2 - b1*b2*l12*l2 - c1*c2*l12*l2 - d1*d2*l12*l2 - e1*e2*l12*l2 - f1*f2*l12*l2 - g1*g2*l12*l2 - h1*h2*l12*l2 - i1*i2*l12*l2 - j1*j2*l12*l2 - k1*k2*l12*l2 + b1^2*l2^2 + b1*b12*l2^2 + c1^2*l2^2 + c1*c12*l2^2 + d1^2*l2^2 + d1*d12*l2^2 + e1^2*l2^2 + e1*e12*l2^2 + f1^2*l2^2 + f1*f12*l2^2 + g1^2*l2^2 + g1*g12*l2^2 + h1^2*l2^2 + h1*h12*l2^2 + i1^2*l2^2 + i1*i12*l2^2 + j1^2*l2^2 + j1*j12*l2^2 + k1^2*l2^2 + k1*k12*l2^2 + a1^2*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2 + i2^2 + j2^2 + k2^2 + l2^2) + a1*a12*(b2^2 + c2^2 + d2^2 + e2^2 + f2^2 + g2^2 + h2^2 + i2^2 + j2^2 + k2^2 + l2^2) - a2*(a12*(b1*b2 + c1*c2 + d1*d2 + e1*e2 + f1*f2 + g1*g2 + h1*h2 + i1*i2 + j1*j2 + k1*k2 + l1*l2) + a1*(2*b1*b2 + b12*b2 + 2*c1*c2 + c12*c2 + 2*d1*d2 + d12*d2 + 2*e1*e2 + e12*e2 + 2*f1*f2 + f12*f2 + 2*g1*g2 + g12*g2 + 2*h1*h2 + h12*h2 + 2*i1*i2 + i12*i2 + 2*j1*j2 + j12*j2 + 2*k1*k2 + k12*k2 + 2*l1*l2 + l12*l2))))
  }
  
  # using the expressions for m12 and m21 above, propagate errors
  m12_errors <- propagate(expr = EXPR1, data = full_matrix, second.order = TRUE, cov=beta_cov, do.sim=FALSE)
  m21_errors <- propagate(expr = EXPR2, data = full_matrix, second.order = TRUE, cov=beta_cov, do.sim=FALSE)
  results <- cbind(markers[1],markers[2],m12,m12_errors$prop[[4]],m21,m21_errors$prop[[4]])
  colnames(results) <- c("marker 1","marker 2","m12","m12_std_dev","m21","m21_std_dev")
  rownames(results)  <- NULL
  return(results)
}
