## ----message=FALSE, include=FALSE---------------------------------------------

rm(list=ls())

library(sf)
library(dplyr)
library(cartograflow)
library(cartography)


## ----flowdata_preprocess, echo=FALSE, message=FALSE, warning=FALSE------------

# Load example
data<-read.csv2("./data/MOBPRO_ETP.csv",
               header=TRUE,
               sep=";",
               stringsAsFactors=FALSE,
               encoding="UTF-8",
               dec=".",
               check.names=FALSE)

# Variable typing
data$i<-as.character(data$i)
data$j<-as.character(data$j)
data$Fij<-as.numeric(data$Fij)
data$count<-as.numeric(data$count)

str(data)


## ----flowdata_reerse, warning=FALSE, echo=TRUE--------------------------------

# Selecting useful variables
tabflow<-data%>%select(i,j,Fij)

# Change matrix format (if necessary)
matflow <-flowtabmat(tabflow,matlist="M")
head(matflow[1:4,1:4])
#dim(matflow)

# reverse Change matrix format : from matrix to list
tabflow<-flowtabmat(tab=matflow, matlist="L")
colnames(tabflow)<-c("i","j","Fij")
head(tabflow)


## ----data_preprocess, warning=FALSE, echo=TRUE--------------------------------

# Load a list of geo codes
ID_CODE<-read.csv2("./data/COD_GEO_EPT.csv",
                   header=TRUE,
                   sep=";",
                   stringsAsFactors=FALSE,
                   encoding="UTF-8",
                   dec=".",
                   check.names=FALSE)
#head(ID_CODE)

CODE<-ID_CODE%>% dplyr::select(COD_GEO_EPT)

colnames(CODE)<-c("CODGEO")
#head(CODE)


## ----data_computing, warning=FALSE, echo=TRUE---------------------------------

# Compute bilateral volum : FSij
matflow_vol<-flowtype(matflow, format="M", x="bivolum")

tabflow_vol<-flowtype(tabflow, origin ="i",destination="j",fij="Fij",format="L", x="bivolum")

# Compute bilateral balance : FSij
tabflow_net<-flowtype(tabflow, origin ="i",destination="j",fij="Fij",format="L", x="bibal")

# Compute all types of bilateral flows, in one 6 columns "L"format matrix
tabflow_all<-flowtype(tabflow, origin ="i",destination="j",fij="Fij",format="L", x="alltypes")
head(tabflow_all) 
 
# Compute flow asymetry
#tabflow_all$FAsy<-(tabflow_all$FDij / tabflow_all$FDij)*100


## ----maps, fig.width=7, message=FALSE, warning=FALSE, include=FALSE-----------

library(sf)
data(flowdata)
map <- st_read(system.file("shape/MGP_TER.shp", package = "cartograflow"))


## ----maps_links, fig.show='hold', fig.width=7, message=TRUE, warning=FALSE, ECHO=FALSE, include=FALSE----

# Add and overlay spatial background 
par(bg = "NA")

knitr::opts_chunk$set(fig.width=6, fig.height=6)
par(mar=c(0,0,1,0))
extent <- c(2800000, 1340000, 6400000, 4800000)
resolution<-150

# Add the corresponding background 
plot(st_geometry(map), col = NA, border=NA, bg="#dfe6e1")
plot(st_geometry(map), col = "light grey", add=TRUE)

# Flowmapping of all links

flowmap(tab=tabflow,
        fij="Fij",
        origin.f = "i",
        destination.f = "j",
        bkg = map,
        code="EPT_NUM",
        nodes.X="X",
        nodes.Y = "Y",
        filter=FALSE,
        add=TRUE
        )

# Map cosmetics
layoutLayer(title = "All origin-destination for commuting in Greater Paris, 2017",
           coltitle ="black",
           author = "Cartograflow, 2020",
           sources = "Data : INSEE, 2017 ; Basemap : APUR, RIATE, 2018.",
           scale = 2,
           tabtitle = FALSE,
           frame = TRUE,
           col = "grey"
            )
# North arrow
north("topright")


## ----maps_flowmean, echo=TRUE, fig.show='hold', fig.width=7, message=FALSE, warning=FALSE, ECHO=FALSE----

library(sf)
data(flowdata)
map <- st_read(system.file("shape/MGP_TER.shp", package = "cartograflow"))

# Add and overlay spatial background 
par(bg = "NA")

knitr::opts_chunk$set(fig.width=6, fig.height=6)
par(mar=c(0,0,1,0))
extent <- c(2800000, 1340000, 6400000, 4800000)
resolution<-150

plot(st_geometry(map), col = NA, border=NA, bg="#dfe6e1")
plot(st_geometry(map), col = "light grey", add=TRUE)

# Flow mapping above-average flows
flowmap(tab=tabflow,
        fij="Fij",
        origin.f = "i",
        destination.f = "j",
        bkg = map,
        code="EPT_NUM",
        nodes.X="X",
        nodes.Y = "Y",
        filter=TRUE,
        threshold =(mean(tabflow$Fij)),  #mean value is the level of threshold
        taille=20,           
        a.head = 1,
        a.length = 0.11,
        a.angle = 30,
        a.col="#138913",
        add=TRUE)

# Map Legend
legendPropLines(pos="topleft",
                title.txt="Flows up to 13220 commuters",
                title.cex=0.8,   
                cex=0.5,
                values.cex= 0.7,  
                var=c(mean(tabflow$Fij),max(tabflow$Fij)), 
                lwd=5, 
                frame = FALSE,
                col="#138913",
                values.rnd = 0
                )

#Map cosmetic

layoutLayer(title = "Commuters up to above-average in Greater Paris",
           coltitle ="black",
           author = "Cartograflow, 2020",
           sources = "Data : INSEE, 2017 ; Basemap : APUR, RIATE, 2018.",
           scale = 2,
           tabtitle = FALSE,
           frame = TRUE,
           col = "grey"
            )

# North arrow
north("topright")


## ----analysis, fig.show='hold',message = TRUE , warning=FALSE-----------------

tabgini<-flowgini(ODpts = tabflow,
                  origin="i",destination = "j",valflow = "Fij",
                  lorenz.plot = FALSE)

# Interpretation ; The flows are quite concentrated on a few links, the Gini coefficent is equal to 73.16% 


## ----analysis_lorenz, fig.show='hold',message = TRUE , warning=FALSE----------

head(tabgini)

flowgini(ODpts = tabflow,
         origin="i",destination = "j",valflow = "Fij",
         lorenz.plot = TRUE)


## ----echo=TRUE, fig.show='hold', message=TRUE, warning=FALSE------------------

flowanalysis(tabgini,
             critflow = 0.8,
             result = "signif")

# Interpretation : Flow values up to 13442 are the 80% largest one corresponding to 23,14 % of the total links' features.


## ----echo=TRUE, fig.show='hold', message=TRUE, warning=FALSE------------------

# Graphic parameters
knitr::opts_chunk$set(fig.width=6, fig.height=6)
par(mar=c(0,0,1,0))
extent <- c(2800000, 1340000, 6400000, 4800000)
resolution<-150

# Overlay a spatial background 
par(bg = "NA")

# Add the corresponding background 
plot(st_geometry(map), col = NA, border=NA, bg="#dfe6e1")
plot(st_geometry(map), col = "light grey", add=TRUE)

# For mapping flow up to 13342

flowmap(tab=tabflow,
        fij="Fij",origin.f = "i",destination.f = "j",
        bkg = map,code="EPT_NUM",nodes.X="X",nodes.Y = "Y",
        add=TRUE,
        filter=TRUE,
        threshold=13442,    
        taille=15,           
        a.head = 1,
        a.length = 0.11,
        a.angle = 30,
        a.col="#138913")

# Map Legend
legendPropLines(pos="topleft",
                title.txt="Commuters up to 13442\n (80% of the largest flows)",
                title.cex=0.8,   
                cex=0.5,
                values.cex= 0.7,  
                var=c(13442,max(tabflow$Fij)), 
                lwd=15, 
                frame = FALSE,
                col="#138913",
                values.rnd = 0
                )

#Map cosmetic

layoutLayer(title = "Significant professional mobility in Greater Paris",
           coltitle ="black",
           author = "Cartograflow, 2020",
           sources = "Data : INSEE, 2017 ; Basemap : APUR, RIATE, 2018.",
           scale = 2,
           tabtitle = FALSE,
           frame = TRUE,
           col = "grey",
            )
# north arrow
north("topright")


## ----fig.show='hold', warning=FALSE, include=FALSE----------------------------

flowanalysis(tabgini,
             critlink = 0.1,
             result = "density")

# Interpretation : Flows up to 45772 are the 58.12% largest one corresponding to 10 % of the total links

# Graphic parameters
knitr::opts_chunk$set(fig.width=6, fig.height=6)
par(mar=c(0,0,1,0))
extent <- c(2800000, 1340000, 6400000, 4800000)
resolution<-150

# Overlay a spatial background 
par(bg = "NA")

# Add the corresponding background 
plot(st_geometry(map), col = NA, border=NA, bg="#dfe6e1")
plot(st_geometry(map), col = "light grey", add=TRUE)

# For mapping 10% of the total features as flows up to 45772 commuters

flowmap(tab=tabgini,
        fij="Fij",origin.f = "i",destination.f = "j",
        bkg = map,code="EPT_NUM",nodes.X="X",nodes.Y = "Y",
        add=TRUE,
        plota = TRUE,
        filter=TRUE,
        threshold=45772,    
        taille=15,           
        a.head = 1,
        a.length = 0.15,
        a.angle = 30,
        a.col="#138913")

# Map Legend
legendPropLines(pos="topleft",
                title.txt="Number of commuters up to 45772\n(10% of the links)",
                title.cex=0.8,   
                cex=0.5,
                values.cex= 0.7,  
                var=c(45772,max(tabgini$Fij)), 
                lwd=15, 
                frame = FALSE,
                col="#138913",
                values.rnd = 0
                )

#Map cosmetic
layoutLayer(title = "Low density professional mobility in Greater Paris",
           coltitle ="black",
           author = "Cartograflow, 2020",
           sources = "Data : INSEE, 2017 ; Basemap : APUR, RIATE, 2018.",
           scale = 2,
           tabtitle = FALSE,
           frame = TRUE,
           col = "grey"
            )

# north arrow
north("topright")


## ----echo=TRUE, fig.show='hold'-----------------------------------------------

head(tabflow)

tab<-flowjointure(geom="area",DF.flow=tabflow,origin = "i",destination = "j",
                   bkg=map,id="EPT_NUM",x="X",y="Y")

tab.distance<-flowdist(tab,
                       dist.method = "euclidian",
                       result = "dist")

tab.distance<-tab.distance %>% select(i,j,distance)
tab<-tab %>% select(i,j,ydata)
head(tab.distance)


## ----echo=TRUE, fig.show='hold'-----------------------------------------------

#reduce the flow dataset from a selected distance travelled (eg. 8.5 km)
library(rlang)

tab.flow<-flowreduct(tab,
                     tab.distance,
                     metric = "continous",
                     d.criteria = "dmax", #max distance parameter 
                     d = 8567)        #max distance value - Q1 : 8567 km

#select for all i,j flow values up to 0
flow.d<-tab.flow %>%
        select(i,j,flowfilter) %>%
        filter(flowfilter !=0)
head(flow.d)


## ----echo=TRUE, fig.show='hold'-----------------------------------------------

# Graphic parameters
knitr::opts_chunk$set(fig.width=6, fig.height=6)
par(mar=c(0,0,1,0))
extent <- c(2800000, 1340000, 6400000, 4800000)
resolution<-150

# Overlay a spatial background 
par(bg = "NA")

# Add the corresponding background 
plot(st_geometry(map), col = NA, border=NA, bg="#dfe6e1")
plot(st_geometry(map), col = "light grey", add=TRUE)

#Flowmap : flow travelled less than 8.5 km  (as the first quartile Q1)

flowmap(tab=flow.d,
        fij="flowfilter",origin.f = "i",destination.f = "j",
        bkg = map,code="EPT_NUM",nodes.X="X",nodes.Y = "Y",
        filter=TRUE,
        taille=8,           
        a.head = 1,
        a.length = 0.11,
        a.col="#f7714f",
        add=TRUE)

#Map legend
legendPropLines(pos="topleft",
                title.txt="Number of commuters\n(distance travelled less than 8,5 km)",
                title.cex=0.8,    
                cex=0.5,
                values.cex= 0.7,  
                var=c(min(flow.d$flowfilter),8567), 
                col="#f7714f",
                lwd=8,
                frame = FALSE,
                values.rnd = 0
                )

#Map cosmetic
layoutLayer(title = "Professional mobility in Greater Paris : short distance travelled",
            author = "Cartograflow, 2020",
            sources = "Data : INSEE, 2017 ; Basemap : APUR, RIATE, 2018.",
            scale = 2,
            tabtitle = FALSE,
            frame = TRUE,
            col = "grey",
            coltitle ="black"
            )

# north arrow
north("topright")


## ----fig.show='hold', message=FALSE, warning=FALSE, include=FALSE-------------

#head(tabflow)

tab<-flowjointure(geom="area",DF.flow=tabflow,origin = "i",destination = "j",
                   bkg=map,id="EPT_NUM",x="X",y="Y")

tab.distance<-flowdist(tab,
                       dist.method = "euclidian",
                       result = "dist")

tab.distance<-tab.distance %>% select(i,j,distance)
tab<-tab %>% select(i,j,ydata)
head(tab.distance)

tab.flow<-flowreduct(tab,
                     tab.distance,
                     metric = "continous",
                     d.criteria = "dmin",  
                     d = 19234)        #Q2 : 14518 km - Q3:19234 km

#select for all i,j flow values above to 0
flow.d<-tab.flow%>%
       select(i,j,flowfilter)%>%
        filter(flowfilter !=0)

# Graphic parameters
knitr::opts_chunk$set(fig.width=6, fig.height=6)
par(mar=c(0,0,1,0))
extent <- c(2800000, 1340000, 6400000, 4800000)
resolution<-150

# Overlay a spatial background 
par(bg = "NA")

# Add the corresponding background 
plot(st_geometry(map), col = NA, border=NA, bg="#dfe6e1")
plot(st_geometry(map), col = "light grey", add=TRUE)

#Flowmap : flow travelled up to 20 km (as the third quartile Q3)

par(mar=c(0,0,1,0))

extent <- c(2800000, 1340000, 6400000, 4800000)
resolution<-150

flowmap(tab=flow.d,
        fij="flowfilter",origin.f = "i",destination.f = "j",
        bkg = map,code="EPT_NUM",nodes.X="X",nodes.Y = "Y",
        filter=TRUE,
        taille=8,           
        a.head = 1,
        a.length = 0.11,
        a.col="#f7714f",
        add=TRUE)

# Map legend
legendPropLines(pos="topleft",
                title.txt="Number of commuters\n(distance travelled more than 19 km)",
                title.cex=0.8,    
                cex=0.5,
                values.cex= 0.7,  
                var=c(19234, max(flow.d$flowfilter)), 
                col="#f7714f",
                lwd=8, 
                frame = FALSE,
                values.rnd = 0
                )
# Map cosmetics
layoutLayer(title = "Professional mobility in Greater Paris : long distance travelled",
            author = "Cartograflow, 2020",
            sources = "Data : INSEE, 2017 ; Basemap : APUR, RIATE, 2018.",
            scale = 2,
            tabtitle = FALSE,
            frame = TRUE,
            col = "grey",
            coltitle ="black")

# north arrow
north("topright")


## ----fig.show='hold', message=FALSE, warning=FALSE, include=FALSE-------------

#head(tabflow)

tab<-flowjointure(geom="area",DF.flow=tabflow,origin = "i",destination = "j",
                   bkg=map,id="EPT_NUM",x="X",y="Y")

tab.distance<-flowdist(tab,
                       dist.method = "euclidian",
                       result = "dist")

tab.distance<-tab.distance %>% select(i,j,distance)
tab<-tab %>% select(i,j,ydata)
head(tab.distance)

tab.flow<-flowreduct(tab,
                     tab.distance,
                     metric = "continous",
                     d.criteria = "dmax",  
                     d = 19234)       #Q3:19234 km

#select for all i,j flow values up to 8567  
flow.d<-tab.flow%>%
       select(i,j,flowfilter)%>%
        filter(flowfilter >8567)      #Q1=8567km

# Graphic parameters
knitr::opts_chunk$set(fig.width=6, fig.height=6)
par(mar=c(0,0,1,0))
extent <- c(2800000, 1340000, 6400000, 4800000)
resolution<-150

# Overlay a spatial background 
par(bg = "NA")

# Add the corresponding background 
plot(st_geometry(map), col = NA, border=NA, bg="#dfe6e1")
plot(st_geometry(map), col = "light grey", add=TRUE)

#Flowmap : flow travelled up to (Q3)

flowmap(tab=flow.d,
        fij="flowfilter",origin.f = "i",destination.f = "j",
        bkg = map,code="EPT_NUM",nodes.X="X",nodes.Y = "Y",
        filter=TRUE,
        taille=8,           
        a.head = 1,
        a.length = 0.11,
        a.col="#f7714f",
        add=TRUE)

# Map legend
legendPropLines(pos="topleft",
                title.txt="Number of commuters\n(distance travelled between 8,5 and 19 km)",
                title.cex=0.8,    
                cex=0.5,
                values.cex= 0.7,  
                var=c(8567, max(flow.d$flowfilter)), 
                col="#f7714f",
                lwd=8, 
                frame = FALSE,
                values.rnd = 0
                )

# Map cosmetic
layoutLayer(title = "Professional mobility in Greater Paris : median distance travelled",
            author = "Cartograflow, 2020",
            sources = "Data : INSEE, 2017 ; Basemap : APUR, RIATE, 2018.",
            scale = 5,
            tabtitle = TRUE,
            frame = TRUE,
            col = "grey",
            coltitle ="black")

## ----echo=TRUE, include=TRUE, message=FALSE, fig.show='hold', warning=FALSE----

library(igraph)
## Neighbouring graph (order k= 1)
graph_ckij_1<-flowcontig(bkg=map, code="EPT_NUM", 
                         k=1, algo = "automatic")
#Max order = "3"

## ----echo=FALSE, include=FALSE, message=FALSE, fig.show='hold', warning=FALSE----

flowmap(tab=graph_ckij_1,
        fij="ordre",origin.f = "i",destination.f = "j",
        bkg = map,code="EPT_NUM",nodes.X="X",nodes.Y = "Y",
        filter=FALSE)

mtext("Study Area Neighbourhood Plot (k=1)", side=3)


## ----echo=TRUE, fig.show='hold', message=FALSE, warning=FALSE-----------------
library(rlang)

#head(tabflow)
#head(graph_ckij_1)

reduc_k1<-flowreduct(tabflow,
                  graph_ckij_1,
                  metric = "ordinal")
head(reduc_k1)

## ----echo=TRUE, fig.show='hold', message=FALSE, warning=FALSE-----------------

# Graphic parameters
knitr::opts_chunk$set(fig.width=6, fig.height=6)
par(mar=c(0,0,1,0))
extent <- c(2800000, 1340000, 6400000, 4800000)
resolution<-150

# Overlay a spatial background 
par(bg = "NA")

# Add the corresponding background 
plot(st_geometry(map), col = NA, border=NA, bg="#dfe6e1")
plot(st_geometry(map), col = "light grey", add=TRUE)

#Flowmap : flow travelled between adjacent areas

flowmap(tab=reduc_k1,
        fij="flow",origin.f = "i",destination.f = "j",
        bkg = map,code="EPT_NUM",nodes.X="X",nodes.Y = "Y",
        filter=TRUE,
        taille=8,
        a.head = 1,
        a.length = 0.11,
        a.col="#0e7fe3",
        add=TRUE
        )

# Map Legend
legendPropLines(pos="topleft",
                title.txt="Number of commuters in adjacent places\n(k=1)",
                title.cex=0.8,    
                cex=0.5,
                values.cex= 0.7,  
                var=c(min(reduc_k1$flow),max(reduc_k1$flow)), 
                col="#0e7fe3",
                lwd=8, 
                frame = FALSE,
                values.rnd = 0
                )

# Map cosmetic
layoutLayer(title = "Professional mobility in Greater Paris between 1-neighbouring municipalities",
            author = "Cartograflow, 2020",
            sources = "Data : INSEE, 2017 ; Basemap : APUR, RIATE, 2018.",
            scale = 2,
            tabtitle = FALSE,
            frame = TRUE,
            col = "grey",
            coltitle ="black")


## ----fig.show='hold', message=FALSE, warning=FALSE, include=FALSE-------------

library(igraph)

## Neighbouring graph (Max(k)=3)
graph_ckij_3<-flowcontig(bkg=map, code="EPT_NUM",k=3)

# Flow reduction
#head(tabflow)

reduc_k3<-flowreduct(tabflow,
                  graph_ckij_3,
                  metric = "ordinal")
#head(reduc_k3)

# Graphic parameters
knitr::opts_chunk$set(fig.width=6, fig.height=6)
par(mar=c(0,0,1,0))
extent <- c(2800000, 1340000, 6400000, 4800000)
resolution<-150

# Overlay a spatial background 
par(bg = "NA")

# Add the corresponding background 
plot(st_geometry(map), col = NA, border=NA, bg="#dfe6e1")
plot(st_geometry(map), col = "light grey", add=TRUE)

#Flowmap : flow travelled between non adjacent areas

flowmap(tab=reduc_k3,
        fij="flow",origin.f = "i",destination.f = "j",
        bkg = map,code="EPT_NUM",nodes.X="X",nodes.Y = "Y",
        filter=TRUE,
        taille=8,
        a.head = 1,
        a.length = 0.11,
        a.col="#0e7fe3",
        add=TRUE 
        )

# Map Legend
legendPropLines(pos="topleft",
                title.txt="Number of commuters between non adjacent places\n(k=3)",
                title.cex=0.8,    
                cex=0.5,
                values.cex= 0.7,  
                var=c(min(reduc_k3$flow),max(reduc_k3$flow)), 
                col="#0e7fe3",
                lwd=8, 
                frame = FALSE,
                values.rnd = 0
                )

# Map cosmetic
layoutLayer(title = "Professional mobility in Greater Paris between 3-neighbouring municipalities",
            author = "Cartograflow, 2020",
            sources = "Data : INSEE, 2017 ; Basemap : APUR, RIATE, 2018.",
            scale = 2,
            tabtitle = FALSE,
            frame = TRUE,
            col = "grey",
            coltitle ="black")


## ----flows_criteria, echo=TRUE, fig.show='hold', fig.width=7, message=FALSE, warning=FALSE, ECHO=FALSE----

#Computes k=1
library(igraph)

#Neighbouring graph k=1
graph_ckij_1<-flowcontig(bkg=map, code="EPT_NUM",k=1)

#Flow reduction
reduc_k1<-flowreduct(tabflow,
                  graph_ckij_1,
                  metric = "ordinal")

#Mean flow value
reduc_k1_mean<-mean(reduc_k1$flow)
mean<-mean(tabflow$Fij)

# mean value of reduc_k1_mean =18591
# mean value of tabflow =13220


## ----echo=TRUE, fig.show='hold', message=FALSE, warning=FALSE-----------------

# Graphic parameters
knitr::opts_chunk$set(fig.width=6, fig.height=6)
par(mar=c(0,0,1,0))
extent <- c(2800000, 1340000, 6400000, 4800000)
resolution<-150

# Overlay a spatial background 
par(bg = "NA")

# Add the corresponding background 
plot(st_geometry(map), col = NA, border=NA, bg="#dfe6e1")
plot(st_geometry(map), col = "light grey", add=TRUE)

#Flowmap : flow travelled between adjacent areas

flowmap(tab=reduc_k1,
        fij="flow",origin.f = "i",destination.f = "j",
        bkg = map,code="EPT_NUM",nodes.X="X",nodes.Y = "Y",
        filter=TRUE,
        threshold = 13220, #reduc_k1 mean value is 18591 ; tabflow mean value is 13220
        taille=8,
        a.head = 1,
        a.length = 0.11,
        a.col="#0e7fe3",
        add=TRUE 
        )

# Map Legend
legendPropLines(pos="topleft",
                title.txt="Number of above-average flows that occur between adjacent areas\n(k=1)",
                title.cex=0.8,    
                cex=0.5,
                values.cex= 0.7,  
                var=c(min(reduc_k1$flow),max(reduc_k1$flow)), 
                col="#0e7fe3",
                lwd=8, 
                frame = FALSE,
                values.rnd = 0
                )

# Map cosmetic
layoutLayer(title = "Professional mobility in Greater Paris between 1-neighbouring municipalities",
            author = "Cartograflow, 2020",
            sources = "Data : INSEE, 2017 ; Basemap : APUR, RIATE, 2018.",
            scale = 2,
            tabtitle = FALSE,
            frame = TRUE,
            col = "grey",
            coltitle ="black")


## ----lecho=TRUE, fig.show='hold'----------------------------------------------

sessionInfo()


