% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitting.R, R/utils.R
\name{fitSmoothHazard}
\alias{fitSmoothHazard}
\alias{fitSmoothHazard.fit}
\alias{prepareX}
\title{Fit smooth-in-time parametric hazard functions.}
\usage{
fitSmoothHazard(
  formula,
  data,
  time,
  family = c("glm", "gam", "glmnet"),
  censored.indicator,
  ratio = 100,
  ...
)

fitSmoothHazard.fit(
  x,
  y,
  formula_time,
  time,
  event,
  family = c("glm", "glmnet"),
  censored.indicator,
  ratio = 100,
  ...
)

prepareX(formula, data)
}
\arguments{
\item{formula}{an object of class "formula" (or one that can be coerced to
that class): a symbolic description of the model to be fitted. The details
of model specification are given under Details.}

\item{data}{a data frame, list or environment containing the variables in the
model. If not found in data, the variables are taken from
\code{environment(formula)}, typically the environment from which
\code{fitSmoothHazard} is called.}

\item{time}{a character string giving the name of the time variable. See
Details.}

\item{family}{a character string specifying the family of regression models
used to fit the hazard.}

\item{censored.indicator}{a character string of length 1 indicating which
value in \code{event} is the censored. This function will use
\code{\link[stats]{relevel}} to set \code{censored.indicator} as the
reference level. This argument is ignored if the \code{event} variable is a
numeric.}

\item{ratio}{integer, giving the ratio of the size of the base series to that
of the case series. Defaults to 100.}

\item{...}{Additional parameters passed to fitting functions (e.g.
\code{glm}, \code{glmnet}, \code{gam}).}

\item{x}{Matrix containing covariates.}

\item{y}{Matrix containing two columns: one corresponding to time, the other
to the event type.}

\item{formula_time}{A formula describing how the hazard depends on time.
Defaults to linear.}

\item{event}{a character string giving the name of the event variable.}
}
\value{
An object of \code{glm} and \code{lm} when there is only one event of
interest, or of class \code{\link{CompRisk}}, which inherits from
\code{vglm}, for a competing risk analysis. As such, functions like
\code{summary}, \code{deviance} and \code{coefficients} give familiar
results.
}
\description{
Miettinen and Hanley (2009) explained how case-base sampling can be used to
estimate smooth-in-time parametric hazard functions. The idea is to sample
person-moments, which may or may not correspond to an event, and then fit the
hazard using logistic regression.
}
\details{
The object \code{data} should either be the output of the function
\code{\link{sampleCaseBase}} or the source dataset on which case-base
sampling will be performed. In the latter case, it is assumed that
\code{data} contains the two columns corresponding to the supplied time and
event variables. The variable \code{time} is used for the sampling the base
series, and therefore it should represent the time variable on its original
(i.e. non transformed) scale. If \code{time} is missing, the function looks
for a column named \code{"time"} in the data. Note that the event variable is
inferred from \code{formula}, since it is the left hand side.

For single-event survival analysis, it is also possible to fit the hazard
function using \code{glmnet} or \code{gam}. The choice of fitting family is
controlled by the parameter \code{family}. The default value is \code{glm},
which corresponds to logistic regression. For competing risk analysis, only
\code{glm} and \code{glmnet} are allowed.

We also provide a matrix interface through \code{fitSmoothHazard.fit}, which
mimics \code{glm.fit}. This is mostly convenient for \code{family =
"glmnet"}, since a formula interface becomes quickly cumbersome as the number
of variables increases. In this setting, the matrix \code{y} should have two
columns and contain the time and event variables (e.g. like the output of
\code{survival::Surv}). We need this linear function of time in order to
perform case-base sampling. Therefore, nonlinear functions of time should be
specified as a one-sided formula through the argument \code{formula_time}
(the left-hand side is always ignored).

\code{prepareX} is a slightly modified version of the same function from
the \code{glmnet} package. It can be used to convert a data.frame to a matrix
with categorical variables converted to dummy variables using one-hot
encoding
}
\examples{
# Simulate censored survival data for two outcome types from exponential
# distributions
library(data.table)
nobs <- 500
tlim <- 20

# simulation parameters
b1 <- 200
b2 <- 50

# event type 0-censored, 1-event of interest, 2-competing event
# t observed time/endpoint
# z is a binary covariate
DT <- data.table(z = rbinom(nobs, 1, 0.5))
DT[, `:=`(
  "t_event" = rweibull(nobs, 1, b1),
  "t_comp" = rweibull(nobs, 1, b2)
)]
DT[, `:=`(
  "event" = 1 * (t_event < t_comp) + 2 * (t_event >= t_comp),
  "time" = pmin(t_event, t_comp)
)]
DT[time >= tlim, `:=`("event" = 0, "time" = tlim)]

out_linear <- fitSmoothHazard(event ~ time + z, DT, ratio = 10)
out_log <- fitSmoothHazard(event ~ log(time) + z, DT, ratio = 10)

# Use GAMs
library(mgcv)
DT[event == 2, event := 1]
out_gam <- fitSmoothHazard(event ~ s(time) + z, DT,
                           ratio = 10, family = "gam")
}
