\name{simulate_rou_model}
\alias{simulate_rou_model}
\title{
Simulate a reflected Ornstein-Uhlenbeck model for continuous trait evolution.
}
\description{
Given a rooted phylogenetic tree and a reflected Ornstein-Uhlenbeck (ROU) model for the evolution of a continuous (numeric) trait, simulate random outcomes of the model on all nodes and/or tips of the tree. The ROU process is similar to the Ornstein-Uhlenbeck process (see \code{\link{simulate_ou_model}}), with the difference that the ROU process cannot fall below a certain value (its "reflection point"), which (in this implementation) is also its deterministic equilibrium point (Hu et al. 2015). The function traverses nodes from root to tips and randomly assigns a state to each node or tip based on its parent's previously assigned state and the specified model parameters. The generated states have joint distributions consistent with the ROU model. Optionally, multiple independent simulations can be performed using the same model.
}
\usage{
simulate_rou_model(tree, reflection_point, spread, decay_rate,
                   include_tips=TRUE, include_nodes=TRUE, 
                   Nsimulations=1, drop_dims=TRUE)
}
\arguments{
\item{tree}{
A rooted tree of class "phylo". The root is assumed to be the unique node with no incoming edge.
}
\item{reflection_point}{
Numeric. The reflection point of the ROU model. In castor, this also happens to be the deterministic equilibrium of the ROU process (i.e. if the decay rate were infinite). For example, if a trait can only be positive (but arbitrarily small), then \code{reflection_point} may be set to 0.
}
\item{spread}{
Numeric. The stationary standard deviation of the corresponding unreflected OU process.
}
\item{decay_rate}{
Numeric. Exponential decay rate (stabilization rate) of the ROU process (in units 1/edge_length_units).
}
\item{include_tips}{
Include random states for the tips. If \code{FALSE}, no states will be returned for tips.
}
\item{include_nodes}{
Include random states for the nodes. If \code{FALSE}, no states will be returned for nodes.
}
\item{Nsimulations}{
Number of random independent simulations to perform. For each node and/or tip, there will be \code{Nsimulations} random states generated.
}
\item{drop_dims}{
Logical, specifying whether the returned \code{tip_states} and \code{node_states} (see below) should be vectors, if \code{Nsimulations==1}. If \code{drop_dims==FALSE}, then \code{tip_states} and \code{tip_nodes} will always be 2D matrices.
}
}


\details{
For each simulation, the state of the root is picked randomly from the stationary distribution of the ROU model, i.e. from a one-sided normal distribution with mode = \code{reflection_point} and standard deviation = \code{stationary_std}.

If \code{tree$edge.length} is missing, each edge in the tree is assumed to have length 1. The tree may include multi-furcations (i.e. nodes with more than 2 children) as well as mono-furcations (i.e. nodes with only one child). The asymptotic time complexity of this function is O(Nedges*Nsimulations), where Nedges is the number of edges in the tree.
}


\value{
A list with the following elements:
\item{tip_states}{
Either \code{NULL} (if \code{include_tips==FALSE}), or a 2D numeric matrix of size Nsimulations x Ntips, where Ntips is the number of tips in the tree. The [r,c]-th entry of this matrix will be the state of tip c generated by the r-th simulation. If \code{drop_dims==TRUE} and \code{Nsimulations==1}, then \code{tip_states} will be a vector.
}
\item{node_states}{
Either \code{NULL} (if \code{include_nodes==FALSE}), or a 2D numeric matrix of size Nsimulations x Nnodes, where Nnodes is the number of nodes in the tree. The [r,c]-th entry of this matrix will be the state of node c generated by the r-th simulation. If \code{drop_dims==TRUE} and \code{Nsimulations==1}, then \code{node_states} will be a vector.
}
}

\author{Stilianos Louca}

\references{
Y. Hu, C. Lee, M. H. Lee, J. Song (2015). Parameter estimation for reflected Ornstein-Uhlenbeck processes with discrete observations. Statistical Inference for Stochastic Processes. 18:279-291.
}

\seealso{
\code{\link{simulate_ou_model}}, \code{\link{simulate_bm_model}}, \code{\link{simulate_mk_model}}
}

\examples{
# generate a random tree
tree = generate_random_tree(list(birth_rate_intercept=1),max_tips=10000)$tree

# simulate evolution of a continuous trait whose value is always >=1
tip_states = simulate_rou_model(tree, reflection_point=1, spread=2, decay_rate=0.1)$tip_states

# plot histogram of simulated tip states
hist(tip_states, breaks=20, xlab="state", main="Trait probability distribution", prob=TRUE)
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{OU model}
\keyword{random}
