#' List announcements
#' @param limit integer optional. Number of results to return. Defaults to 10. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to released_at. Must be one of: released_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this announcement}
#' \item{subject}{string, The subject of this announcement.}
#' \item{body}{string, The body of this announcement.}
#' \item{releasedAt}{string, The date and time this announcement was released.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
admin_list_announcements <- function(limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/admin/announcements"
  path_params  <- list()
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Post an announcement
#' @param subject string required. The subject of this announcement.
#' @param body string required. The body of this announcement.
#' @param released_at string optional. The date and time this announcement was released.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this announcement}
#' \item{subject}{string, The subject of this announcement.}
#' \item{body}{string, The body of this announcement.}
#' \item{releasedAt}{string, The date and time this announcement was released.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
admin_post_announcements <- function(subject, body, released_at = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/admin/announcements"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(subject = subject, body = body, releasedAt = released_at)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a particular announcement
#' @param id integer required. The ID of this announcement
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this announcement}
#' \item{subject}{string, The subject of this announcement.}
#' \item{body}{string, The body of this announcement.}
#' \item{releasedAt}{string, The date and time this announcement was released.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
admin_get_announcements <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/admin/announcements/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Edit an announcement
#' @param id integer required. The ID of this announcement
#' @param subject string optional. The subject of this announcement.
#' @param body string optional. The body of this announcement.
#' @param released_at string optional. The date and time this announcement was released.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this announcement}
#' \item{subject}{string, The subject of this announcement.}
#' \item{body}{string, The body of this announcement.}
#' \item{releasedAt}{string, The date and time this announcement was released.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
admin_patch_announcements <- function(id, subject = NULL, body = NULL, released_at = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/admin/announcements/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(subject = subject, body = body, releasedAt = released_at)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Delete an announcement
#' @param id integer required. The ID of this announcement
#' 
#' @return  An empty HTTP response
#' @export
admin_delete_announcements <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/admin/announcements/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a theme
#' @param name string required. The name of this theme.
#' @param settings_json string required. The JSON-encoded theme configuration.
#' @param organization_ids array optional. List of organization ID's allowed to see this theme.
#' @param logo_file_id string optional. The ID of the logo image file.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this theme.}
#' \item{name}{string, The name of this theme.}
#' \item{organizationIds}{array, List of organization ID's allowed to use this theme.}
#' \item{settings}{string, The theme configuration object.}
#' \item{logoFile}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of the logo image file.
#' \item downloadUrl string, The URL of the logo image file.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
admin_post_themes <- function(name, settings_json, organization_ids = NULL, logo_file_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/admin/themes"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, settingsJson = settings_json, organizationIds = organization_ids, logoFileId = logo_file_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Edit a theme
#' @param id integer required. The ID of this theme.
#' @param name string optional. The name of this theme.
#' @param organization_ids array optional. List of organization ID's allowed to see this theme.
#' @param settings_json string optional. The JSON-encoded theme configuration.
#' @param logo_file_id string optional. The ID of the logo image file.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this theme.}
#' \item{name}{string, The name of this theme.}
#' \item{organizationIds}{array, List of organization ID's allowed to use this theme.}
#' \item{settings}{string, The theme configuration object.}
#' \item{logoFile}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of the logo image file.
#' \item downloadUrl string, The URL of the logo image file.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
admin_patch_themes <- function(id, name = NULL, organization_ids = NULL, settings_json = NULL, logo_file_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/admin/themes/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, organizationIds = organization_ids, settingsJson = settings_json, logoFileId = logo_file_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' List organizations
#' @param status array optional. The status of the organization (active/trial/inactive).
#' @param org_type array optional. The organization type (platform/ads/survey_vendor/other).
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this organization.}
#' \item{name}{string, The name of this organization.}
#' \item{slug}{string, The slug of this organization.}
#' \item{accountManagerId}{integer, The user ID of the Account Manager.}
#' \item{csSpecialistId}{integer, The user ID of the Client Success Specialist.}
#' \item{status}{string, The status of the organization (active/trial/inactive).}
#' \item{orgType}{string, The organization type (platform/ads/survey_vendor/other).}
#' \item{customBranding}{string, The custom branding settings.}
#' \item{contractSize}{integer, The monthly contract size.}
#' \item{maxAnalystUsers}{integer, The max number of full platform users for the org.}
#' \item{maxReportUsers}{integer, The max number of report-only platform users for the org.}
#' \item{vertical}{string, The business vertical that the organization belongs to.}
#' \item{csMetadata}{string, Additional metadata about the organization in JSON format.}
#' \item{removeFooterInEmails}{boolean, If true, emails sent by platform will not include Civis text.}
#' \item{salesforceAccountId}{string, The SalesForce Account ID for this organization.}
#' \item{tableauSiteId}{string, The Tableau Site ID for this organization.}
#' \item{fedrampEnabled}{boolean, Flag denoting whether this organization is FedRAMP compliant.}
#' \item{createdById}{integer, The ID of the user who created this organization}
#' \item{lastUpdatedById}{integer, The ID of the user who last updated this organization}
#' \item{advancedSettings}{list, A list containing the following elements: 
#' \itemize{
#' \item dedicatedDjPoolEnabled boolean, If true, the Organization has a dedicated delayed jobs pool. Defaults to false.
#' }}
#' \item{tableauRefreshHistory}{array, The number of tableau refreshes used this month.}
#' @export
admin_list_organizations <- function(status = NULL, org_type = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/admin/organizations"
  path_params  <- list()
  query_params <- list(status = status, org_type = org_type)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
aliases_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/aliases/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
aliases_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/aliases/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
aliases_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/aliases/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
aliases_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/aliases/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
aliases_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/aliases/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
aliases_list_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/aliases/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
aliases_put_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/aliases/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Aliases
#' @param object_type string optional. Filter results by object type. Pass multiple object types with a comma-separatedlist. Valid types include: cass_ncoa, container_script, geocode, python_script, r_script, salesforce_export, javascript_script, sql_script, project, notebook, workflow, template_script, template_report, service, report, tableau and service_report.
#' @param limit integer optional. Number of results to return. Defaults to 50. Maximum allowed is 1000.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id, object_type.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The id of the Alias object.}
#' \item{objectId}{integer, The id of the object}
#' \item{objectType}{string, The type of the object. Valid types include: cass_ncoa, container_script, geocode, python_script, r_script, salesforce_export, javascript_script, sql_script, project, notebook, workflow, template_script, template_report, service, report, tableau and service_report.}
#' \item{alias}{string, The alias of the object}
#' \item{userId}{integer, The id of the user who created the alias}
#' \item{displayName}{string, The display name of the Alias object. Defaults to object name if not provided.}
#' @export
aliases_list <- function(object_type = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/aliases/"
  path_params  <- list()
  query_params <- list(object_type = object_type, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create an Alias
#' @param object_id integer required. The id of the object
#' @param object_type string required. The type of the object. Valid types include: cass_ncoa, container_script, geocode, python_script, r_script, salesforce_export, javascript_script, sql_script, project, notebook, workflow, template_script, template_report, service, report, tableau and service_report.
#' @param alias string required. The alias of the object
#' @param display_name string optional. The display name of the Alias object. Defaults to object name if not provided.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The id of the Alias object.}
#' \item{objectId}{integer, The id of the object}
#' \item{objectType}{string, The type of the object. Valid types include: cass_ncoa, container_script, geocode, python_script, r_script, salesforce_export, javascript_script, sql_script, project, notebook, workflow, template_script, template_report, service, report, tableau and service_report.}
#' \item{alias}{string, The alias of the object}
#' \item{userId}{integer, The id of the user who created the alias}
#' \item{displayName}{string, The display name of the Alias object. Defaults to object name if not provided.}
#' @export
aliases_post <- function(object_id, object_type, alias, display_name = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/aliases/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(objectId = object_id, objectType = object_type, alias = alias, displayName = display_name)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get an Alias
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The id of the Alias object.}
#' \item{objectId}{integer, The id of the object}
#' \item{objectType}{string, The type of the object. Valid types include: cass_ncoa, container_script, geocode, python_script, r_script, salesforce_export, javascript_script, sql_script, project, notebook, workflow, template_script, template_report, service, report, tableau and service_report.}
#' \item{alias}{string, The alias of the object}
#' \item{userId}{integer, The id of the user who created the alias}
#' \item{displayName}{string, The display name of the Alias object. Defaults to object name if not provided.}
#' @export
aliases_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/aliases/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this Alias
#' @param id integer required. The id of the Alias object.
#' @param object_id integer required. The id of the object
#' @param object_type string required. The type of the object. Valid types include: cass_ncoa, container_script, geocode, python_script, r_script, salesforce_export, javascript_script, sql_script, project, notebook, workflow, template_script, template_report, service, report, tableau and service_report.
#' @param alias string required. The alias of the object
#' @param display_name string optional. The display name of the Alias object. Defaults to object name if not provided.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The id of the Alias object.}
#' \item{objectId}{integer, The id of the object}
#' \item{objectType}{string, The type of the object. Valid types include: cass_ncoa, container_script, geocode, python_script, r_script, salesforce_export, javascript_script, sql_script, project, notebook, workflow, template_script, template_report, service, report, tableau and service_report.}
#' \item{alias}{string, The alias of the object}
#' \item{userId}{integer, The id of the user who created the alias}
#' \item{displayName}{string, The display name of the Alias object. Defaults to object name if not provided.}
#' @export
aliases_put <- function(id, object_id, object_type, alias, display_name = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/aliases/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(objectId = object_id, objectType = object_type, alias = alias, displayName = display_name)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this Alias
#' @param id integer required. The id of the Alias object.
#' @param object_id integer optional. The id of the object
#' @param object_type string optional. The type of the object. Valid types include: cass_ncoa, container_script, geocode, python_script, r_script, salesforce_export, javascript_script, sql_script, project, notebook, workflow, template_script, template_report, service, report, tableau and service_report.
#' @param alias string optional. The alias of the object
#' @param display_name string optional. The display name of the Alias object. Defaults to object name if not provided.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The id of the Alias object.}
#' \item{objectId}{integer, The id of the object}
#' \item{objectType}{string, The type of the object. Valid types include: cass_ncoa, container_script, geocode, python_script, r_script, salesforce_export, javascript_script, sql_script, project, notebook, workflow, template_script, template_report, service, report, tableau and service_report.}
#' \item{alias}{string, The alias of the object}
#' \item{userId}{integer, The id of the user who created the alias}
#' \item{displayName}{string, The display name of the Alias object. Defaults to object name if not provided.}
#' @export
aliases_patch <- function(id, object_id = NULL, object_type = NULL, alias = NULL, display_name = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/aliases/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(objectId = object_id, objectType = object_type, alias = alias, displayName = display_name)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Delete an alias
#' @param id integer required. The id of the Alias object.
#' 
#' @return  An empty HTTP response
#' @export
aliases_delete <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/aliases/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about an alias within an FCO type
#' @param object_type string required. The type of the object. Valid types include: cass_ncoa, container_script, geocode, python_script, r_script, salesforce_export, javascript_script, sql_script, project, notebook, workflow, template_script, template_report, service, report, tableau and service_report.
#' @param alias string required. The alias of the object
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The id of the Alias object.}
#' \item{objectId}{integer, The id of the object}
#' \item{objectType}{string, The type of the object. Valid types include: cass_ncoa, container_script, geocode, python_script, r_script, salesforce_export, javascript_script, sql_script, project, notebook, workflow, template_script, template_report, service, report, tableau and service_report.}
#' \item{alias}{string, The alias of the object}
#' \item{userId}{integer, The id of the user who created the alias}
#' \item{displayName}{string, The display name of the Alias object. Defaults to object name if not provided.}
#' @export
aliases_get_object_type <- function(object_type, alias) {

  args <- as.list(match.call())[-1]
  path <- "/aliases/{object_type}/{alias}"
  path_params  <- list(object_type = object_type, alias = alias)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List announcements
#' @param limit integer optional. Number of results to return. Defaults to 10. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to released_at. Must be one of: released_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this announcement}
#' \item{subject}{string, The subject of this announcement.}
#' \item{body}{string, The body of this announcement.}
#' \item{releasedAt}{string, The date and time this announcement was released.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
announcements_list <- function(limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/announcements/"
  path_params  <- list()
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Kubernetes Clusters
#' @param organization_id integer optional. The ID of this cluster's organization. Cannot be used along with the organization slug.
#' @param organization_slug string optional. The slug of this cluster's organization. Cannot be used along with the organization ID.
#' @param raw_cluster_slug string optional. The slug of this cluster's raw configuration.
#' @param exclude_inactive_orgs boolean optional. When true, excludes KubeClusters associated with inactive orgs. Defaults to false.
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to organization_id. Must be one of: organization_id, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this cluster.}
#' \item{organizationId}{string, The id of this cluster's organization.}
#' \item{organizationName}{string, The name of this cluster's organization.}
#' \item{organizationSlug}{string, The slug of this cluster's organization.}
#' \item{rawClusterSlug}{string, The slug of this cluster's raw configuration.}
#' \item{customPartitions}{boolean, Whether this cluster has a custom partition configuration.}
#' \item{clusterPartitions}{array, An array containing the following fields: 
#' \itemize{
#' \item clusterPartitionId integer, The ID of this cluster partition.
#' \item name string, The name of the cluster partition.
#' \item labels array, Labels associated with this partition.
#' \item instanceConfigs array, The instances configured for this cluster partition.
#' \item defaultInstanceConfigId integer, The id of the InstanceConfig that is the default for this partition.
#' }}
#' \item{isNatEnabled}{boolean, Whether this cluster needs a NAT gateway or not.}
#' @export
clusters_list_kubernetes <- function(organization_id = NULL, organization_slug = NULL, raw_cluster_slug = NULL, exclude_inactive_orgs = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes"
  path_params  <- list()
  query_params <- list(organization_id = organization_id, organization_slug = organization_slug, raw_cluster_slug = raw_cluster_slug, exclude_inactive_orgs = exclude_inactive_orgs, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Kubernetes Cluster
#' @param organization_id string optional. The id of this cluster's organization.
#' @param organization_slug string optional. The slug of this cluster's organization.
#' @param raw_cluster_slug string optional. The slug of this cluster's raw configuration.
#' @param is_nat_enabled boolean optional. Whether this cluster needs a NAT gateway or not.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this cluster.}
#' \item{organizationId}{string, The id of this cluster's organization.}
#' \item{organizationName}{string, The name of this cluster's organization.}
#' \item{organizationSlug}{string, The slug of this cluster's organization.}
#' \item{rawClusterSlug}{string, The slug of this cluster's raw configuration.}
#' \item{customPartitions}{boolean, Whether this cluster has a custom partition configuration.}
#' \item{clusterPartitions}{array, An array containing the following fields: 
#' \itemize{
#' \item clusterPartitionId integer, The ID of this cluster partition.
#' \item name string, The name of the cluster partition.
#' \item labels array, Labels associated with this partition.
#' \item instanceConfigs array, The instances configured for this cluster partition.
#' \item defaultInstanceConfigId integer, The id of the InstanceConfig that is the default for this partition.
#' }}
#' \item{isNatEnabled}{boolean, Whether this cluster needs a NAT gateway or not.}
#' \item{hours}{number, The number of hours used this month for this cluster.}
#' @export
clusters_post_kubernetes <- function(organization_id = NULL, organization_slug = NULL, raw_cluster_slug = NULL, is_nat_enabled = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(organizationId = organization_id, organizationSlug = organization_slug, rawClusterSlug = raw_cluster_slug, isNatEnabled = is_nat_enabled)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Describe a Kubernetes Cluster
#' @param id integer required. 
#' @param include_usage_stats boolean optional. When true, usage stats are returned in instance config objects. Defaults to false.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this cluster.}
#' \item{organizationId}{string, The id of this cluster's organization.}
#' \item{organizationName}{string, The name of this cluster's organization.}
#' \item{organizationSlug}{string, The slug of this cluster's organization.}
#' \item{rawClusterSlug}{string, The slug of this cluster's raw configuration.}
#' \item{customPartitions}{boolean, Whether this cluster has a custom partition configuration.}
#' \item{clusterPartitions}{array, An array containing the following fields: 
#' \itemize{
#' \item clusterPartitionId integer, The ID of this cluster partition.
#' \item name string, The name of the cluster partition.
#' \item labels array, Labels associated with this partition.
#' \item instanceConfigs array, The instances configured for this cluster partition.
#' \item defaultInstanceConfigId integer, The id of the InstanceConfig that is the default for this partition.
#' }}
#' \item{isNatEnabled}{boolean, Whether this cluster needs a NAT gateway or not.}
#' \item{hours}{number, The number of hours used this month for this cluster.}
#' @export
clusters_get_kubernetes <- function(id, include_usage_stats = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes/{id}"
  path_params  <- list(id = id)
  query_params <- list(include_usage_stats = include_usage_stats)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a Kubernetes Cluster
#' @param id integer required. The ID of this cluster.
#' @param raw_cluster_slug string optional. The slug of this cluster's raw configuration.
#' @param is_nat_enabled boolean optional. Whether this cluster needs a NAT gateway or not.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this cluster.}
#' \item{organizationId}{string, The id of this cluster's organization.}
#' \item{organizationName}{string, The name of this cluster's organization.}
#' \item{organizationSlug}{string, The slug of this cluster's organization.}
#' \item{rawClusterSlug}{string, The slug of this cluster's raw configuration.}
#' \item{customPartitions}{boolean, Whether this cluster has a custom partition configuration.}
#' \item{clusterPartitions}{array, An array containing the following fields: 
#' \itemize{
#' \item clusterPartitionId integer, The ID of this cluster partition.
#' \item name string, The name of the cluster partition.
#' \item labels array, Labels associated with this partition.
#' \item instanceConfigs array, The instances configured for this cluster partition.
#' \item defaultInstanceConfigId integer, The id of the InstanceConfig that is the default for this partition.
#' }}
#' \item{isNatEnabled}{boolean, Whether this cluster needs a NAT gateway or not.}
#' \item{hours}{number, The number of hours used this month for this cluster.}
#' @export
clusters_patch_kubernetes <- function(id, raw_cluster_slug = NULL, is_nat_enabled = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(rawClusterSlug = raw_cluster_slug, isNatEnabled = is_nat_enabled)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the deployments associated with a Kubernetes Cluster
#' @param id integer required. The id of the cluster.
#' @param base_type string optional. If specified, return deployments of these base types. It accepts a comma-separated list, possible values are 'Notebook', 'Service', 'Run'.
#' @param state string optional. If specified, return deployments in these states. It accepts a comma-separated list, possible values are pending, running, terminated, sleeping
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to created_at. Must be one of: created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The id of this deployment.}
#' \item{name}{string, The name of the deployment.}
#' \item{baseId}{integer, The id of the base object associated with the deployment.}
#' \item{baseType}{string, The base type of this deployment.}
#' \item{state}{string, The state of the deployment.}
#' \item{cpu}{integer, The CPU in millicores required by the deployment.}
#' \item{memory}{integer, The memory in MB required by the deployment.}
#' \item{diskSpace}{integer, The disk space in GB required by the deployment.}
#' \item{instanceType}{string, The EC2 instance type requested for the deployment.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{maxMemoryUsage}{number, If the deployment has finished, the maximum amount of memory used during the deployment, in MB.}
#' \item{maxCpuUsage}{number, If the deployment has finished, the maximum amount of cpu used during the deployment, in millicores.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
clusters_list_kubernetes_deployments <- function(id, base_type = NULL, state = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes/{id}/deployments"
  path_params  <- list(id = id)
  query_params <- list(base_type = base_type, state = state, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get stats about deployments associated with a Kubernetes Cluster
#' @param id integer required. The ID of this cluster.
#' 
#' @return  An array containing the following fields:
#' \item{baseType}{string, The base type of this deployment}
#' \item{state}{string, State of the deployment}
#' \item{count}{integer, Number of deployments of base type and state}
#' \item{totalCpu}{integer, Total amount of CPU in millicores for deployments of base type and state}
#' \item{totalMemory}{integer, Total amount of Memory in megabytes for deployments of base type and state}
#' @export
clusters_list_kubernetes_deployment_stats <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes/{id}/deployment_stats"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Cluster Partitions for given cluster
#' @param id integer required. 
#' @param include_usage_stats boolean optional. When true, usage stats are returned in instance config objects. Defaults to false.
#' 
#' @return  An array containing the following fields:
#' \item{clusterPartitionId}{integer, The ID of this cluster partition.}
#' \item{name}{string, The name of the cluster partition.}
#' \item{labels}{array, Labels associated with this partition.}
#' \item{instanceConfigs}{array, An array containing the following fields: 
#' \itemize{
#' \item instanceConfigId integer, The ID of this InstanceConfig.
#' \item instanceType string, An EC2 instance type. Possible values include t2.large, m4.xlarge, m4.2xlarge, m4.4xlarge, m5.12xlarge, and c5.18xlarge.
#' \item minInstances integer, The minimum number of instances of that type in this cluster.
#' \item maxInstances integer, The maximum number of instances of that type in this cluster.
#' \item instanceMaxMemory integer, The amount of memory (RAM) available to a single instance of that type in megabytes.
#' \item instanceMaxCpu integer, The number of processor shares available to a single instance of that type in millicores.
#' \item instanceMaxDisk integer, The amount of disk available to a single instance of that type in gigabytes.
#' \item usageStats object, 
#' }}
#' \item{defaultInstanceConfigId}{integer, The id of the InstanceConfig that is the default for this partition.}
#' @export
clusters_list_kubernetes_partitions <- function(id, include_usage_stats = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes/{id}/partitions"
  path_params  <- list(id = id)
  query_params <- list(include_usage_stats = include_usage_stats)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Cluster Partition for given cluster
#' @param id integer required. The ID of the cluster which this partition belongs to.
#' @param instance_configs array required. An array containing the following fields: 
#' \itemize{
#' \item instanceType string, An EC2 instance type. Possible values include t2.large, m4.xlarge, m4.2xlarge, m4.4xlarge, m5.12xlarge, and c5.18xlarge.
#' \item minInstances integer, The minimum number of instances of that type in this cluster.
#' \item maxInstances integer, The maximum number of instances of that type in this cluster.
#' }
#' @param name string required. The name of the cluster partition.
#' @param labels array required. Labels associated with this partition.
#' 
#' @return  A list containing the following elements:
#' \item{clusterPartitionId}{integer, The ID of this cluster partition.}
#' \item{name}{string, The name of the cluster partition.}
#' \item{labels}{array, Labels associated with this partition.}
#' \item{instanceConfigs}{array, An array containing the following fields: 
#' \itemize{
#' \item instanceConfigId integer, The ID of this InstanceConfig.
#' \item instanceType string, An EC2 instance type. Possible values include t2.large, m4.xlarge, m4.2xlarge, m4.4xlarge, m5.12xlarge, and c5.18xlarge.
#' \item minInstances integer, The minimum number of instances of that type in this cluster.
#' \item maxInstances integer, The maximum number of instances of that type in this cluster.
#' \item instanceMaxMemory integer, The amount of memory (RAM) available to a single instance of that type in megabytes.
#' \item instanceMaxCpu integer, The number of processor shares available to a single instance of that type in millicores.
#' \item instanceMaxDisk integer, The amount of disk available to a single instance of that type in gigabytes.
#' \item usageStats object, 
#' }}
#' \item{defaultInstanceConfigId}{integer, The id of the InstanceConfig that is the default for this partition.}
#' @export
clusters_post_kubernetes_partitions <- function(id, instance_configs, name, labels) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes/{id}/partitions"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(instanceConfigs = instance_configs, name = name, labels = labels)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a Cluster Partition
#' @param id integer required. The ID of the cluster which this partition belongs to.
#' @param cluster_partition_id integer required. The ID of this cluster partition.
#' @param instance_configs array optional. An array containing the following fields: 
#' \itemize{
#' \item instanceType string, An EC2 instance type. Possible values include t2.large, m4.xlarge, m4.2xlarge, m4.4xlarge, m5.12xlarge, and c5.18xlarge.
#' \item minInstances integer, The minimum number of instances of that type in this cluster.
#' \item maxInstances integer, The maximum number of instances of that type in this cluster.
#' }
#' @param name string optional. The name of the cluster partition.
#' @param labels array optional. Labels associated with this partition.
#' 
#' @return  A list containing the following elements:
#' \item{clusterPartitionId}{integer, The ID of this cluster partition.}
#' \item{name}{string, The name of the cluster partition.}
#' \item{labels}{array, Labels associated with this partition.}
#' \item{instanceConfigs}{array, An array containing the following fields: 
#' \itemize{
#' \item instanceConfigId integer, The ID of this InstanceConfig.
#' \item instanceType string, An EC2 instance type. Possible values include t2.large, m4.xlarge, m4.2xlarge, m4.4xlarge, m5.12xlarge, and c5.18xlarge.
#' \item minInstances integer, The minimum number of instances of that type in this cluster.
#' \item maxInstances integer, The maximum number of instances of that type in this cluster.
#' \item instanceMaxMemory integer, The amount of memory (RAM) available to a single instance of that type in megabytes.
#' \item instanceMaxCpu integer, The number of processor shares available to a single instance of that type in millicores.
#' \item instanceMaxDisk integer, The amount of disk available to a single instance of that type in gigabytes.
#' \item usageStats object, 
#' }}
#' \item{defaultInstanceConfigId}{integer, The id of the InstanceConfig that is the default for this partition.}
#' @export
clusters_patch_kubernetes_partitions <- function(id, cluster_partition_id, instance_configs = NULL, name = NULL, labels = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes/{id}/partitions/{cluster_partition_id}"
  path_params  <- list(id = id, cluster_partition_id = cluster_partition_id)
  query_params <- list()
  body_params  <- list(instanceConfigs = instance_configs, name = name, labels = labels)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Delete a Cluster Partition
#' @param id integer required. The ID of the cluster which this partition belongs to.
#' @param cluster_partition_id integer required. The ID of this cluster partition.
#' 
#' @return  An empty HTTP response
#' @export
clusters_delete_kubernetes_partitions <- function(id, cluster_partition_id) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes/{id}/partitions/{cluster_partition_id}"
  path_params  <- list(id = id, cluster_partition_id = cluster_partition_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Describe a Cluster Partition
#' @param id integer required. The ID of the cluster which this partition belongs to.
#' @param cluster_partition_id integer required. The ID of this cluster partition.
#' @param include_usage_stats boolean optional. When true, usage stats are returned in instance config objects. Defaults to false.
#' 
#' @return  A list containing the following elements:
#' \item{clusterPartitionId}{integer, The ID of this cluster partition.}
#' \item{name}{string, The name of the cluster partition.}
#' \item{labels}{array, Labels associated with this partition.}
#' \item{instanceConfigs}{array, An array containing the following fields: 
#' \itemize{
#' \item instanceConfigId integer, The ID of this InstanceConfig.
#' \item instanceType string, An EC2 instance type. Possible values include t2.large, m4.xlarge, m4.2xlarge, m4.4xlarge, m5.12xlarge, and c5.18xlarge.
#' \item minInstances integer, The minimum number of instances of that type in this cluster.
#' \item maxInstances integer, The maximum number of instances of that type in this cluster.
#' \item instanceMaxMemory integer, The amount of memory (RAM) available to a single instance of that type in megabytes.
#' \item instanceMaxCpu integer, The number of processor shares available to a single instance of that type in millicores.
#' \item instanceMaxDisk integer, The amount of disk available to a single instance of that type in gigabytes.
#' \item usageStats object, 
#' }}
#' \item{defaultInstanceConfigId}{integer, The id of the InstanceConfig that is the default for this partition.}
#' @export
clusters_get_kubernetes_partitions <- function(id, cluster_partition_id, include_usage_stats = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes/{id}/partitions/{cluster_partition_id}"
  path_params  <- list(id = id, cluster_partition_id = cluster_partition_id)
  query_params <- list(include_usage_stats = include_usage_stats)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Describe an Instance Config
#' @param instance_config_id integer required. The ID of this instance config.
#' @param include_usage_stats boolean optional. When true, usage stats are returned in instance config objects. Defaults to false.
#' 
#' @return  A list containing the following elements:
#' \item{instanceConfigId}{integer, The ID of this InstanceConfig.}
#' \item{instanceType}{string, An EC2 instance type. Possible values include t2.large, m4.xlarge, m4.2xlarge, m4.4xlarge, m5.12xlarge, and c5.18xlarge.}
#' \item{minInstances}{integer, The minimum number of instances of that type in this cluster.}
#' \item{maxInstances}{integer, The maximum number of instances of that type in this cluster.}
#' \item{instanceMaxMemory}{integer, The amount of memory (RAM) available to a single instance of that type in megabytes.}
#' \item{instanceMaxCpu}{integer, The number of processor shares available to a single instance of that type in millicores.}
#' \item{instanceMaxDisk}{integer, The amount of disk available to a single instance of that type in gigabytes.}
#' \item{usageStats}{list, A list containing the following elements: 
#' \itemize{
#' \item pendingMemoryRequested integer, The sum of memory requests (in MB) for pending deployments in this instance config.
#' \item pendingCpuRequested integer, The sum of cpu requests (in millicores) for pending deployments in this instance config.
#' \item runningMemoryRequested integer, The sum of memory requests (in MB) for running deployments in this instance config.
#' \item runningCpuRequested integer, The sum of cpu requests (in millicores) for running deployments in this instance config.
#' \item pendingDeployments integer, The number of pending deployments in this instance config.
#' \item runningDeployments integer, The number of running deployments in this instance config.
#' }}
#' \item{clusterPartitionId}{integer, The ID of this InstanceConfig's cluster partition}
#' \item{clusterPartitionName}{string, The name of this InstanceConfig's cluster partition}
#' @export
clusters_get_kubernetes_instance_configs <- function(instance_config_id, include_usage_stats = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes/instance_configs/{instance_config_id}"
  path_params  <- list(instance_config_id = instance_config_id)
  query_params <- list(include_usage_stats = include_usage_stats)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List active workloads in an Instance Config
#' @param id integer required. The id of the instance config.
#' @param state string optional. If specified, return workloads in these states. It accepts a comma-separated list, possible values are pending, running
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The id of this deployment.}
#' \item{baseType}{string, The base type of this deployment.}
#' \item{baseId}{integer, The id of the base object associated with this deployment.}
#' \item{baseObjectName}{string, The name of the base object associated with this deployment. Null if you do not have permission to read the object.}
#' \item{jobType}{string, If the base object is a job run you have permission to read, the type of the job. One of "python_script", "r_script", "container_script", or "custom_script".}
#' \item{jobId}{integer, If the base object is a job run you have permission to read, the id of the job.}
#' \item{jobCancelRequestedAt}{string, If the base object is a job run you have permission to read, and it was requested to be cancelled, the timestamp of that request.}
#' \item{state}{string, The state of this deployment.}
#' \item{cpu}{integer, The CPU in millicores requested by this deployment.}
#' \item{memory}{integer, The memory in MB requested by this deployment.}
#' \item{diskSpace}{integer, The disk space in GB requested by this deployment.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, The timestamp of when the deployment began.}
#' \item{cancellable}{boolean, True if you have permission to cancel this deployment.}
#' @export
clusters_list_kubernetes_instance_configs_active_workloads <- function(id, state = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes/instance_configs/{id}/active_workloads"
  path_params  <- list(id = id)
  query_params <- list(state = state)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get statistics about the current users of an Instance Config
#' @param instance_config_id integer required. The ID of this instance config.
#' @param order string optional. The field on which to order the result set. Defaults to running_deployments. Must be one of pending_memory_requested, pending_cpu_requested, running_memory_requested, running_cpu_requested, pending_deployments, running_deployments.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending). Defaults to desc.
#' 
#' @return  An array containing the following fields:
#' \item{userId}{string, The owning user's ID}
#' \item{userName}{string, The owning user's name}
#' \item{pendingDeployments}{integer, The number of deployments belonging to the owning user in "pending" state}
#' \item{pendingMemoryRequested}{integer, The sum of memory requests (in MB) for deployments belonging to the owning user in "pending" state}
#' \item{pendingCpuRequested}{integer, The sum of CPU requests (in millicores) for deployments belonging to the owning user in "pending" state}
#' \item{runningDeployments}{integer, The number of deployments belonging to the owning user in "running" state}
#' \item{runningMemoryRequested}{integer, The sum of memory requests (in MB) for deployments belonging to the owning user in "running" state}
#' \item{runningCpuRequested}{integer, The sum of CPU requests (in millicores) for deployments belonging to the owning user in "running" state}
#' @export
clusters_list_kubernetes_instance_configs_user_statistics <- function(instance_config_id, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes/instance_configs/{instance_config_id}/user_statistics"
  path_params  <- list(instance_config_id = instance_config_id)
  query_params <- list(order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get graphs of historical resource usage in an Instance Config
#' @param instance_config_id integer required. The ID of this instance config.
#' @param timeframe string optional. The span of time that the graphs cover. Must be one of 1_day, 1_week.
#' 
#' @return  A list containing the following elements:
#' \item{cpuGraphUrl}{string, URL for the graph of historical CPU usage in this instance config.}
#' \item{memGraphUrl}{string, URL for the graph of historical memory usage in this instance config.}
#' @export
clusters_list_kubernetes_instance_configs_historical_graphs <- function(instance_config_id, timeframe = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/clusters/kubernetes/instance_configs/{instance_config_id}/historical_graphs"
  path_params  <- list(instance_config_id = instance_config_id)
  query_params <- list(timeframe = timeframe)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get list of Credential Types
#' 
#' @return  A list containing the following elements:
#' \item{types}{array, list of acceptable credential types}
#' @export
credentials_list_types <- function() {

  args <- as.list(match.call())[-1]
  path <- "/credentials/types"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List credentials
#' @param type string optional. The type (or types) of credentials to return. One or more of: Amazon Web Services S3, Bitbucket, CASS/NCOA PAF, Certificate, Civis Platform, Custom, Database, Google, Github, Salesforce User, Salesforce Client, and TableauUser. Specify multiple values as a comma-separated list (e.g., "A,B").
#' @param remote_host_id integer optional. The ID of the remote host associated with the credentials to return.
#' @param default boolean optional. If true, will return a list with a single credential which is the current user's default credential.
#' @param system_credentials boolean optional. If true, will only return system credentials. System credentials can only be created and viewed by Civis Admins.
#' @param users string optional. A comma-separated list of user ids. If specified, returns set of credentials owned by the users that requesting user has at least read access on.
#' @param name string optional. If specified, will be used to filter the credentials returned. Will search across name and will return any full name containing the search string.
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 1000.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, created_at, name.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the credential.}
#' \item{name}{string, The name identifying the credential}
#' \item{type}{string, The credential's type.}
#' \item{username}{string, The username for the credential.}
#' \item{description}{string, A long description of the credential.}
#' \item{owner}{string, The username of the user who this credential belongs to. Using user.username is preferred.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{remoteHostId}{integer, The ID of the remote host associated with this credential.}
#' \item{remoteHostName}{string, The name of the remote host associated with this credential.}
#' \item{state}{string, The U.S. state for the credential. Only for VAN credentials.}
#' \item{createdAt}{string, The creation time for this credential.}
#' \item{updatedAt}{string, The last modification time for this credential.}
#' \item{default}{boolean, Whether or not the credential is a default. Only for Database credentials.}
#' @export
credentials_list <- function(type = NULL, remote_host_id = NULL, default = NULL, system_credentials = NULL, users = NULL, name = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/"
  path_params  <- list()
  query_params <- list(type = type, remote_host_id = remote_host_id, default = default, system_credentials = system_credentials, users = users, name = name, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a credential
#' @param type string required. The type of credential. Note: only these credentials can be created or edited via this API ["Amazon Web Services S3", "CASS/NCOA PAF", "Certificate", "Civis Platform", "Custom", "Database", "Google", "Salesforce User", "Salesforce Client", "TableauUser"]
#' @param username string required. The username for the credential.
#' @param password string required. The password for the credential.
#' @param name string optional. The name identifying the credential.
#' @param description string optional. A long description of the credential.
#' @param remote_host_id integer optional. The ID of the remote host associated with the credential.
#' @param user_id integer optional. The ID of the user the credential is created for. Note: This attribute is only accepted if you are a Civis Admin User.
#' @param state string optional. The U.S. state for the credential. Only for VAN credentials.
#' @param system_credential boolean optional. Boolean flag that sets a credential to be a system credential. System credentials can only be created by Civis Admins and will create a credential owned by the Civis Robot user.
#' @param default boolean optional. Whether or not the credential is a default. Only for Database credentials.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the credential.}
#' \item{name}{string, The name identifying the credential}
#' \item{type}{string, The credential's type.}
#' \item{username}{string, The username for the credential.}
#' \item{description}{string, A long description of the credential.}
#' \item{owner}{string, The username of the user who this credential belongs to. Using user.username is preferred.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{remoteHostId}{integer, The ID of the remote host associated with this credential.}
#' \item{remoteHostName}{string, The name of the remote host associated with this credential.}
#' \item{state}{string, The U.S. state for the credential. Only for VAN credentials.}
#' \item{createdAt}{string, The creation time for this credential.}
#' \item{updatedAt}{string, The last modification time for this credential.}
#' \item{default}{boolean, Whether or not the credential is a default. Only for Database credentials.}
#' @export
credentials_post <- function(type, username, password, name = NULL, description = NULL, remote_host_id = NULL, user_id = NULL, state = NULL, system_credential = NULL, default = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(type = type, username = username, password = password, name = name, description = description, remoteHostId = remote_host_id, userId = user_id, state = state, systemCredential = system_credential, default = default)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update an existing credential
#' @param id integer required. The ID of the credential.
#' @param type string required. The type of credential. Note: only these credentials can be created or edited via this API ["Amazon Web Services S3", "CASS/NCOA PAF", "Certificate", "Civis Platform", "Custom", "Database", "Google", "Salesforce User", "Salesforce Client", "TableauUser"]
#' @param username string required. The username for the credential.
#' @param password string required. The password for the credential.
#' @param name string optional. The name identifying the credential.
#' @param description string optional. A long description of the credential.
#' @param remote_host_id integer optional. The ID of the remote host associated with the credential.
#' @param user_id integer optional. The ID of the user the credential is created for. Note: This attribute is only accepted if you are a Civis Admin User.
#' @param state string optional. The U.S. state for the credential. Only for VAN credentials.
#' @param system_credential boolean optional. Boolean flag that sets a credential to be a system credential. System credentials can only be created by Civis Admins and will create a credential owned by the Civis Robot user.
#' @param default boolean optional. Whether or not the credential is a default. Only for Database credentials.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the credential.}
#' \item{name}{string, The name identifying the credential}
#' \item{type}{string, The credential's type.}
#' \item{username}{string, The username for the credential.}
#' \item{description}{string, A long description of the credential.}
#' \item{owner}{string, The username of the user who this credential belongs to. Using user.username is preferred.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{remoteHostId}{integer, The ID of the remote host associated with this credential.}
#' \item{remoteHostName}{string, The name of the remote host associated with this credential.}
#' \item{state}{string, The U.S. state for the credential. Only for VAN credentials.}
#' \item{createdAt}{string, The creation time for this credential.}
#' \item{updatedAt}{string, The last modification time for this credential.}
#' \item{default}{boolean, Whether or not the credential is a default. Only for Database credentials.}
#' @export
credentials_put <- function(id, type, username, password, name = NULL, description = NULL, remote_host_id = NULL, user_id = NULL, state = NULL, system_credential = NULL, default = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(type = type, username = username, password = password, name = name, description = description, remoteHostId = remote_host_id, userId = user_id, state = state, systemCredential = system_credential, default = default)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of a credential
#' @param id integer required. The ID of the credential.
#' @param name string optional. The name identifying the credential.
#' @param type string optional. The type of credential. Note: only these credentials can be created or edited via this API ["Amazon Web Services S3", "CASS/NCOA PAF", "Certificate", "Civis Platform", "Custom", "Database", "Google", "Salesforce User", "Salesforce Client", "TableauUser"]
#' @param description string optional. A long description of the credential.
#' @param username string optional. The username for the credential.
#' @param password string optional. The password for the credential.
#' @param remote_host_id integer optional. The ID of the remote host associated with the credential.
#' @param user_id integer optional. The ID of the user the credential is created for. Note: This attribute is only accepted if you are a Civis Admin User.
#' @param state string optional. The U.S. state for the credential. Only for VAN credentials.
#' @param system_credential boolean optional. Boolean flag that sets a credential to be a system credential. System credentials can only be created by Civis Admins and will create a credential owned by the Civis Robot user.
#' @param default boolean optional. Whether or not the credential is a default. Only for Database credentials.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the credential.}
#' \item{name}{string, The name identifying the credential}
#' \item{type}{string, The credential's type.}
#' \item{username}{string, The username for the credential.}
#' \item{description}{string, A long description of the credential.}
#' \item{owner}{string, The username of the user who this credential belongs to. Using user.username is preferred.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{remoteHostId}{integer, The ID of the remote host associated with this credential.}
#' \item{remoteHostName}{string, The name of the remote host associated with this credential.}
#' \item{state}{string, The U.S. state for the credential. Only for VAN credentials.}
#' \item{createdAt}{string, The creation time for this credential.}
#' \item{updatedAt}{string, The last modification time for this credential.}
#' \item{default}{boolean, Whether or not the credential is a default. Only for Database credentials.}
#' @export
credentials_patch <- function(id, name = NULL, type = NULL, description = NULL, username = NULL, password = NULL, remote_host_id = NULL, user_id = NULL, state = NULL, system_credential = NULL, default = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, type = type, description = description, username = username, password = password, remoteHostId = remote_host_id, userId = user_id, state = state, systemCredential = system_credential, default = default)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a credential
#' @param id integer required. The ID of the credential.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the credential.}
#' \item{name}{string, The name identifying the credential}
#' \item{type}{string, The credential's type.}
#' \item{username}{string, The username for the credential.}
#' \item{description}{string, A long description of the credential.}
#' \item{owner}{string, The username of the user who this credential belongs to. Using user.username is preferred.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{remoteHostId}{integer, The ID of the remote host associated with this credential.}
#' \item{remoteHostName}{string, The name of the remote host associated with this credential.}
#' \item{state}{string, The U.S. state for the credential. Only for VAN credentials.}
#' \item{createdAt}{string, The creation time for this credential.}
#' \item{updatedAt}{string, The last modification time for this credential.}
#' \item{default}{boolean, Whether or not the credential is a default. Only for Database credentials.}
#' @export
credentials_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Delete a credential
#' @param id integer required. The ID of the credential.
#' 
#' @return  An empty HTTP response
#' @export
credentials_delete <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Authenticate against a remote host
#' @param url string required. The URL to your host.
#' @param remote_host_type string required. The type of remote host. One of: RemoteHostTypes::Bigquery, RemoteHostTypes::Bitbucket, RemoteHostTypes::GitSSH, RemoteHostTypes::Github, RemoteHostTypes::GoogleDoc, RemoteHostTypes::JDBC, RemoteHostTypes::Postgres, RemoteHostTypes::Redshift, RemoteHostTypes::S3Storage, and RemoteHostTypes::Salesforce
#' @param username string required. The username for the credential.
#' @param password string required. The password for the credential.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the credential.}
#' \item{name}{string, The name identifying the credential}
#' \item{type}{string, The credential's type.}
#' \item{username}{string, The username for the credential.}
#' \item{description}{string, A long description of the credential.}
#' \item{owner}{string, The username of the user who this credential belongs to. Using user.username is preferred.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{remoteHostId}{integer, The ID of the remote host associated with this credential.}
#' \item{remoteHostName}{string, The name of the remote host associated with this credential.}
#' \item{state}{string, The U.S. state for the credential. Only for VAN credentials.}
#' \item{createdAt}{string, The creation time for this credential.}
#' \item{updatedAt}{string, The last modification time for this credential.}
#' \item{default}{boolean, Whether or not the credential is a default. Only for Database credentials.}
#' @export
credentials_post_authenticate <- function(url, remote_host_type, username, password) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/authenticate"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(url = url, remoteHostType = remote_host_type, username = username, password = password)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Generate a temporary credential for accessing S3
#' @param id integer required. The ID of the credential.
#' @param duration integer optional. The number of seconds the temporary credential should be valid. Defaults to 15 minutes. Must not be less than 15 minutes or greater than 36 hours.
#' 
#' @return  A list containing the following elements:
#' \item{accessKey}{string, The identifier of the credential.}
#' \item{secretAccessKey}{string, The secret part of the credential.}
#' \item{sessionToken}{string, The session token identifier.}
#' @export
credentials_post_temporary <- function(id, duration = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}/temporary"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(duration = duration)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
credentials_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
credentials_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
credentials_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
credentials_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
credentials_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
credentials_list_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
credentials_put_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/credentials/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List databases
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for the database.}
#' \item{name}{string, The name of the database.}
#' \item{adapter}{string, The type of the database.}
#' @export
databases_list <- function() {

  args <- as.list(match.call())[-1]
  path <- "/databases/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show database information
#' @param id integer required. The ID for the database.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the database.}
#' \item{name}{string, The name of the database.}
#' \item{adapter}{string, The type of the database.}
#' @export
databases_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List schemas in this database
#' @param id integer required. The ID of the database.
#' @param name string optional. If specified, will be used to filter the schemas returned. Substring matching is supported (e.g., "name=schema" will return both "schema1" and "schema2").
#' @param credential_id integer optional. If provided, schemas will be filtered based on the given credential.
#' 
#' @return  An array containing the following fields:
#' \item{schema}{string, The name of a schema.}
#' @export
databases_list_schemas <- function(id, name = NULL, credential_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/schemas"
  path_params  <- list(id = id)
  query_params <- list(name = name, credential_id = credential_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Creates and enqueues a schema scanner job
#' @param id integer required. The ID of the database.
#' @param schema string required. The name of the schema.
#' @param stats_priority string optional. When to sync table statistics for every table in the schema. Valid options are the following. Option: 'flag' means to flag stats for the next scheduled run of a full table scan on the database. Option: 'block' means to block this job on stats syncing. Option: 'queue' means to queue a separate job for syncing stats and do not block this job on the queued job. Defaults to 'flag'
#' 
#' @return  A list containing the following elements:
#' \item{jobId}{integer, The ID of the job created.}
#' \item{runId}{integer, The ID of the run created.}
#' @export
databases_post_schemas_scan <- function(id, schema, stats_priority = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/schemas/scan"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(schema = schema, statsPriority = stats_priority)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List tables in the specified database, deprecated use "GET /tables" instead
#' @param id integer required. The ID of the database.
#' @param name string optional. If specified, will be used to filter the tables returned. Substring matching is supported (e.g., "name=table" will return both "table1" and "my table").
#' @param limit integer optional. Number of results to return. Defaults to 200. Maximum allowed is 1000.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to name. Must be one of: name.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the table.}
#' \item{databaseId}{integer, The ID of the database.}
#' \item{schema}{string, The name of the schema containing the table.}
#' \item{name}{string, Name of the table.}
#' \item{description}{string, The description of the table, as specified by the table owner}
#' \item{isView}{boolean, True if this table represents a view. False if it represents a regular table.}
#' \item{rowCount}{integer, The number of rows in the table.}
#' \item{columnCount}{integer, The number of columns in the table.}
#' \item{sizeMb}{number, The size of the table in megabytes.}
#' \item{owner}{string, The database username of the table's owner.}
#' \item{distkey}{string, The column used as the Amazon Redshift distkey.}
#' \item{sortkeys}{string, The column used as the Amazon Redshift sortkey.}
#' \item{refreshStatus}{string, How up-to-date the table's statistics on row counts, null counts, distinct counts, and values distributions are. One of: refreshing, stale, or current.}
#' \item{lastRefresh}{string, The time of the last statistics refresh.}
#' \item{refreshId}{string, The ID of the most recent statistics refresh.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{tableTags}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, Table Tag ID
#' \item name string, Table Tag Name
#' }}
#' @export
databases_list_tables <- function(id, name = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/tables"
  path_params  <- list(id = id)
  query_params <- list(name = name, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List tables in the specified database, deprecated use "GET /tables" instead
#' @param id integer required. The ID of the database.
#' @param name string optional. If specified, will be used to filter the tables returned. Substring matching is supported (e.g., "name=table" will return both "table1" and "my table").
#' @param column_name string optional. Search for tables containing a column with the given name.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the table.}
#' \item{databaseId}{integer, The ID of the database.}
#' \item{schema}{string, The name of the schema containing the table.}
#' \item{name}{string, Name of the table.}
#' \item{description}{string, The description of the table, as specified by the table owner}
#' \item{isView}{boolean, True if this table represents a view. False if it represents a regular table.}
#' \item{rowCount}{integer, The number of rows in the table.}
#' \item{columnCount}{integer, The number of columns in the table.}
#' \item{sizeMb}{number, The size of the table in megabytes.}
#' \item{owner}{string, The database username of the table's owner.}
#' \item{distkey}{string, The column used as the Amazon Redshift distkey.}
#' \item{sortkeys}{string, The column used as the Amazon Redshift sortkey.}
#' \item{refreshStatus}{string, How up-to-date the table's statistics on row counts, null counts, distinct counts, and values distributions are. One of: refreshing, stale, or current.}
#' \item{lastRefresh}{string, The time of the last statistics refresh.}
#' \item{refreshId}{string, The ID of the most recent statistics refresh.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{tableTags}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, Table Tag ID
#' \item name string, Table Tag Name
#' }}
#' \item{columnNames}{array, The names of each column in the table.}
#' @export
databases_list_tables_search <- function(id, name = NULL, column_name = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/tables-search"
  path_params  <- list(id = id)
  query_params <- list(name = name, column_name = column_name)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show table privileges
#' @param id integer required. The ID of the database
#' @param schema_name string required. The name of the schema
#' @param table_name string required. The name of the table
#' 
#' @return  A list containing the following elements:
#' \item{grantee}{string, Name of the granted user or group}
#' \item{granteeType}{string, User or group}
#' \item{privileges}{array, Privileges that the grantee has on this resource}
#' \item{grantablePrivileges}{array, Privileges that the grantee can grant to others for this resource}
#' @export
databases_get_table_privilegesschema_name <- function(id, schema_name, table_name) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/table_privileges/{schema_name}/{table_name}"
  path_params  <- list(id = id, schema_name = schema_name, table_name = table_name)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show schema privileges
#' @param id integer required. The ID of the database
#' @param schema_name string required. The name of the schema
#' 
#' @return  A list containing the following elements:
#' \item{grantee}{string, Name of the granted user or group}
#' \item{granteeType}{string, User or group}
#' \item{privileges}{array, Privileges that the grantee has on this resource}
#' \item{grantablePrivileges}{array, Privileges that the grantee can grant to others for this resource}
#' @export
databases_get_schema_privileges <- function(id, schema_name) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/schema_privileges/{schema_name}"
  path_params  <- list(id = id, schema_name = schema_name)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List whitelisted IPs for the specified database
#' @param id integer required. The ID for the database.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this whitelisted IP address.}
#' \item{remoteHostId}{integer, The ID of the database this rule is applied to.}
#' \item{securityGroupId}{string, The ID of the security group this rule is applied to.}
#' \item{subnetMask}{string, The subnet mask that is allowed by this rule.}
#' \item{createdAt}{string, The time this rule was created.}
#' \item{updatedAt}{string, The time this rule was last updated.}
#' @export
databases_list_whitelist_ips <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/whitelist-ips"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' View details about a whitelisted IP
#' @param id integer required. The ID of the database this rule is applied to.
#' @param whitelisted_ip_id integer required. The ID of this whitelisted IP address.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this whitelisted IP address.}
#' \item{remoteHostId}{integer, The ID of the database this rule is applied to.}
#' \item{securityGroupId}{string, The ID of the security group this rule is applied to.}
#' \item{subnetMask}{string, The subnet mask that is allowed by this rule.}
#' \item{authorizedBy}{string, The user who authorized this rule.}
#' \item{isActive}{boolean, True if the rule is applied, false if it has been revoked.}
#' \item{createdAt}{string, The time this rule was created.}
#' \item{updatedAt}{string, The time this rule was last updated.}
#' @export
databases_get_whitelist_ips <- function(id, whitelisted_ip_id) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/whitelist-ips/{whitelisted_ip_id}"
  path_params  <- list(id = id, whitelisted_ip_id = whitelisted_ip_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the advanced settings for this database
#' @param id integer required. The ID of the database this advanced settings object belongs to.
#' 
#' @return  A list containing the following elements:
#' \item{exportCachingEnabled}{boolean, Whether or not caching is enabled for export jobs run on this database server.}
#' @export
databases_list_advanced_settings <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/advanced-settings"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the advanced settings for this database
#' @param id integer required. The ID of the database this advanced settings object belongs to.
#' @param export_caching_enabled boolean optional. Whether or not caching is enabled for export jobs run on this database server.
#' 
#' @return  A list containing the following elements:
#' \item{exportCachingEnabled}{boolean, Whether or not caching is enabled for export jobs run on this database server.}
#' @export
databases_patch_advanced_settings <- function(id, export_caching_enabled = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/advanced-settings"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(exportCachingEnabled = export_caching_enabled)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Edit the advanced settings for this database
#' @param id integer required. The ID of the database this advanced settings object belongs to.
#' @param export_caching_enabled boolean required. Whether or not caching is enabled for export jobs run on this database server.
#' 
#' @return  A list containing the following elements:
#' \item{exportCachingEnabled}{boolean, Whether or not caching is enabled for export jobs run on this database server.}
#' @export
databases_put_advanced_settings <- function(id, export_caching_enabled) {

  args <- as.list(match.call())[-1]
  path <- "/databases/{id}/advanced-settings"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(exportCachingEnabled = export_caching_enabled)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List API endpoints
#' 
#' @return  An empty HTTP response
#' @export
endpoints_list <- function() {

  args <- as.list(match.call())[-1]
  path <- "/endpoints/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Civis Data Match Enhancement
#' @param name string required. The name of the enhancement job.
#' @param input_field_mapping list required. The field (i.e., column) mapping for the input table. See https://api.civisanalytics.com/enhancements/field-mapping for a list of valid field types and descriptions. Each field type should be mapped to a string specifying a column name in the input table. For field types that support multiple values (e.g., the "phone" field), a list of column names can be provided (e.g., {"phone": ["home_phone", "mobile_phone"], ...}).
#' @param input_table list required. A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }
#' @param match_target_id integer required. The ID of the Civis Data match target. See /match_targets for IDs.
#' @param output_table list required. A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }
#' @param schedule list optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications list optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param max_matches integer optional. The maximum number of matches per record in the input table to return. Must be between 0 and 10. 0 returns all matches.
#' @param threshold number optional. The score threshold (between 0 and 1). Matches below this threshold will not be returned. The default value is 0.5.
#' @param archived boolean optional. Whether the Civis Data Match Job has been archived.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{inputFieldMapping}{list, The field (i.e., column) mapping for the input table. See https://api.civisanalytics.com/enhancements/field-mapping for a list of valid field types and descriptions. Each field type should be mapped to a string specifying a column name in the input table. For field types that support multiple values (e.g., the "phone" field), a list of column names can be provided (e.g., {"phone": ["home_phone", "mobile_phone"], ...}).}
#' \item{inputTable}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }}
#' \item{matchTargetId}{integer, The ID of the Civis Data match target. See /match_targets for IDs.}
#' \item{outputTable}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }}
#' \item{maxMatches}{integer, The maximum number of matches per record in the input table to return. Must be between 0 and 10. 0 returns all matches.}
#' \item{threshold}{number, The score threshold (between 0 and 1). Matches below this threshold will not be returned. The default value is 0.5.}
#' \item{archived}{boolean, Whether the Civis Data Match Job has been archived.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' @export
enhancements_post_civis_data_match <- function(name, input_field_mapping, input_table, match_target_id, output_table, schedule = NULL, parent_id = NULL, notifications = NULL, max_matches = NULL, threshold = NULL, archived = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, inputFieldMapping = input_field_mapping, inputTable = input_table, matchTargetId = match_target_id, outputTable = output_table, schedule = schedule, parentId = parent_id, notifications = notifications, maxMatches = max_matches, threshold = threshold, archived = archived)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a Civis Data Match Enhancement
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{inputFieldMapping}{list, The field (i.e., column) mapping for the input table. See https://api.civisanalytics.com/enhancements/field-mapping for a list of valid field types and descriptions. Each field type should be mapped to a string specifying a column name in the input table. For field types that support multiple values (e.g., the "phone" field), a list of column names can be provided (e.g., {"phone": ["home_phone", "mobile_phone"], ...}).}
#' \item{inputTable}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }}
#' \item{matchTargetId}{integer, The ID of the Civis Data match target. See /match_targets for IDs.}
#' \item{outputTable}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }}
#' \item{maxMatches}{integer, The maximum number of matches per record in the input table to return. Must be between 0 and 10. 0 returns all matches.}
#' \item{threshold}{number, The score threshold (between 0 and 1). Matches below this threshold will not be returned. The default value is 0.5.}
#' \item{archived}{boolean, Whether the Civis Data Match Job has been archived.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' @export
enhancements_get_civis_data_match <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this Civis Data Match Enhancement
#' @param id integer required. The ID for the enhancement.
#' @param name string required. The name of the enhancement job.
#' @param input_field_mapping list required. The field (i.e., column) mapping for the input table. See https://api.civisanalytics.com/enhancements/field-mapping for a list of valid field types and descriptions. Each field type should be mapped to a string specifying a column name in the input table. For field types that support multiple values (e.g., the "phone" field), a list of column names can be provided (e.g., {"phone": ["home_phone", "mobile_phone"], ...}).
#' @param input_table list required. A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }
#' @param match_target_id integer required. The ID of the Civis Data match target. See /match_targets for IDs.
#' @param output_table list required. A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }
#' @param schedule list optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications list optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param max_matches integer optional. The maximum number of matches per record in the input table to return. Must be between 0 and 10. 0 returns all matches.
#' @param threshold number optional. The score threshold (between 0 and 1). Matches below this threshold will not be returned. The default value is 0.5.
#' @param archived boolean optional. Whether the Civis Data Match Job has been archived.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{inputFieldMapping}{list, The field (i.e., column) mapping for the input table. See https://api.civisanalytics.com/enhancements/field-mapping for a list of valid field types and descriptions. Each field type should be mapped to a string specifying a column name in the input table. For field types that support multiple values (e.g., the "phone" field), a list of column names can be provided (e.g., {"phone": ["home_phone", "mobile_phone"], ...}).}
#' \item{inputTable}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }}
#' \item{matchTargetId}{integer, The ID of the Civis Data match target. See /match_targets for IDs.}
#' \item{outputTable}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }}
#' \item{maxMatches}{integer, The maximum number of matches per record in the input table to return. Must be between 0 and 10. 0 returns all matches.}
#' \item{threshold}{number, The score threshold (between 0 and 1). Matches below this threshold will not be returned. The default value is 0.5.}
#' \item{archived}{boolean, Whether the Civis Data Match Job has been archived.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' @export
enhancements_put_civis_data_match <- function(id, name, input_field_mapping, input_table, match_target_id, output_table, schedule = NULL, parent_id = NULL, notifications = NULL, max_matches = NULL, threshold = NULL, archived = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, inputFieldMapping = input_field_mapping, inputTable = input_table, matchTargetId = match_target_id, outputTable = output_table, schedule = schedule, parentId = parent_id, notifications = notifications, maxMatches = max_matches, threshold = threshold, archived = archived)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this Civis Data Match Enhancement
#' @param id integer required. The ID for the enhancement.
#' @param name string optional. The name of the enhancement job.
#' @param schedule list optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications list optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param input_field_mapping list optional. The field (i.e., column) mapping for the input table. See https://api.civisanalytics.com/enhancements/field-mapping for a list of valid field types and descriptions. Each field type should be mapped to a string specifying a column name in the input table. For field types that support multiple values (e.g., the "phone" field), a list of column names can be provided (e.g., {"phone": ["home_phone", "mobile_phone"], ...}).
#' @param input_table list optional. A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }
#' @param match_target_id integer optional. The ID of the Civis Data match target. See /match_targets for IDs.
#' @param output_table list optional. A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }
#' @param max_matches integer optional. The maximum number of matches per record in the input table to return. Must be between 0 and 10. 0 returns all matches.
#' @param threshold number optional. The score threshold (between 0 and 1). Matches below this threshold will not be returned. The default value is 0.5.
#' @param archived boolean optional. Whether the Civis Data Match Job has been archived.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{inputFieldMapping}{list, The field (i.e., column) mapping for the input table. See https://api.civisanalytics.com/enhancements/field-mapping for a list of valid field types and descriptions. Each field type should be mapped to a string specifying a column name in the input table. For field types that support multiple values (e.g., the "phone" field), a list of column names can be provided (e.g., {"phone": ["home_phone", "mobile_phone"], ...}).}
#' \item{inputTable}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }}
#' \item{matchTargetId}{integer, The ID of the Civis Data match target. See /match_targets for IDs.}
#' \item{outputTable}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }}
#' \item{maxMatches}{integer, The maximum number of matches per record in the input table to return. Must be between 0 and 10. 0 returns all matches.}
#' \item{threshold}{number, The score threshold (between 0 and 1). Matches below this threshold will not be returned. The default value is 0.5.}
#' \item{archived}{boolean, Whether the Civis Data Match Job has been archived.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' @export
enhancements_patch_civis_data_match <- function(id, name = NULL, schedule = NULL, parent_id = NULL, notifications = NULL, input_field_mapping = NULL, input_table = NULL, match_target_id = NULL, output_table = NULL, max_matches = NULL, threshold = NULL, archived = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, schedule = schedule, parentId = parent_id, notifications = notifications, inputFieldMapping = input_field_mapping, inputTable = input_table, matchTargetId = match_target_id, outputTable = output_table, maxMatches = max_matches, threshold = threshold, archived = archived)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Clone this Civis Data Match Enhancement
#' @param id integer required. The ID for the enhancement.
#' @param clone_schedule boolean optional. If true, also copy the schedule to the new enhancement.
#' @param clone_triggers boolean optional. If true, also copy the triggers to the new enhancement.
#' @param clone_notifications boolean optional. If true, also copy the notifications to the new enhancement.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{inputFieldMapping}{list, The field (i.e., column) mapping for the input table. See https://api.civisanalytics.com/enhancements/field-mapping for a list of valid field types and descriptions. Each field type should be mapped to a string specifying a column name in the input table. For field types that support multiple values (e.g., the "phone" field), a list of column names can be provided (e.g., {"phone": ["home_phone", "mobile_phone"], ...}).}
#' \item{inputTable}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }}
#' \item{matchTargetId}{integer, The ID of the Civis Data match target. See /match_targets for IDs.}
#' \item{outputTable}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }}
#' \item{maxMatches}{integer, The maximum number of matches per record in the input table to return. Must be between 0 and 10. 0 returns all matches.}
#' \item{threshold}{number, The score threshold (between 0 and 1). Matches below this threshold will not be returned. The default value is 0.5.}
#' \item{archived}{boolean, Whether the Civis Data Match Job has been archived.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' @export
enhancements_post_civis_data_match_clone <- function(id, clone_schedule = NULL, clone_triggers = NULL, clone_notifications = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/clone"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(cloneSchedule = clone_schedule, cloneTriggers = clone_triggers, cloneNotifications = clone_notifications)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the civis_data_match.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{civisDataMatchId}{integer, The ID of the civis_data_match.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_post_civis_data_match_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given civis_data_match
#' @param id integer required. The ID of the civis_data_match.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the run.}
#' \item{civisDataMatchId}{integer, The ID of the civis_data_match.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_list_civis_data_match_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the civis_data_match.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{civisDataMatchId}{integer, The ID of the civis_data_match.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_get_civis_data_match_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the civis_data_match.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_civis_data_match_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the civis_data_match.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
enhancements_list_civis_data_match_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{state}{string, The state of the run, one of 'queued', 'running' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' @export
enhancements_post_civis_data_match_cancel <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/cancel"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the outputs for a run
#' @param id integer required. The ID of the job.
#' @param run_id integer required. The ID of the run.
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to created_at. Must be one of: created_at, id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, The type of the output. Valid values are File, Table, Report, Project, Credential, or JSONValue}
#' \item{objectId}{integer, The ID of the output.}
#' \item{name}{string, The name of the output.}
#' \item{link}{string, The hypermedia link to the output.}
#' \item{value}{string, }
#' @export
enhancements_list_civis_data_match_runs_outputs <- function(id, run_id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/runs/{run_id}/outputs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_list_civis_data_match_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_put_civis_data_match_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_civis_data_match_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_put_civis_data_match_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_civis_data_match_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
enhancements_list_civis_data_match_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
enhancements_put_civis_data_match_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{inputFieldMapping}{list, The field (i.e., column) mapping for the input table. See https://api.civisanalytics.com/enhancements/field-mapping for a list of valid field types and descriptions. Each field type should be mapped to a string specifying a column name in the input table. For field types that support multiple values (e.g., the "phone" field), a list of column names can be provided (e.g., {"phone": ["home_phone", "mobile_phone"], ...}).}
#' \item{inputTable}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }}
#' \item{matchTargetId}{integer, The ID of the Civis Data match target. See /match_targets for IDs.}
#' \item{outputTable}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseName string, The Redshift database name for the table.
#' \item schema string, The schema name for the table.
#' \item table string, The table name.
#' }}
#' \item{maxMatches}{integer, The maximum number of matches per record in the input table to return. Must be between 0 and 10. 0 returns all matches.}
#' \item{threshold}{number, The score threshold (between 0 and 1). Matches below this threshold will not be returned. The default value is 0.5.}
#' \item{archived}{boolean, Whether the Civis Data Match Job has been archived.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' @export
enhancements_put_civis_data_match_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a Civis Data Match Enhancement belongs to
#' @param id integer required. The ID of the Civis Data Match Enhancement.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
enhancements_list_civis_data_match_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a Civis Data Match Enhancement to a project
#' @param id integer required. The ID of the Civis Data Match Enhancement.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_put_civis_data_match_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a Civis Data Match Enhancement from a project
#' @param id integer required. The ID of the Civis Data Match Enhancement.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_civis_data_match_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/civis-data-match/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List available enhancement types
#' 
#' @return  An array containing the following fields:
#' \item{name}{string, The name of the type.}
#' @export
enhancements_list_types <- function() {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/types"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the fields in a field mapping for Civis Data Match, Data Unification, and Table Deduplication jobs
#' 
#' @return  An array containing the following fields:
#' \item{field}{string, The name of the field.}
#' \item{description}{string, The description of the field.}
#' @export
enhancements_list_field_mapping <- function() {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/field-mapping"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Enhancements
#' @param type string optional. If specified, return items of these types.
#' @param author string optional. If specified, return items from any of these authors. It accepts a comma-separated list of user IDs.
#' @param status string optional. If specified, returns items with one of these statuses. It accepts a comma-separated list, possible values are 'running', 'failed', 'succeeded', 'idle', 'scheduled'.
#' @param archived string optional. The archival status of the requested item(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at, last_run.updated_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
enhancements_list <- function(type = NULL, author = NULL, status = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/"
  path_params  <- list()
  query_params <- list(type = type, author = author, status = status, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a CASS/NCOA Enhancement
#' @param name string required. The name of the enhancement job.
#' @param source list required. A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name of the source table.
#' \item table string, The name of the source table.
#' \item remoteHostId integer, The ID of the database host for the table.
#' \item credentialId integer, The id of the credentials to be used when performing the enhancement.
#' \item multipartKey array, The source table primary key.
#' }
#' }
#' @param schedule list optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications list optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param destination list optional. A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name for the output data.
#' \item table string, The table name for the output data.
#' }
#' }
#' @param column_mapping list optional. A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }
#' @param use_default_column_mapping boolean optional. Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.
#' @param perform_ncoa boolean optional. Whether to update addresses for records matching the National Change of Address (NCOA) database.
#' @param ncoa_credential_id integer optional. Credential to use when performing NCOA updates. Required if 'performNcoa' is true.
#' @param output_level string optional. The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.
#' @param limiting_sql string optional. The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').
#' @param chunk_size integer optional. The maximum number of records processed at a time. Note that this parameter is not available to all users.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name of the source table.
#' \item table string, The name of the source table.
#' \item remoteHostId integer, The ID of the database host for the table.
#' \item credentialId integer, The id of the credentials to be used when performing the enhancement.
#' \item multipartKey array, The source table primary key.
#' }
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name for the output data.
#' \item table string, The table name for the output data.
#' }
#' }}
#' \item{columnMapping}{list, A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }}
#' \item{useDefaultColumnMapping}{boolean, Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.}
#' \item{performNcoa}{boolean, Whether to update addresses for records matching the National Change of Address (NCOA) database.}
#' \item{ncoaCredentialId}{integer, Credential to use when performing NCOA updates. Required if 'performNcoa' is true.}
#' \item{outputLevel}{string, The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.}
#' \item{limitingSQL}{string, The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').}
#' \item{chunkSize}{integer, The maximum number of records processed at a time. Note that this parameter is not available to all users.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
enhancements_post_cass_ncoa <- function(name, source, schedule = NULL, parent_id = NULL, notifications = NULL, destination = NULL, column_mapping = NULL, use_default_column_mapping = NULL, perform_ncoa = NULL, ncoa_credential_id = NULL, output_level = NULL, limiting_sql = NULL, chunk_size = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, source = source, schedule = schedule, parentId = parent_id, notifications = notifications, destination = destination, columnMapping = column_mapping, useDefaultColumnMapping = use_default_column_mapping, performNcoa = perform_ncoa, ncoaCredentialId = ncoa_credential_id, outputLevel = output_level, limitingSQL = limiting_sql, chunkSize = chunk_size)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a CASS/NCOA Enhancement
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name of the source table.
#' \item table string, The name of the source table.
#' \item remoteHostId integer, The ID of the database host for the table.
#' \item credentialId integer, The id of the credentials to be used when performing the enhancement.
#' \item multipartKey array, The source table primary key.
#' }
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name for the output data.
#' \item table string, The table name for the output data.
#' }
#' }}
#' \item{columnMapping}{list, A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }}
#' \item{useDefaultColumnMapping}{boolean, Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.}
#' \item{performNcoa}{boolean, Whether to update addresses for records matching the National Change of Address (NCOA) database.}
#' \item{ncoaCredentialId}{integer, Credential to use when performing NCOA updates. Required if 'performNcoa' is true.}
#' \item{outputLevel}{string, The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.}
#' \item{limitingSQL}{string, The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').}
#' \item{chunkSize}{integer, The maximum number of records processed at a time. Note that this parameter is not available to all users.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
enhancements_get_cass_ncoa <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this CASS/NCOA Enhancement
#' @param id integer required. The ID for the enhancement.
#' @param name string required. The name of the enhancement job.
#' @param source list required. A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name of the source table.
#' \item table string, The name of the source table.
#' \item remoteHostId integer, The ID of the database host for the table.
#' \item credentialId integer, The id of the credentials to be used when performing the enhancement.
#' \item multipartKey array, The source table primary key.
#' }
#' }
#' @param schedule list optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications list optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param destination list optional. A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name for the output data.
#' \item table string, The table name for the output data.
#' }
#' }
#' @param column_mapping list optional. A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }
#' @param use_default_column_mapping boolean optional. Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.
#' @param perform_ncoa boolean optional. Whether to update addresses for records matching the National Change of Address (NCOA) database.
#' @param ncoa_credential_id integer optional. Credential to use when performing NCOA updates. Required if 'performNcoa' is true.
#' @param output_level string optional. The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.
#' @param limiting_sql string optional. The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').
#' @param chunk_size integer optional. The maximum number of records processed at a time. Note that this parameter is not available to all users.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name of the source table.
#' \item table string, The name of the source table.
#' \item remoteHostId integer, The ID of the database host for the table.
#' \item credentialId integer, The id of the credentials to be used when performing the enhancement.
#' \item multipartKey array, The source table primary key.
#' }
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name for the output data.
#' \item table string, The table name for the output data.
#' }
#' }}
#' \item{columnMapping}{list, A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }}
#' \item{useDefaultColumnMapping}{boolean, Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.}
#' \item{performNcoa}{boolean, Whether to update addresses for records matching the National Change of Address (NCOA) database.}
#' \item{ncoaCredentialId}{integer, Credential to use when performing NCOA updates. Required if 'performNcoa' is true.}
#' \item{outputLevel}{string, The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.}
#' \item{limitingSQL}{string, The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').}
#' \item{chunkSize}{integer, The maximum number of records processed at a time. Note that this parameter is not available to all users.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
enhancements_put_cass_ncoa <- function(id, name, source, schedule = NULL, parent_id = NULL, notifications = NULL, destination = NULL, column_mapping = NULL, use_default_column_mapping = NULL, perform_ncoa = NULL, ncoa_credential_id = NULL, output_level = NULL, limiting_sql = NULL, chunk_size = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, source = source, schedule = schedule, parentId = parent_id, notifications = notifications, destination = destination, columnMapping = column_mapping, useDefaultColumnMapping = use_default_column_mapping, performNcoa = perform_ncoa, ncoaCredentialId = ncoa_credential_id, outputLevel = output_level, limitingSQL = limiting_sql, chunkSize = chunk_size)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this CASS/NCOA Enhancement
#' @param id integer required. The ID for the enhancement.
#' @param name string optional. The name of the enhancement job.
#' @param schedule list optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications list optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param source list optional. A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name of the source table.
#' \item table string, The name of the source table.
#' \item remoteHostId integer, The ID of the database host for the table.
#' \item credentialId integer, The id of the credentials to be used when performing the enhancement.
#' \item multipartKey array, The source table primary key.
#' }
#' }
#' @param destination list optional. A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name for the output data.
#' \item table string, The table name for the output data.
#' }
#' }
#' @param column_mapping list optional. A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }
#' @param use_default_column_mapping boolean optional. Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.
#' @param perform_ncoa boolean optional. Whether to update addresses for records matching the National Change of Address (NCOA) database.
#' @param ncoa_credential_id integer optional. Credential to use when performing NCOA updates. Required if 'performNcoa' is true.
#' @param output_level string optional. The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.
#' @param limiting_sql string optional. The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').
#' @param chunk_size integer optional. The maximum number of records processed at a time. Note that this parameter is not available to all users.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name of the source table.
#' \item table string, The name of the source table.
#' \item remoteHostId integer, The ID of the database host for the table.
#' \item credentialId integer, The id of the credentials to be used when performing the enhancement.
#' \item multipartKey array, The source table primary key.
#' }
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name for the output data.
#' \item table string, The table name for the output data.
#' }
#' }}
#' \item{columnMapping}{list, A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }}
#' \item{useDefaultColumnMapping}{boolean, Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.}
#' \item{performNcoa}{boolean, Whether to update addresses for records matching the National Change of Address (NCOA) database.}
#' \item{ncoaCredentialId}{integer, Credential to use when performing NCOA updates. Required if 'performNcoa' is true.}
#' \item{outputLevel}{string, The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.}
#' \item{limitingSQL}{string, The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').}
#' \item{chunkSize}{integer, The maximum number of records processed at a time. Note that this parameter is not available to all users.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
enhancements_patch_cass_ncoa <- function(id, name = NULL, schedule = NULL, parent_id = NULL, notifications = NULL, source = NULL, destination = NULL, column_mapping = NULL, use_default_column_mapping = NULL, perform_ncoa = NULL, ncoa_credential_id = NULL, output_level = NULL, limiting_sql = NULL, chunk_size = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, schedule = schedule, parentId = parent_id, notifications = notifications, source = source, destination = destination, columnMapping = column_mapping, useDefaultColumnMapping = use_default_column_mapping, performNcoa = perform_ncoa, ncoaCredentialId = ncoa_credential_id, outputLevel = output_level, limitingSQL = limiting_sql, chunkSize = chunk_size)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the cass_ncoa.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{cassNcoaId}{integer, The ID of the cass_ncoa.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_post_cass_ncoa_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given cass_ncoa
#' @param id integer required. The ID of the cass_ncoa.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the run.}
#' \item{cassNcoaId}{integer, The ID of the cass_ncoa.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_list_cass_ncoa_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the cass_ncoa.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{cassNcoaId}{integer, The ID of the cass_ncoa.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_get_cass_ncoa_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the cass_ncoa.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_cass_ncoa_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the cass_ncoa.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
enhancements_list_cass_ncoa_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{state}{string, The state of the run, one of 'queued', 'running' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' @export
enhancements_post_cass_ncoa_cancel <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/cancel"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the outputs for a run
#' @param id integer required. The ID of the job.
#' @param run_id integer required. The ID of the run.
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to created_at. Must be one of: created_at, id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, The type of the output. Valid values are File, Table, Report, Project, Credential, or JSONValue}
#' \item{objectId}{integer, The ID of the output.}
#' \item{name}{string, The name of the output.}
#' \item{link}{string, The hypermedia link to the output.}
#' \item{value}{string, }
#' @export
enhancements_list_cass_ncoa_runs_outputs <- function(id, run_id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/runs/{run_id}/outputs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Geocode Enhancement
#' @param name string required. The name of the enhancement job.
#' @param remote_host_id integer required. The ID of the remote host.
#' @param credential_id integer required. The ID of the remote host credential.
#' @param source_schema_and_table string required. The source database schema and table.
#' @param schedule list optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications list optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param multipart_key array optional. The source table primary key.
#' @param limiting_sql string optional. The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').
#' @param target_schema string optional. The output table schema.
#' @param target_table string optional. The output table name.
#' @param country string optional. The country of the addresses to be geocoded; either 'us' or 'ca'.
#' @param provider string optional. The geocoding provider; one of postgis, nominatim, and geocoder_ca.
#' @param output_address boolean optional. Whether to output the parsed address. Only guaranteed for the 'postgis' provider.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{remoteHostId}{integer, The ID of the remote host.}
#' \item{credentialId}{integer, The ID of the remote host credential.}
#' \item{sourceSchemaAndTable}{string, The source database schema and table.}
#' \item{multipartKey}{array, The source table primary key.}
#' \item{limitingSQL}{string, The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').}
#' \item{targetSchema}{string, The output table schema.}
#' \item{targetTable}{string, The output table name.}
#' \item{country}{string, The country of the addresses to be geocoded; either 'us' or 'ca'.}
#' \item{provider}{string, The geocoding provider; one of postgis, nominatim, and geocoder_ca.}
#' \item{outputAddress}{boolean, Whether to output the parsed address. Only guaranteed for the 'postgis' provider.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
enhancements_post_geocode <- function(name, remote_host_id, credential_id, source_schema_and_table, schedule = NULL, parent_id = NULL, notifications = NULL, multipart_key = NULL, limiting_sql = NULL, target_schema = NULL, target_table = NULL, country = NULL, provider = NULL, output_address = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, remoteHostId = remote_host_id, credentialId = credential_id, sourceSchemaAndTable = source_schema_and_table, schedule = schedule, parentId = parent_id, notifications = notifications, multipartKey = multipart_key, limitingSQL = limiting_sql, targetSchema = target_schema, targetTable = target_table, country = country, provider = provider, outputAddress = output_address)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a Geocode Enhancement
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{remoteHostId}{integer, The ID of the remote host.}
#' \item{credentialId}{integer, The ID of the remote host credential.}
#' \item{sourceSchemaAndTable}{string, The source database schema and table.}
#' \item{multipartKey}{array, The source table primary key.}
#' \item{limitingSQL}{string, The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').}
#' \item{targetSchema}{string, The output table schema.}
#' \item{targetTable}{string, The output table name.}
#' \item{country}{string, The country of the addresses to be geocoded; either 'us' or 'ca'.}
#' \item{provider}{string, The geocoding provider; one of postgis, nominatim, and geocoder_ca.}
#' \item{outputAddress}{boolean, Whether to output the parsed address. Only guaranteed for the 'postgis' provider.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
enhancements_get_geocode <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this Geocode Enhancement
#' @param id integer required. The ID for the enhancement.
#' @param name string required. The name of the enhancement job.
#' @param remote_host_id integer required. The ID of the remote host.
#' @param credential_id integer required. The ID of the remote host credential.
#' @param source_schema_and_table string required. The source database schema and table.
#' @param schedule list optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications list optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param multipart_key array optional. The source table primary key.
#' @param limiting_sql string optional. The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').
#' @param target_schema string optional. The output table schema.
#' @param target_table string optional. The output table name.
#' @param country string optional. The country of the addresses to be geocoded; either 'us' or 'ca'.
#' @param provider string optional. The geocoding provider; one of postgis, nominatim, and geocoder_ca.
#' @param output_address boolean optional. Whether to output the parsed address. Only guaranteed for the 'postgis' provider.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{remoteHostId}{integer, The ID of the remote host.}
#' \item{credentialId}{integer, The ID of the remote host credential.}
#' \item{sourceSchemaAndTable}{string, The source database schema and table.}
#' \item{multipartKey}{array, The source table primary key.}
#' \item{limitingSQL}{string, The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').}
#' \item{targetSchema}{string, The output table schema.}
#' \item{targetTable}{string, The output table name.}
#' \item{country}{string, The country of the addresses to be geocoded; either 'us' or 'ca'.}
#' \item{provider}{string, The geocoding provider; one of postgis, nominatim, and geocoder_ca.}
#' \item{outputAddress}{boolean, Whether to output the parsed address. Only guaranteed for the 'postgis' provider.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
enhancements_put_geocode <- function(id, name, remote_host_id, credential_id, source_schema_and_table, schedule = NULL, parent_id = NULL, notifications = NULL, multipart_key = NULL, limiting_sql = NULL, target_schema = NULL, target_table = NULL, country = NULL, provider = NULL, output_address = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, remoteHostId = remote_host_id, credentialId = credential_id, sourceSchemaAndTable = source_schema_and_table, schedule = schedule, parentId = parent_id, notifications = notifications, multipartKey = multipart_key, limitingSQL = limiting_sql, targetSchema = target_schema, targetTable = target_table, country = country, provider = provider, outputAddress = output_address)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this Geocode Enhancement
#' @param id integer required. The ID for the enhancement.
#' @param name string optional. The name of the enhancement job.
#' @param schedule list optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }
#' @param parent_id integer optional. Parent ID that triggers this enhancement.
#' @param notifications list optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param remote_host_id integer optional. The ID of the remote host.
#' @param credential_id integer optional. The ID of the remote host credential.
#' @param source_schema_and_table string optional. The source database schema and table.
#' @param multipart_key array optional. The source table primary key.
#' @param limiting_sql string optional. The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').
#' @param target_schema string optional. The output table schema.
#' @param target_table string optional. The output table name.
#' @param country string optional. The country of the addresses to be geocoded; either 'us' or 'ca'.
#' @param provider string optional. The geocoding provider; one of postgis, nominatim, and geocoder_ca.
#' @param output_address boolean optional. Whether to output the parsed address. Only guaranteed for the 'postgis' provider.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{remoteHostId}{integer, The ID of the remote host.}
#' \item{credentialId}{integer, The ID of the remote host credential.}
#' \item{sourceSchemaAndTable}{string, The source database schema and table.}
#' \item{multipartKey}{array, The source table primary key.}
#' \item{limitingSQL}{string, The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').}
#' \item{targetSchema}{string, The output table schema.}
#' \item{targetTable}{string, The output table name.}
#' \item{country}{string, The country of the addresses to be geocoded; either 'us' or 'ca'.}
#' \item{provider}{string, The geocoding provider; one of postgis, nominatim, and geocoder_ca.}
#' \item{outputAddress}{boolean, Whether to output the parsed address. Only guaranteed for the 'postgis' provider.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
enhancements_patch_geocode <- function(id, name = NULL, schedule = NULL, parent_id = NULL, notifications = NULL, remote_host_id = NULL, credential_id = NULL, source_schema_and_table = NULL, multipart_key = NULL, limiting_sql = NULL, target_schema = NULL, target_table = NULL, country = NULL, provider = NULL, output_address = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, schedule = schedule, parentId = parent_id, notifications = notifications, remoteHostId = remote_host_id, credentialId = credential_id, sourceSchemaAndTable = source_schema_and_table, multipartKey = multipart_key, limitingSQL = limiting_sql, targetSchema = target_schema, targetTable = target_table, country = country, provider = provider, outputAddress = output_address)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the geocode.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{geocodeId}{integer, The ID of the geocode.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_post_geocode_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given geocode
#' @param id integer required. The ID of the geocode.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the run.}
#' \item{geocodeId}{integer, The ID of the geocode.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_list_geocode_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the geocode.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{geocodeId}{integer, The ID of the geocode.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
enhancements_get_geocode_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the geocode.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_geocode_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the geocode.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
enhancements_list_geocode_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{state}{string, The state of the run, one of 'queued', 'running' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' @export
enhancements_post_geocode_cancel <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/cancel"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the outputs for a run
#' @param id integer required. The ID of the job.
#' @param run_id integer required. The ID of the run.
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to created_at. Must be one of: created_at, id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, The type of the output. Valid values are File, Table, Report, Project, Credential, or JSONValue}
#' \item{objectId}{integer, The ID of the output.}
#' \item{name}{string, The name of the output.}
#' \item{link}{string, The hypermedia link to the output.}
#' \item{value}{string, }
#' @export
enhancements_list_geocode_runs_outputs <- function(id, run_id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/runs/{run_id}/outputs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_list_cass_ncoa_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_put_cass_ncoa_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_cass_ncoa_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_put_cass_ncoa_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_cass_ncoa_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
enhancements_list_cass_ncoa_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
enhancements_put_cass_ncoa_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a CASS/NCOA Enhancement belongs to
#' @param id integer required. The ID of the CASS/NCOA Enhancement.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
enhancements_list_cass_ncoa_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a CASS/NCOA Enhancement to a project
#' @param id integer required. The ID of the CASS/NCOA Enhancement.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_put_cass_ncoa_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a CASS/NCOA Enhancement from a project
#' @param id integer required. The ID of the CASS/NCOA Enhancement.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_cass_ncoa_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name of the source table.
#' \item table string, The name of the source table.
#' \item remoteHostId integer, The ID of the database host for the table.
#' \item credentialId integer, The id of the credentials to be used when performing the enhancement.
#' \item multipartKey array, The source table primary key.
#' }
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The schema name for the output data.
#' \item table string, The table name for the output data.
#' }
#' }}
#' \item{columnMapping}{list, A list containing the following elements: 
#' \itemize{
#' \item address1 string, The first address line.
#' \item address2 string, The second address line.
#' \item city string, The city of an address.
#' \item state string, The state of an address.
#' \item zip string, The zip code of an address.
#' \item name string, The full name of the resident at this address. If needed, separate multiple columns with `+`, e.g. `first_name+last_name`
#' \item company string, The name of the company located at this address.
#' }}
#' \item{useDefaultColumnMapping}{boolean, Defaults to true, where the existing column mapping on the input table will be used. If false, a custom column mapping must be provided.}
#' \item{performNcoa}{boolean, Whether to update addresses for records matching the National Change of Address (NCOA) database.}
#' \item{ncoaCredentialId}{integer, Credential to use when performing NCOA updates. Required if 'performNcoa' is true.}
#' \item{outputLevel}{string, The set of fields persisted by a CASS or NCOA enhancement.For CASS enhancements, one of 'cass' or 'all.'For NCOA enhancements, one of 'cass', 'ncoa' , 'coalesced' or 'all'.By default, all fields will be returned.}
#' \item{limitingSQL}{string, The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').}
#' \item{chunkSize}{integer, The maximum number of records processed at a time. Note that this parameter is not available to all users.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
enhancements_put_cass_ncoa_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/cass-ncoa/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_list_geocode_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_put_geocode_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_geocode_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
enhancements_put_geocode_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_geocode_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
enhancements_list_geocode_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
enhancements_put_geocode_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a Geocode Enhancement belongs to
#' @param id integer required. The ID of the Geocode Enhancement.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
enhancements_list_geocode_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a Geocode Enhancement to a project
#' @param id integer required. The ID of the Geocode Enhancement.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_put_geocode_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a Geocode Enhancement from a project
#' @param id integer required. The ID of the Geocode Enhancement.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
enhancements_delete_geocode_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the enhancement.}
#' \item{name}{string, The name of the enhancement job.}
#' \item{type}{string, The type of the enhancement (e.g CASS-NCOA)}
#' \item{createdAt}{string, The time this enhancement was created.}
#' \item{updatedAt}{string, The time the enhancement was last updated.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The status of the enhancement's last run}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, Parent ID that triggers this enhancement.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{remoteHostId}{integer, The ID of the remote host.}
#' \item{credentialId}{integer, The ID of the remote host credential.}
#' \item{sourceSchemaAndTable}{string, The source database schema and table.}
#' \item{multipartKey}{array, The source table primary key.}
#' \item{limitingSQL}{string, The limiting SQL for the source table. "WHERE" should be omitted (e.g. state='IL').}
#' \item{targetSchema}{string, The output table schema.}
#' \item{targetTable}{string, The output table name.}
#' \item{country}{string, The country of the addresses to be geocoded; either 'us' or 'ca'.}
#' \item{provider}{string, The geocoding provider; one of postgis, nominatim, and geocoder_ca.}
#' \item{outputAddress}{boolean, Whether to output the parsed address. Only guaranteed for the 'postgis' provider.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
enhancements_put_geocode_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/enhancements/geocode/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List 
#' @param type string optional. If specified, return exports of these types. It accepts a comma-separated list, possible values are 'database' and 'gdoc'.
#' @param status string optional. If specified, returns export with one of these statuses. It accepts a comma-separated list, possible values are 'running', 'failed', 'succeeded', 'idle', 'scheduled'.
#' @param author string optional. If specified, return items from any of these authors. It accepts a comma-separated list of user IDs.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' @param archived string optional. The archival status of the requested item(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at, last_run.updated_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this export.}
#' \item{name}{string, The name of this export.}
#' \item{type}{string, The type of export.}
#' \item{createdAt}{string, The creation time for this export.}
#' \item{updatedAt}{string, The last modification time for this export.}
#' \item{state}{string, }
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' @export
exports_list <- function(type = NULL, status = NULL, author = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/exports/"
  path_params  <- list()
  query_params <- list(type = type, status = status, author = author, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the csv_export.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{state}{string, }
#' \item{createdAt}{string, The time that the run was queued.}
#' \item{startedAt}{string, The time that the run started.}
#' \item{finishedAt}{string, The time that the run completed.}
#' \item{error}{string, The error message for this run, if present.}
#' \item{outputCachedOn}{string, The time that the output was originally exported, if a cache entry was used by the run.}
#' @export
exports_post_files_csv_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/exports/files/csv/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given csv_export
#' @param id integer required. The ID of the csv_export.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, }
#' \item{state}{string, }
#' \item{createdAt}{string, The time that the run was queued.}
#' \item{startedAt}{string, The time that the run started.}
#' \item{finishedAt}{string, The time that the run completed.}
#' \item{error}{string, The error message for this run, if present.}
#' @export
exports_list_files_csv_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/exports/files/csv/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the csv_export.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{state}{string, }
#' \item{createdAt}{string, The time that the run was queued.}
#' \item{startedAt}{string, The time that the run started.}
#' \item{finishedAt}{string, The time that the run completed.}
#' \item{error}{string, The error message for this run, if present.}
#' \item{outputCachedOn}{string, The time that the output was originally exported, if a cache entry was used by the run.}
#' @export
exports_get_files_csv_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/exports/files/csv/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the csv_export.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
exports_delete_files_csv_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/exports/files/csv/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the csv_export.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
exports_list_files_csv_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/exports/files/csv/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the outputs for a run
#' @param id integer required. The ID of the csv_export.
#' @param run_id integer required. The ID of the run.
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to created_at. Must be one of: created_at, id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, The type of the output. Valid values are File, Table, Report, Project, Credential, or JSONValue}
#' \item{objectId}{integer, The ID of the output.}
#' \item{name}{string, The name of the output.}
#' \item{link}{string, The hypermedia link to the output.}
#' \item{value}{string, }
#' @export
exports_list_files_csv_runs_outputs <- function(id, run_id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/exports/files/csv/{id}/runs/{run_id}/outputs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a CSV Export
#' @param source list required. A list containing the following elements: 
#' \itemize{
#' \item sql string, The SQL query for this Csv Export job
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' }
#' @param destination list required. A list containing the following elements: 
#' \itemize{
#' \item filenamePrefix string, The prefix of the name of the file returned to the user.
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item filePath string, The path within the bucket where the exported file will be saved. E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"
#' \item storageHostId integer, The ID of the destination storage host.
#' \item credentialId integer, The ID of the credentials for the destination storage host.
#' \item existingFiles string, Notifies the job of what to do in the case that the exported file already exists at the provided path.One of: fail, append, overwrite. Default: fail. If "append" is specified,the new file will always be added to the provided path. If "overwrite" is specifiedall existing files at the provided path will be deleted and the new file will be added.By default, or if "fail" is specified, the export will fail if a file exists at the provided path.
#' }
#' }
#' @param name string optional. The name of this Csv Export job.
#' @param include_header boolean optional. A boolean value indicating whether or not the header should be included. Defaults to true.
#' @param compression string optional. The compression of the output file. Valid arguments are "gzip" and "none". Defaults to "gzip".
#' @param column_delimiter string optional. The column delimiter for the output file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".
#' @param hidden boolean optional. A boolean value indicating whether or not this request should be hidden. Defaults to false.
#' @param force_multifile boolean optional. Whether or not the csv should be split into multiple files. Default: false
#' @param max_file_size integer optional. The max file size, in MB, created files will be. Only available when force_multifile is true. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this Csv Export job.}
#' \item{name}{string, The name of this Csv Export job.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item sql string, The SQL query for this Csv Export job
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item filenamePrefix string, The prefix of the name of the file returned to the user.
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item filePath string, The path within the bucket where the exported file will be saved. E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"
#' \item storageHostId integer, The ID of the destination storage host.
#' \item credentialId integer, The ID of the credentials for the destination storage host.
#' \item existingFiles string, Notifies the job of what to do in the case that the exported file already exists at the provided path.One of: fail, append, overwrite. Default: fail. If "append" is specified,the new file will always be added to the provided path. If "overwrite" is specifiedall existing files at the provided path will be deleted and the new file will be added.By default, or if "fail" is specified, the export will fail if a file exists at the provided path.
#' }
#' }}
#' \item{includeHeader}{boolean, A boolean value indicating whether or not the header should be included. Defaults to true.}
#' \item{compression}{string, The compression of the output file. Valid arguments are "gzip" and "none". Defaults to "gzip".}
#' \item{columnDelimiter}{string, The column delimiter for the output file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".}
#' \item{hidden}{boolean, A boolean value indicating whether or not this request should be hidden. Defaults to false.}
#' \item{forceMultifile}{boolean, Whether or not the csv should be split into multiple files. Default: false}
#' \item{maxFileSize}{integer, The max file size, in MB, created files will be. Only available when force_multifile is true. }
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
exports_post_files_csv <- function(source, destination, name = NULL, include_header = NULL, compression = NULL, column_delimiter = NULL, hidden = NULL, force_multifile = NULL, max_file_size = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/exports/files/csv"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(source = source, destination = destination, name = name, includeHeader = include_header, compression = compression, columnDelimiter = column_delimiter, hidden = hidden, forceMultifile = force_multifile, maxFileSize = max_file_size)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a CSV Export
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this Csv Export job.}
#' \item{name}{string, The name of this Csv Export job.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item sql string, The SQL query for this Csv Export job
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item filenamePrefix string, The prefix of the name of the file returned to the user.
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item filePath string, The path within the bucket where the exported file will be saved. E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"
#' \item storageHostId integer, The ID of the destination storage host.
#' \item credentialId integer, The ID of the credentials for the destination storage host.
#' \item existingFiles string, Notifies the job of what to do in the case that the exported file already exists at the provided path.One of: fail, append, overwrite. Default: fail. If "append" is specified,the new file will always be added to the provided path. If "overwrite" is specifiedall existing files at the provided path will be deleted and the new file will be added.By default, or if "fail" is specified, the export will fail if a file exists at the provided path.
#' }
#' }}
#' \item{includeHeader}{boolean, A boolean value indicating whether or not the header should be included. Defaults to true.}
#' \item{compression}{string, The compression of the output file. Valid arguments are "gzip" and "none". Defaults to "gzip".}
#' \item{columnDelimiter}{string, The column delimiter for the output file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".}
#' \item{hidden}{boolean, A boolean value indicating whether or not this request should be hidden. Defaults to false.}
#' \item{forceMultifile}{boolean, Whether or not the csv should be split into multiple files. Default: false}
#' \item{maxFileSize}{integer, The max file size, in MB, created files will be. Only available when force_multifile is true. }
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
exports_get_files_csv <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/exports/files/csv/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this CSV Export
#' @param id integer required. The ID of this Csv Export job.
#' @param source list required. A list containing the following elements: 
#' \itemize{
#' \item sql string, The SQL query for this Csv Export job
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' }
#' @param destination list required. A list containing the following elements: 
#' \itemize{
#' \item filenamePrefix string, The prefix of the name of the file returned to the user.
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item filePath string, The path within the bucket where the exported file will be saved. E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"
#' \item storageHostId integer, The ID of the destination storage host.
#' \item credentialId integer, The ID of the credentials for the destination storage host.
#' \item existingFiles string, Notifies the job of what to do in the case that the exported file already exists at the provided path.One of: fail, append, overwrite. Default: fail. If "append" is specified,the new file will always be added to the provided path. If "overwrite" is specifiedall existing files at the provided path will be deleted and the new file will be added.By default, or if "fail" is specified, the export will fail if a file exists at the provided path.
#' }
#' }
#' @param name string optional. The name of this Csv Export job.
#' @param include_header boolean optional. A boolean value indicating whether or not the header should be included. Defaults to true.
#' @param compression string optional. The compression of the output file. Valid arguments are "gzip" and "none". Defaults to "gzip".
#' @param column_delimiter string optional. The column delimiter for the output file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".
#' @param hidden boolean optional. A boolean value indicating whether or not this request should be hidden. Defaults to false.
#' @param force_multifile boolean optional. Whether or not the csv should be split into multiple files. Default: false
#' @param max_file_size integer optional. The max file size, in MB, created files will be. Only available when force_multifile is true. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this Csv Export job.}
#' \item{name}{string, The name of this Csv Export job.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item sql string, The SQL query for this Csv Export job
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item filenamePrefix string, The prefix of the name of the file returned to the user.
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item filePath string, The path within the bucket where the exported file will be saved. E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"
#' \item storageHostId integer, The ID of the destination storage host.
#' \item credentialId integer, The ID of the credentials for the destination storage host.
#' \item existingFiles string, Notifies the job of what to do in the case that the exported file already exists at the provided path.One of: fail, append, overwrite. Default: fail. If "append" is specified,the new file will always be added to the provided path. If "overwrite" is specifiedall existing files at the provided path will be deleted and the new file will be added.By default, or if "fail" is specified, the export will fail if a file exists at the provided path.
#' }
#' }}
#' \item{includeHeader}{boolean, A boolean value indicating whether or not the header should be included. Defaults to true.}
#' \item{compression}{string, The compression of the output file. Valid arguments are "gzip" and "none". Defaults to "gzip".}
#' \item{columnDelimiter}{string, The column delimiter for the output file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".}
#' \item{hidden}{boolean, A boolean value indicating whether or not this request should be hidden. Defaults to false.}
#' \item{forceMultifile}{boolean, Whether or not the csv should be split into multiple files. Default: false}
#' \item{maxFileSize}{integer, The max file size, in MB, created files will be. Only available when force_multifile is true. }
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
exports_put_files_csv <- function(id, source, destination, name = NULL, include_header = NULL, compression = NULL, column_delimiter = NULL, hidden = NULL, force_multifile = NULL, max_file_size = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/exports/files/csv/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(source = source, destination = destination, name = name, includeHeader = include_header, compression = compression, columnDelimiter = column_delimiter, hidden = hidden, forceMultifile = force_multifile, maxFileSize = max_file_size)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this CSV Export
#' @param id integer required. The ID of this Csv Export job.
#' @param name string optional. The name of this Csv Export job.
#' @param source list optional. A list containing the following elements: 
#' \itemize{
#' \item sql string, The SQL query for this Csv Export job
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' }
#' @param destination list optional. A list containing the following elements: 
#' \itemize{
#' \item filenamePrefix string, The prefix of the name of the file returned to the user.
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item filePath string, The path within the bucket where the exported file will be saved. E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"
#' \item storageHostId integer, The ID of the destination storage host.
#' \item credentialId integer, The ID of the credentials for the destination storage host.
#' \item existingFiles string, Notifies the job of what to do in the case that the exported file already exists at the provided path.One of: fail, append, overwrite. Default: fail. If "append" is specified,the new file will always be added to the provided path. If "overwrite" is specifiedall existing files at the provided path will be deleted and the new file will be added.By default, or if "fail" is specified, the export will fail if a file exists at the provided path.
#' }
#' }
#' @param include_header boolean optional. A boolean value indicating whether or not the header should be included. Defaults to true.
#' @param compression string optional. The compression of the output file. Valid arguments are "gzip" and "none". Defaults to "gzip".
#' @param column_delimiter string optional. The column delimiter for the output file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".
#' @param hidden boolean optional. A boolean value indicating whether or not this request should be hidden. Defaults to false.
#' @param force_multifile boolean optional. Whether or not the csv should be split into multiple files. Default: false
#' @param max_file_size integer optional. The max file size, in MB, created files will be. Only available when force_multifile is true. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this Csv Export job.}
#' \item{name}{string, The name of this Csv Export job.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item sql string, The SQL query for this Csv Export job
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item filenamePrefix string, The prefix of the name of the file returned to the user.
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item filePath string, The path within the bucket where the exported file will be saved. E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"
#' \item storageHostId integer, The ID of the destination storage host.
#' \item credentialId integer, The ID of the credentials for the destination storage host.
#' \item existingFiles string, Notifies the job of what to do in the case that the exported file already exists at the provided path.One of: fail, append, overwrite. Default: fail. If "append" is specified,the new file will always be added to the provided path. If "overwrite" is specifiedall existing files at the provided path will be deleted and the new file will be added.By default, or if "fail" is specified, the export will fail if a file exists at the provided path.
#' }
#' }}
#' \item{includeHeader}{boolean, A boolean value indicating whether or not the header should be included. Defaults to true.}
#' \item{compression}{string, The compression of the output file. Valid arguments are "gzip" and "none". Defaults to "gzip".}
#' \item{columnDelimiter}{string, The column delimiter for the output file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".}
#' \item{hidden}{boolean, A boolean value indicating whether or not this request should be hidden. Defaults to false.}
#' \item{forceMultifile}{boolean, Whether or not the csv should be split into multiple files. Default: false}
#' \item{maxFileSize}{integer, The max file size, in MB, created files will be. Only available when force_multifile is true. }
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
exports_patch_files_csv <- function(id, name = NULL, source = NULL, destination = NULL, include_header = NULL, compression = NULL, column_delimiter = NULL, hidden = NULL, force_multifile = NULL, max_file_size = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/exports/files/csv/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, source = source, destination = destination, includeHeader = include_header, compression = compression, columnDelimiter = column_delimiter, hidden = hidden, forceMultifile = force_multifile, maxFileSize = max_file_size)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this Csv Export job.}
#' \item{name}{string, The name of this Csv Export job.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item sql string, The SQL query for this Csv Export job
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item filenamePrefix string, The prefix of the name of the file returned to the user.
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item filePath string, The path within the bucket where the exported file will be saved. E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"
#' \item storageHostId integer, The ID of the destination storage host.
#' \item credentialId integer, The ID of the credentials for the destination storage host.
#' \item existingFiles string, Notifies the job of what to do in the case that the exported file already exists at the provided path.One of: fail, append, overwrite. Default: fail. If "append" is specified,the new file will always be added to the provided path. If "overwrite" is specifiedall existing files at the provided path will be deleted and the new file will be added.By default, or if "fail" is specified, the export will fail if a file exists at the provided path.
#' }
#' }}
#' \item{includeHeader}{boolean, A boolean value indicating whether or not the header should be included. Defaults to true.}
#' \item{compression}{string, The compression of the output file. Valid arguments are "gzip" and "none". Defaults to "gzip".}
#' \item{columnDelimiter}{string, The column delimiter for the output file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".}
#' \item{hidden}{boolean, A boolean value indicating whether or not this request should be hidden. Defaults to false.}
#' \item{forceMultifile}{boolean, Whether or not the csv should be split into multiple files. Default: false}
#' \item{maxFileSize}{integer, The max file size, in MB, created files will be. Only available when force_multifile is true. }
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
exports_put_files_csv_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/exports/files/csv/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List feature flags
#' 
#' @return  An array containing the following fields:
#' \item{name}{string, The name of the feature.}
#' \item{description}{string, }
#' \item{activeForMe}{boolean, Whether the feature is active for the current user.}
#' \item{userCount}{integer, The number of users with this feature flag enabled.}
#' \item{team}{string, }
#' \item{jira}{string, }
#' \item{added}{string, }
#' \item{groupCount}{integer, }
#' \item{organizationCount}{integer, }
#' \item{percentage}{integer, The target percentage of users who should have this feature flag enabled.}
#' @export
feature_flags_list <- function() {

  args <- as.list(match.call())[-1]
  path <- "/feature_flags/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show a feature flag
#' @param name string required. The name of the feature.
#' 
#' @return  A list containing the following elements:
#' \item{name}{string, The name of the feature.}
#' \item{description}{string, }
#' \item{organizations}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, Organization ID
#' \item name string, Organization name
#' \item slug string, Organization slug
#' }}
#' \item{groups}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, Group ID
#' \item name string, Group name
#' \item slug string, Group slug
#' \item users array, Users within the group
#' }}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, User ID
#' \item username string, Username
#' }}
#' \item{percentage}{integer, The target percentage of users who should have this feature flag enabled.}
#' @export
feature_flags_get <- function(name) {

  args <- as.list(match.call())[-1]
  path <- "/feature_flags/{name}"
  path_params  <- list(name = name)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Activate a feature for a user
#' @param flag_name string required. The feature name.
#' @param user_id integer required. The user ID.
#' 
#' @return  A list containing the following elements:
#' \item{name}{string, The name of the feature.}
#' \item{description}{string, }
#' \item{activeForMe}{boolean, Whether the feature is active for the current user.}
#' \item{userCount}{integer, The number of users with this feature flag enabled.}
#' \item{team}{string, }
#' \item{jira}{string, }
#' \item{added}{string, }
#' \item{groupCount}{integer, }
#' \item{organizationCount}{integer, }
#' \item{percentage}{integer, The target percentage of users who should have this feature flag enabled.}
#' @export
feature_flags_put_users <- function(flag_name, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/feature_flags/{flag_name}/users/{user_id}"
  path_params  <- list(flag_name = flag_name, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Deactivate a feature for a user
#' @param flag_name string required. The feature name.
#' @param user_id integer required. The user ID.
#' 
#' @return  An empty HTTP response
#' @export
feature_flags_delete_users <- function(flag_name, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/feature_flags/{flag_name}/users/{user_id}"
  path_params  <- list(flag_name = flag_name, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Activate a feature for a group
#' @param flag_name string required. The feature flag name.
#' @param group_id integer required. Group ID.
#' 
#' @return  A list containing the following elements:
#' \item{name}{string, The name of the feature.}
#' \item{description}{string, }
#' \item{activeForMe}{boolean, Whether the feature is active for the current user.}
#' \item{userCount}{integer, The number of users with this feature flag enabled.}
#' \item{team}{string, }
#' \item{jira}{string, }
#' \item{added}{string, }
#' \item{groupCount}{integer, }
#' \item{organizationCount}{integer, }
#' \item{percentage}{integer, The target percentage of users who should have this feature flag enabled.}
#' @export
feature_flags_put_groups <- function(flag_name, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/feature_flags/{flag_name}/groups/{group_id}"
  path_params  <- list(flag_name = flag_name, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Deactivate a feature for a group
#' @param flag_name string required. The feature flag name.
#' @param group_id integer required. Group ID.
#' 
#' @return  An empty HTTP response
#' @export
feature_flags_delete_groups <- function(flag_name, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/feature_flags/{flag_name}/groups/{group_id}"
  path_params  <- list(flag_name = flag_name, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Activate a feature for a organization
#' @param flag_name string required. The feature flag name.
#' @param organization_id integer required. Organization ID.
#' 
#' @return  A list containing the following elements:
#' \item{name}{string, The name of the feature.}
#' \item{description}{string, }
#' \item{activeForMe}{boolean, Whether the feature is active for the current user.}
#' \item{userCount}{integer, The number of users with this feature flag enabled.}
#' \item{team}{string, }
#' \item{jira}{string, }
#' \item{added}{string, }
#' \item{groupCount}{integer, }
#' \item{organizationCount}{integer, }
#' \item{percentage}{integer, The target percentage of users who should have this feature flag enabled.}
#' @export
feature_flags_put_organizations <- function(flag_name, organization_id) {

  args <- as.list(match.call())[-1]
  path <- "/feature_flags/{flag_name}/organizations/{organization_id}"
  path_params  <- list(flag_name = flag_name, organization_id = organization_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Deactivate a feature for a organization
#' @param flag_name string required. The feature flag name.
#' @param organization_id integer required. Organization ID.
#' 
#' @return  An empty HTTP response
#' @export
feature_flags_delete_organizations <- function(flag_name, organization_id) {

  args <- as.list(match.call())[-1]
  path <- "/feature_flags/{flag_name}/organizations/{organization_id}"
  path_params  <- list(flag_name = flag_name, organization_id = organization_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a File belongs to
#' @param id integer required. The ID of the File.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
files_list_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a File to a project
#' @param id integer required. The ID of the File.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
files_put_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a File from a project
#' @param id integer required. The ID of the File.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
files_delete_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
files_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
files_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
files_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
files_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
files_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
files_list_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
files_put_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Initiate an upload of a file into the platform
#' @param name string required. The file name.
#' @param expires_at string optional. The date and time the file will expire. If not specified, the file will expire in 30 days. To keep a file indefinitely, specify null.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the file.}
#' \item{name}{string, The file name.}
#' \item{createdAt}{string, The date and time the file was created.}
#' \item{fileSize}{integer, The file size.}
#' \item{expiresAt}{string, The date and time the file will expire. If not specified, the file will expire in 30 days. To keep a file indefinitely, specify null.}
#' \item{uploadUrl}{string, The URL that may be used to upload a file. To use the upload URL, initiate a POST request to the given URL with the file you wish to import as the "file" form field.}
#' \item{uploadFields}{list, A hash containing the form fields to be included with the POST request.}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
files_post <- function(name, expires_at = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, expiresAt = expires_at)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Initiate a multipart upload
#' @param name string required. The file name.
#' @param num_parts integer required. The number of parts in which the file will be uploaded. This parameter determines the number of presigned URLs that are returned.
#' @param expires_at string optional. The date and time the file will expire. If not specified, the file will expire in 30 days. To keep a file indefinitely, specify null.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the file.}
#' \item{name}{string, The file name.}
#' \item{createdAt}{string, The date and time the file was created.}
#' \item{fileSize}{integer, The file size.}
#' \item{expiresAt}{string, The date and time the file will expire. If not specified, the file will expire in 30 days. To keep a file indefinitely, specify null.}
#' \item{uploadUrls}{array, An array of URLs that may be used to upload file parts. Use separate PUT requests to complete the part uploads. Links expire after 12 hours.}
#' @export
files_post_multipart <- function(name, num_parts, expires_at = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/multipart"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, numParts = num_parts, expiresAt = expires_at)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Complete a multipart upload
#' @param id integer required. The ID of the file.
#' 
#' @return  An empty HTTP response
#' @export
files_post_multipart_complete <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/files/multipart/{id}/complete"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about a file
#' @param id integer required. The ID of the file.
#' @param link_expires_at string optional. The date and time the download link will expire. Must be a time between now and 36 hours from now. Defaults to 30 minutes from now.
#' @param inline boolean optional. If true, will return a url that can be displayed inline in HTML
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the file.}
#' \item{name}{string, The file name.}
#' \item{createdAt}{string, The date and time the file was created.}
#' \item{fileSize}{integer, The file size.}
#' \item{expiresAt}{string, The date and time the file will expire. If not specified, the file will expire in 30 days. To keep a file indefinitely, specify null.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{downloadUrl}{string, A JSON string containing information about the URL of the file.}
#' \item{fileUrl}{string, The URL that may be used to download the file.}
#' \item{detectedInfo}{list, A list containing the following elements: 
#' \itemize{
#' \item includeHeader boolean, A boolean value indicating whether or not the first row of the file is a header row.
#' \item columnDelimiter string, The column delimiter for the file. One of "comma", "tab", or "pipe".
#' \item compression string, The type of compression of the file. One of "gzip", or "none".
#' \item tableColumns array, An array of hashes corresponding to the columns in the file. Each hash should have keys for column "name" and "sql_type"
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
files_get <- function(id, link_expires_at = NULL, inline = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}"
  path_params  <- list(id = id)
  query_params <- list(link_expires_at = link_expires_at, inline = inline)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update details about a file
#' @param id integer required. The ID of the file.
#' @param name string required. The file name. The extension must match the previous extension.
#' @param expires_at string required. The date and time the file will expire. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the file.}
#' \item{name}{string, The file name.}
#' \item{createdAt}{string, The date and time the file was created.}
#' \item{fileSize}{integer, The file size.}
#' \item{expiresAt}{string, The date and time the file will expire. If not specified, the file will expire in 30 days. To keep a file indefinitely, specify null.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{downloadUrl}{string, A JSON string containing information about the URL of the file.}
#' \item{fileUrl}{string, The URL that may be used to download the file.}
#' \item{detectedInfo}{list, A list containing the following elements: 
#' \itemize{
#' \item includeHeader boolean, A boolean value indicating whether or not the first row of the file is a header row.
#' \item columnDelimiter string, The column delimiter for the file. One of "comma", "tab", or "pipe".
#' \item compression string, The type of compression of the file. One of "gzip", or "none".
#' \item tableColumns array, An array of hashes corresponding to the columns in the file. Each hash should have keys for column "name" and "sql_type"
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
files_put <- function(id, name, expires_at) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, expiresAt = expires_at)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update details about a file
#' @param id integer required. The ID of the file.
#' @param name string optional. The file name. The extension must match the previous extension.
#' @param expires_at string optional. The date and time the file will expire. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the file.}
#' \item{name}{string, The file name.}
#' \item{createdAt}{string, The date and time the file was created.}
#' \item{fileSize}{integer, The file size.}
#' \item{expiresAt}{string, The date and time the file will expire. If not specified, the file will expire in 30 days. To keep a file indefinitely, specify null.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{downloadUrl}{string, A JSON string containing information about the URL of the file.}
#' \item{fileUrl}{string, The URL that may be used to download the file.}
#' \item{detectedInfo}{list, A list containing the following elements: 
#' \itemize{
#' \item includeHeader boolean, A boolean value indicating whether or not the first row of the file is a header row.
#' \item columnDelimiter string, The column delimiter for the file. One of "comma", "tab", or "pipe".
#' \item compression string, The type of compression of the file. One of "gzip", or "none".
#' \item tableColumns array, An array of hashes corresponding to the columns in the file. Each hash should have keys for column "name" and "sql_type"
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
files_patch <- function(id, name = NULL, expires_at = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, expiresAt = expires_at)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Preprocess CSV
#' @param file_id integer required. The ID of the file.
#' @param in_place boolean optional. If true, the file is cleaned in place. If false, a new file ID is created. Defaults to true.
#' @param detect_table_columns boolean optional. If true, detect the table columns in the file including the sql types. If false, skip table column detection.Defaults to false.
#' @param force_character_set_conversion boolean optional. If true, the file will always be converted to UTF-8 and any character that cannot be converted will be discarded. If false, the character set conversion will only run if the detected character set is not compatible with UTF-8 (e.g., UTF-8, ASCII).
#' @param include_header boolean optional. A boolean value indicating whether or not the first row of the file is a header row. If not provided, will attempt to auto-detect whether a header row is present.
#' @param column_delimiter string optional. The column delimiter for the file. One of "comma", "tab", or "pipe". If not provided, the column delimiter will be auto-detected.
#' @param hidden boolean optional. The hidden status of the item.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the job created.}
#' \item{fileId}{integer, The ID of the file.}
#' \item{inPlace}{boolean, If true, the file is cleaned in place. If false, a new file ID is created. Defaults to true.}
#' \item{detectTableColumns}{boolean, If true, detect the table columns in the file including the sql types. If false, skip table column detection.Defaults to false.}
#' \item{forceCharacterSetConversion}{boolean, If true, the file will always be converted to UTF-8 and any character that cannot be converted will be discarded. If false, the character set conversion will only run if the detected character set is not compatible with UTF-8 (e.g., UTF-8, ASCII).}
#' \item{includeHeader}{boolean, A boolean value indicating whether or not the first row of the file is a header row. If not provided, will attempt to auto-detect whether a header row is present.}
#' \item{columnDelimiter}{string, The column delimiter for the file. One of "comma", "tab", or "pipe". If not provided, the column delimiter will be auto-detected.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' @export
files_post_preprocess_csv <- function(file_id, in_place = NULL, detect_table_columns = NULL, force_character_set_conversion = NULL, include_header = NULL, column_delimiter = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/preprocess/csv"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(fileId = file_id, inPlace = in_place, detectTableColumns = detect_table_columns, forceCharacterSetConversion = force_character_set_conversion, includeHeader = include_header, columnDelimiter = column_delimiter, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a Preprocess CSV
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the job created.}
#' \item{fileId}{integer, The ID of the file.}
#' \item{inPlace}{boolean, If true, the file is cleaned in place. If false, a new file ID is created. Defaults to true.}
#' \item{detectTableColumns}{boolean, If true, detect the table columns in the file including the sql types. If false, skip table column detection.Defaults to false.}
#' \item{forceCharacterSetConversion}{boolean, If true, the file will always be converted to UTF-8 and any character that cannot be converted will be discarded. If false, the character set conversion will only run if the detected character set is not compatible with UTF-8 (e.g., UTF-8, ASCII).}
#' \item{includeHeader}{boolean, A boolean value indicating whether or not the first row of the file is a header row. If not provided, will attempt to auto-detect whether a header row is present.}
#' \item{columnDelimiter}{string, The column delimiter for the file. One of "comma", "tab", or "pipe". If not provided, the column delimiter will be auto-detected.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' @export
files_get_preprocess_csv <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/files/preprocess/csv/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this Preprocess CSV
#' @param id integer required. The ID of the job created.
#' @param file_id integer required. The ID of the file.
#' @param in_place boolean optional. If true, the file is cleaned in place. If false, a new file ID is created. Defaults to true.
#' @param detect_table_columns boolean optional. If true, detect the table columns in the file including the sql types. If false, skip table column detection.Defaults to false.
#' @param force_character_set_conversion boolean optional. If true, the file will always be converted to UTF-8 and any character that cannot be converted will be discarded. If false, the character set conversion will only run if the detected character set is not compatible with UTF-8 (e.g., UTF-8, ASCII).
#' @param include_header boolean optional. A boolean value indicating whether or not the first row of the file is a header row. If not provided, will attempt to auto-detect whether a header row is present.
#' @param column_delimiter string optional. The column delimiter for the file. One of "comma", "tab", or "pipe". If not provided, the column delimiter will be auto-detected.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the job created.}
#' \item{fileId}{integer, The ID of the file.}
#' \item{inPlace}{boolean, If true, the file is cleaned in place. If false, a new file ID is created. Defaults to true.}
#' \item{detectTableColumns}{boolean, If true, detect the table columns in the file including the sql types. If false, skip table column detection.Defaults to false.}
#' \item{forceCharacterSetConversion}{boolean, If true, the file will always be converted to UTF-8 and any character that cannot be converted will be discarded. If false, the character set conversion will only run if the detected character set is not compatible with UTF-8 (e.g., UTF-8, ASCII).}
#' \item{includeHeader}{boolean, A boolean value indicating whether or not the first row of the file is a header row. If not provided, will attempt to auto-detect whether a header row is present.}
#' \item{columnDelimiter}{string, The column delimiter for the file. One of "comma", "tab", or "pipe". If not provided, the column delimiter will be auto-detected.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' @export
files_put_preprocess_csv <- function(id, file_id, in_place = NULL, detect_table_columns = NULL, force_character_set_conversion = NULL, include_header = NULL, column_delimiter = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/preprocess/csv/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(fileId = file_id, inPlace = in_place, detectTableColumns = detect_table_columns, forceCharacterSetConversion = force_character_set_conversion, includeHeader = include_header, columnDelimiter = column_delimiter)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this Preprocess CSV
#' @param id integer required. The ID of the job created.
#' @param file_id integer optional. The ID of the file.
#' @param in_place boolean optional. If true, the file is cleaned in place. If false, a new file ID is created. Defaults to true.
#' @param detect_table_columns boolean optional. If true, detect the table columns in the file including the sql types. If false, skip table column detection.Defaults to false.
#' @param force_character_set_conversion boolean optional. If true, the file will always be converted to UTF-8 and any character that cannot be converted will be discarded. If false, the character set conversion will only run if the detected character set is not compatible with UTF-8 (e.g., UTF-8, ASCII).
#' @param include_header boolean optional. A boolean value indicating whether or not the first row of the file is a header row. If not provided, will attempt to auto-detect whether a header row is present.
#' @param column_delimiter string optional. The column delimiter for the file. One of "comma", "tab", or "pipe". If not provided, the column delimiter will be auto-detected.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the job created.}
#' \item{fileId}{integer, The ID of the file.}
#' \item{inPlace}{boolean, If true, the file is cleaned in place. If false, a new file ID is created. Defaults to true.}
#' \item{detectTableColumns}{boolean, If true, detect the table columns in the file including the sql types. If false, skip table column detection.Defaults to false.}
#' \item{forceCharacterSetConversion}{boolean, If true, the file will always be converted to UTF-8 and any character that cannot be converted will be discarded. If false, the character set conversion will only run if the detected character set is not compatible with UTF-8 (e.g., UTF-8, ASCII).}
#' \item{includeHeader}{boolean, A boolean value indicating whether or not the first row of the file is a header row. If not provided, will attempt to auto-detect whether a header row is present.}
#' \item{columnDelimiter}{string, The column delimiter for the file. One of "comma", "tab", or "pipe". If not provided, the column delimiter will be auto-detected.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' @export
files_patch_preprocess_csv <- function(id, file_id = NULL, in_place = NULL, detect_table_columns = NULL, force_character_set_conversion = NULL, include_header = NULL, column_delimiter = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/files/preprocess/csv/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(fileId = file_id, inPlace = in_place, detectTableColumns = detect_table_columns, forceCharacterSetConversion = force_character_set_conversion, includeHeader = include_header, columnDelimiter = column_delimiter)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive a Preprocess CSV (deprecated, use archiving endpoints instead)
#' @param id integer required. 
#' 
#' @return  An empty HTTP response
#' @export
files_delete_preprocess_csv <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/files/preprocess/csv/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the job created.}
#' \item{fileId}{integer, The ID of the file.}
#' \item{inPlace}{boolean, If true, the file is cleaned in place. If false, a new file ID is created. Defaults to true.}
#' \item{detectTableColumns}{boolean, If true, detect the table columns in the file including the sql types. If false, skip table column detection.Defaults to false.}
#' \item{forceCharacterSetConversion}{boolean, If true, the file will always be converted to UTF-8 and any character that cannot be converted will be discarded. If false, the character set conversion will only run if the detected character set is not compatible with UTF-8 (e.g., UTF-8, ASCII).}
#' \item{includeHeader}{boolean, A boolean value indicating whether or not the first row of the file is a header row. If not provided, will attempt to auto-detect whether a header row is present.}
#' \item{columnDelimiter}{string, The column delimiter for the file. One of "comma", "tab", or "pipe". If not provided, the column delimiter will be auto-detected.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' @export
files_put_preprocess_csv_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/files/preprocess/csv/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List bookmarked git repositories
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to repo_url. Must be one of: repo_url, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this git repository.}
#' \item{repoUrl}{string, The URL for this git repository.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
git_repos_list <- function(limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/git_repos/"
  path_params  <- list()
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Bookmark a git repository
#' @param repo_url string required. The URL for this git repository.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this git repository.}
#' \item{repoUrl}{string, The URL for this git repository.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
git_repos_post <- function(repo_url) {

  args <- as.list(match.call())[-1]
  path <- "/git_repos/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(repoUrl = repo_url)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a bookmarked git repository
#' @param id integer required. The ID for this git repository.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this git repository.}
#' \item{repoUrl}{string, The URL for this git repository.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
git_repos_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/git_repos/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove the bookmark on a git repository
#' @param id integer required. The ID for this git repository.
#' 
#' @return  An empty HTTP response
#' @export
git_repos_delete <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/git_repos/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get all branches and tags of a bookmarked git repository
#' @param id integer required. The ID for this git repository.
#' 
#' @return  A list containing the following elements:
#' \item{branches}{array, List of branch names of this git repository.}
#' \item{tags}{array, List of tag names of this git repository.}
#' @export
git_repos_list_refs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/git_repos/{id}/refs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Groups
#' @param query string optional. If specified, it will filter the groups returned.
#' @param permission string optional. A permissions string, one of "read", "write", or "manage". Lists only groups for which the current user has that permission.
#' @param include_members boolean optional. Show members of the group.
#' @param organization_id integer optional. The organization by which to filter groups.
#' @param user_ids array optional. A list of user IDs to filter groups by.Groups will be returned if any of the users is a member
#' @param limit integer optional. Number of results to return. Defaults to 50. Maximum allowed is 1000.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to name. Must be one of: name, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this group.}
#' \item{name}{string, This group's name.}
#' \item{createdAt}{string, The date and time when this group was created.}
#' \item{updatedAt}{string, The date and time when this group was last updated.}
#' \item{description}{string, The description of the group.}
#' \item{slug}{string, The slug for this group.}
#' \item{organizationId}{integer, The ID of the organization this group belongs to.}
#' \item{organizationName}{string, The name of the organization this group belongs to.}
#' \item{memberCount}{integer, The number of active members in this group.}
#' \item{totalMemberCount}{integer, The total number of members in this group.}
#' \item{lastUpdatedById}{integer, The ID of the user who last updated this group.}
#' \item{createdById}{integer, The ID of the user who created this group.}
#' \item{members}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' @export
groups_list <- function(query = NULL, permission = NULL, include_members = NULL, organization_id = NULL, user_ids = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/groups/"
  path_params  <- list()
  query_params <- list(query = query, permission = permission, include_members = include_members, organization_id = organization_id, user_ids = user_ids, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Group
#' @param name string required. This group's name.
#' @param description string optional. The description of the group.
#' @param slug string optional. The slug for this group.
#' @param organization_id integer optional. The ID of the organization this group belongs to.
#' @param default_otp_required_for_login boolean optional. The two factor authentication requirement for this group.
#' @param role_ids array optional. An array of ids of all the roles this group has.
#' @param default_time_zone string optional. The default time zone of this group.
#' @param default_jobs_label string optional. The default partition label for jobs of this group.
#' @param default_notebooks_label string optional. The default partition label for notebooks of this group.
#' @param default_services_label string optional. The default partition label for services of this group.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this group.}
#' \item{name}{string, This group's name.}
#' \item{createdAt}{string, The date and time when this group was created.}
#' \item{updatedAt}{string, The date and time when this group was last updated.}
#' \item{description}{string, The description of the group.}
#' \item{slug}{string, The slug for this group.}
#' \item{organizationId}{integer, The ID of the organization this group belongs to.}
#' \item{organizationName}{string, The name of the organization this group belongs to.}
#' \item{memberCount}{integer, The number of active members in this group.}
#' \item{totalMemberCount}{integer, The total number of members in this group.}
#' \item{defaultOtpRequiredForLogin}{boolean, The two factor authentication requirement for this group.}
#' \item{roleIds}{array, An array of ids of all the roles this group has.}
#' \item{defaultTimeZone}{string, The default time zone of this group.}
#' \item{defaultJobsLabel}{string, The default partition label for jobs of this group.}
#' \item{defaultNotebooksLabel}{string, The default partition label for notebooks of this group.}
#' \item{defaultServicesLabel}{string, The default partition label for services of this group.}
#' \item{lastUpdatedById}{integer, The ID of the user who last updated this group.}
#' \item{createdById}{integer, The ID of the user who created this group.}
#' \item{members}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' \item email string, This user's email address.
#' \item primaryGroupId integer, The ID of the primary group of this user.
#' \item active boolean, Whether this user account is active or deactivated.
#' }}
#' @export
groups_post <- function(name, description = NULL, slug = NULL, organization_id = NULL, default_otp_required_for_login = NULL, role_ids = NULL, default_time_zone = NULL, default_jobs_label = NULL, default_notebooks_label = NULL, default_services_label = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/groups/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, description = description, slug = slug, organizationId = organization_id, defaultOtpRequiredForLogin = default_otp_required_for_login, roleIds = role_ids, defaultTimeZone = default_time_zone, defaultJobsLabel = default_jobs_label, defaultNotebooksLabel = default_notebooks_label, defaultServicesLabel = default_services_label)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a Group
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this group.}
#' \item{name}{string, This group's name.}
#' \item{createdAt}{string, The date and time when this group was created.}
#' \item{updatedAt}{string, The date and time when this group was last updated.}
#' \item{description}{string, The description of the group.}
#' \item{slug}{string, The slug for this group.}
#' \item{organizationId}{integer, The ID of the organization this group belongs to.}
#' \item{organizationName}{string, The name of the organization this group belongs to.}
#' \item{memberCount}{integer, The number of active members in this group.}
#' \item{totalMemberCount}{integer, The total number of members in this group.}
#' \item{defaultOtpRequiredForLogin}{boolean, The two factor authentication requirement for this group.}
#' \item{roleIds}{array, An array of ids of all the roles this group has.}
#' \item{defaultTimeZone}{string, The default time zone of this group.}
#' \item{defaultJobsLabel}{string, The default partition label for jobs of this group.}
#' \item{defaultNotebooksLabel}{string, The default partition label for notebooks of this group.}
#' \item{defaultServicesLabel}{string, The default partition label for services of this group.}
#' \item{lastUpdatedById}{integer, The ID of the user who last updated this group.}
#' \item{createdById}{integer, The ID of the user who created this group.}
#' \item{members}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' \item email string, This user's email address.
#' \item primaryGroupId integer, The ID of the primary group of this user.
#' \item active boolean, Whether this user account is active or deactivated.
#' }}
#' @export
groups_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/groups/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this Group
#' @param id integer required. The ID of this group.
#' @param name string required. This group's name.
#' @param description string optional. The description of the group.
#' @param slug string optional. The slug for this group.
#' @param organization_id integer optional. The ID of the organization this group belongs to.
#' @param default_otp_required_for_login boolean optional. The two factor authentication requirement for this group.
#' @param role_ids array optional. An array of ids of all the roles this group has.
#' @param default_time_zone string optional. The default time zone of this group.
#' @param default_jobs_label string optional. The default partition label for jobs of this group.
#' @param default_notebooks_label string optional. The default partition label for notebooks of this group.
#' @param default_services_label string optional. The default partition label for services of this group.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this group.}
#' \item{name}{string, This group's name.}
#' \item{createdAt}{string, The date and time when this group was created.}
#' \item{updatedAt}{string, The date and time when this group was last updated.}
#' \item{description}{string, The description of the group.}
#' \item{slug}{string, The slug for this group.}
#' \item{organizationId}{integer, The ID of the organization this group belongs to.}
#' \item{organizationName}{string, The name of the organization this group belongs to.}
#' \item{memberCount}{integer, The number of active members in this group.}
#' \item{totalMemberCount}{integer, The total number of members in this group.}
#' \item{defaultOtpRequiredForLogin}{boolean, The two factor authentication requirement for this group.}
#' \item{roleIds}{array, An array of ids of all the roles this group has.}
#' \item{defaultTimeZone}{string, The default time zone of this group.}
#' \item{defaultJobsLabel}{string, The default partition label for jobs of this group.}
#' \item{defaultNotebooksLabel}{string, The default partition label for notebooks of this group.}
#' \item{defaultServicesLabel}{string, The default partition label for services of this group.}
#' \item{lastUpdatedById}{integer, The ID of the user who last updated this group.}
#' \item{createdById}{integer, The ID of the user who created this group.}
#' \item{members}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' \item email string, This user's email address.
#' \item primaryGroupId integer, The ID of the primary group of this user.
#' \item active boolean, Whether this user account is active or deactivated.
#' }}
#' @export
groups_put <- function(id, name, description = NULL, slug = NULL, organization_id = NULL, default_otp_required_for_login = NULL, role_ids = NULL, default_time_zone = NULL, default_jobs_label = NULL, default_notebooks_label = NULL, default_services_label = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/groups/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, description = description, slug = slug, organizationId = organization_id, defaultOtpRequiredForLogin = default_otp_required_for_login, roleIds = role_ids, defaultTimeZone = default_time_zone, defaultJobsLabel = default_jobs_label, defaultNotebooksLabel = default_notebooks_label, defaultServicesLabel = default_services_label)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this Group
#' @param id integer required. The ID of this group.
#' @param name string optional. This group's name.
#' @param description string optional. The description of the group.
#' @param slug string optional. The slug for this group.
#' @param organization_id integer optional. The ID of the organization this group belongs to.
#' @param default_otp_required_for_login boolean optional. The two factor authentication requirement for this group.
#' @param role_ids array optional. An array of ids of all the roles this group has.
#' @param default_time_zone string optional. The default time zone of this group.
#' @param default_jobs_label string optional. The default partition label for jobs of this group.
#' @param default_notebooks_label string optional. The default partition label for notebooks of this group.
#' @param default_services_label string optional. The default partition label for services of this group.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this group.}
#' \item{name}{string, This group's name.}
#' \item{createdAt}{string, The date and time when this group was created.}
#' \item{updatedAt}{string, The date and time when this group was last updated.}
#' \item{description}{string, The description of the group.}
#' \item{slug}{string, The slug for this group.}
#' \item{organizationId}{integer, The ID of the organization this group belongs to.}
#' \item{organizationName}{string, The name of the organization this group belongs to.}
#' \item{memberCount}{integer, The number of active members in this group.}
#' \item{totalMemberCount}{integer, The total number of members in this group.}
#' \item{defaultOtpRequiredForLogin}{boolean, The two factor authentication requirement for this group.}
#' \item{roleIds}{array, An array of ids of all the roles this group has.}
#' \item{defaultTimeZone}{string, The default time zone of this group.}
#' \item{defaultJobsLabel}{string, The default partition label for jobs of this group.}
#' \item{defaultNotebooksLabel}{string, The default partition label for notebooks of this group.}
#' \item{defaultServicesLabel}{string, The default partition label for services of this group.}
#' \item{lastUpdatedById}{integer, The ID of the user who last updated this group.}
#' \item{createdById}{integer, The ID of the user who created this group.}
#' \item{members}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' \item email string, This user's email address.
#' \item primaryGroupId integer, The ID of the primary group of this user.
#' \item active boolean, Whether this user account is active or deactivated.
#' }}
#' @export
groups_patch <- function(id, name = NULL, description = NULL, slug = NULL, organization_id = NULL, default_otp_required_for_login = NULL, role_ids = NULL, default_time_zone = NULL, default_jobs_label = NULL, default_notebooks_label = NULL, default_services_label = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/groups/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, description = description, slug = slug, organizationId = organization_id, defaultOtpRequiredForLogin = default_otp_required_for_login, roleIds = role_ids, defaultTimeZone = default_time_zone, defaultJobsLabel = default_jobs_label, defaultNotebooksLabel = default_notebooks_label, defaultServicesLabel = default_services_label)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Delete a Group (deprecated)
#' @param id integer required. 
#' 
#' @return  An empty HTTP response
#' @export
groups_delete <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/groups/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
groups_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/groups/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
groups_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/groups/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
groups_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/groups/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
groups_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/groups/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
groups_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/groups/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a user to a group
#' @param id integer required. The ID of the group.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this group.}
#' \item{name}{string, This group's name.}
#' \item{createdAt}{string, The date and time when this group was created.}
#' \item{updatedAt}{string, The date and time when this group was last updated.}
#' \item{description}{string, The description of the group.}
#' \item{slug}{string, The slug for this group.}
#' \item{organizationId}{integer, The ID of the organization this group belongs to.}
#' \item{organizationName}{string, The name of the organization this group belongs to.}
#' \item{memberCount}{integer, The number of active members in this group.}
#' \item{totalMemberCount}{integer, The total number of members in this group.}
#' \item{defaultOtpRequiredForLogin}{boolean, The two factor authentication requirement for this group.}
#' \item{roleIds}{array, An array of ids of all the roles this group has.}
#' \item{defaultTimeZone}{string, The default time zone of this group.}
#' \item{defaultJobsLabel}{string, The default partition label for jobs of this group.}
#' \item{defaultNotebooksLabel}{string, The default partition label for notebooks of this group.}
#' \item{defaultServicesLabel}{string, The default partition label for services of this group.}
#' \item{lastUpdatedById}{integer, The ID of the user who last updated this group.}
#' \item{createdById}{integer, The ID of the user who created this group.}
#' \item{members}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' \item email string, This user's email address.
#' \item primaryGroupId integer, The ID of the primary group of this user.
#' \item active boolean, Whether this user account is active or deactivated.
#' }}
#' @export
groups_put_members <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/groups/{id}/members/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a user from a group
#' @param id integer required. The ID of the group.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
groups_delete_members <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/groups/{id}/members/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get child groups of this group
#' @param id integer required. The ID of this group.
#' 
#' @return  A list containing the following elements:
#' \item{manageable}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item name string, 
#' }}
#' \item{writeable}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item name string, 
#' }}
#' \item{readable}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item name string, 
#' }}
#' @export
groups_list_child_groups <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/groups/{id}/child_groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
imports_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
imports_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
imports_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
imports_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
imports_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
imports_list_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
imports_put_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects an Import belongs to
#' @param id integer required. The ID of the Import.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
imports_list_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add an Import to a project
#' @param id integer required. The ID of the Import.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
imports_put_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove an Import from a project
#' @param id integer required. The ID of the Import.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
imports_delete_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{name}{string, The name of the import.}
#' \item{syncType}{string, The type of sync to perform; one of Dbsync, AutoImport, GdocImport, GdocExport, and Salesforce.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id. For DB Syncs, the first element is an SSL private key credential id, and the second element is the corresponding public key credential id.
#' \item name string, 
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id. For DB Syncs, the first element is an SSL private key credential id, and the second element is the corresponding public key credential id.
#' \item name string, 
#' }}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{parentId}{integer, Parent id to trigger this import from}
#' \item{id}{integer, The ID for the import.}
#' \item{isOutbound}{boolean, }
#' \item{jobType}{string, The job type of this import.}
#' \item{syncs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item source object, 
#' \item destination object, 
#' \item advancedOptions object, 
#' }}
#' \item{state}{string, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this import.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
imports_put_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Imports
#' @param type string optional. If specified, return imports of these types. It accepts a comma-separated list, possible values are 'AutoImport', 'DbSync', 'Salesforce', 'GdocImport'.
#' @param destination string optional. If specified, returns imports with one of these destinations. It accepts a comma-separated list of remote host ids.
#' @param source string optional. If specified, returns imports with one of these sources. It accepts a comma-separated list of remote host ids. 'DbSync' must be specified for 'type'.
#' @param status string optional. If specified, returns imports with one of these statuses. It accepts a comma-separated list, possible values are 'running', 'failed', 'succeeded', 'idle', 'scheduled'.
#' @param author string optional. If specified, return items from any of these authors. It accepts a comma-separated list of user IDs.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' @param archived string optional. The archival status of the requested item(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at, last_run.updated_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{name}{string, The name of the import.}
#' \item{syncType}{string, The type of sync to perform; one of Dbsync, AutoImport, GdocImport, GdocExport, and Salesforce.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id. For DB Syncs, the first element is an SSL private key credential id, and the second element is the corresponding public key credential id.
#' \item name string, 
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id. For DB Syncs, the first element is an SSL private key credential id, and the second element is the corresponding public key credential id.
#' \item name string, 
#' }}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{id}{integer, The ID for the import.}
#' \item{isOutbound}{boolean, }
#' \item{jobType}{string, The job type of this import.}
#' \item{state}{string, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{timeZone}{string, The time zone of this import.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
imports_list <- function(type = NULL, destination = NULL, source = NULL, status = NULL, author = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/"
  path_params  <- list()
  query_params <- list(type = type, destination = destination, source = source, status = status, author = author, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a new import configuration
#' @param name string required. The name of the import.
#' @param sync_type string required. The type of sync to perform; one of Dbsync, AutoImport, GdocImport, GdocExport, and Salesforce.
#' @param is_outbound boolean required. 
#' @param source list optional. A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id. For DB Syncs, the first element is an SSL private key credential id, and the second element is the corresponding public key credential id.
#' }
#' @param destination list optional. A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id. For DB Syncs, the first element is an SSL private key credential id, and the second element is the corresponding public key credential id.
#' }
#' @param schedule list optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }
#' @param notifications list optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param parent_id integer optional. Parent id to trigger this import from
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this import.
#' @param hidden boolean optional. The hidden status of the item.
#' 
#' @return  A list containing the following elements:
#' \item{name}{string, The name of the import.}
#' \item{syncType}{string, The type of sync to perform; one of Dbsync, AutoImport, GdocImport, GdocExport, and Salesforce.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id. For DB Syncs, the first element is an SSL private key credential id, and the second element is the corresponding public key credential id.
#' \item name string, 
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id. For DB Syncs, the first element is an SSL private key credential id, and the second element is the corresponding public key credential id.
#' \item name string, 
#' }}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{parentId}{integer, Parent id to trigger this import from}
#' \item{id}{integer, The ID for the import.}
#' \item{isOutbound}{boolean, }
#' \item{jobType}{string, The job type of this import.}
#' \item{syncs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item source object, 
#' \item destination object, 
#' \item advancedOptions object, 
#' }}
#' \item{state}{string, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this import.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
imports_post <- function(name, sync_type, is_outbound, source = NULL, destination = NULL, schedule = NULL, notifications = NULL, parent_id = NULL, next_run_at = NULL, time_zone = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, syncType = sync_type, isOutbound = is_outbound, source = source, destination = destination, schedule = schedule, notifications = notifications, parentId = parent_id, nextRunAt = next_run_at, timeZone = time_zone, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Initate an import of a tabular file into the platform
#' @param schema string required. The schema of the destination table.
#' @param name string required. The name of the destination table.
#' @param remote_host_id integer required. The id of the destination database host.
#' @param credential_id integer required. The id of the credentials to be used when performing the database import.
#' @param max_errors integer optional. The maximum number of rows with errors to remove from the import before failing.
#' @param existing_table_rows string optional. The behaviour if a table with the requested name already exists.  One of "fail", "truncate", "append", or "drop".Defaults to "fail".
#' @param diststyle string optional. The diststyle to use for the table. One of "even", "all", or "key".
#' @param distkey string optional. The column to use as the distkey for the table.
#' @param sortkey1 string optional. The column to use as the sort key for the table.
#' @param sortkey2 string optional. The second column in a compound sortkey for the table.
#' @param column_delimiter string optional. The column delimiter of the file. If column_delimiter is null or omitted, it will be auto-detected. Valid arguments are "comma", "tab", and "pipe".
#' @param first_row_is_header boolean optional. A boolean value indicating whether or not the first row is a header row. If first_row_is_header is null or omitted, it will be auto-detected.
#' @param multipart boolean optional. If true, the upload URI will require a `multipart/form-data` POST request. Defaults to false.
#' @param escaped boolean optional. If true, escape quotes with a backslash; otherwise, escape quotes by double-quoting. Defaults to false.
#' @param hidden boolean optional. The hidden status of the item.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The id of the import.}
#' \item{uploadUri}{string, The URI which may be used to upload a tabular file for import. You must use this URI to upload the file you wish imported and then inform the Civis API when your upload is complete using the URI given by the runUri field of this response.}
#' \item{runUri}{string, The URI to POST to once the file upload is complete. After uploading the file using the URI given in the uploadUri attribute of the response, POST to this URI to initiate the import of your uploaded file into the platform.}
#' \item{uploadFields}{list, If multipart was set to true, these fields should be included in the multipart upload.}
#' @export
imports_post_files <- function(schema, name, remote_host_id, credential_id, max_errors = NULL, existing_table_rows = NULL, diststyle = NULL, distkey = NULL, sortkey1 = NULL, sortkey2 = NULL, column_delimiter = NULL, first_row_is_header = NULL, multipart = NULL, escaped = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(schema = schema, name = name, remoteHostId = remote_host_id, credentialId = credential_id, maxErrors = max_errors, existingTableRows = existing_table_rows, diststyle = diststyle, distkey = distkey, sortkey1 = sortkey1, sortkey2 = sortkey2, columnDelimiter = column_delimiter, firstRowIsHeader = first_row_is_header, multipart = multipart, escaped = escaped, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the import.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{importId}{integer, The ID of the import.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
imports_post_files_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given import
#' @param id integer required. The ID of the import.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the run.}
#' \item{importId}{integer, The ID of the import.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
imports_list_files_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the import.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{importId}{integer, The ID of the import.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
imports_get_files_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the import.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
imports_delete_files_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the import.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
imports_list_files_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the import.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
imports_list_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a CSV Import
#' @param source list required. A list containing the following elements: 
#' \itemize{
#' \item fileIds array, The file ID(s) to import, if importing Civis file(s).
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item storageHostId integer, The ID of the source storage host.
#' \item credentialId integer, The ID of the credentials for the source storage host.
#' \item filePaths array, The file or directory path(s) within the bucket from which to import.  E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"If specifying a directory path, the job will import every file found under that path. All files must have the same column layout and file format (e.g., compression, columnDelimiter, etc.).
#' }
#' }
#' @param destination list required. A list containing the following elements: 
#' \itemize{
#' \item schema string, The destination schema name.
#' \item table string, The destination table name.
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' \item primaryKeys array, A list of column(s) which together uniquely identify a row in the destination table.These columns must not contain NULL values. If the import mode is "upsert", this field is required;see the Civis Helpdesk article on "Advanced CSV Imports via the Civis API" for more information.
#' \item lastModifiedKeys array, A list of the columns indicating a record has been updated.If the destination table does not exist, and the import mode is "upsert", this field is required.
#' }
#' @param first_row_is_header boolean required. A boolean value indicating whether or not the first row of the source file is a header row.
#' @param name string optional. The name of the import.
#' @param column_delimiter string optional. The column delimiter for the file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".
#' @param escaped boolean optional. A boolean value indicating whether or not the source file has quotes escaped with a backslash.Defaults to false.
#' @param compression string optional. The type of compression of the source file. Valid arguments are "gzip" and "none". Defaults to "none".
#' @param existing_table_rows string optional. The behavior if a destination table with the requested name already exists.  One of "fail", "truncate", "append", "drop", or "upsert".Defaults to "fail".
#' @param max_errors integer optional. The maximum number of rows with errors to ignore before failing. This option is not supported for Postgres databases.
#' @param table_columns array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The column name.
#' \item sqlType string, The SQL type of the column.
#' }
#' @param loosen_types boolean optional. If true, SQL types with precisions/lengths will have these values increased to accommodate data growth in future loads. Type loosening only occurs on table creation. Defaults to false.
#' @param execution string optional. In upsert mode, controls the movement of data in upsert mode. If set to "delayed", the data will be moved after a brief delay. If set to "immediate", the data will be moved immediately. In non-upsert modes, controls the speed at which detailed column stats appear in the data catalogue. Defaults to "delayed", to accommodate concurrent upserts to the same table and speedier non-upsert imports.
#' @param redshift_destination_options list optional. A list containing the following elements: 
#' \itemize{
#' \item diststyle string, The diststyle to use for the table. One of "even", "all", or "key".
#' \item distkey string, Distkey for this table in Redshift
#' \item sortkeys array, Sortkeys for this table in Redshift. Please provide a maximum of two.
#' }
#' @param hidden boolean optional. The hidden status of the item.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the import.}
#' \item{name}{string, The name of the import.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item fileIds array, The file ID(s) to import, if importing Civis file(s).
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item storageHostId integer, The ID of the source storage host.
#' \item credentialId integer, The ID of the credentials for the source storage host.
#' \item filePaths array, The file or directory path(s) within the bucket from which to import.  E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"If specifying a directory path, the job will import every file found under that path. All files must have the same column layout and file format (e.g., compression, columnDelimiter, etc.).
#' }
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item schema string, The destination schema name.
#' \item table string, The destination table name.
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' \item primaryKeys array, A list of column(s) which together uniquely identify a row in the destination table.These columns must not contain NULL values. If the import mode is "upsert", this field is required;see the Civis Helpdesk article on "Advanced CSV Imports via the Civis API" for more information.
#' \item lastModifiedKeys array, A list of the columns indicating a record has been updated.If the destination table does not exist, and the import mode is "upsert", this field is required.
#' }}
#' \item{firstRowIsHeader}{boolean, A boolean value indicating whether or not the first row of the source file is a header row.}
#' \item{columnDelimiter}{string, The column delimiter for the file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".}
#' \item{escaped}{boolean, A boolean value indicating whether or not the source file has quotes escaped with a backslash.Defaults to false.}
#' \item{compression}{string, The type of compression of the source file. Valid arguments are "gzip" and "none". Defaults to "none".}
#' \item{existingTableRows}{string, The behavior if a destination table with the requested name already exists.  One of "fail", "truncate", "append", "drop", or "upsert".Defaults to "fail".}
#' \item{maxErrors}{integer, The maximum number of rows with errors to ignore before failing. This option is not supported for Postgres databases.}
#' \item{tableColumns}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The column name.
#' \item sqlType string, The SQL type of the column.
#' }}
#' \item{loosenTypes}{boolean, If true, SQL types with precisions/lengths will have these values increased to accommodate data growth in future loads. Type loosening only occurs on table creation. Defaults to false.}
#' \item{execution}{string, In upsert mode, controls the movement of data in upsert mode. If set to "delayed", the data will be moved after a brief delay. If set to "immediate", the data will be moved immediately. In non-upsert modes, controls the speed at which detailed column stats appear in the data catalogue. Defaults to "delayed", to accommodate concurrent upserts to the same table and speedier non-upsert imports.}
#' \item{redshiftDestinationOptions}{list, A list containing the following elements: 
#' \itemize{
#' \item diststyle string, The diststyle to use for the table. One of "even", "all", or "key".
#' \item distkey string, Distkey for this table in Redshift
#' \item sortkeys array, Sortkeys for this table in Redshift. Please provide a maximum of two.
#' }}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
imports_post_files_csv <- function(source, destination, first_row_is_header, name = NULL, column_delimiter = NULL, escaped = NULL, compression = NULL, existing_table_rows = NULL, max_errors = NULL, table_columns = NULL, loosen_types = NULL, execution = NULL, redshift_destination_options = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/csv"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(source = source, destination = destination, firstRowIsHeader = first_row_is_header, name = name, columnDelimiter = column_delimiter, escaped = escaped, compression = compression, existingTableRows = existing_table_rows, maxErrors = max_errors, tableColumns = table_columns, loosenTypes = loosen_types, execution = execution, redshiftDestinationOptions = redshift_destination_options, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a CSV Import
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the import.}
#' \item{name}{string, The name of the import.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item fileIds array, The file ID(s) to import, if importing Civis file(s).
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item storageHostId integer, The ID of the source storage host.
#' \item credentialId integer, The ID of the credentials for the source storage host.
#' \item filePaths array, The file or directory path(s) within the bucket from which to import.  E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"If specifying a directory path, the job will import every file found under that path. All files must have the same column layout and file format (e.g., compression, columnDelimiter, etc.).
#' }
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item schema string, The destination schema name.
#' \item table string, The destination table name.
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' \item primaryKeys array, A list of column(s) which together uniquely identify a row in the destination table.These columns must not contain NULL values. If the import mode is "upsert", this field is required;see the Civis Helpdesk article on "Advanced CSV Imports via the Civis API" for more information.
#' \item lastModifiedKeys array, A list of the columns indicating a record has been updated.If the destination table does not exist, and the import mode is "upsert", this field is required.
#' }}
#' \item{firstRowIsHeader}{boolean, A boolean value indicating whether or not the first row of the source file is a header row.}
#' \item{columnDelimiter}{string, The column delimiter for the file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".}
#' \item{escaped}{boolean, A boolean value indicating whether or not the source file has quotes escaped with a backslash.Defaults to false.}
#' \item{compression}{string, The type of compression of the source file. Valid arguments are "gzip" and "none". Defaults to "none".}
#' \item{existingTableRows}{string, The behavior if a destination table with the requested name already exists.  One of "fail", "truncate", "append", "drop", or "upsert".Defaults to "fail".}
#' \item{maxErrors}{integer, The maximum number of rows with errors to ignore before failing. This option is not supported for Postgres databases.}
#' \item{tableColumns}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The column name.
#' \item sqlType string, The SQL type of the column.
#' }}
#' \item{loosenTypes}{boolean, If true, SQL types with precisions/lengths will have these values increased to accommodate data growth in future loads. Type loosening only occurs on table creation. Defaults to false.}
#' \item{execution}{string, In upsert mode, controls the movement of data in upsert mode. If set to "delayed", the data will be moved after a brief delay. If set to "immediate", the data will be moved immediately. In non-upsert modes, controls the speed at which detailed column stats appear in the data catalogue. Defaults to "delayed", to accommodate concurrent upserts to the same table and speedier non-upsert imports.}
#' \item{redshiftDestinationOptions}{list, A list containing the following elements: 
#' \itemize{
#' \item diststyle string, The diststyle to use for the table. One of "even", "all", or "key".
#' \item distkey string, Distkey for this table in Redshift
#' \item sortkeys array, Sortkeys for this table in Redshift. Please provide a maximum of two.
#' }}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
imports_get_files_csv <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/csv/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this CSV Import
#' @param id integer required. The ID for the import.
#' @param source list required. A list containing the following elements: 
#' \itemize{
#' \item fileIds array, The file ID(s) to import, if importing Civis file(s).
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item storageHostId integer, The ID of the source storage host.
#' \item credentialId integer, The ID of the credentials for the source storage host.
#' \item filePaths array, The file or directory path(s) within the bucket from which to import.  E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"If specifying a directory path, the job will import every file found under that path. All files must have the same column layout and file format (e.g., compression, columnDelimiter, etc.).
#' }
#' }
#' @param destination list required. A list containing the following elements: 
#' \itemize{
#' \item schema string, The destination schema name.
#' \item table string, The destination table name.
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' \item primaryKeys array, A list of column(s) which together uniquely identify a row in the destination table.These columns must not contain NULL values. If the import mode is "upsert", this field is required;see the Civis Helpdesk article on "Advanced CSV Imports via the Civis API" for more information.
#' \item lastModifiedKeys array, A list of the columns indicating a record has been updated.If the destination table does not exist, and the import mode is "upsert", this field is required.
#' }
#' @param first_row_is_header boolean required. A boolean value indicating whether or not the first row of the source file is a header row.
#' @param name string optional. The name of the import.
#' @param column_delimiter string optional. The column delimiter for the file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".
#' @param escaped boolean optional. A boolean value indicating whether or not the source file has quotes escaped with a backslash.Defaults to false.
#' @param compression string optional. The type of compression of the source file. Valid arguments are "gzip" and "none". Defaults to "none".
#' @param existing_table_rows string optional. The behavior if a destination table with the requested name already exists.  One of "fail", "truncate", "append", "drop", or "upsert".Defaults to "fail".
#' @param max_errors integer optional. The maximum number of rows with errors to ignore before failing. This option is not supported for Postgres databases.
#' @param table_columns array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The column name.
#' \item sqlType string, The SQL type of the column.
#' }
#' @param loosen_types boolean optional. If true, SQL types with precisions/lengths will have these values increased to accommodate data growth in future loads. Type loosening only occurs on table creation. Defaults to false.
#' @param execution string optional. In upsert mode, controls the movement of data in upsert mode. If set to "delayed", the data will be moved after a brief delay. If set to "immediate", the data will be moved immediately. In non-upsert modes, controls the speed at which detailed column stats appear in the data catalogue. Defaults to "delayed", to accommodate concurrent upserts to the same table and speedier non-upsert imports.
#' @param redshift_destination_options list optional. A list containing the following elements: 
#' \itemize{
#' \item diststyle string, The diststyle to use for the table. One of "even", "all", or "key".
#' \item distkey string, Distkey for this table in Redshift
#' \item sortkeys array, Sortkeys for this table in Redshift. Please provide a maximum of two.
#' }
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the import.}
#' \item{name}{string, The name of the import.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item fileIds array, The file ID(s) to import, if importing Civis file(s).
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item storageHostId integer, The ID of the source storage host.
#' \item credentialId integer, The ID of the credentials for the source storage host.
#' \item filePaths array, The file or directory path(s) within the bucket from which to import.  E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"If specifying a directory path, the job will import every file found under that path. All files must have the same column layout and file format (e.g., compression, columnDelimiter, etc.).
#' }
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item schema string, The destination schema name.
#' \item table string, The destination table name.
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' \item primaryKeys array, A list of column(s) which together uniquely identify a row in the destination table.These columns must not contain NULL values. If the import mode is "upsert", this field is required;see the Civis Helpdesk article on "Advanced CSV Imports via the Civis API" for more information.
#' \item lastModifiedKeys array, A list of the columns indicating a record has been updated.If the destination table does not exist, and the import mode is "upsert", this field is required.
#' }}
#' \item{firstRowIsHeader}{boolean, A boolean value indicating whether or not the first row of the source file is a header row.}
#' \item{columnDelimiter}{string, The column delimiter for the file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".}
#' \item{escaped}{boolean, A boolean value indicating whether or not the source file has quotes escaped with a backslash.Defaults to false.}
#' \item{compression}{string, The type of compression of the source file. Valid arguments are "gzip" and "none". Defaults to "none".}
#' \item{existingTableRows}{string, The behavior if a destination table with the requested name already exists.  One of "fail", "truncate", "append", "drop", or "upsert".Defaults to "fail".}
#' \item{maxErrors}{integer, The maximum number of rows with errors to ignore before failing. This option is not supported for Postgres databases.}
#' \item{tableColumns}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The column name.
#' \item sqlType string, The SQL type of the column.
#' }}
#' \item{loosenTypes}{boolean, If true, SQL types with precisions/lengths will have these values increased to accommodate data growth in future loads. Type loosening only occurs on table creation. Defaults to false.}
#' \item{execution}{string, In upsert mode, controls the movement of data in upsert mode. If set to "delayed", the data will be moved after a brief delay. If set to "immediate", the data will be moved immediately. In non-upsert modes, controls the speed at which detailed column stats appear in the data catalogue. Defaults to "delayed", to accommodate concurrent upserts to the same table and speedier non-upsert imports.}
#' \item{redshiftDestinationOptions}{list, A list containing the following elements: 
#' \itemize{
#' \item diststyle string, The diststyle to use for the table. One of "even", "all", or "key".
#' \item distkey string, Distkey for this table in Redshift
#' \item sortkeys array, Sortkeys for this table in Redshift. Please provide a maximum of two.
#' }}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
imports_put_files_csv <- function(id, source, destination, first_row_is_header, name = NULL, column_delimiter = NULL, escaped = NULL, compression = NULL, existing_table_rows = NULL, max_errors = NULL, table_columns = NULL, loosen_types = NULL, execution = NULL, redshift_destination_options = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/csv/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(source = source, destination = destination, firstRowIsHeader = first_row_is_header, name = name, columnDelimiter = column_delimiter, escaped = escaped, compression = compression, existingTableRows = existing_table_rows, maxErrors = max_errors, tableColumns = table_columns, loosenTypes = loosen_types, execution = execution, redshiftDestinationOptions = redshift_destination_options)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this CSV Import
#' @param id integer required. The ID for the import.
#' @param name string optional. The name of the import.
#' @param source list optional. A list containing the following elements: 
#' \itemize{
#' \item fileIds array, The file ID(s) to import, if importing Civis file(s).
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item storageHostId integer, The ID of the source storage host.
#' \item credentialId integer, The ID of the credentials for the source storage host.
#' \item filePaths array, The file or directory path(s) within the bucket from which to import.  E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"If specifying a directory path, the job will import every file found under that path. All files must have the same column layout and file format (e.g., compression, columnDelimiter, etc.).
#' }
#' }
#' @param destination list optional. A list containing the following elements: 
#' \itemize{
#' \item schema string, The destination schema name.
#' \item table string, The destination table name.
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' \item primaryKeys array, A list of column(s) which together uniquely identify a row in the destination table.These columns must not contain NULL values. If the import mode is "upsert", this field is required;see the Civis Helpdesk article on "Advanced CSV Imports via the Civis API" for more information.
#' \item lastModifiedKeys array, A list of the columns indicating a record has been updated.If the destination table does not exist, and the import mode is "upsert", this field is required.
#' }
#' @param first_row_is_header boolean optional. A boolean value indicating whether or not the first row of the source file is a header row.
#' @param column_delimiter string optional. The column delimiter for the file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".
#' @param escaped boolean optional. A boolean value indicating whether or not the source file has quotes escaped with a backslash.Defaults to false.
#' @param compression string optional. The type of compression of the source file. Valid arguments are "gzip" and "none". Defaults to "none".
#' @param existing_table_rows string optional. The behavior if a destination table with the requested name already exists.  One of "fail", "truncate", "append", "drop", or "upsert".Defaults to "fail".
#' @param max_errors integer optional. The maximum number of rows with errors to ignore before failing. This option is not supported for Postgres databases.
#' @param table_columns array optional. An array containing the following fields: 
#' \itemize{
#' \item name string, The column name.
#' \item sqlType string, The SQL type of the column.
#' }
#' @param loosen_types boolean optional. If true, SQL types with precisions/lengths will have these values increased to accommodate data growth in future loads. Type loosening only occurs on table creation. Defaults to false.
#' @param execution string optional. In upsert mode, controls the movement of data in upsert mode. If set to "delayed", the data will be moved after a brief delay. If set to "immediate", the data will be moved immediately. In non-upsert modes, controls the speed at which detailed column stats appear in the data catalogue. Defaults to "delayed", to accommodate concurrent upserts to the same table and speedier non-upsert imports.
#' @param redshift_destination_options list optional. A list containing the following elements: 
#' \itemize{
#' \item diststyle string, The diststyle to use for the table. One of "even", "all", or "key".
#' \item distkey string, Distkey for this table in Redshift
#' \item sortkeys array, Sortkeys for this table in Redshift. Please provide a maximum of two.
#' }
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the import.}
#' \item{name}{string, The name of the import.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item fileIds array, The file ID(s) to import, if importing Civis file(s).
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item storageHostId integer, The ID of the source storage host.
#' \item credentialId integer, The ID of the credentials for the source storage host.
#' \item filePaths array, The file or directory path(s) within the bucket from which to import.  E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"If specifying a directory path, the job will import every file found under that path. All files must have the same column layout and file format (e.g., compression, columnDelimiter, etc.).
#' }
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item schema string, The destination schema name.
#' \item table string, The destination table name.
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' \item primaryKeys array, A list of column(s) which together uniquely identify a row in the destination table.These columns must not contain NULL values. If the import mode is "upsert", this field is required;see the Civis Helpdesk article on "Advanced CSV Imports via the Civis API" for more information.
#' \item lastModifiedKeys array, A list of the columns indicating a record has been updated.If the destination table does not exist, and the import mode is "upsert", this field is required.
#' }}
#' \item{firstRowIsHeader}{boolean, A boolean value indicating whether or not the first row of the source file is a header row.}
#' \item{columnDelimiter}{string, The column delimiter for the file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".}
#' \item{escaped}{boolean, A boolean value indicating whether or not the source file has quotes escaped with a backslash.Defaults to false.}
#' \item{compression}{string, The type of compression of the source file. Valid arguments are "gzip" and "none". Defaults to "none".}
#' \item{existingTableRows}{string, The behavior if a destination table with the requested name already exists.  One of "fail", "truncate", "append", "drop", or "upsert".Defaults to "fail".}
#' \item{maxErrors}{integer, The maximum number of rows with errors to ignore before failing. This option is not supported for Postgres databases.}
#' \item{tableColumns}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The column name.
#' \item sqlType string, The SQL type of the column.
#' }}
#' \item{loosenTypes}{boolean, If true, SQL types with precisions/lengths will have these values increased to accommodate data growth in future loads. Type loosening only occurs on table creation. Defaults to false.}
#' \item{execution}{string, In upsert mode, controls the movement of data in upsert mode. If set to "delayed", the data will be moved after a brief delay. If set to "immediate", the data will be moved immediately. In non-upsert modes, controls the speed at which detailed column stats appear in the data catalogue. Defaults to "delayed", to accommodate concurrent upserts to the same table and speedier non-upsert imports.}
#' \item{redshiftDestinationOptions}{list, A list containing the following elements: 
#' \itemize{
#' \item diststyle string, The diststyle to use for the table. One of "even", "all", or "key".
#' \item distkey string, Distkey for this table in Redshift
#' \item sortkeys array, Sortkeys for this table in Redshift. Please provide a maximum of two.
#' }}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
imports_patch_files_csv <- function(id, name = NULL, source = NULL, destination = NULL, first_row_is_header = NULL, column_delimiter = NULL, escaped = NULL, compression = NULL, existing_table_rows = NULL, max_errors = NULL, table_columns = NULL, loosen_types = NULL, execution = NULL, redshift_destination_options = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/csv/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, source = source, destination = destination, firstRowIsHeader = first_row_is_header, columnDelimiter = column_delimiter, escaped = escaped, compression = compression, existingTableRows = existing_table_rows, maxErrors = max_errors, tableColumns = table_columns, loosenTypes = loosen_types, execution = execution, redshiftDestinationOptions = redshift_destination_options)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive a CSV Import (deprecated, use archiving endpoints instead)
#' @param id integer required. 
#' 
#' @return  An empty HTTP response
#' @export
imports_delete_files_csv <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/csv/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the import.}
#' \item{name}{string, The name of the import.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item fileIds array, The file ID(s) to import, if importing Civis file(s).
#' \item storagePath list . A list containing the following elements: 
#' \itemize{
#' \item storageHostId integer, The ID of the source storage host.
#' \item credentialId integer, The ID of the credentials for the source storage host.
#' \item filePaths array, The file or directory path(s) within the bucket from which to import.  E.g. the file_path for "s3://mybucket/files/all/" would be "/files/all/"If specifying a directory path, the job will import every file found under that path. All files must have the same column layout and file format (e.g., compression, columnDelimiter, etc.).
#' }
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item schema string, The destination schema name.
#' \item table string, The destination table name.
#' \item remoteHostId integer, The ID of the destination database host.
#' \item credentialId integer, The ID of the credentials for the destination database.
#' \item primaryKeys array, A list of column(s) which together uniquely identify a row in the destination table.These columns must not contain NULL values. If the import mode is "upsert", this field is required;see the Civis Helpdesk article on "Advanced CSV Imports via the Civis API" for more information.
#' \item lastModifiedKeys array, A list of the columns indicating a record has been updated.If the destination table does not exist, and the import mode is "upsert", this field is required.
#' }}
#' \item{firstRowIsHeader}{boolean, A boolean value indicating whether or not the first row of the source file is a header row.}
#' \item{columnDelimiter}{string, The column delimiter for the file. Valid arguments are "comma", "tab", and "pipe". Defaults to "comma".}
#' \item{escaped}{boolean, A boolean value indicating whether or not the source file has quotes escaped with a backslash.Defaults to false.}
#' \item{compression}{string, The type of compression of the source file. Valid arguments are "gzip" and "none". Defaults to "none".}
#' \item{existingTableRows}{string, The behavior if a destination table with the requested name already exists.  One of "fail", "truncate", "append", "drop", or "upsert".Defaults to "fail".}
#' \item{maxErrors}{integer, The maximum number of rows with errors to ignore before failing. This option is not supported for Postgres databases.}
#' \item{tableColumns}{array, An array containing the following fields: 
#' \itemize{
#' \item name string, The column name.
#' \item sqlType string, The SQL type of the column.
#' }}
#' \item{loosenTypes}{boolean, If true, SQL types with precisions/lengths will have these values increased to accommodate data growth in future loads. Type loosening only occurs on table creation. Defaults to false.}
#' \item{execution}{string, In upsert mode, controls the movement of data in upsert mode. If set to "delayed", the data will be moved after a brief delay. If set to "immediate", the data will be moved immediately. In non-upsert modes, controls the speed at which detailed column stats appear in the data catalogue. Defaults to "delayed", to accommodate concurrent upserts to the same table and speedier non-upsert imports.}
#' \item{redshiftDestinationOptions}{list, A list containing the following elements: 
#' \itemize{
#' \item diststyle string, The diststyle to use for the table. One of "even", "all", or "key".
#' \item distkey string, Distkey for this table in Redshift
#' \item sortkeys array, Sortkeys for this table in Redshift. Please provide a maximum of two.
#' }}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
imports_put_files_csv_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/csv/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the csv_import.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{csvImportId}{integer, The ID of the csv_import.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
imports_post_files_csv_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/csv/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given csv_import
#' @param id integer required. The ID of the csv_import.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the run.}
#' \item{csvImportId}{integer, The ID of the csv_import.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
imports_list_files_csv_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/csv/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the csv_import.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{csvImportId}{integer, The ID of the csv_import.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
imports_get_files_csv_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/csv/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the csv_import.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
imports_delete_files_csv_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/csv/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the csv_import.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
imports_list_files_csv_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/files/csv/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List batch imports
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for the import.}
#' \item{schema}{string, The destination schema name. This schema must already exist in Redshift.}
#' \item{table}{string, The destination table name, without the schema prefix. This table must already exist in Redshift.}
#' \item{remoteHostId}{integer, The ID of the destination database host.}
#' \item{state}{string, The state of the run; one of "queued", "running", "succeeded", "failed", or "cancelled".}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error returned by the run, if any.}
#' @export
imports_list_batches <- function(hidden = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/batches"
  path_params  <- list()
  query_params <- list(hidden = hidden, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Upload multiple files to Civis
#' @param file_ids array required. The file IDs for the import.
#' @param schema string required. The destination schema name. This schema must already exist in Redshift.
#' @param table string required. The destination table name, without the schema prefix. This table must already exist in Redshift.
#' @param remote_host_id integer required. The ID of the destination database host.
#' @param credential_id integer required. The ID of the credentials to be used when performing the database import.
#' @param column_delimiter string optional. The column delimiter for the file. Valid arguments are "comma", "tab", and "pipe". If unspecified, defaults to "comma".
#' @param first_row_is_header boolean optional. A boolean value indicating whether or not the first row is a header row. If unspecified, defaults to false.
#' @param compression string optional. The type of compression. Valid arguments are "gzip", "zip", and "none". If unspecified, defaults to "gzip".
#' @param hidden boolean optional. The hidden status of the item.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the import.}
#' \item{schema}{string, The destination schema name. This schema must already exist in Redshift.}
#' \item{table}{string, The destination table name, without the schema prefix. This table must already exist in Redshift.}
#' \item{remoteHostId}{integer, The ID of the destination database host.}
#' \item{state}{string, The state of the run; one of "queued", "running", "succeeded", "failed", or "cancelled".}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error returned by the run, if any.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' @export
imports_post_batches <- function(file_ids, schema, table, remote_host_id, credential_id, column_delimiter = NULL, first_row_is_header = NULL, compression = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/batches"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(fileIds = file_ids, schema = schema, table = table, remoteHostId = remote_host_id, credentialId = credential_id, columnDelimiter = column_delimiter, firstRowIsHeader = first_row_is_header, compression = compression, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about a batch import
#' @param id integer required. The ID for the import.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the import.}
#' \item{schema}{string, The destination schema name. This schema must already exist in Redshift.}
#' \item{table}{string, The destination table name, without the schema prefix. This table must already exist in Redshift.}
#' \item{remoteHostId}{integer, The ID of the destination database host.}
#' \item{state}{string, The state of the run; one of "queued", "running", "succeeded", "failed", or "cancelled".}
#' \item{startedAt}{string, The time the last run started at.}
#' \item{finishedAt}{string, The time the last run completed.}
#' \item{error}{string, The error returned by the run, if any.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' @export
imports_get_batches <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/batches/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about an import
#' @param id integer required. The ID for the import.
#' 
#' @return  A list containing the following elements:
#' \item{name}{string, The name of the import.}
#' \item{syncType}{string, The type of sync to perform; one of Dbsync, AutoImport, GdocImport, GdocExport, and Salesforce.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id. For DB Syncs, the first element is an SSL private key credential id, and the second element is the corresponding public key credential id.
#' \item name string, 
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id. For DB Syncs, the first element is an SSL private key credential id, and the second element is the corresponding public key credential id.
#' \item name string, 
#' }}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{parentId}{integer, Parent id to trigger this import from}
#' \item{id}{integer, The ID for the import.}
#' \item{isOutbound}{boolean, }
#' \item{jobType}{string, The job type of this import.}
#' \item{syncs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item source object, 
#' \item destination object, 
#' \item advancedOptions object, 
#' }}
#' \item{state}{string, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this import.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
imports_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update an import
#' @param id integer required. The ID for the import.
#' @param name string required. The name of the import.
#' @param sync_type string required. The type of sync to perform; one of Dbsync, AutoImport, GdocImport, GdocExport, and Salesforce.
#' @param is_outbound boolean required. 
#' @param source list optional. A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id. For DB Syncs, the first element is an SSL private key credential id, and the second element is the corresponding public key credential id.
#' }
#' @param destination list optional. A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id. For DB Syncs, the first element is an SSL private key credential id, and the second element is the corresponding public key credential id.
#' }
#' @param schedule list optional. A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }
#' @param notifications list optional. A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }
#' @param parent_id integer optional. Parent id to trigger this import from
#' @param next_run_at string optional. The time of the next scheduled run.
#' @param time_zone string optional. The time zone of this import.
#' 
#' @return  A list containing the following elements:
#' \item{name}{string, The name of the import.}
#' \item{syncType}{string, The type of sync to perform; one of Dbsync, AutoImport, GdocImport, GdocExport, and Salesforce.}
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id. For DB Syncs, the first element is an SSL private key credential id, and the second element is the corresponding public key credential id.
#' \item name string, 
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item remoteHostId integer, 
#' \item credentialId integer, 
#' \item additionalCredentials array, Array that holds additional credentials used for specific imports. For salesforce imports, the first and only element is the client credential id. For DB Syncs, the first element is an SSL private key credential id, and the second element is the corresponding public key credential id.
#' \item name string, 
#' }}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{parentId}{integer, Parent id to trigger this import from}
#' \item{id}{integer, The ID for the import.}
#' \item{isOutbound}{boolean, }
#' \item{jobType}{string, The job type of this import.}
#' \item{syncs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item source object, 
#' \item destination object, 
#' \item advancedOptions object, 
#' }}
#' \item{state}{string, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{nextRunAt}{string, The time of the next scheduled run.}
#' \item{timeZone}{string, The time zone of this import.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
imports_put <- function(id, name, sync_type, is_outbound, source = NULL, destination = NULL, schedule = NULL, notifications = NULL, parent_id = NULL, next_run_at = NULL, time_zone = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, syncType = sync_type, isOutbound = is_outbound, source = source, destination = destination, schedule = schedule, notifications = notifications, parentId = parent_id, nextRunAt = next_run_at, timeZone = time_zone)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the run history of this import
#' @param id integer required. 
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, }
#' \item{state}{string, }
#' \item{createdAt}{string, The time that the run was queued.}
#' \item{startedAt}{string, The time that the run started.}
#' \item{finishedAt}{string, The time that the run completed.}
#' \item{error}{string, The error message for this run, if present.}
#' @export
imports_list_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Run an import
#' @param id integer required. The ID of the import to run.
#' 
#' @return  A list containing the following elements:
#' \item{runId}{integer, The ID of the new run triggered.}
#' @export
imports_post_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{state}{string, The state of the run, one of 'queued', 'running' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' @export
imports_post_cancel <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/cancel"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a sync
#' @param id integer required. 
#' @param source list required. A list containing the following elements: 
#' \itemize{
#' \item path string, The path of the dataset to sync from; for a database source, schema.tablename. If you are doing a Google Sheet export, this can be blank. This is a legacy parameter, it is recommended you use one of the following: databaseTable, file, googleWorksheet, salesforce
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The database schema name.
#' \item table string, The database table name.
#' \item useWithoutSchema boolean, This attribute is no longer available; defaults to false but cannot be used.
#' }
#' \item file list . 
#' \item googleWorksheet list . A list containing the following elements: 
#' \itemize{
#' \item spreadsheet string, The spreadsheet document name.
#' \item spreadsheetId string, The spreadsheet document id.
#' \item worksheet string, The worksheet tab name.
#' \item worksheetId integer, The worksheet tab id.
#' }
#' \item salesforce list . A list containing the following elements: 
#' \itemize{
#' \item objectName string, The Salesforce object name.
#' }
#' }
#' @param destination list required. A list containing the following elements: 
#' \itemize{
#' \item path string, The schema.tablename to sync to. If you are doing a Google Sheet export, this is the spreadsheet and sheet name separated by a period. i.e. if you have a spreadsheet named "MySpreadsheet" and a sheet called "Sheet1" this field would be "MySpreadsheet.Sheet1". This is a legacy parameter, it is recommended you use one of the following: databaseTable, googleWorksheet
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The database schema name.
#' \item table string, The database table name.
#' \item useWithoutSchema boolean, This attribute is no longer available; defaults to false but cannot be used.
#' }
#' \item googleWorksheet list . A list containing the following elements: 
#' \itemize{
#' \item spreadsheet string, The spreadsheet document name.
#' \item spreadsheetId string, The spreadsheet document id.
#' \item worksheet string, The worksheet tab name.
#' \item worksheetId integer, The worksheet tab id.
#' }
#' }
#' @param advanced_options list optional. A list containing the following elements: 
#' \itemize{
#' \item maxErrors integer, 
#' \item existingTableRows string, 
#' \item diststyle string, 
#' \item distkey string, 
#' \item sortkey1 string, 
#' \item sortkey2 string, 
#' \item columnDelimiter string, 
#' \item columnOverrides object, Hash used for overriding auto-detected names and types, with keys being the index of the column being overridden.
#' \item escaped boolean, If true, escape quotes with a backslash; otherwise, escape quotes by double-quoting. Defaults to false.
#' \item identityColumn string, 
#' \item rowChunkSize integer, 
#' \item wipeDestinationTable boolean, 
#' \item truncateLongLines boolean, 
#' \item invalidCharReplacement string, 
#' \item verifyTableRowCounts boolean, 
#' \item partitionColumnName string, This parameter is deprecated
#' \item partitionSchemaName string, This parameter is deprecated
#' \item partitionTableName string, This parameter is deprecated
#' \item partitionTablePartitionColumnMinName string, This parameter is deprecated
#' \item partitionTablePartitionColumnMaxName string, This parameter is deprecated
#' \item lastModifiedColumn string, 
#' \item mysqlCatalogMatchesSchema boolean, This attribute is no longer available; defaults to true but cannot be used.
#' \item chunkingMethod string, This parameter is deprecated
#' \item firstRowIsHeader boolean, 
#' \item exportAction string, The kind of export action you want to have the export execute. Set to "newsprsht" if you want a new worksheet inside a new spreadsheet. Set to "newwksht" if you want a new worksheet inside an existing spreadsheet. Set to "updatewksht" if you want to overwrite an existing worksheet inside an existing spreadsheet. Set to "appendwksht" if you want to append to the end of an existing worksheet inside an existing spreadsheet. Default is set to "newsprsht"
#' \item sqlQuery string, If you are doing a Google Sheet export, this is your SQL query.
#' \item contactLists string, 
#' \item soqlQuery string, 
#' \item includeDeletedRecords boolean, 
#' }
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of the table or file, if available.
#' \item path string, The path of the dataset to sync from; for a database source, schema.tablename. If you are doing a Google Sheet export, this can be blank. This is a legacy parameter, it is recommended you use one of the following: databaseTable, file, googleWorksheet, salesforce
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The database schema name.
#' \item table string, The database table name.
#' \item useWithoutSchema boolean, This attribute is no longer available; defaults to false but cannot be used.
#' }
#' \item file list . A list containing the following elements: 
#' \itemize{
#' \item id integer, The file id.
#' }
#' \item googleWorksheet list . A list containing the following elements: 
#' \itemize{
#' \item spreadsheet string, The spreadsheet document name.
#' \item spreadsheetId string, The spreadsheet document id.
#' \item worksheet string, The worksheet tab name.
#' \item worksheetId integer, The worksheet tab id.
#' }
#' \item salesforce list . A list containing the following elements: 
#' \itemize{
#' \item objectName string, The Salesforce object name.
#' }
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item path string, The schema.tablename to sync to. If you are doing a Google Sheet export, this is the spreadsheet and sheet name separated by a period. i.e. if you have a spreadsheet named "MySpreadsheet" and a sheet called "Sheet1" this field would be "MySpreadsheet.Sheet1". This is a legacy parameter, it is recommended you use one of the following: databaseTable, googleWorksheet
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The database schema name.
#' \item table string, The database table name.
#' \item useWithoutSchema boolean, This attribute is no longer available; defaults to false but cannot be used.
#' }
#' \item googleWorksheet list . A list containing the following elements: 
#' \itemize{
#' \item spreadsheet string, The spreadsheet document name.
#' \item spreadsheetId string, The spreadsheet document id.
#' \item worksheet string, The worksheet tab name.
#' \item worksheetId integer, The worksheet tab id.
#' }
#' }}
#' \item{advancedOptions}{list, A list containing the following elements: 
#' \itemize{
#' \item maxErrors integer, 
#' \item existingTableRows string, 
#' \item diststyle string, 
#' \item distkey string, 
#' \item sortkey1 string, 
#' \item sortkey2 string, 
#' \item columnDelimiter string, 
#' \item columnOverrides object, Hash used for overriding auto-detected names and types, with keys being the index of the column being overridden.
#' \item escaped boolean, If true, escape quotes with a backslash; otherwise, escape quotes by double-quoting. Defaults to false.
#' \item identityColumn string, 
#' \item rowChunkSize integer, 
#' \item wipeDestinationTable boolean, 
#' \item truncateLongLines boolean, 
#' \item invalidCharReplacement string, 
#' \item verifyTableRowCounts boolean, 
#' \item partitionColumnName string, This parameter is deprecated
#' \item partitionSchemaName string, This parameter is deprecated
#' \item partitionTableName string, This parameter is deprecated
#' \item partitionTablePartitionColumnMinName string, This parameter is deprecated
#' \item partitionTablePartitionColumnMaxName string, This parameter is deprecated
#' \item lastModifiedColumn string, 
#' \item mysqlCatalogMatchesSchema boolean, This attribute is no longer available; defaults to true but cannot be used.
#' \item chunkingMethod string, This parameter is deprecated
#' \item firstRowIsHeader boolean, 
#' \item exportAction string, The kind of export action you want to have the export execute. Set to "newsprsht" if you want a new worksheet inside a new spreadsheet. Set to "newwksht" if you want a new worksheet inside an existing spreadsheet. Set to "updatewksht" if you want to overwrite an existing worksheet inside an existing spreadsheet. Set to "appendwksht" if you want to append to the end of an existing worksheet inside an existing spreadsheet. Default is set to "newsprsht"
#' \item sqlQuery string, If you are doing a Google Sheet export, this is your SQL query.
#' \item contactLists string, 
#' \item soqlQuery string, 
#' \item includeDeletedRecords boolean, 
#' }}
#' @export
imports_post_syncs <- function(id, source, destination, advanced_options = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/syncs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(source = source, destination = destination, advancedOptions = advanced_options)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a sync
#' @param id integer required. The ID of the import to fetch.
#' @param sync_id integer required. The ID of the sync to fetch.
#' @param source list required. A list containing the following elements: 
#' \itemize{
#' \item path string, The path of the dataset to sync from; for a database source, schema.tablename. If you are doing a Google Sheet export, this can be blank. This is a legacy parameter, it is recommended you use one of the following: databaseTable, file, googleWorksheet, salesforce
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The database schema name.
#' \item table string, The database table name.
#' \item useWithoutSchema boolean, This attribute is no longer available; defaults to false but cannot be used.
#' }
#' \item file list . 
#' \item googleWorksheet list . A list containing the following elements: 
#' \itemize{
#' \item spreadsheet string, The spreadsheet document name.
#' \item spreadsheetId string, The spreadsheet document id.
#' \item worksheet string, The worksheet tab name.
#' \item worksheetId integer, The worksheet tab id.
#' }
#' \item salesforce list . A list containing the following elements: 
#' \itemize{
#' \item objectName string, The Salesforce object name.
#' }
#' }
#' @param destination list required. A list containing the following elements: 
#' \itemize{
#' \item path string, The schema.tablename to sync to. If you are doing a Google Sheet export, this is the spreadsheet and sheet name separated by a period. i.e. if you have a spreadsheet named "MySpreadsheet" and a sheet called "Sheet1" this field would be "MySpreadsheet.Sheet1". This is a legacy parameter, it is recommended you use one of the following: databaseTable, googleWorksheet
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The database schema name.
#' \item table string, The database table name.
#' \item useWithoutSchema boolean, This attribute is no longer available; defaults to false but cannot be used.
#' }
#' \item googleWorksheet list . A list containing the following elements: 
#' \itemize{
#' \item spreadsheet string, The spreadsheet document name.
#' \item spreadsheetId string, The spreadsheet document id.
#' \item worksheet string, The worksheet tab name.
#' \item worksheetId integer, The worksheet tab id.
#' }
#' }
#' @param advanced_options list optional. A list containing the following elements: 
#' \itemize{
#' \item maxErrors integer, 
#' \item existingTableRows string, 
#' \item diststyle string, 
#' \item distkey string, 
#' \item sortkey1 string, 
#' \item sortkey2 string, 
#' \item columnDelimiter string, 
#' \item columnOverrides object, Hash used for overriding auto-detected names and types, with keys being the index of the column being overridden.
#' \item escaped boolean, If true, escape quotes with a backslash; otherwise, escape quotes by double-quoting. Defaults to false.
#' \item identityColumn string, 
#' \item rowChunkSize integer, 
#' \item wipeDestinationTable boolean, 
#' \item truncateLongLines boolean, 
#' \item invalidCharReplacement string, 
#' \item verifyTableRowCounts boolean, 
#' \item partitionColumnName string, This parameter is deprecated
#' \item partitionSchemaName string, This parameter is deprecated
#' \item partitionTableName string, This parameter is deprecated
#' \item partitionTablePartitionColumnMinName string, This parameter is deprecated
#' \item partitionTablePartitionColumnMaxName string, This parameter is deprecated
#' \item lastModifiedColumn string, 
#' \item mysqlCatalogMatchesSchema boolean, This attribute is no longer available; defaults to true but cannot be used.
#' \item chunkingMethod string, This parameter is deprecated
#' \item firstRowIsHeader boolean, 
#' \item exportAction string, The kind of export action you want to have the export execute. Set to "newsprsht" if you want a new worksheet inside a new spreadsheet. Set to "newwksht" if you want a new worksheet inside an existing spreadsheet. Set to "updatewksht" if you want to overwrite an existing worksheet inside an existing spreadsheet. Set to "appendwksht" if you want to append to the end of an existing worksheet inside an existing spreadsheet. Default is set to "newsprsht"
#' \item sqlQuery string, If you are doing a Google Sheet export, this is your SQL query.
#' \item contactLists string, 
#' \item soqlQuery string, 
#' \item includeDeletedRecords boolean, 
#' }
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of the table or file, if available.
#' \item path string, The path of the dataset to sync from; for a database source, schema.tablename. If you are doing a Google Sheet export, this can be blank. This is a legacy parameter, it is recommended you use one of the following: databaseTable, file, googleWorksheet, salesforce
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The database schema name.
#' \item table string, The database table name.
#' \item useWithoutSchema boolean, This attribute is no longer available; defaults to false but cannot be used.
#' }
#' \item file list . A list containing the following elements: 
#' \itemize{
#' \item id integer, The file id.
#' }
#' \item googleWorksheet list . A list containing the following elements: 
#' \itemize{
#' \item spreadsheet string, The spreadsheet document name.
#' \item spreadsheetId string, The spreadsheet document id.
#' \item worksheet string, The worksheet tab name.
#' \item worksheetId integer, The worksheet tab id.
#' }
#' \item salesforce list . A list containing the following elements: 
#' \itemize{
#' \item objectName string, The Salesforce object name.
#' }
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item path string, The schema.tablename to sync to. If you are doing a Google Sheet export, this is the spreadsheet and sheet name separated by a period. i.e. if you have a spreadsheet named "MySpreadsheet" and a sheet called "Sheet1" this field would be "MySpreadsheet.Sheet1". This is a legacy parameter, it is recommended you use one of the following: databaseTable, googleWorksheet
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The database schema name.
#' \item table string, The database table name.
#' \item useWithoutSchema boolean, This attribute is no longer available; defaults to false but cannot be used.
#' }
#' \item googleWorksheet list . A list containing the following elements: 
#' \itemize{
#' \item spreadsheet string, The spreadsheet document name.
#' \item spreadsheetId string, The spreadsheet document id.
#' \item worksheet string, The worksheet tab name.
#' \item worksheetId integer, The worksheet tab id.
#' }
#' }}
#' \item{advancedOptions}{list, A list containing the following elements: 
#' \itemize{
#' \item maxErrors integer, 
#' \item existingTableRows string, 
#' \item diststyle string, 
#' \item distkey string, 
#' \item sortkey1 string, 
#' \item sortkey2 string, 
#' \item columnDelimiter string, 
#' \item columnOverrides object, Hash used for overriding auto-detected names and types, with keys being the index of the column being overridden.
#' \item escaped boolean, If true, escape quotes with a backslash; otherwise, escape quotes by double-quoting. Defaults to false.
#' \item identityColumn string, 
#' \item rowChunkSize integer, 
#' \item wipeDestinationTable boolean, 
#' \item truncateLongLines boolean, 
#' \item invalidCharReplacement string, 
#' \item verifyTableRowCounts boolean, 
#' \item partitionColumnName string, This parameter is deprecated
#' \item partitionSchemaName string, This parameter is deprecated
#' \item partitionTableName string, This parameter is deprecated
#' \item partitionTablePartitionColumnMinName string, This parameter is deprecated
#' \item partitionTablePartitionColumnMaxName string, This parameter is deprecated
#' \item lastModifiedColumn string, 
#' \item mysqlCatalogMatchesSchema boolean, This attribute is no longer available; defaults to true but cannot be used.
#' \item chunkingMethod string, This parameter is deprecated
#' \item firstRowIsHeader boolean, 
#' \item exportAction string, The kind of export action you want to have the export execute. Set to "newsprsht" if you want a new worksheet inside a new spreadsheet. Set to "newwksht" if you want a new worksheet inside an existing spreadsheet. Set to "updatewksht" if you want to overwrite an existing worksheet inside an existing spreadsheet. Set to "appendwksht" if you want to append to the end of an existing worksheet inside an existing spreadsheet. Default is set to "newsprsht"
#' \item sqlQuery string, If you are doing a Google Sheet export, this is your SQL query.
#' \item contactLists string, 
#' \item soqlQuery string, 
#' \item includeDeletedRecords boolean, 
#' }}
#' @export
imports_put_syncs <- function(id, sync_id, source, destination, advanced_options = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/syncs/{sync_id}"
  path_params  <- list(id = id, sync_id = sync_id)
  query_params <- list()
  body_params  <- list(source = source, destination = destination, advancedOptions = advanced_options)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive a sync (deprecated, use the /archive endpoint instead)
#' @param id integer required. The ID of the import to fetch.
#' @param sync_id integer required. The ID of the sync to fetch.
#' 
#' @return  An empty HTTP response
#' @export
imports_delete_syncs <- function(id, sync_id) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/syncs/{sync_id}"
  path_params  <- list(id = id, sync_id = sync_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this sync
#' @param id integer required. The ID of the import to fetch.
#' @param sync_id integer required. The ID of the sync to fetch.
#' @param status boolean optional. The desired archived status of the sync.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{source}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of the table or file, if available.
#' \item path string, The path of the dataset to sync from; for a database source, schema.tablename. If you are doing a Google Sheet export, this can be blank. This is a legacy parameter, it is recommended you use one of the following: databaseTable, file, googleWorksheet, salesforce
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The database schema name.
#' \item table string, The database table name.
#' \item useWithoutSchema boolean, This attribute is no longer available; defaults to false but cannot be used.
#' }
#' \item file list . A list containing the following elements: 
#' \itemize{
#' \item id integer, The file id.
#' }
#' \item googleWorksheet list . A list containing the following elements: 
#' \itemize{
#' \item spreadsheet string, The spreadsheet document name.
#' \item spreadsheetId string, The spreadsheet document id.
#' \item worksheet string, The worksheet tab name.
#' \item worksheetId integer, The worksheet tab id.
#' }
#' \item salesforce list . A list containing the following elements: 
#' \itemize{
#' \item objectName string, The Salesforce object name.
#' }
#' }}
#' \item{destination}{list, A list containing the following elements: 
#' \itemize{
#' \item path string, The schema.tablename to sync to. If you are doing a Google Sheet export, this is the spreadsheet and sheet name separated by a period. i.e. if you have a spreadsheet named "MySpreadsheet" and a sheet called "Sheet1" this field would be "MySpreadsheet.Sheet1". This is a legacy parameter, it is recommended you use one of the following: databaseTable, googleWorksheet
#' \item databaseTable list . A list containing the following elements: 
#' \itemize{
#' \item schema string, The database schema name.
#' \item table string, The database table name.
#' \item useWithoutSchema boolean, This attribute is no longer available; defaults to false but cannot be used.
#' }
#' \item googleWorksheet list . A list containing the following elements: 
#' \itemize{
#' \item spreadsheet string, The spreadsheet document name.
#' \item spreadsheetId string, The spreadsheet document id.
#' \item worksheet string, The worksheet tab name.
#' \item worksheetId integer, The worksheet tab id.
#' }
#' }}
#' \item{advancedOptions}{list, A list containing the following elements: 
#' \itemize{
#' \item maxErrors integer, 
#' \item existingTableRows string, 
#' \item diststyle string, 
#' \item distkey string, 
#' \item sortkey1 string, 
#' \item sortkey2 string, 
#' \item columnDelimiter string, 
#' \item columnOverrides object, Hash used for overriding auto-detected names and types, with keys being the index of the column being overridden.
#' \item escaped boolean, If true, escape quotes with a backslash; otherwise, escape quotes by double-quoting. Defaults to false.
#' \item identityColumn string, 
#' \item rowChunkSize integer, 
#' \item wipeDestinationTable boolean, 
#' \item truncateLongLines boolean, 
#' \item invalidCharReplacement string, 
#' \item verifyTableRowCounts boolean, 
#' \item partitionColumnName string, This parameter is deprecated
#' \item partitionSchemaName string, This parameter is deprecated
#' \item partitionTableName string, This parameter is deprecated
#' \item partitionTablePartitionColumnMinName string, This parameter is deprecated
#' \item partitionTablePartitionColumnMaxName string, This parameter is deprecated
#' \item lastModifiedColumn string, 
#' \item mysqlCatalogMatchesSchema boolean, This attribute is no longer available; defaults to true but cannot be used.
#' \item chunkingMethod string, This parameter is deprecated
#' \item firstRowIsHeader boolean, 
#' \item exportAction string, The kind of export action you want to have the export execute. Set to "newsprsht" if you want a new worksheet inside a new spreadsheet. Set to "newwksht" if you want a new worksheet inside an existing spreadsheet. Set to "updatewksht" if you want to overwrite an existing worksheet inside an existing spreadsheet. Set to "appendwksht" if you want to append to the end of an existing worksheet inside an existing spreadsheet. Default is set to "newsprsht"
#' \item sqlQuery string, If you are doing a Google Sheet export, this is your SQL query.
#' \item contactLists string, 
#' \item soqlQuery string, 
#' \item includeDeletedRecords boolean, 
#' }}
#' @export
imports_put_syncs_archive <- function(id, sync_id, status = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/imports/{id}/syncs/{sync_id}/archive"
  path_params  <- list(id = id, sync_id = sync_id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Jobs
#' @param state string optional. The job's state. One or more of queued, running, succeeded, failed, and cancelled. Specify multiple values as a comma-separated list (e.g., "A,B").
#' @param type string optional. The job's type. Specify multiple values as a comma-separated list (e.g., "A,B").
#' @param q string optional. Query string to search on the id, name, and job type.
#' @param permission string optional. A permissions string, one of "read", "write", or "manage". Lists only jobs for which the current user has that permission.
#' @param scheduled boolean optional. If the item is scheduled.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' @param archived string optional. The archival status of the requested item(s).
#' @param author string optional. If specified, return items from any of these authors. It accepts a comma-separated list of user IDs.
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, }
#' \item{name}{string, }
#' \item{type}{string, }
#' \item{fromTemplateId}{integer, }
#' \item{state}{string, Whether the job is idle, queued, running, cancelled, or failed.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' @export
jobs_list <- function(state = NULL, type = NULL, q = NULL, permission = NULL, scheduled = NULL, hidden = NULL, archived = NULL, author = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/"
  path_params  <- list()
  query_params <- list(state = state, type = type, q = q, permission = permission, scheduled = scheduled, hidden = hidden, archived = archived, author = author, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show basic job info
#' @param id integer required. The ID for this job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{name}{string, }
#' \item{type}{string, }
#' \item{fromTemplateId}{integer, }
#' \item{state}{string, Whether the job is idle, queued, running, cancelled, or failed.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{successEmailSubject}{string, }
#' \item{successEmailBody}{string, }
#' \item{runningAsUser}{string, }
#' \item{runByUser}{string, }
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' @export
jobs_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Generate and retrieve trigger email address
#' @param id integer required. The ID for this job.
#' 
#' @return  A list containing the following elements:
#' \item{triggerEmail}{string, Email address which may be used to trigger this job to run.}
#' @export
jobs_post_trigger_email <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/trigger_email"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show chain of parents as a list that this job triggers from
#' @param id integer required. The ID for this job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{name}{string, }
#' \item{type}{string, }
#' \item{fromTemplateId}{integer, }
#' \item{state}{string, Whether the job is idle, queued, running, cancelled, or failed.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{successEmailSubject}{string, }
#' \item{successEmailBody}{string, }
#' \item{runningAsUser}{string, }
#' \item{runByUser}{string, }
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' @export
jobs_list_parents <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/parents"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show nested tree of children that this job triggers
#' @param id integer required. The ID for this job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{name}{string, }
#' \item{type}{string, }
#' \item{fromTemplateId}{integer, }
#' \item{state}{string, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{children}{array, }
#' @export
jobs_list_children <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/children"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given job
#' @param id integer required. The ID for this job.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, }
#' \item{state}{string, }
#' \item{createdAt}{string, The time that the run was queued.}
#' \item{startedAt}{string, The time that the run started.}
#' \item{finishedAt}{string, The time that the run completed.}
#' \item{error}{string, The error message for this run, if present.}
#' @export
jobs_list_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Run a job
#' @param id integer required. The ID for this job.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{state}{string, }
#' \item{createdAt}{string, The time that the run was queued.}
#' \item{startedAt}{string, The time that the run started.}
#' \item{finishedAt}{string, The time that the run completed.}
#' \item{error}{string, The error message for this run, if present.}
#' @export
jobs_post_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a job
#' @param id integer required. The ID of the Job.
#' @param run_id integer required. The ID of the Run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{state}{string, }
#' \item{createdAt}{string, The time that the run was queued.}
#' \item{startedAt}{string, The time that the run started.}
#' \item{finishedAt}{string, The time that the run completed.}
#' \item{error}{string, The error message for this run, if present.}
#' @export
jobs_get_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the Job.
#' @param run_id integer required. The ID of the Run.
#' 
#' @return  An empty HTTP response
#' @export
jobs_delete_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the outputs for a run
#' @param id integer required. The ID of the job.
#' @param run_id integer required. The ID of the run.
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to created_at. Must be one of: created_at, id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, The type of the output. Valid values are File, Table, Report, Project, Credential, or JSONValue}
#' \item{objectId}{integer, The ID of the output.}
#' \item{name}{string, The name of the output.}
#' \item{link}{string, The hypermedia link to the output.}
#' \item{value}{string, }
#' @export
jobs_list_runs_outputs <- function(id, run_id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/runs/{run_id}/outputs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the job.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
jobs_list_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the workflows a job belongs to
#' @param id integer required. 
#' @param archived string optional. The archival status of the requested item(s).
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this workflow.}
#' \item{name}{string, The name of this workflow.}
#' \item{description}{string, A description of the workflow.}
#' \item{valid}{boolean, The validity of the workflow definition.}
#' \item{fileId}{string, The file id for the s3 file containing the workflow configuration.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{state}{string, The state of the workflow. State is "running" if any execution is running, otherwise reflects most recent execution state.}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{allowConcurrentExecutions}{boolean, Whether the workflow can execute when already running.}
#' \item{timeZone}{string, The time zone of this workflow.}
#' \item{nextExecutionAt}{string, The time of the next scheduled execution.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
jobs_list_workflows <- function(id, archived = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/workflows"
  path_params  <- list(id = id)
  query_params <- list(archived = archived)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
jobs_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
jobs_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
jobs_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
jobs_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
jobs_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
jobs_list_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
jobs_put_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a Job belongs to
#' @param id integer required. The ID of the Job.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
jobs_list_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a Job to a project
#' @param id integer required. The ID of the Job.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
jobs_put_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a Job from a project
#' @param id integer required. The ID of the Job.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
jobs_delete_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, }
#' \item{name}{string, }
#' \item{type}{string, }
#' \item{fromTemplateId}{integer, }
#' \item{state}{string, Whether the job is idle, queued, running, cancelled, or failed.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{successEmailSubject}{string, }
#' \item{successEmailBody}{string, }
#' \item{runningAsUser}{string, }
#' \item{runByUser}{string, }
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' @export
jobs_put_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/jobs/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a JSON Value
#' @param value_str string required. The JSON value to store. Should be a serialized JSON string. Limited to 1000000 bytes.
#' @param name string optional. The name of the JSON Value.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the JSON Value.}
#' \item{name}{string, The name of the JSON Value.}
#' \item{value}{string, The deserialized JSON value.}
#' @export
json_values_post <- function(value_str, name = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/json_values/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(valueStr = value_str, name = name)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about a JSON Value
#' @param id integer required. The ID of the JSON Value.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the JSON Value.}
#' \item{name}{string, The name of the JSON Value.}
#' \item{value}{string, The deserialized JSON value.}
#' @export
json_values_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/json_values/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this JSON Value
#' @param id integer required. The ID of the JSON Value.
#' @param name string optional. The name of the JSON Value.
#' @param value_str string optional. The JSON value to store. Should be a serialized JSON string. Limited to 1000000 bytes.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the JSON Value.}
#' \item{name}{string, The name of the JSON Value.}
#' \item{value}{string, The deserialized JSON value.}
#' @export
json_values_patch <- function(id, name = NULL, value_str = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/json_values/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, valueStr = value_str)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
json_values_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/json_values/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
json_values_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/json_values/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
json_values_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/json_values/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
json_values_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/json_values/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
json_values_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/json_values/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
json_values_list_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/json_values/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
json_values_put_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/json_values/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
match_targets_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/match_targets/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
match_targets_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/match_targets/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
match_targets_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/match_targets/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
match_targets_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/match_targets/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
match_targets_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/match_targets/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the match target}
#' \item{name}{string, The name of the match target}
#' \item{targetFileName}{string, The name of the target file}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{boolean, Whether the match target has been archived.}
#' @export
match_targets_put_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/match_targets/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List match targets
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the match target}
#' \item{name}{string, The name of the match target}
#' \item{targetFileName}{string, The name of the target file}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{boolean, Whether the match target has been archived.}
#' @export
match_targets_list <- function() {

  args <- as.list(match.call())[-1]
  path <- "/match_targets/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a new match target
#' @param name string required. The name of the match target
#' @param target_file_name string optional. The name of the target file
#' @param archived boolean optional. Whether the match target has been archived.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the match target}
#' \item{name}{string, The name of the match target}
#' \item{targetFileName}{string, The name of the target file}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{boolean, Whether the match target has been archived.}
#' @export
match_targets_post <- function(name, target_file_name = NULL, archived = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/match_targets/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, targetFileName = target_file_name, archived = archived)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show Match Target info
#' @param id integer required. The ID of the match target
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the match target}
#' \item{name}{string, The name of the match target}
#' \item{targetFileName}{string, The name of the target file}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{boolean, Whether the match target has been archived.}
#' @export
match_targets_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/match_targets/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a match target
#' @param id integer required. The ID of the match target
#' @param name string optional. The name of the match target
#' @param target_file_name string optional. The name of the target file
#' @param archived boolean optional. Whether the match target has been archived.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the match target}
#' \item{name}{string, The name of the match target}
#' \item{targetFileName}{string, The name of the target file}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{boolean, Whether the match target has been archived.}
#' @export
match_targets_patch <- function(id, name = NULL, target_file_name = NULL, archived = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/match_targets/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, targetFileName = target_file_name, archived = archived)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_list_spot_orders_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_put_spot_orders_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
media_delete_spot_orders_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_put_spot_orders_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
media_delete_spot_orders_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the spot order.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{csvS3Uri}{string, S3 URI for the spot order CSV file.}
#' \item{jsonS3Uri}{string, S3 URI for the spot order JSON file.}
#' \item{xmlArchiveS3Uri}{string, S3 URI for the spot order XML archive.}
#' \item{lastTransformJobId}{integer, ID of the spot order transformation job.}
#' @export
media_put_spot_orders_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_list_optimizations_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_put_optimizations_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
media_delete_optimizations_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_put_optimizations_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
media_delete_optimizations_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The optimization ID.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of the optimization.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{spotOrderId}{integer, The ID for the spot order produced by the optimization.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{reportLink}{string, A link to the visual report for the optimization.}
#' \item{spotOrderLink}{string, A link to the json version of the spot order.}
#' \item{fileLinks}{array, Links to the csv and xml versions of the spot order.}
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item marketId integer, The market ID.
#' \item startDate string, The start date for the media run.
#' \item endDate string, The end date for the media run.
#' \item forceCpm boolean, Whether to force optimization to use CPM data even if partition data is available.
#' \item reachAlpha number, A tuning parameter used to adjust RF.
#' \item syscodes array, The syscodes for the media run.
#' \item rateCards array, The ratecards for the media run.
#' \item constraints array, The constraints for the media run.
#' }}
#' \item{programs}{array, An array of programs that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_programs is not also set.}
#' \item{networks}{array, An array of networks that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_networks is not also set.}
#' \item{excludePrograms}{boolean, If Civis Media Optimizer should exclude the programs in the programs parameter.If this value is set to false, it will make the optimization limit itself to the programs supplied through the programs parameter.An error will be thrown if programs is not also set.}
#' \item{excludeNetworks}{boolean, If Civis Media Optimizer should exclude the networks in the networks parameter.If this value is set to false, it will make the optimization limit itself to the networks supplied through the networks.An error will be thrown if networks is not also set.}
#' \item{timeSlotPercentages}{list, The maximum amount of the budget spent on that particular day of the week, daypart, or specific time slot for broadcast and cable.}
#' @export
media_put_optimizations_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_list_ratecards_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_put_ratecards_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
media_delete_ratecards_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
media_put_ratecards_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
media_delete_ratecards_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ratecard ID.}
#' \item{filename}{string, Name of the ratecard file.}
#' \item{startOn}{string, First day to which the ratecard applies.}
#' \item{endOn}{string, Last day to which the ratecard applies.}
#' \item{dmaNumber}{integer, Number of the DMA associated with the ratecard.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
media_put_ratecards_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List all optimizations
#' @param archived string optional. The archival status of the requested item(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to created_at. Must be one of: created_at, author, name.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The optimization ID.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of the optimization.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{spotOrderId}{integer, The ID for the spot order produced by the optimization.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
media_list_optimizations <- function(archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations"
  path_params  <- list()
  query_params <- list(archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a new optimization
#' @param runs array required. An array containing the following fields: 
#' \itemize{
#' \item marketId integer, The market ID.
#' \item startDate string, The start date for the media run.
#' \item endDate string, The end date for the media run.
#' \item forceCpm boolean, Whether to force optimization to use CPM data even if partition data is available.
#' \item reachAlpha number, A tuning parameter used to adjust RF.
#' \item syscodes array, The syscodes for the media run.
#' \item rateCards array, The ratecards for the media run.
#' \item constraints array, The constraints for the media run.
#' }
#' @param name string optional. The name of the optimization.
#' @param programs array optional. An array of programs that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_programs is not also set.
#' @param networks array optional. An array of networks that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_networks is not also set.
#' @param exclude_programs boolean optional. If Civis Media Optimizer should exclude the programs in the programs parameter.If this value is set to false, it will make the optimization limit itself to the programs supplied through the programs parameter.An error will be thrown if programs is not also set.
#' @param exclude_networks boolean optional. If Civis Media Optimizer should exclude the networks in the networks parameter.If this value is set to false, it will make the optimization limit itself to the networks supplied through the networks.An error will be thrown if networks is not also set.
#' @param time_slot_percentages list optional. The maximum amount of the budget spent on that particular day of the week, daypart, or specific time slot for broadcast and cable.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The optimization ID.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of the optimization.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{spotOrderId}{integer, The ID for the spot order produced by the optimization.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{reportLink}{string, A link to the visual report for the optimization.}
#' \item{spotOrderLink}{string, A link to the json version of the spot order.}
#' \item{fileLinks}{array, Links to the csv and xml versions of the spot order.}
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item marketId integer, The market ID.
#' \item startDate string, The start date for the media run.
#' \item endDate string, The end date for the media run.
#' \item forceCpm boolean, Whether to force optimization to use CPM data even if partition data is available.
#' \item reachAlpha number, A tuning parameter used to adjust RF.
#' \item syscodes array, The syscodes for the media run.
#' \item rateCards array, The ratecards for the media run.
#' \item constraints array, The constraints for the media run.
#' }}
#' \item{programs}{array, An array of programs that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_programs is not also set.}
#' \item{networks}{array, An array of networks that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_networks is not also set.}
#' \item{excludePrograms}{boolean, If Civis Media Optimizer should exclude the programs in the programs parameter.If this value is set to false, it will make the optimization limit itself to the programs supplied through the programs parameter.An error will be thrown if programs is not also set.}
#' \item{excludeNetworks}{boolean, If Civis Media Optimizer should exclude the networks in the networks parameter.If this value is set to false, it will make the optimization limit itself to the networks supplied through the networks.An error will be thrown if networks is not also set.}
#' \item{timeSlotPercentages}{list, The maximum amount of the budget spent on that particular day of the week, daypart, or specific time slot for broadcast and cable.}
#' @export
media_post_optimizations <- function(runs, name = NULL, programs = NULL, networks = NULL, exclude_programs = NULL, exclude_networks = NULL, time_slot_percentages = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(runs = runs, name = name, programs = programs, networks = networks, excludePrograms = exclude_programs, excludeNetworks = exclude_networks, timeSlotPercentages = time_slot_percentages)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show a single optimization
#' @param id integer required. The optimization ID.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The optimization ID.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of the optimization.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{spotOrderId}{integer, The ID for the spot order produced by the optimization.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{reportLink}{string, A link to the visual report for the optimization.}
#' \item{spotOrderLink}{string, A link to the json version of the spot order.}
#' \item{fileLinks}{array, Links to the csv and xml versions of the spot order.}
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item marketId integer, The market ID.
#' \item startDate string, The start date for the media run.
#' \item endDate string, The end date for the media run.
#' \item forceCpm boolean, Whether to force optimization to use CPM data even if partition data is available.
#' \item reachAlpha number, A tuning parameter used to adjust RF.
#' \item syscodes array, The syscodes for the media run.
#' \item rateCards array, The ratecards for the media run.
#' \item constraints array, The constraints for the media run.
#' }}
#' \item{programs}{array, An array of programs that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_programs is not also set.}
#' \item{networks}{array, An array of networks that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_networks is not also set.}
#' \item{excludePrograms}{boolean, If Civis Media Optimizer should exclude the programs in the programs parameter.If this value is set to false, it will make the optimization limit itself to the programs supplied through the programs parameter.An error will be thrown if programs is not also set.}
#' \item{excludeNetworks}{boolean, If Civis Media Optimizer should exclude the networks in the networks parameter.If this value is set to false, it will make the optimization limit itself to the networks supplied through the networks.An error will be thrown if networks is not also set.}
#' \item{timeSlotPercentages}{list, The maximum amount of the budget spent on that particular day of the week, daypart, or specific time slot for broadcast and cable.}
#' @export
media_get_optimizations <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Edit an existing optimization
#' @param id integer required. The optimization ID.
#' @param name string optional. The name of the optimization.
#' @param runs array optional. An array containing the following fields: 
#' \itemize{
#' \item marketId integer, The market ID.
#' \item startDate string, The start date for the media run.
#' \item endDate string, The end date for the media run.
#' \item forceCpm boolean, Whether to force optimization to use CPM data even if partition data is available.
#' \item reachAlpha number, A tuning parameter used to adjust RF.
#' \item syscodes array, The syscodes for the media run.
#' \item rateCards array, The ratecards for the media run.
#' \item constraints array, The constraints for the media run.
#' }
#' @param programs array optional. An array of programs that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_programs is not also set.
#' @param networks array optional. An array of networks that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_networks is not also set.
#' @param exclude_programs boolean optional. If Civis Media Optimizer should exclude the programs in the programs parameter.If this value is set to false, it will make the optimization limit itself to the programs supplied through the programs parameter.An error will be thrown if programs is not also set.
#' @param exclude_networks boolean optional. If Civis Media Optimizer should exclude the networks in the networks parameter.If this value is set to false, it will make the optimization limit itself to the networks supplied through the networks.An error will be thrown if networks is not also set.
#' @param time_slot_percentages list optional. The maximum amount of the budget spent on that particular day of the week, daypart, or specific time slot for broadcast and cable.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The optimization ID.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of the optimization.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{spotOrderId}{integer, The ID for the spot order produced by the optimization.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{reportLink}{string, A link to the visual report for the optimization.}
#' \item{spotOrderLink}{string, A link to the json version of the spot order.}
#' \item{fileLinks}{array, Links to the csv and xml versions of the spot order.}
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item marketId integer, The market ID.
#' \item startDate string, The start date for the media run.
#' \item endDate string, The end date for the media run.
#' \item forceCpm boolean, Whether to force optimization to use CPM data even if partition data is available.
#' \item reachAlpha number, A tuning parameter used to adjust RF.
#' \item syscodes array, The syscodes for the media run.
#' \item rateCards array, The ratecards for the media run.
#' \item constraints array, The constraints for the media run.
#' }}
#' \item{programs}{array, An array of programs that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_programs is not also set.}
#' \item{networks}{array, An array of networks that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_networks is not also set.}
#' \item{excludePrograms}{boolean, If Civis Media Optimizer should exclude the programs in the programs parameter.If this value is set to false, it will make the optimization limit itself to the programs supplied through the programs parameter.An error will be thrown if programs is not also set.}
#' \item{excludeNetworks}{boolean, If Civis Media Optimizer should exclude the networks in the networks parameter.If this value is set to false, it will make the optimization limit itself to the networks supplied through the networks.An error will be thrown if networks is not also set.}
#' \item{timeSlotPercentages}{list, The maximum amount of the budget spent on that particular day of the week, daypart, or specific time slot for broadcast and cable.}
#' @export
media_patch_optimizations <- function(id, name = NULL, runs = NULL, programs = NULL, networks = NULL, exclude_programs = NULL, exclude_networks = NULL, time_slot_percentages = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, runs = runs, programs = programs, networks = networks, excludePrograms = exclude_programs, excludeNetworks = exclude_networks, timeSlotPercentages = time_slot_percentages)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Clone an existing optimization
#' @param id integer required. The optimization ID.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The optimization ID.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of the optimization.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{spotOrderId}{integer, The ID for the spot order produced by the optimization.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{reportLink}{string, A link to the visual report for the optimization.}
#' \item{spotOrderLink}{string, A link to the json version of the spot order.}
#' \item{fileLinks}{array, Links to the csv and xml versions of the spot order.}
#' \item{runs}{array, An array containing the following fields: 
#' \itemize{
#' \item marketId integer, The market ID.
#' \item startDate string, The start date for the media run.
#' \item endDate string, The end date for the media run.
#' \item forceCpm boolean, Whether to force optimization to use CPM data even if partition data is available.
#' \item reachAlpha number, A tuning parameter used to adjust RF.
#' \item syscodes array, The syscodes for the media run.
#' \item rateCards array, The ratecards for the media run.
#' \item constraints array, The constraints for the media run.
#' }}
#' \item{programs}{array, An array of programs that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_programs is not also set.}
#' \item{networks}{array, An array of networks that the Civis Media Optimizer either exclude or limit to.An error will be thrown if exclude_networks is not also set.}
#' \item{excludePrograms}{boolean, If Civis Media Optimizer should exclude the programs in the programs parameter.If this value is set to false, it will make the optimization limit itself to the programs supplied through the programs parameter.An error will be thrown if programs is not also set.}
#' \item{excludeNetworks}{boolean, If Civis Media Optimizer should exclude the networks in the networks parameter.If this value is set to false, it will make the optimization limit itself to the networks supplied through the networks.An error will be thrown if networks is not also set.}
#' \item{timeSlotPercentages}{list, The maximum amount of the budget spent on that particular day of the week, daypart, or specific time slot for broadcast and cable.}
#' @export
media_post_optimizations_clone <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/clone"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the optimization.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{optimizationId}{integer, The ID of the optimization.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
media_post_optimizations_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given optimization
#' @param id integer required. The ID of the optimization.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the run.}
#' \item{optimizationId}{integer, The ID of the optimization.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
media_list_optimizations_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the optimization.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{optimizationId}{integer, The ID of the optimization.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
media_get_optimizations_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the optimization.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
media_delete_optimizations_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the optimization.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
media_list_optimizations_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/optimizations/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List all spot orders
#' @param id integer optional. The ID for the spot order.
#' @param archived string optional. The archival status of the requested item(s).
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for the spot order.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
media_list_spot_orders <- function(id = NULL, archived = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders"
  path_params  <- list()
  query_params <- list(id = id, archived = archived)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a spot order
#' @param body string optional. CSV body of a spot order.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the spot order.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{csvS3Uri}{string, S3 URI for the spot order CSV file.}
#' \item{jsonS3Uri}{string, S3 URI for the spot order JSON file.}
#' \item{xmlArchiveS3Uri}{string, S3 URI for the spot order XML archive.}
#' \item{lastTransformJobId}{integer, ID of the spot order transformation job.}
#' @export
media_post_spot_orders <- function(body = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(body = body)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show a single spot order
#' @param id integer required. The ID for the spot order.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the spot order.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{csvS3Uri}{string, S3 URI for the spot order CSV file.}
#' \item{jsonS3Uri}{string, S3 URI for the spot order JSON file.}
#' \item{xmlArchiveS3Uri}{string, S3 URI for the spot order XML archive.}
#' \item{lastTransformJobId}{integer, ID of the spot order transformation job.}
#' @export
media_get_spot_orders <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Edit the specified spot order
#' @param id integer required. The ID for the spot order.
#' @param body string optional. CSV body of a spot order.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for the spot order.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{csvS3Uri}{string, S3 URI for the spot order CSV file.}
#' \item{jsonS3Uri}{string, S3 URI for the spot order JSON file.}
#' \item{xmlArchiveS3Uri}{string, S3 URI for the spot order XML archive.}
#' \item{lastTransformJobId}{integer, ID of the spot order transformation job.}
#' @export
media_put_spot_orders <- function(id, body = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/spot_orders/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(body = body)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List all ratecards
#' @param archived string optional. The archival status of the requested item(s).
#' @param filename string optional. If specified, will be used to filter the ratecards returned. Substring matching is supported with "\%" and "*" wildcards (e.g., "filename=\%ratecard\%" will return both "ratecard 1" and "my ratecard").
#' @param dma_number integer optional. If specified, will be used to filter the ratecards by DMA.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ratecard ID.}
#' \item{filename}{string, Name of the ratecard file.}
#' \item{startOn}{string, First day to which the ratecard applies.}
#' \item{endOn}{string, Last day to which the ratecard applies.}
#' \item{dmaNumber}{integer, Number of the DMA associated with the ratecard.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
media_list_ratecards <- function(archived = NULL, filename = NULL, dma_number = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards"
  path_params  <- list()
  query_params <- list(archived = archived, filename = filename, dma_number = dma_number)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Ratecard
#' @param filename string required. Name of the ratecard file.
#' @param start_on string required. First day to which the ratecard applies.
#' @param end_on string required. Last day to which the ratecard applies.
#' @param dma_number integer required. Number of the DMA associated with the ratecard.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ratecard ID.}
#' \item{filename}{string, Name of the ratecard file.}
#' \item{startOn}{string, First day to which the ratecard applies.}
#' \item{endOn}{string, Last day to which the ratecard applies.}
#' \item{dmaNumber}{integer, Number of the DMA associated with the ratecard.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
media_post_ratecards <- function(filename, start_on, end_on, dma_number) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(filename = filename, startOn = start_on, endOn = end_on, dmaNumber = dma_number)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a Ratecard
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ratecard ID.}
#' \item{filename}{string, Name of the ratecard file.}
#' \item{startOn}{string, First day to which the ratecard applies.}
#' \item{endOn}{string, Last day to which the ratecard applies.}
#' \item{dmaNumber}{integer, Number of the DMA associated with the ratecard.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
media_get_ratecards <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this Ratecard
#' @param id integer required. The ratecard ID.
#' @param filename string required. Name of the ratecard file.
#' @param start_on string required. First day to which the ratecard applies.
#' @param end_on string required. Last day to which the ratecard applies.
#' @param dma_number integer required. Number of the DMA associated with the ratecard.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ratecard ID.}
#' \item{filename}{string, Name of the ratecard file.}
#' \item{startOn}{string, First day to which the ratecard applies.}
#' \item{endOn}{string, Last day to which the ratecard applies.}
#' \item{dmaNumber}{integer, Number of the DMA associated with the ratecard.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
media_put_ratecards <- function(id, filename, start_on, end_on, dma_number) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(filename = filename, startOn = start_on, endOn = end_on, dmaNumber = dma_number)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this Ratecard
#' @param id integer required. The ratecard ID.
#' @param filename string optional. Name of the ratecard file.
#' @param start_on string optional. First day to which the ratecard applies.
#' @param end_on string optional. Last day to which the ratecard applies.
#' @param dma_number integer optional. Number of the DMA associated with the ratecard.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ratecard ID.}
#' \item{filename}{string, Name of the ratecard file.}
#' \item{startOn}{string, First day to which the ratecard applies.}
#' \item{endOn}{string, Last day to which the ratecard applies.}
#' \item{dmaNumber}{integer, Number of the DMA associated with the ratecard.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
media_patch_ratecards <- function(id, filename = NULL, start_on = NULL, end_on = NULL, dma_number = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/ratecards/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(filename = filename, startOn = start_on, endOn = end_on, dmaNumber = dma_number)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' List all Designated Market Areas
#' @param name string optional. If specified, will be used to filter the DMAs returned. Substring matching is supported with "\%" and "*" wildcards (e.g., "name=\%region\%" will return both "region1" and "my region").
#' @param number integer optional. If specified, will be used to filter the DMAS by number.
#' 
#' @return  An array containing the following fields:
#' \item{name}{string, Name for the DMA region.}
#' \item{number}{integer, Identifier number for a DMA.}
#' @export
media_list_dmas <- function(name = NULL, number = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/dmas"
  path_params  <- list()
  query_params <- list(name = name, number = number)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List all Media Targets
#' @param name string optional. The name of the target.
#' @param identifier string optional. A unique identifier for this target.
#' @param data_source string optional. The source of viewership data for this target.
#' 
#' @return  An array containing the following fields:
#' \item{name}{string, The name of the target.}
#' \item{identifier}{string, A unique identifier for this target.}
#' \item{dataSource}{string, The source of viewership data for this target.}
#' @export
media_list_targets <- function(name = NULL, identifier = NULL, data_source = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/media/targets"
  path_params  <- list()
  query_params <- list(name = name, identifier = identifier, data_source = data_source)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List all available model types
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the model type.}
#' \item{algorithm}{string, The name of the algorithm used to train the model.}
#' \item{dvType}{string, The type of dependent variable predicted by the model.}
#' \item{fintAllowed}{boolean, Whether this model type supports searching for interaction terms.}
#' @export
models_list_types <- function() {

  args <- as.list(match.call())[-1]
  path <- "/models/types"
  path_params  <- list()
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List 
#' @param model_name string optional. If specified, will be used to filter the models returned. Substring matching is supported. (e.g., "modelName=model" will return both "model1" and "my model").
#' @param training_table_name string optional. If specified, will be used to filter the models returned by the training dataset table name. Substring matching is supported. (e.g., "trainingTableName=table" will return both "table1" and "my_table").
#' @param dependent_variable string optional. If specified, will be used to filter the models returned by the dependent variable column name. Substring matching is supported. (e.g., "dependentVariable=predictor" will return both "predictor" and "my predictor").
#' @param status string optional. If specified, returns models with one of these statuses. It accepts a comma-separated list, possible values are 'running', 'failed', 'succeeded', 'idle', 'scheduled'.
#' @param author string optional. If specified, return items from any of these authors. It accepts a comma-separated list of user IDs.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' @param archived string optional. The archival status of the requested item(s).
#' @param limit integer optional. Number of results to return. Defaults to its maximum of 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at, last_run.updated_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the model.}
#' \item{tableName}{string, The qualified name of the table containing the training set from which to build the model.}
#' \item{databaseId}{integer, The ID of the database holding the training set table used to build the model.}
#' \item{credentialId}{integer, The ID of the credential used to read the target table. Defaults to the user's default credential.}
#' \item{modelName}{string, The name of the model.}
#' \item{description}{string, A description of the model.}
#' \item{interactionTerms}{boolean, Whether to search for interaction terms.}
#' \item{boxCoxTransformation}{boolean, Whether to transform data so that it assumes a normal distribution. Valid only with continuous models.}
#' \item{modelTypeId}{integer, The ID of the model's type.}
#' \item{primaryKey}{string, The unique ID (primary key) of the training dataset.}
#' \item{dependentVariable}{string, The dependent variable of the training dataset.}
#' \item{dependentVariableOrder}{array, The order of dependent variables, especially useful for Ordinal Modeling.}
#' \item{excludedColumns}{array, A list of columns which will be considered ineligible to be independent variables.}
#' \item{limitingSQL}{string, A custom SQL WHERE clause used to filter the rows used to build the model. (e.g., "id > 105").}
#' \item{crossValidationParameters}{list, Cross validation parameter grid for tree methods, e.g. {"n_estimators": [100, 200, 500], "learning_rate": [0.01, 0.1], "max_depth": [2, 3]}.}
#' \item{numberOfFolds}{integer, Number of folds for cross validation. Default value is 5.}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this model.}
#' \item{timeZone}{string, The time zone of this model.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, The time the model was created.}
#' \item{updatedAt}{string, The time the model was updated.}
#' \item{currentBuildState}{string, The status of the current model build. One of "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.}
#' \item{currentBuildException}{string, Exception message, if applicable, of the current model build.}
#' \item{builds}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the model build.
#' \item name string, The name of the model build.
#' \item createdAt string, The time the model build was created.
#' \item description string, A description of the model build.
#' \item rootMeanSquaredError number, A key metric for continuous models. Nil for other model types.
#' \item rSquaredError number, A key metric for continuous models. Nil for other model types.
#' \item rocAuc number, A key metric for binary, multinomial, and ordinal models. Nil for other model types.
#' }}
#' \item{predictions}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the model to which to apply the prediction.
#' \item tableName string, The qualified name of the table on which to apply the predictive model.
#' \item primaryKey array, The primary key or composite keys of the table being predicted.
#' \item limitingSQL string, A SQL WHERE clause used to scope the rows to be predicted.
#' \item outputTable string, The qualified name of the table to be created which will contain the model's predictions.
#' \item state string, The status of the prediction. One of: "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.
#' }}
#' \item{lastOutputLocation}{string, The output JSON for the last build.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
models_list <- function(model_name = NULL, training_table_name = NULL, dependent_variable = NULL, status = NULL, author = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/"
  path_params  <- list()
  query_params <- list(model_name = model_name, training_table_name = training_table_name, dependent_variable = dependent_variable, status = status, author = author, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Retrieve model configuration
#' @param id integer required. The ID of the model.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the model.}
#' \item{tableName}{string, The qualified name of the table containing the training set from which to build the model.}
#' \item{databaseId}{integer, The ID of the database holding the training set table used to build the model.}
#' \item{credentialId}{integer, The ID of the credential used to read the target table. Defaults to the user's default credential.}
#' \item{modelName}{string, The name of the model.}
#' \item{description}{string, A description of the model.}
#' \item{interactionTerms}{boolean, Whether to search for interaction terms.}
#' \item{boxCoxTransformation}{boolean, Whether to transform data so that it assumes a normal distribution. Valid only with continuous models.}
#' \item{modelTypeId}{integer, The ID of the model's type.}
#' \item{primaryKey}{string, The unique ID (primary key) of the training dataset.}
#' \item{dependentVariable}{string, The dependent variable of the training dataset.}
#' \item{dependentVariableOrder}{array, The order of dependent variables, especially useful for Ordinal Modeling.}
#' \item{excludedColumns}{array, A list of columns which will be considered ineligible to be independent variables.}
#' \item{limitingSQL}{string, A custom SQL WHERE clause used to filter the rows used to build the model. (e.g., "id > 105").}
#' \item{activeBuildId}{integer, The ID of the current active build, the build used to score predictions.}
#' \item{crossValidationParameters}{list, Cross validation parameter grid for tree methods, e.g. {"n_estimators": [100, 200, 500], "learning_rate": [0.01, 0.1], "max_depth": [2, 3]}.}
#' \item{numberOfFolds}{integer, Number of folds for cross validation. Default value is 5.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this model.}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{timeZone}{string, The time zone of this model.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, The time the model was created.}
#' \item{updatedAt}{string, The time the model was updated.}
#' \item{currentBuildState}{string, The status of the current model build. One of "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.}
#' \item{currentBuildException}{string, Exception message, if applicable, of the current model build.}
#' \item{builds}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the model build.
#' \item name string, The name of the model build.
#' \item createdAt string, The time the model build was created.
#' \item description string, A description of the model build.
#' \item rootMeanSquaredError number, A key metric for continuous models. Nil for other model types.
#' \item rSquaredError number, A key metric for continuous models. Nil for other model types.
#' \item rocAuc number, A key metric for binary, multinomial, and ordinal models. Nil for other model types.
#' }}
#' \item{predictions}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the model to which to apply the prediction.
#' \item tableName string, The qualified name of the table on which to apply the predictive model.
#' \item primaryKey array, The primary key or composite keys of the table being predicted.
#' \item limitingSQL string, A SQL WHERE clause used to scope the rows to be predicted.
#' \item outputTable string, The qualified name of the table to be created which will contain the model's predictions.
#' \item schedule object, 
#' \item state string, The status of the prediction. One of: "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.
#' }}
#' \item{lastOutputLocation}{string, The output JSON for the last build.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
models_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a build
#' @param id integer required. The ID of the model.
#' @param build_id integer required. The ID of the build.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the model build.}
#' \item{state}{string, The state of the model build.one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{error}{string, The error, if any, returned by the build.}
#' \item{name}{string, The name of the model build.}
#' \item{createdAt}{string, The time the model build was created.}
#' \item{description}{string, A description of the model build.}
#' \item{rootMeanSquaredError}{number, A key metric for continuous models. Nil for other model types.}
#' \item{rSquaredError}{number, A key metric for continuous models. Nil for other model types.}
#' \item{rocAuc}{number, A key metric for binary, multinomial, and ordinal models. Nil for other model types.}
#' \item{transformationMetadata}{string, A string representing the full JSON output of the metadata for transformation of column names}
#' \item{output}{string, A string representing the JSON output for the specified build. Only present when smaller than 10KB in size.}
#' \item{outputLocation}{string, A URL representing the location of the full JSON output for the specified build.The URL link will be valid for 5 minutes.}
#' @export
models_get_builds <- function(id, build_id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/builds/{build_id}"
  path_params  <- list(id = id, build_id = build_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a build
#' @param id integer required. The ID of the model.
#' @param build_id integer required. The ID of the build.
#' 
#' @return  An empty HTTP response
#' @export
models_delete_builds <- function(id, build_id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/builds/{build_id}"
  path_params  <- list(id = id, build_id = build_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List builds for the given model
#' @param id integer required. The ID of the model.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the model build.}
#' \item{state}{string, The state of the model build.one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{error}{string, The error, if any, returned by the build.}
#' \item{name}{string, The name of the model build.}
#' \item{createdAt}{string, The time the model build was created.}
#' \item{description}{string, A description of the model build.}
#' \item{rootMeanSquaredError}{number, A key metric for continuous models. Nil for other model types.}
#' \item{rSquaredError}{number, A key metric for continuous models. Nil for other model types.}
#' \item{rocAuc}{number, A key metric for binary, multinomial, and ordinal models. Nil for other model types.}
#' \item{transformationMetadata}{string, A string representing the full JSON output of the metadata for transformation of column names}
#' \item{output}{string, A string representing the JSON output for the specified build. Only present when smaller than 10KB in size.}
#' \item{outputLocation}{string, A URL representing the location of the full JSON output for the specified build.The URL link will be valid for 5 minutes.}
#' @export
models_list_builds <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/builds"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a build
#' @param id integer required. The ID of the model.
#' @param build_id integer required. The ID of the build.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
models_list_builds_logs <- function(id, build_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/builds/{build_id}/logs"
  path_params  <- list(id = id, build_id = build_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
models_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
models_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
models_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
models_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
models_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
models_list_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
models_put_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a Model belongs to
#' @param id integer required. The ID of the Model.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
models_list_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a Model to a project
#' @param id integer required. The ID of the Model.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
models_put_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a Model from a project
#' @param id integer required. The ID of the Model.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
models_delete_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the model.}
#' \item{tableName}{string, The qualified name of the table containing the training set from which to build the model.}
#' \item{databaseId}{integer, The ID of the database holding the training set table used to build the model.}
#' \item{credentialId}{integer, The ID of the credential used to read the target table. Defaults to the user's default credential.}
#' \item{modelName}{string, The name of the model.}
#' \item{description}{string, A description of the model.}
#' \item{interactionTerms}{boolean, Whether to search for interaction terms.}
#' \item{boxCoxTransformation}{boolean, Whether to transform data so that it assumes a normal distribution. Valid only with continuous models.}
#' \item{modelTypeId}{integer, The ID of the model's type.}
#' \item{primaryKey}{string, The unique ID (primary key) of the training dataset.}
#' \item{dependentVariable}{string, The dependent variable of the training dataset.}
#' \item{dependentVariableOrder}{array, The order of dependent variables, especially useful for Ordinal Modeling.}
#' \item{excludedColumns}{array, A list of columns which will be considered ineligible to be independent variables.}
#' \item{limitingSQL}{string, A custom SQL WHERE clause used to filter the rows used to build the model. (e.g., "id > 105").}
#' \item{activeBuildId}{integer, The ID of the current active build, the build used to score predictions.}
#' \item{crossValidationParameters}{list, Cross validation parameter grid for tree methods, e.g. {"n_estimators": [100, 200, 500], "learning_rate": [0.01, 0.1], "max_depth": [2, 3]}.}
#' \item{numberOfFolds}{integer, Number of folds for cross validation. Default value is 5.}
#' \item{notifications}{list, A list containing the following elements: 
#' \itemize{
#' \item urls array, URLs to receive a POST request at job completion
#' \item successEmailSubject string, Custom subject line for success e-mail.
#' \item successEmailBody string, Custom body text for success e-mail, written in Markdown.
#' \item successEmailAddresses array, Addresses to notify by e-mail when the job completes successfully.
#' \item successEmailFromName string, Name from which success emails are sent; defaults to "Civis."
#' \item successEmailReplyTo string, Address for replies to success emails; defaults to the author of the job.
#' \item failureEmailAddresses array, Addresses to notify by e-mail when the job fails.
#' \item stallWarningMinutes integer, Stall warning emails will be sent after this amount of minutes.
#' \item successOn boolean, If success email notifications are on.
#' \item failureOn boolean, If failure email notifications are on.
#' }}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{parentId}{integer, The ID of the parent job that will trigger this model.}
#' \item{runningAs}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{timeZone}{string, The time zone of this model.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, The time the model was created.}
#' \item{updatedAt}{string, The time the model was updated.}
#' \item{currentBuildState}{string, The status of the current model build. One of "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.}
#' \item{currentBuildException}{string, Exception message, if applicable, of the current model build.}
#' \item{builds}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the model build.
#' \item name string, The name of the model build.
#' \item createdAt string, The time the model build was created.
#' \item description string, A description of the model build.
#' \item rootMeanSquaredError number, A key metric for continuous models. Nil for other model types.
#' \item rSquaredError number, A key metric for continuous models. Nil for other model types.
#' \item rocAuc number, A key metric for binary, multinomial, and ordinal models. Nil for other model types.
#' }}
#' \item{predictions}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the model to which to apply the prediction.
#' \item tableName string, The qualified name of the table on which to apply the predictive model.
#' \item primaryKey array, The primary key or composite keys of the table being predicted.
#' \item limitingSQL string, A SQL WHERE clause used to scope the rows to be predicted.
#' \item outputTable string, The qualified name of the table to be created which will contain the model's predictions.
#' \item schedule object, 
#' \item state string, The status of the prediction. One of: "succeeded", "failed", "queued", or "running,"or "idle", if no build has been attempted.
#' }}
#' \item{lastOutputLocation}{string, The output JSON for the last build.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
models_put_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show the model build schedule
#' @param id integer required. The ID of the model associated with this schedule.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the model associated with this schedule.}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' @export
models_list_schedules <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/models/{id}/schedules"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Notebooks
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' @param archived string optional. The archival status of the requested item(s).
#' @param author string optional. If specified, return items from any of these authors. It accepts a comma-separated list of user IDs.
#' @param status string optional. If specified, returns notebooks with one of these statuses. It accepts a comma-separated list, possible values are 'running', 'pending', 'idle'.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this notebook.}
#' \item{name}{string, The name of this notebook.}
#' \item{language}{string, The kernel language of this notebook.}
#' \item{description}{string, The description of this notebook.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{mostRecentDeployment}{list, A list containing the following elements: 
#' \itemize{
#' \item deploymentId integer, The ID for this deployment.
#' \item userId integer, The ID of the owner.
#' \item host string, Domain of the deployment.
#' \item name string, Name of the deployment.
#' \item dockerImageName string, The name of the docker image to pull from DockerHub.
#' \item dockerImageTag string, The tag of the docker image to pull from DockerHub (default: latest).
#' \item instanceType string, The EC2 instance type requested for the deployment.
#' \item memory integer, The memory allocated to the deployment, in MB.
#' \item cpu integer, The cpu allocated to the deployment, in millicores.
#' \item state string, The state of the deployment.
#' \item stateMessage string, A detailed description of the state.
#' \item maxMemoryUsage number, If the deployment has finished, the maximum amount of memory used during the deployment, in MB.
#' \item maxCpuUsage number, If the deployment has finished, the maximum amount of cpu used during the deployment, in millicores.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item notebookId integer, The ID of owning Notebook
#' }}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
notebooks_list <- function(hidden = NULL, archived = NULL, author = NULL, status = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/"
  path_params  <- list()
  query_params <- list(hidden = hidden, archived = archived, author = author, status = status, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Notebook
#' @param name string optional. The name of this notebook.
#' @param language string optional. The kernel language of this notebook.
#' @param description string optional. The description of this notebook.
#' @param file_id string optional. The file ID for the S3 file containing the .ipynb file.
#' @param requirements_file_id string optional. The file ID for the S3 file containing the requirements.txt file.
#' @param requirements string optional. The requirements txt file.
#' @param docker_image_name string optional. The name of the docker image to pull from DockerHub.
#' @param docker_image_tag string optional. The tag of the docker image to pull from DockerHub (default: latest).
#' @param instance_type string optional. The EC2 instance type to deploy to.
#' @param memory integer optional. The amount of memory allocated to the notebook.
#' @param cpu integer optional. The amount of cpu allocated to the the notebook.
#' @param credentials array optional. A list of credential IDs to pass to the notebook.
#' @param environment_variables list optional. Environment variables to be passed into the Notebook.
#' @param idle_timeout integer optional. How long the notebook will stay alive without any kernel activity.
#' @param partition_label string optional. The partition label used to run this object.
#' @param git_repo_url string optional. The url of the git repository
#' @param git_ref string optional. The git reference if git repo is specified
#' @param git_path string optional. The path to the .ipynb file in the git repo that will be started up on notebook launch
#' @param hidden boolean optional. The hidden status of the item.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this notebook.}
#' \item{name}{string, The name of this notebook.}
#' \item{language}{string, The kernel language of this notebook.}
#' \item{description}{string, The description of this notebook.}
#' \item{notebookUrl}{string, Time-limited URL to get the .ipynb file for this notebook.}
#' \item{notebookPreviewUrl}{string, Time-limited URL to get the .htm preview file for this notebook.}
#' \item{requirementsUrl}{string, Time-limited URL to get the requirements.txt file for this notebook.}
#' \item{fileId}{string, The file ID for the S3 file containing the .ipynb file.}
#' \item{requirementsFileId}{string, The file ID for the S3 file containing the requirements.txt file.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{instanceType}{string, The EC2 instance type to deploy to.}
#' \item{memory}{integer, The amount of memory allocated to the notebook.}
#' \item{cpu}{integer, The amount of cpu allocated to the the notebook.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{mostRecentDeployment}{list, A list containing the following elements: 
#' \itemize{
#' \item deploymentId integer, The ID for this deployment.
#' \item userId integer, The ID of the owner.
#' \item host string, Domain of the deployment.
#' \item name string, Name of the deployment.
#' \item dockerImageName string, The name of the docker image to pull from DockerHub.
#' \item dockerImageTag string, The tag of the docker image to pull from DockerHub (default: latest).
#' \item displayUrl string, A signed URL for viewing the deployed item.
#' \item instanceType string, The EC2 instance type requested for the deployment.
#' \item memory integer, The memory allocated to the deployment, in MB.
#' \item cpu integer, The cpu allocated to the deployment, in millicores.
#' \item state string, The state of the deployment.
#' \item stateMessage string, A detailed description of the state.
#' \item maxMemoryUsage number, If the deployment has finished, the maximum amount of memory used during the deployment, in MB.
#' \item maxCpuUsage number, If the deployment has finished, the maximum amount of cpu used during the deployment, in millicores.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item notebookId integer, The ID of owning Notebook
#' }}
#' \item{credentials}{array, A list of credential IDs to pass to the notebook.}
#' \item{environmentVariables}{list, Environment variables to be passed into the Notebook.}
#' \item{idleTimeout}{integer, How long the notebook will stay alive without any kernel activity.}
#' \item{partitionLabel}{string, The partition label used to run this object.}
#' \item{gitRepoId}{integer, The ID of the git repository.}
#' \item{gitRepoUrl}{string, The url of the git repository}
#' \item{gitRef}{string, The git reference if git repo is specified}
#' \item{gitPath}{string, The path to the .ipynb file in the git repo that will be started up on notebook launch}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{hidden}{boolean, The hidden status of the item.}
#' @export
notebooks_post <- function(name = NULL, language = NULL, description = NULL, file_id = NULL, requirements_file_id = NULL, requirements = NULL, docker_image_name = NULL, docker_image_tag = NULL, instance_type = NULL, memory = NULL, cpu = NULL, credentials = NULL, environment_variables = NULL, idle_timeout = NULL, partition_label = NULL, git_repo_url = NULL, git_ref = NULL, git_path = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, language = language, description = description, fileId = file_id, requirementsFileId = requirements_file_id, requirements = requirements, dockerImageName = docker_image_name, dockerImageTag = docker_image_tag, instanceType = instance_type, memory = memory, cpu = cpu, credentials = credentials, environmentVariables = environment_variables, idleTimeout = idle_timeout, partitionLabel = partition_label, gitRepoUrl = git_repo_url, gitRef = git_ref, gitPath = git_path, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a Notebook
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this notebook.}
#' \item{name}{string, The name of this notebook.}
#' \item{language}{string, The kernel language of this notebook.}
#' \item{description}{string, The description of this notebook.}
#' \item{notebookUrl}{string, Time-limited URL to get the .ipynb file for this notebook.}
#' \item{notebookPreviewUrl}{string, Time-limited URL to get the .htm preview file for this notebook.}
#' \item{requirementsUrl}{string, Time-limited URL to get the requirements.txt file for this notebook.}
#' \item{fileId}{string, The file ID for the S3 file containing the .ipynb file.}
#' \item{requirementsFileId}{string, The file ID for the S3 file containing the requirements.txt file.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{instanceType}{string, The EC2 instance type to deploy to.}
#' \item{memory}{integer, The amount of memory allocated to the notebook.}
#' \item{cpu}{integer, The amount of cpu allocated to the the notebook.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{mostRecentDeployment}{list, A list containing the following elements: 
#' \itemize{
#' \item deploymentId integer, The ID for this deployment.
#' \item userId integer, The ID of the owner.
#' \item host string, Domain of the deployment.
#' \item name string, Name of the deployment.
#' \item dockerImageName string, The name of the docker image to pull from DockerHub.
#' \item dockerImageTag string, The tag of the docker image to pull from DockerHub (default: latest).
#' \item displayUrl string, A signed URL for viewing the deployed item.
#' \item instanceType string, The EC2 instance type requested for the deployment.
#' \item memory integer, The memory allocated to the deployment, in MB.
#' \item cpu integer, The cpu allocated to the deployment, in millicores.
#' \item state string, The state of the deployment.
#' \item stateMessage string, A detailed description of the state.
#' \item maxMemoryUsage number, If the deployment has finished, the maximum amount of memory used during the deployment, in MB.
#' \item maxCpuUsage number, If the deployment has finished, the maximum amount of cpu used during the deployment, in millicores.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item notebookId integer, The ID of owning Notebook
#' }}
#' \item{credentials}{array, A list of credential IDs to pass to the notebook.}
#' \item{environmentVariables}{list, Environment variables to be passed into the Notebook.}
#' \item{idleTimeout}{integer, How long the notebook will stay alive without any kernel activity.}
#' \item{partitionLabel}{string, The partition label used to run this object.}
#' \item{gitRepoId}{integer, The ID of the git repository.}
#' \item{gitRepoUrl}{string, The url of the git repository}
#' \item{gitRef}{string, The git reference if git repo is specified}
#' \item{gitPath}{string, The path to the .ipynb file in the git repo that will be started up on notebook launch}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{hidden}{boolean, The hidden status of the item.}
#' @export
notebooks_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this Notebook
#' @param id integer required. The ID for this notebook.
#' @param name string optional. The name of this notebook.
#' @param language string optional. The kernel language of this notebook.
#' @param description string optional. The description of this notebook.
#' @param file_id string optional. The file ID for the S3 file containing the .ipynb file.
#' @param requirements_file_id string optional. The file ID for the S3 file containing the requirements.txt file.
#' @param requirements string optional. The requirements txt file.
#' @param docker_image_name string optional. The name of the docker image to pull from DockerHub.
#' @param docker_image_tag string optional. The tag of the docker image to pull from DockerHub (default: latest).
#' @param instance_type string optional. The EC2 instance type to deploy to.
#' @param memory integer optional. The amount of memory allocated to the notebook.
#' @param cpu integer optional. The amount of cpu allocated to the the notebook.
#' @param credentials array optional. A list of credential IDs to pass to the notebook.
#' @param environment_variables list optional. Environment variables to be passed into the Notebook.
#' @param idle_timeout integer optional. How long the notebook will stay alive without any kernel activity.
#' @param partition_label string optional. The partition label used to run this object.
#' @param git_repo_url string optional. The url of the git repository
#' @param git_ref string optional. The git reference if git repo is specified
#' @param git_path string optional. The path to the .ipynb file in the git repo that will be started up on notebook launch
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this notebook.}
#' \item{name}{string, The name of this notebook.}
#' \item{language}{string, The kernel language of this notebook.}
#' \item{description}{string, The description of this notebook.}
#' \item{notebookUrl}{string, Time-limited URL to get the .ipynb file for this notebook.}
#' \item{notebookPreviewUrl}{string, Time-limited URL to get the .htm preview file for this notebook.}
#' \item{requirementsUrl}{string, Time-limited URL to get the requirements.txt file for this notebook.}
#' \item{fileId}{string, The file ID for the S3 file containing the .ipynb file.}
#' \item{requirementsFileId}{string, The file ID for the S3 file containing the requirements.txt file.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{instanceType}{string, The EC2 instance type to deploy to.}
#' \item{memory}{integer, The amount of memory allocated to the notebook.}
#' \item{cpu}{integer, The amount of cpu allocated to the the notebook.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{mostRecentDeployment}{list, A list containing the following elements: 
#' \itemize{
#' \item deploymentId integer, The ID for this deployment.
#' \item userId integer, The ID of the owner.
#' \item host string, Domain of the deployment.
#' \item name string, Name of the deployment.
#' \item dockerImageName string, The name of the docker image to pull from DockerHub.
#' \item dockerImageTag string, The tag of the docker image to pull from DockerHub (default: latest).
#' \item displayUrl string, A signed URL for viewing the deployed item.
#' \item instanceType string, The EC2 instance type requested for the deployment.
#' \item memory integer, The memory allocated to the deployment, in MB.
#' \item cpu integer, The cpu allocated to the deployment, in millicores.
#' \item state string, The state of the deployment.
#' \item stateMessage string, A detailed description of the state.
#' \item maxMemoryUsage number, If the deployment has finished, the maximum amount of memory used during the deployment, in MB.
#' \item maxCpuUsage number, If the deployment has finished, the maximum amount of cpu used during the deployment, in millicores.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item notebookId integer, The ID of owning Notebook
#' }}
#' \item{credentials}{array, A list of credential IDs to pass to the notebook.}
#' \item{environmentVariables}{list, Environment variables to be passed into the Notebook.}
#' \item{idleTimeout}{integer, How long the notebook will stay alive without any kernel activity.}
#' \item{partitionLabel}{string, The partition label used to run this object.}
#' \item{gitRepoId}{integer, The ID of the git repository.}
#' \item{gitRepoUrl}{string, The url of the git repository}
#' \item{gitRef}{string, The git reference if git repo is specified}
#' \item{gitPath}{string, The path to the .ipynb file in the git repo that will be started up on notebook launch}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{hidden}{boolean, The hidden status of the item.}
#' @export
notebooks_put <- function(id, name = NULL, language = NULL, description = NULL, file_id = NULL, requirements_file_id = NULL, requirements = NULL, docker_image_name = NULL, docker_image_tag = NULL, instance_type = NULL, memory = NULL, cpu = NULL, credentials = NULL, environment_variables = NULL, idle_timeout = NULL, partition_label = NULL, git_repo_url = NULL, git_ref = NULL, git_path = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, language = language, description = description, fileId = file_id, requirementsFileId = requirements_file_id, requirements = requirements, dockerImageName = docker_image_name, dockerImageTag = docker_image_tag, instanceType = instance_type, memory = memory, cpu = cpu, credentials = credentials, environmentVariables = environment_variables, idleTimeout = idle_timeout, partitionLabel = partition_label, gitRepoUrl = git_repo_url, gitRef = git_ref, gitPath = git_path)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this Notebook
#' @param id integer required. The ID for this notebook.
#' @param name string optional. The name of this notebook.
#' @param language string optional. The kernel language of this notebook.
#' @param description string optional. The description of this notebook.
#' @param file_id string optional. The file ID for the S3 file containing the .ipynb file.
#' @param requirements_file_id string optional. The file ID for the S3 file containing the requirements.txt file.
#' @param requirements string optional. The requirements txt file.
#' @param docker_image_name string optional. The name of the docker image to pull from DockerHub.
#' @param docker_image_tag string optional. The tag of the docker image to pull from DockerHub (default: latest).
#' @param instance_type string optional. The EC2 instance type to deploy to.
#' @param memory integer optional. The amount of memory allocated to the notebook.
#' @param cpu integer optional. The amount of cpu allocated to the the notebook.
#' @param credentials array optional. A list of credential IDs to pass to the notebook.
#' @param environment_variables list optional. Environment variables to be passed into the Notebook.
#' @param idle_timeout integer optional. How long the notebook will stay alive without any kernel activity.
#' @param partition_label string optional. The partition label used to run this object.
#' @param git_repo_url string optional. The url of the git repository
#' @param git_ref string optional. The git reference if git repo is specified
#' @param git_path string optional. The path to the .ipynb file in the git repo that will be started up on notebook launch
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this notebook.}
#' \item{name}{string, The name of this notebook.}
#' \item{language}{string, The kernel language of this notebook.}
#' \item{description}{string, The description of this notebook.}
#' \item{notebookUrl}{string, Time-limited URL to get the .ipynb file for this notebook.}
#' \item{notebookPreviewUrl}{string, Time-limited URL to get the .htm preview file for this notebook.}
#' \item{requirementsUrl}{string, Time-limited URL to get the requirements.txt file for this notebook.}
#' \item{fileId}{string, The file ID for the S3 file containing the .ipynb file.}
#' \item{requirementsFileId}{string, The file ID for the S3 file containing the requirements.txt file.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{instanceType}{string, The EC2 instance type to deploy to.}
#' \item{memory}{integer, The amount of memory allocated to the notebook.}
#' \item{cpu}{integer, The amount of cpu allocated to the the notebook.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{mostRecentDeployment}{list, A list containing the following elements: 
#' \itemize{
#' \item deploymentId integer, The ID for this deployment.
#' \item userId integer, The ID of the owner.
#' \item host string, Domain of the deployment.
#' \item name string, Name of the deployment.
#' \item dockerImageName string, The name of the docker image to pull from DockerHub.
#' \item dockerImageTag string, The tag of the docker image to pull from DockerHub (default: latest).
#' \item displayUrl string, A signed URL for viewing the deployed item.
#' \item instanceType string, The EC2 instance type requested for the deployment.
#' \item memory integer, The memory allocated to the deployment, in MB.
#' \item cpu integer, The cpu allocated to the deployment, in millicores.
#' \item state string, The state of the deployment.
#' \item stateMessage string, A detailed description of the state.
#' \item maxMemoryUsage number, If the deployment has finished, the maximum amount of memory used during the deployment, in MB.
#' \item maxCpuUsage number, If the deployment has finished, the maximum amount of cpu used during the deployment, in millicores.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item notebookId integer, The ID of owning Notebook
#' }}
#' \item{credentials}{array, A list of credential IDs to pass to the notebook.}
#' \item{environmentVariables}{list, Environment variables to be passed into the Notebook.}
#' \item{idleTimeout}{integer, How long the notebook will stay alive without any kernel activity.}
#' \item{partitionLabel}{string, The partition label used to run this object.}
#' \item{gitRepoId}{integer, The ID of the git repository.}
#' \item{gitRepoUrl}{string, The url of the git repository}
#' \item{gitRef}{string, The git reference if git repo is specified}
#' \item{gitPath}{string, The path to the .ipynb file in the git repo that will be started up on notebook launch}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{hidden}{boolean, The hidden status of the item.}
#' @export
notebooks_patch <- function(id, name = NULL, language = NULL, description = NULL, file_id = NULL, requirements_file_id = NULL, requirements = NULL, docker_image_name = NULL, docker_image_tag = NULL, instance_type = NULL, memory = NULL, cpu = NULL, credentials = NULL, environment_variables = NULL, idle_timeout = NULL, partition_label = NULL, git_repo_url = NULL, git_ref = NULL, git_path = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, language = language, description = description, fileId = file_id, requirementsFileId = requirements_file_id, requirements = requirements, dockerImageName = docker_image_name, dockerImageTag = docker_image_tag, instanceType = instance_type, memory = memory, cpu = cpu, credentials = credentials, environmentVariables = environment_variables, idleTimeout = idle_timeout, partitionLabel = partition_label, gitRepoUrl = git_repo_url, gitRef = git_ref, gitPath = git_path)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive a Notebook (deprecated, use archiving endpoints instead)
#' @param id integer required. 
#' 
#' @return  An empty HTTP response
#' @export
notebooks_delete <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get URLs to update notebook
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{updateUrl}{string, Time-limited URL to PUT new contents of the .ipynb file for this notebook.}
#' \item{updatePreviewUrl}{string, Time-limited URL to PUT new contents of the .htm preview file for this notebook.}
#' @export
notebooks_list_update_links <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/update-links"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Clone this Notebook
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this notebook.}
#' \item{name}{string, The name of this notebook.}
#' \item{language}{string, The kernel language of this notebook.}
#' \item{description}{string, The description of this notebook.}
#' \item{notebookUrl}{string, Time-limited URL to get the .ipynb file for this notebook.}
#' \item{notebookPreviewUrl}{string, Time-limited URL to get the .htm preview file for this notebook.}
#' \item{requirementsUrl}{string, Time-limited URL to get the requirements.txt file for this notebook.}
#' \item{fileId}{string, The file ID for the S3 file containing the .ipynb file.}
#' \item{requirementsFileId}{string, The file ID for the S3 file containing the requirements.txt file.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{instanceType}{string, The EC2 instance type to deploy to.}
#' \item{memory}{integer, The amount of memory allocated to the notebook.}
#' \item{cpu}{integer, The amount of cpu allocated to the the notebook.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{mostRecentDeployment}{list, A list containing the following elements: 
#' \itemize{
#' \item deploymentId integer, The ID for this deployment.
#' \item userId integer, The ID of the owner.
#' \item host string, Domain of the deployment.
#' \item name string, Name of the deployment.
#' \item dockerImageName string, The name of the docker image to pull from DockerHub.
#' \item dockerImageTag string, The tag of the docker image to pull from DockerHub (default: latest).
#' \item displayUrl string, A signed URL for viewing the deployed item.
#' \item instanceType string, The EC2 instance type requested for the deployment.
#' \item memory integer, The memory allocated to the deployment, in MB.
#' \item cpu integer, The cpu allocated to the deployment, in millicores.
#' \item state string, The state of the deployment.
#' \item stateMessage string, A detailed description of the state.
#' \item maxMemoryUsage number, If the deployment has finished, the maximum amount of memory used during the deployment, in MB.
#' \item maxCpuUsage number, If the deployment has finished, the maximum amount of cpu used during the deployment, in millicores.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item notebookId integer, The ID of owning Notebook
#' }}
#' \item{credentials}{array, A list of credential IDs to pass to the notebook.}
#' \item{environmentVariables}{list, Environment variables to be passed into the Notebook.}
#' \item{idleTimeout}{integer, How long the notebook will stay alive without any kernel activity.}
#' \item{partitionLabel}{string, The partition label used to run this object.}
#' \item{gitRepoId}{integer, The ID of the git repository.}
#' \item{gitRepoUrl}{string, The url of the git repository}
#' \item{gitRef}{string, The git reference if git repo is specified}
#' \item{gitPath}{string, The path to the .ipynb file in the git repo that will be started up on notebook launch}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{hidden}{boolean, The hidden status of the item.}
#' @export
notebooks_post_clone <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/clone"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
notebooks_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
notebooks_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
notebooks_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
notebooks_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
notebooks_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
notebooks_list_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
notebooks_put_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this notebook.}
#' \item{name}{string, The name of this notebook.}
#' \item{language}{string, The kernel language of this notebook.}
#' \item{description}{string, The description of this notebook.}
#' \item{notebookUrl}{string, Time-limited URL to get the .ipynb file for this notebook.}
#' \item{notebookPreviewUrl}{string, Time-limited URL to get the .htm preview file for this notebook.}
#' \item{requirementsUrl}{string, Time-limited URL to get the requirements.txt file for this notebook.}
#' \item{fileId}{string, The file ID for the S3 file containing the .ipynb file.}
#' \item{requirementsFileId}{string, The file ID for the S3 file containing the requirements.txt file.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{instanceType}{string, The EC2 instance type to deploy to.}
#' \item{memory}{integer, The amount of memory allocated to the notebook.}
#' \item{cpu}{integer, The amount of cpu allocated to the the notebook.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{mostRecentDeployment}{list, A list containing the following elements: 
#' \itemize{
#' \item deploymentId integer, The ID for this deployment.
#' \item userId integer, The ID of the owner.
#' \item host string, Domain of the deployment.
#' \item name string, Name of the deployment.
#' \item dockerImageName string, The name of the docker image to pull from DockerHub.
#' \item dockerImageTag string, The tag of the docker image to pull from DockerHub (default: latest).
#' \item displayUrl string, A signed URL for viewing the deployed item.
#' \item instanceType string, The EC2 instance type requested for the deployment.
#' \item memory integer, The memory allocated to the deployment, in MB.
#' \item cpu integer, The cpu allocated to the deployment, in millicores.
#' \item state string, The state of the deployment.
#' \item stateMessage string, A detailed description of the state.
#' \item maxMemoryUsage number, If the deployment has finished, the maximum amount of memory used during the deployment, in MB.
#' \item maxCpuUsage number, If the deployment has finished, the maximum amount of cpu used during the deployment, in millicores.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item notebookId integer, The ID of owning Notebook
#' }}
#' \item{credentials}{array, A list of credential IDs to pass to the notebook.}
#' \item{environmentVariables}{list, Environment variables to be passed into the Notebook.}
#' \item{idleTimeout}{integer, How long the notebook will stay alive without any kernel activity.}
#' \item{partitionLabel}{string, The partition label used to run this object.}
#' \item{gitRepoId}{integer, The ID of the git repository.}
#' \item{gitRepoUrl}{string, The url of the git repository}
#' \item{gitRef}{string, The git reference if git repo is specified}
#' \item{gitPath}{string, The path to the .ipynb file in the git repo that will be started up on notebook launch}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{hidden}{boolean, The hidden status of the item.}
#' @export
notebooks_put_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a Notebook belongs to
#' @param id integer required. The ID of the Notebook.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
notebooks_list_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a Notebook to a project
#' @param id integer required. The ID of the Notebook.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
notebooks_put_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a Notebook from a project
#' @param id integer required. The ID of the Notebook.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
notebooks_delete_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List deployments for a Notebook
#' @param notebook_id integer required. The ID of the owning Notebook
#' @param deployment_id integer optional. The ID for this deployment
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to created_at. Must be one of: created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{deploymentId}{integer, The ID for this deployment.}
#' \item{userId}{integer, The ID of the owner.}
#' \item{host}{string, Domain of the deployment.}
#' \item{name}{string, Name of the deployment.}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{instanceType}{string, The EC2 instance type requested for the deployment.}
#' \item{memory}{integer, The memory allocated to the deployment, in MB.}
#' \item{cpu}{integer, The cpu allocated to the deployment, in millicores.}
#' \item{state}{string, The state of the deployment.}
#' \item{stateMessage}{string, A detailed description of the state.}
#' \item{maxMemoryUsage}{number, If the deployment has finished, the maximum amount of memory used during the deployment, in MB.}
#' \item{maxCpuUsage}{number, If the deployment has finished, the maximum amount of cpu used during the deployment, in millicores.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{notebookId}{integer, The ID of owning Notebook}
#' @export
notebooks_list_deployments <- function(notebook_id, deployment_id = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{notebook_id}/deployments"
  path_params  <- list(notebook_id = notebook_id)
  query_params <- list(deployment_id = deployment_id, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Deploy a Notebook
#' @param notebook_id integer required. The ID of the owning Notebook
#' @param deployment_id integer optional. The ID for this deployment
#' 
#' @return  A list containing the following elements:
#' \item{deploymentId}{integer, The ID for this deployment.}
#' \item{userId}{integer, The ID of the owner.}
#' \item{host}{string, Domain of the deployment.}
#' \item{name}{string, Name of the deployment.}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{displayUrl}{string, A signed URL for viewing the deployed item.}
#' \item{instanceType}{string, The EC2 instance type requested for the deployment.}
#' \item{memory}{integer, The memory allocated to the deployment, in MB.}
#' \item{cpu}{integer, The cpu allocated to the deployment, in millicores.}
#' \item{state}{string, The state of the deployment.}
#' \item{stateMessage}{string, A detailed description of the state.}
#' \item{maxMemoryUsage}{number, If the deployment has finished, the maximum amount of memory used during the deployment, in MB.}
#' \item{maxCpuUsage}{number, If the deployment has finished, the maximum amount of cpu used during the deployment, in millicores.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{notebookId}{integer, The ID of owning Notebook}
#' @export
notebooks_post_deployments <- function(notebook_id, deployment_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{notebook_id}/deployments"
  path_params  <- list(notebook_id = notebook_id)
  query_params <- list()
  body_params  <- list(deploymentId = deployment_id)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about a Notebook deployment
#' @param notebook_id integer required. The ID of the owning Notebook
#' @param deployment_id integer required. The ID for this deployment
#' 
#' @return  A list containing the following elements:
#' \item{deploymentId}{integer, The ID for this deployment.}
#' \item{userId}{integer, The ID of the owner.}
#' \item{host}{string, Domain of the deployment.}
#' \item{name}{string, Name of the deployment.}
#' \item{dockerImageName}{string, The name of the docker image to pull from DockerHub.}
#' \item{dockerImageTag}{string, The tag of the docker image to pull from DockerHub (default: latest).}
#' \item{displayUrl}{string, A signed URL for viewing the deployed item.}
#' \item{instanceType}{string, The EC2 instance type requested for the deployment.}
#' \item{memory}{integer, The memory allocated to the deployment, in MB.}
#' \item{cpu}{integer, The cpu allocated to the deployment, in millicores.}
#' \item{state}{string, The state of the deployment.}
#' \item{stateMessage}{string, A detailed description of the state.}
#' \item{maxMemoryUsage}{number, If the deployment has finished, the maximum amount of memory used during the deployment, in MB.}
#' \item{maxCpuUsage}{number, If the deployment has finished, the maximum amount of cpu used during the deployment, in millicores.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{notebookId}{integer, The ID of owning Notebook}
#' @export
notebooks_get_deployments <- function(notebook_id, deployment_id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{notebook_id}/deployments/{deployment_id}"
  path_params  <- list(notebook_id = notebook_id, deployment_id = deployment_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Delete a Notebook deployment
#' @param notebook_id integer required. The ID of the owning Notebook
#' @param deployment_id integer required. The ID for this deployment
#' 
#' @return  An empty HTTP response
#' @export
notebooks_delete_deployments <- function(notebook_id, deployment_id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{notebook_id}/deployments/{deployment_id}"
  path_params  <- list(notebook_id = notebook_id, deployment_id = deployment_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a Notebook deployment
#' @param id integer required. The ID of the owning Notebook.
#' @param deployment_id integer required. The ID for this deployment.
#' @param start_at string optional. Log entries with a lower timestamp will be omitted.
#' @param end_at string optional. Log entries with a higher timestamp will be omitted.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{message}{string, The log message.}
#' \item{stream}{string, The stream of the log. One of "stdout", "stderr".}
#' \item{createdAt}{string, The time the log was created.}
#' \item{source}{string, The source of the log. One of "system", "user".}
#' @export
notebooks_list_deployments_logs <- function(id, deployment_id, start_at = NULL, end_at = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/deployments/{deployment_id}/logs"
  path_params  <- list(id = id, deployment_id = deployment_id)
  query_params <- list(start_at = start_at, end_at = end_at, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the git metadata attached to an item
#' @param id integer required. The ID of the file.
#' 
#' @return  A list containing the following elements:
#' \item{gitRef}{string, A git reference specifying an unambiguous version of the file. Can be a branch name, tag or the full or shortened SHA of a commit.}
#' \item{gitBranch}{string, The git branch that the file is on.}
#' \item{gitPath}{string, The path of the file in the repository.}
#' \item{gitRepo}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for this git repository.
#' \item repoUrl string, The URL for this git repository.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{gitRefType}{string, Specifies if the file is versioned by branch or tag.}
#' \item{pullFromGit}{boolean, Automatically pull latest commit from git. Only works for scripts and workflows (assuming you have the feature enabled)}
#' @export
notebooks_list_git <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/git"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Attach an item to a file in a git repo
#' @param id integer required. The ID of the file.
#' @param git_ref string optional. A git reference specifying an unambiguous version of the file. Can be a branch name, or the full or shortened SHA of a commit.
#' @param git_branch string optional. The git branch that the file is on.
#' @param git_path string optional. The path of the file in the repository.
#' @param git_repo_url string optional. The URL of the git repository.
#' @param git_ref_type string optional. Specifies if the file is versioned by branch or tag.
#' @param pull_from_git boolean optional. Automatically pull latest commit from git. Only works for scripts.
#' 
#' @return  A list containing the following elements:
#' \item{gitRef}{string, A git reference specifying an unambiguous version of the file. Can be a branch name, tag or the full or shortened SHA of a commit.}
#' \item{gitBranch}{string, The git branch that the file is on.}
#' \item{gitPath}{string, The path of the file in the repository.}
#' \item{gitRepo}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for this git repository.
#' \item repoUrl string, The URL for this git repository.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{gitRefType}{string, Specifies if the file is versioned by branch or tag.}
#' \item{pullFromGit}{boolean, Automatically pull latest commit from git. Only works for scripts and workflows (assuming you have the feature enabled)}
#' @export
notebooks_put_git <- function(id, git_ref = NULL, git_branch = NULL, git_path = NULL, git_repo_url = NULL, git_ref_type = NULL, pull_from_git = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/git"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(gitRef = git_ref, gitBranch = git_branch, gitPath = git_path, gitRepoUrl = git_repo_url, gitRefType = git_ref_type, pullFromGit = pull_from_git)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update an attached git file
#' @param id integer required. The ID of the file.
#' @param git_ref string optional. A git reference specifying an unambiguous version of the file. Can be a branch name, or the full or shortened SHA of a commit.
#' @param git_branch string optional. The git branch that the file is on.
#' @param git_path string optional. The path of the file in the repository.
#' @param git_repo_url string optional. The URL of the git repository.
#' @param git_ref_type string optional. Specifies if the file is versioned by branch or tag.
#' @param pull_from_git boolean optional. Automatically pull latest commit from git. Only works for scripts.
#' 
#' @return  A list containing the following elements:
#' \item{gitRef}{string, A git reference specifying an unambiguous version of the file. Can be a branch name, tag or the full or shortened SHA of a commit.}
#' \item{gitBranch}{string, The git branch that the file is on.}
#' \item{gitPath}{string, The path of the file in the repository.}
#' \item{gitRepo}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for this git repository.
#' \item repoUrl string, The URL for this git repository.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{gitRefType}{string, Specifies if the file is versioned by branch or tag.}
#' \item{pullFromGit}{boolean, Automatically pull latest commit from git. Only works for scripts and workflows (assuming you have the feature enabled)}
#' @export
notebooks_patch_git <- function(id, git_ref = NULL, git_branch = NULL, git_path = NULL, git_repo_url = NULL, git_ref_type = NULL, pull_from_git = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/git"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(gitRef = git_ref, gitBranch = git_branch, gitPath = git_path, gitRepoUrl = git_repo_url, gitRefType = git_ref_type, pullFromGit = pull_from_git)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the git commits for an item on the current branch
#' @param id integer required. The ID of the file.
#' 
#' @return  A list containing the following elements:
#' \item{commitHash}{string, The SHA of the commit.}
#' \item{authorName}{string, The name of the commit's author.}
#' \item{date}{string, The commit's timestamp.}
#' \item{message}{string, The commit message.}
#' @export
notebooks_list_git_commits <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/git/commits"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Commit and push a new version of the file
#' @param id integer required. The ID of the file.
#' @param content string required. The contents to commit to the file.
#' @param message string required. A commit message describing the changes being made.
#' @param file_hash string required. The full SHA of the file being replaced.
#' 
#' @return  A list containing the following elements:
#' \item{content}{string, The file's contents.}
#' \item{type}{string, The file's type.}
#' \item{size}{integer, The file's size.}
#' \item{fileHash}{string, The SHA of the file.}
#' @export
notebooks_post_git_commits <- function(id, content, message, file_hash) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/git/commits"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(content = content, message = message, fileHash = file_hash)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get file contents at git ref
#' @param id integer required. The ID of the file.
#' @param commit_hash string required. The SHA (full or shortened) of the desired git commit.
#' 
#' @return  A list containing the following elements:
#' \item{content}{string, The file's contents.}
#' \item{type}{string, The file's type.}
#' \item{size}{integer, The file's size.}
#' \item{fileHash}{string, The SHA of the file.}
#' @export
notebooks_get_git_commits <- function(id, commit_hash) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/git/commits/{commit_hash}"
  path_params  <- list(id = id, commit_hash = commit_hash)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Checkout latest commit on the current branch of a script or workflow
#' @param id integer required. The ID of the file.
#' 
#' @return  A list containing the following elements:
#' \item{content}{string, The file's contents.}
#' \item{type}{string, The file's type.}
#' \item{size}{integer, The file's size.}
#' \item{fileHash}{string, The SHA of the file.}
#' @export
notebooks_post_git_checkout_latest <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/git/checkout-latest"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Checkout content that the existing git_ref points to and save to the object
#' @param id integer required. The ID of the file.
#' 
#' @return  A list containing the following elements:
#' \item{content}{string, The file's contents.}
#' \item{type}{string, The file's type.}
#' \item{size}{integer, The file's size.}
#' \item{fileHash}{string, The SHA of the file.}
#' @export
notebooks_post_git_checkout <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/notebooks/{id}/git/checkout"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Receive a stream of notifications as they come in
#' @param last_event_id string optional. allows browser to keep track of last event fired
#' @param r string optional. specifies retry/reconnect timeout
#' @param mock string optional. used for testing
#' 
#' @return  An empty HTTP response
#' @export
notifications_list <- function(last_event_id = NULL, r = NULL, mock = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/notifications/"
  path_params  <- list()
  query_params <- list(last_event_id = last_event_id, r = r, mock = mock)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the ontology of column names Civis uses
#' @param subset string optional. A subset of fields to return.
#' 
#' @return  An array containing the following fields:
#' \item{key}{string, }
#' \item{title}{string, }
#' \item{desc}{string, A description of this field.}
#' \item{aliases}{array, }
#' @export
ontology_list <- function(subset = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/ontology/"
  path_params  <- list()
  query_params <- list(subset = subset)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Permission Sets
#' @param archived string optional. The archival status of the requested item(s).
#' @param author string optional. If specified, return items from any of these authors. It accepts a comma-separated list of user IDs.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this permission set.}
#' \item{name}{string, The name of this permission set.}
#' \item{description}{string, A description of this permission set.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
permission_sets_list <- function(archived = NULL, author = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/"
  path_params  <- list()
  query_params <- list(archived = archived, author = author, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Permission Set
#' @param name string required. The name of this permission set.
#' @param description string optional. A description of this permission set.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this permission set.}
#' \item{name}{string, The name of this permission set.}
#' \item{description}{string, A description of this permission set.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
permission_sets_post <- function(name, description = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, description = description)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a Permission Set
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this permission set.}
#' \item{name}{string, The name of this permission set.}
#' \item{description}{string, A description of this permission set.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
permission_sets_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this Permission Set
#' @param id integer required. The ID for this permission set.
#' @param name string required. The name of this permission set.
#' @param description string optional. A description of this permission set.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this permission set.}
#' \item{name}{string, The name of this permission set.}
#' \item{description}{string, A description of this permission set.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
permission_sets_put <- function(id, name, description = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, description = description)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this Permission Set
#' @param id integer required. The ID for this permission set.
#' @param name string optional. The name of this permission set.
#' @param description string optional. A description of this permission set.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this permission set.}
#' \item{name}{string, The name of this permission set.}
#' \item{description}{string, A description of this permission set.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
permission_sets_patch <- function(id, name = NULL, description = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, description = description)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
permission_sets_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
permission_sets_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
permission_sets_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
permission_sets_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
permission_sets_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
permission_sets_list_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
permission_sets_put_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this permission set.}
#' \item{name}{string, The name of this permission set.}
#' \item{description}{string, A description of this permission set.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
permission_sets_put_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get all permissions for a user, in this permission set
#' @param id integer required. The ID for this permission set.
#' @param user_id integer required. The ID for the user.
#' 
#' @return  An array containing the following fields:
#' \item{resourceName}{string, The name of the resource.}
#' \item{read}{boolean, If true, the user has read permission on this resource.}
#' \item{write}{boolean, If true, the user has write permission on this resource.}
#' \item{manage}{boolean, If true, the user has manage permission on this resource.}
#' @export
permission_sets_list_users_permissions <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/users/{user_id}/permissions"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List resources in a permission set
#' @param id integer required. The ID for this permission set.
#' @param limit integer optional. Number of results to return. Defaults to 50. Maximum allowed is 1000.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to name. Must be one of: name, id, updated_at, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{permissionSetId}{integer, The ID for the permission set this resource belongs to.}
#' \item{name}{string, The name of this resource.}
#' \item{description}{string, A description of this resource.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
permission_sets_list_resources <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/resources"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a resource in a permission set
#' @param id integer required. The ID for this permission set.
#' @param name string required. The name of this resource.
#' @param description string optional. A description of this resource.
#' 
#' @return  A list containing the following elements:
#' \item{permissionSetId}{integer, The ID for the permission set this resource belongs to.}
#' \item{name}{string, The name of this resource.}
#' \item{description}{string, A description of this resource.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
permission_sets_post_resources <- function(id, name, description = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/resources"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, description = description)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a resource in a permission set
#' @param id integer required. The ID for this permission set.
#' @param name string required. The name of this resource.
#' 
#' @return  A list containing the following elements:
#' \item{permissionSetId}{integer, The ID for the permission set this resource belongs to.}
#' \item{name}{string, The name of this resource.}
#' \item{description}{string, A description of this resource.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
permission_sets_get_resources <- function(id, name) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/resources/{name}"
  path_params  <- list(id = id, name = name)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a resource in a permission set
#' @param id integer required. The ID for this permission set.
#' @param name string required. The name of this resource.
#' @param description string optional. A description of this resource.
#' 
#' @return  A list containing the following elements:
#' \item{permissionSetId}{integer, The ID for the permission set this resource belongs to.}
#' \item{name}{string, The name of this resource.}
#' \item{description}{string, A description of this resource.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
permission_sets_patch_resources <- function(id, name, description = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/resources/{name}"
  path_params  <- list(id = id, name = name)
  query_params <- list()
  body_params  <- list(description = description)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Delete a resource in a permission set
#' @param id integer required. The ID for this permission set.
#' @param name string required. The name of this resource.
#' 
#' @return  An empty HTTP response
#' @export
permission_sets_delete_resources <- function(id, name) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/resources/{name}"
  path_params  <- list(id = id, name = name)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID for this permission set.
#' @param name string required. The name of this resource.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
permission_sets_list_resources_shares <- function(id, name) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/resources/{name}/shares"
  path_params  <- list(id = id, name = name)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID for this permission set.
#' @param name string required. The name of this resource.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
permission_sets_put_resources_shares_users <- function(id, name, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/resources/{name}/shares/users"
  path_params  <- list(id = id, name = name)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID for this permission set.
#' @param name string required. The name of this resource.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
permission_sets_delete_resources_shares_users <- function(id, name, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/resources/{name}/shares/users/{user_id}"
  path_params  <- list(id = id, name = name, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID for this permission set.
#' @param name string required. The name of this resource.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
permission_sets_put_resources_shares_groups <- function(id, name, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/resources/{name}/shares/groups"
  path_params  <- list(id = id, name = name)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID for this permission set.
#' @param name string required. The name of this resource.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
permission_sets_delete_resources_shares_groups <- function(id, name, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/permission_sets/{id}/resources/{name}/shares/groups/{group_id}"
  path_params  <- list(id = id, name = name, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List predictions
#' @param model_id integer optional. If specified, only return predictions associated with this model ID.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the prediction.}
#' \item{modelId}{integer, The ID of the model used for this prediction.}
#' \item{scoredTableId}{integer, The ID of the source table for this prediction.}
#' \item{scoredTableName}{string, The name of the source table for this prediction.}
#' \item{outputTableName}{string, The name of the output table for this prediction.}
#' \item{state}{string, The state of the last run of this prediction.}
#' \item{error}{string, The error, if any, of the last run of this prediction.}
#' \item{startedAt}{string, The start time of the last run of this prediction.}
#' \item{finishedAt}{string, The end time of the last run of this prediction.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' @export
predictions_list <- function(model_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/predictions/"
  path_params  <- list()
  query_params <- list(model_id = model_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show the specified prediction
#' @param id integer required. The ID of the prediction.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the prediction.}
#' \item{modelId}{integer, The ID of the model used for this prediction.}
#' \item{scoredTableId}{integer, The ID of the source table for this prediction.}
#' \item{scoredTableName}{string, The name of the source table for this prediction.}
#' \item{outputTableName}{string, The name of the output table for this prediction.}
#' \item{state}{string, The state of the last run of this prediction.}
#' \item{error}{string, The error, if any, of the last run of this prediction.}
#' \item{startedAt}{string, The start time of the last run of this prediction.}
#' \item{finishedAt}{string, The end time of the last run of this prediction.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{scoredTables}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of the table with created predictions.
#' \item schema string, The schema of table with created predictions.
#' \item name string, The name of table with created predictions.
#' \item createdAt string, The time when the table with created predictions was created.
#' \item scoreStats array, An array of metrics on the created predictions.
#' }}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{limitingSQL}{string, A SQL WHERE clause used to scope the rows to be predicted.}
#' \item{primaryKey}{array, The primary key or composite keys of the table being predicted.}
#' @export
predictions_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/predictions/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show the prediction schedule
#' @param id integer required. ID of the prediction associated with this schedule.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, ID of the prediction associated with this schedule.}
#' \item{schedule}{list, A list containing the following elements: 
#' \itemize{
#' \item scheduled boolean, If the item is scheduled.
#' \item scheduledDays array, Days of the week, based on numeric value starting at 0 for Sunday. Mutually exclusive with scheduledDaysOfMonth
#' \item scheduledHours array, Hours of the day it is scheduled on.
#' \item scheduledMinutes array, Minutes of the day it is scheduled on.
#' \item scheduledRunsPerHour integer, Deprecated in favor of scheduled minutes.
#' \item scheduledDaysOfMonth array, Days of the month it is scheduled on, mutually exclusive with scheduledDays.
#' }}
#' \item{scoreOnModelBuild}{boolean, Whether the prediction will run after a rebuild of the associated model.}
#' @export
predictions_list_schedules <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/predictions/{id}/schedules"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List projects
#' @param permission string optional. A permissions string, one of "read", "write", or "manage". Lists only projects for which the current user has that permission.
#' @param auto_share boolean optional. Used to filter projects based on whether the project is autoshare enabled or not.
#' @param author string optional. If specified, return items from any of these authors. It accepts a comma-separated list of user IDs.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' @param archived string optional. The archival status of the requested item(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 1000.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
projects_list <- function(permission = NULL, auto_share = NULL, author = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/projects/"
  path_params  <- list()
  query_params <- list(permission = permission, auto_share = auto_share, author = author, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a project
#' @param name string required. The name of this project.
#' @param description string required. A description of the project.
#' @param note string optional. Notes for the project.
#' @param auto_share boolean optional. If true, objects within the project will be automatically shared when the project is shared or objects are added.
#' @param hidden boolean optional. The hidden status of the item.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{tables}{array, An array containing the following fields: 
#' \itemize{
#' \item schema string, 
#' \item name string, 
#' \item rowCount integer, 
#' \item columnCount integer, 
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{surveys}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{scripts}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{imports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{exports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{models}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{notebooks}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' \item lastDeploy object, 
#' }}
#' \item{services}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' \item lastDeploy object, 
#' }}
#' \item{workflows}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' \item lastExecution object, 
#' }}
#' \item{reports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{scriptTemplates}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' }}
#' \item{files}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item fileName string, 
#' \item fileSize integer, 
#' \item expired boolean, 
#' }}
#' \item{enhancements}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item lastRun object, 
#' }}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item description string, 
#' }}
#' \item{allObjects}{array, An array containing the following fields: 
#' \itemize{
#' \item projectId integer, 
#' \item objectId integer, 
#' \item objectType string, 
#' \item fcoType string, 
#' \item subType string, 
#' \item name string, 
#' \item icon string, 
#' \item author string, 
#' \item updatedAt string, 
#' \item autoShare boolean, 
#' \item archived string, The archival status of the requested item(s).
#' \item hidden boolean, The hidden status of the item.
#' \item myPermissionLevel string, Your permission level on the object. One of "read", "write", or "manage".
#' }}
#' \item{note}{string, }
#' \item{canCurrentUserEnableAutoShare}{boolean, A flag for if the current user can enable auto-sharing mode for this project.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{parentProject}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The parent project's ID.
#' \item name integer, The parent project's name.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
projects_post <- function(name, description, note = NULL, auto_share = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/projects/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, description = description, note = note, autoShare = auto_share, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Clone this 
#' @param id integer required. The ID for this project.
#' @param clone_schedule boolean optional. If true, also copy the schedule for all applicable project objects.
#' @param clone_notifications boolean optional. If true, also copy the notifications for all applicable project objects.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{tables}{array, An array containing the following fields: 
#' \itemize{
#' \item schema string, 
#' \item name string, 
#' \item rowCount integer, 
#' \item columnCount integer, 
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{surveys}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{scripts}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{imports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{exports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{models}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{notebooks}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' \item lastDeploy object, 
#' }}
#' \item{services}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' \item lastDeploy object, 
#' }}
#' \item{workflows}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' \item lastExecution object, 
#' }}
#' \item{reports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{scriptTemplates}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' }}
#' \item{files}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item fileName string, 
#' \item fileSize integer, 
#' \item expired boolean, 
#' }}
#' \item{enhancements}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item lastRun object, 
#' }}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item description string, 
#' }}
#' \item{allObjects}{array, An array containing the following fields: 
#' \itemize{
#' \item projectId integer, 
#' \item objectId integer, 
#' \item objectType string, 
#' \item fcoType string, 
#' \item subType string, 
#' \item name string, 
#' \item icon string, 
#' \item author string, 
#' \item updatedAt string, 
#' \item autoShare boolean, 
#' \item archived string, The archival status of the requested item(s).
#' \item hidden boolean, The hidden status of the item.
#' \item myPermissionLevel string, Your permission level on the object. One of "read", "write", or "manage".
#' }}
#' \item{note}{string, }
#' \item{canCurrentUserEnableAutoShare}{boolean, A flag for if the current user can enable auto-sharing mode for this project.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{parentProject}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The parent project's ID.
#' \item name integer, The parent project's name.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
projects_post_clone <- function(id, clone_schedule = NULL, clone_notifications = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/clone"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(cloneSchedule = clone_schedule, cloneNotifications = clone_notifications)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a detailed view of a project and the objects in it
#' @param project_id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{tables}{array, An array containing the following fields: 
#' \itemize{
#' \item schema string, 
#' \item name string, 
#' \item rowCount integer, 
#' \item columnCount integer, 
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{surveys}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{scripts}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{imports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{exports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{models}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{notebooks}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' \item lastDeploy object, 
#' }}
#' \item{services}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' \item lastDeploy object, 
#' }}
#' \item{workflows}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' \item lastExecution object, 
#' }}
#' \item{reports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{scriptTemplates}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' }}
#' \item{files}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item fileName string, 
#' \item fileSize integer, 
#' \item expired boolean, 
#' }}
#' \item{enhancements}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item lastRun object, 
#' }}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item description string, 
#' }}
#' \item{allObjects}{array, An array containing the following fields: 
#' \itemize{
#' \item projectId integer, 
#' \item objectId integer, 
#' \item objectType string, 
#' \item fcoType string, 
#' \item subType string, 
#' \item name string, 
#' \item icon string, 
#' \item author string, 
#' \item updatedAt string, 
#' \item autoShare boolean, 
#' \item archived string, The archival status of the requested item(s).
#' \item hidden boolean, The hidden status of the item.
#' \item myPermissionLevel string, Your permission level on the object. One of "read", "write", or "manage".
#' }}
#' \item{note}{string, }
#' \item{canCurrentUserEnableAutoShare}{boolean, A flag for if the current user can enable auto-sharing mode for this project.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{parentProject}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The parent project's ID.
#' \item name integer, The parent project's name.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
projects_get <- function(project_id) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{project_id}"
  path_params  <- list(project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a project
#' @param project_id integer required. 
#' @param name string optional. The name of this project.
#' @param description string optional. A description of the project.
#' @param note string optional. Notes for the project.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{tables}{array, An array containing the following fields: 
#' \itemize{
#' \item schema string, 
#' \item name string, 
#' \item rowCount integer, 
#' \item columnCount integer, 
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{surveys}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{scripts}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{imports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{exports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{models}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{notebooks}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' \item lastDeploy object, 
#' }}
#' \item{services}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' \item lastDeploy object, 
#' }}
#' \item{workflows}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' \item lastExecution object, 
#' }}
#' \item{reports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{scriptTemplates}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' }}
#' \item{files}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item fileName string, 
#' \item fileSize integer, 
#' \item expired boolean, 
#' }}
#' \item{enhancements}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item lastRun object, 
#' }}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item description string, 
#' }}
#' \item{allObjects}{array, An array containing the following fields: 
#' \itemize{
#' \item projectId integer, 
#' \item objectId integer, 
#' \item objectType string, 
#' \item fcoType string, 
#' \item subType string, 
#' \item name string, 
#' \item icon string, 
#' \item author string, 
#' \item updatedAt string, 
#' \item autoShare boolean, 
#' \item archived string, The archival status of the requested item(s).
#' \item hidden boolean, The hidden status of the item.
#' \item myPermissionLevel string, Your permission level on the object. One of "read", "write", or "manage".
#' }}
#' \item{note}{string, }
#' \item{canCurrentUserEnableAutoShare}{boolean, A flag for if the current user can enable auto-sharing mode for this project.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{parentProject}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The parent project's ID.
#' \item name integer, The parent project's name.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
projects_put <- function(project_id, name = NULL, description = NULL, note = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{project_id}"
  path_params  <- list(project_id = project_id)
  query_params <- list()
  body_params  <- list(name = name, description = description, note = note)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Archive a project (deprecated, use the /archive endpoint instead)
#' @param project_id integer required. 
#' 
#' @return  An empty HTTP response
#' @export
projects_delete <- function(project_id) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{project_id}"
  path_params  <- list(project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Enable or disable Auto-Share on a project
#' @param project_id integer required. 
#' @param auto_share boolean required. A toggle for sharing the objects within the project when the project is shared or objects are added.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{tables}{array, An array containing the following fields: 
#' \itemize{
#' \item schema string, 
#' \item name string, 
#' \item rowCount integer, 
#' \item columnCount integer, 
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{surveys}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{scripts}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{imports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{exports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{models}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{notebooks}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' \item lastDeploy object, 
#' }}
#' \item{services}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' \item lastDeploy object, 
#' }}
#' \item{workflows}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' \item lastExecution object, 
#' }}
#' \item{reports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{scriptTemplates}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' }}
#' \item{files}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item fileName string, 
#' \item fileSize integer, 
#' \item expired boolean, 
#' }}
#' \item{enhancements}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item lastRun object, 
#' }}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item description string, 
#' }}
#' \item{allObjects}{array, An array containing the following fields: 
#' \itemize{
#' \item projectId integer, 
#' \item objectId integer, 
#' \item objectType string, 
#' \item fcoType string, 
#' \item subType string, 
#' \item name string, 
#' \item icon string, 
#' \item author string, 
#' \item updatedAt string, 
#' \item autoShare boolean, 
#' \item archived string, The archival status of the requested item(s).
#' \item hidden boolean, The hidden status of the item.
#' \item myPermissionLevel string, Your permission level on the object. One of "read", "write", or "manage".
#' }}
#' \item{note}{string, }
#' \item{canCurrentUserEnableAutoShare}{boolean, A flag for if the current user can enable auto-sharing mode for this project.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{parentProject}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The parent project's ID.
#' \item name integer, The parent project's name.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
projects_put_auto_share <- function(project_id, auto_share) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{project_id}/auto_share"
  path_params  <- list(project_id = project_id)
  query_params <- list()
  body_params  <- list(autoShare = auto_share)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
projects_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
projects_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
projects_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
projects_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
projects_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
projects_list_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
projects_put_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{tables}{array, An array containing the following fields: 
#' \itemize{
#' \item schema string, 
#' \item name string, 
#' \item rowCount integer, 
#' \item columnCount integer, 
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{surveys}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{scripts}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{imports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{exports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item type string, 
#' \item finishedAt string, 
#' \item state string, 
#' \item lastRun object, 
#' }}
#' \item{models}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{notebooks}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' \item lastDeploy object, 
#' }}
#' \item{services}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item currentDeploymentId integer, 
#' \item lastDeploy object, 
#' }}
#' \item{workflows}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' \item lastExecution object, 
#' }}
#' \item{reports}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item state string, 
#' }}
#' \item{scriptTemplates}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' }}
#' \item{files}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item fileName string, 
#' \item fileSize integer, 
#' \item expired boolean, 
#' }}
#' \item{enhancements}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item lastRun object, 
#' }}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The item's ID.
#' \item createdAt string, 
#' \item updatedAt string, 
#' \item name string, 
#' \item description string, 
#' }}
#' \item{allObjects}{array, An array containing the following fields: 
#' \itemize{
#' \item projectId integer, 
#' \item objectId integer, 
#' \item objectType string, 
#' \item fcoType string, 
#' \item subType string, 
#' \item name string, 
#' \item icon string, 
#' \item author string, 
#' \item updatedAt string, 
#' \item autoShare boolean, 
#' \item archived string, The archival status of the requested item(s).
#' \item hidden boolean, The hidden status of the item.
#' \item myPermissionLevel string, Your permission level on the object. One of "read", "write", or "manage".
#' }}
#' \item{note}{string, }
#' \item{canCurrentUserEnableAutoShare}{boolean, A flag for if the current user can enable auto-sharing mode for this project.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{parentProject}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The parent project's ID.
#' \item name integer, The parent project's name.
#' }}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' @export
projects_put_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the Parent Projects an item belongs to
#' @param id integer required. The ID of the item.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
projects_list_parent_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/parent_projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add an item to a Parent Project
#' @param id integer required. The ID of the item.
#' @param parent_project_id integer required. The ID of the Parent Project.
#' 
#' @return  An empty HTTP response
#' @export
projects_put_parent_projects <- function(id, parent_project_id) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/parent_projects/{parent_project_id}"
  path_params  <- list(id = id, parent_project_id = parent_project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove an item from a Parent Project
#' @param id integer required. The ID of the item.
#' @param parent_project_id integer required. The ID of the Parent Project.
#' 
#' @return  An empty HTTP response
#' @export
projects_delete_parent_projects <- function(id, parent_project_id) {

  args <- as.list(match.call())[-1]
  path <- "/projects/{id}/parent_projects/{parent_project_id}"
  path_params  <- list(id = id, parent_project_id = parent_project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List queries
#' @param query string optional. Space delimited query for searching queries by their SQL. Supports wild card characters "?" for any single character, and "*" for zero or more characters.
#' @param database_id integer optional. The database ID.
#' @param credential_id integer optional. The credential ID.
#' @param author_id integer optional. The author of the query.
#' @param created_before string optional. An upper bound for the creation date of the query.
#' @param created_after string optional. A lower bound for the creation date of the query.
#' @param started_before string optional. An upper bound for the start date of the last run.
#' @param started_after string optional. A lower bound for the start date of the last run.
#' @param state array optional. The state of the last run. One or more of queued, running, succeeded, failed, and cancelled. Specify multiple values as a comma-separated list (e.g., "A,B").
#' @param exclude_results boolean optional. If true, does not return cached query results.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' @param archived string optional. The archival status of the requested item(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to created_at. Must be one of: created_at, started_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The query ID.}
#' \item{database}{integer, The database ID.}
#' \item{sql}{string, The SQL to execute.}
#' \item{credential}{integer, The credential ID.}
#' \item{resultRows}{array, A preview of rows returned by the query.}
#' \item{resultColumns}{array, A preview of columns returned by the query.}
#' \item{scriptId}{integer, The ID of the script associated with this query.}
#' \item{exception}{string, Deprecated and not used.}
#' \item{error}{string, The error message for this run, if present.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{startedAt}{string, The start time of the last run.}
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run. One of queued, running, succeeded, failed, and cancelled.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{previewRows}{integer, The number of rows to save from the query's result (maximum: 100).}
#' \item{reportId}{integer, The ID of the report associated with this query.}
#' @export
queries_list <- function(query = NULL, database_id = NULL, credential_id = NULL, author_id = NULL, created_before = NULL, created_after = NULL, started_before = NULL, started_after = NULL, state = NULL, exclude_results = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/queries/"
  path_params  <- list()
  query_params <- list(query = query, database_id = database_id, credential_id = credential_id, author_id = author_id, created_before = created_before, created_after = created_after, started_before = started_before, started_after = started_after, state = state, exclude_results = exclude_results, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Execute a query
#' @param database integer required. The database ID.
#' @param sql string required. The SQL to execute.
#' @param preview_rows integer required. The number of rows to save from the query's result (maximum: 100).
#' @param credential integer optional. The credential ID.
#' @param hidden boolean optional. The hidden status of the item.
#' @param interactive boolean optional. Deprecated and not used.
#' @param include_header boolean optional. Whether the CSV output should include a header row [default: true].
#' @param compression string optional. The type of compression. One of gzip or zip, or none [default: gzip].
#' @param column_delimiter string optional. The delimiter to use. One of comma or tab, or pipe [default: comma].
#' @param unquoted boolean optional. If true, will not quote fields.
#' @param filename_prefix string optional. The output filename prefix.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The query ID.}
#' \item{database}{integer, The database ID.}
#' \item{sql}{string, The SQL to execute.}
#' \item{credential}{integer, The credential ID.}
#' \item{resultRows}{array, A preview of rows returned by the query.}
#' \item{resultColumns}{array, A preview of columns returned by the query.}
#' \item{scriptId}{integer, The ID of the script associated with this query.}
#' \item{exception}{string, Deprecated and not used.}
#' \item{error}{string, The error message for this run, if present.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{startedAt}{string, The start time of the last run.}
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run. One of queued, running, succeeded, failed, and cancelled.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{interactive}{boolean, Deprecated and not used.}
#' \item{previewRows}{integer, The number of rows to save from the query's result (maximum: 100).}
#' \item{includeHeader}{boolean, Whether the CSV output should include a header row [default: true].}
#' \item{compression}{string, The type of compression. One of gzip or zip, or none [default: gzip].}
#' \item{columnDelimiter}{string, The delimiter to use. One of comma or tab, or pipe [default: comma].}
#' \item{unquoted}{boolean, If true, will not quote fields.}
#' \item{filenamePrefix}{string, The output filename prefix.}
#' \item{reportId}{integer, The ID of the report associated with this query.}
#' @export
queries_post <- function(database, sql, preview_rows, credential = NULL, hidden = NULL, interactive = NULL, include_header = NULL, compression = NULL, column_delimiter = NULL, unquoted = NULL, filename_prefix = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/queries/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(database = database, sql = sql, previewRows = preview_rows, credential = credential, hidden = hidden, interactive = interactive, includeHeader = include_header, compression = compression, columnDelimiter = column_delimiter, unquoted = unquoted, filenamePrefix = filename_prefix)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Start a run
#' @param id integer required. The ID of the query.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{queryId}{integer, The ID of the query.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
queries_post_runs <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List runs for the given query
#' @param id integer required. The ID of the query.
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 100.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the run.}
#' \item{queryId}{integer, The ID of the query.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
queries_list_runs <- function(id, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}/runs"
  path_params  <- list(id = id)
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Check status of a run
#' @param id integer required. The ID of the query.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the run.}
#' \item{queryId}{integer, The ID of the query.}
#' \item{state}{string, The state of the run, one of 'queued' 'running' 'succeeded' 'failed' or 'cancelled'.}
#' \item{isCancelRequested}{boolean, True if run cancel requested, else false.}
#' \item{createdAt}{string, The time the run was created.}
#' \item{startedAt}{string, The time the run started at.}
#' \item{finishedAt}{string, The time the run completed.}
#' \item{error}{string, The error, if any, returned by the run.}
#' @export
queries_get_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Cancel a run
#' @param id integer required. The ID of the query.
#' @param run_id integer required. The ID of the run.
#' 
#' @return  An empty HTTP response
#' @export
queries_delete_runs <- function(id, run_id) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}/runs/{run_id}"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the logs for a run
#' @param id integer required. The ID of the query.
#' @param run_id integer required. The ID of the run.
#' @param last_id integer optional. The ID of the last log message received. Log entries with this ID value or lower will be omitted.Logs are sorted by ID if this value is provided, and are otherwise sorted by createdAt.
#' @param limit integer optional. The maximum number of log messages to return. Default of 10000.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the log.}
#' \item{createdAt}{string, The time the log was created.}
#' \item{message}{string, The log message.}
#' \item{level}{string, The level of the log. One of unknown,fatal,error,warn,info,debug.}
#' @export
queries_list_runs_logs <- function(id, run_id, last_id = NULL, limit = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}/runs/{run_id}/logs"
  path_params  <- list(id = id, run_id = run_id)
  query_params <- list(last_id = last_id, limit = limit)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the query's associated script
#' @param id integer required. The query ID.
#' @param script_id integer required. The ID of the script associated with this query.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The query ID.}
#' \item{database}{integer, The database ID.}
#' \item{sql}{string, The SQL to execute.}
#' \item{credential}{integer, The credential ID.}
#' \item{resultRows}{array, A preview of rows returned by the query.}
#' \item{resultColumns}{array, A preview of columns returned by the query.}
#' \item{scriptId}{integer, The ID of the script associated with this query.}
#' \item{exception}{string, Deprecated and not used.}
#' \item{error}{string, The error message for this run, if present.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{startedAt}{string, The start time of the last run.}
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run. One of queued, running, succeeded, failed, and cancelled.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{name}{string, The name of the query.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{reportId}{integer, The ID of the report associated with this query.}
#' @export
queries_put_scripts <- function(id, script_id) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}/scripts/{script_id}"
  path_params  <- list(id = id, script_id = script_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get details about a query
#' @param id integer required. The query ID.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The query ID.}
#' \item{database}{integer, The database ID.}
#' \item{sql}{string, The SQL to execute.}
#' \item{credential}{integer, The credential ID.}
#' \item{resultRows}{array, A preview of rows returned by the query.}
#' \item{resultColumns}{array, A preview of columns returned by the query.}
#' \item{scriptId}{integer, The ID of the script associated with this query.}
#' \item{exception}{string, Deprecated and not used.}
#' \item{error}{string, The error message for this run, if present.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{startedAt}{string, The start time of the last run.}
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run. One of queued, running, succeeded, failed, and cancelled.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{name}{string, The name of the query.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{reportId}{integer, The ID of the report associated with this query.}
#' @export
queries_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Sets Query Hidden to true
#' @param id integer required. The query ID.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The query ID.}
#' \item{database}{integer, The database ID.}
#' \item{sql}{string, The SQL to execute.}
#' \item{credential}{integer, The credential ID.}
#' \item{resultRows}{array, A preview of rows returned by the query.}
#' \item{resultColumns}{array, A preview of columns returned by the query.}
#' \item{scriptId}{integer, The ID of the script associated with this query.}
#' \item{exception}{string, Deprecated and not used.}
#' \item{error}{string, The error message for this run, if present.}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{startedAt}{string, The start time of the last run.}
#' \item{finishedAt}{string, The end time of the last run.}
#' \item{state}{string, The state of the last run. One of queued, running, succeeded, failed, and cancelled.}
#' \item{lastRunId}{integer, The ID of the last run.}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{name}{string, The name of the query.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{reportId}{integer, The ID of the report associated with this query.}
#' @export
queries_delete <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/queries/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Remote Hosts
#' @param type string optional. The type of remote host. One of: RemoteHostTypes::Bigquery, RemoteHostTypes::Bitbucket, RemoteHostTypes::GitSSH, RemoteHostTypes::Github, RemoteHostTypes::GoogleDoc, RemoteHostTypes::JDBC, RemoteHostTypes::Postgres, RemoteHostTypes::Redshift, RemoteHostTypes::S3Storage, and RemoteHostTypes::Salesforce
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of the remote host.}
#' \item{name}{string, The human readable name for the remote host.}
#' \item{type}{string, The type of remote host. One of: RemoteHostTypes::Bigquery, RemoteHostTypes::Bitbucket, RemoteHostTypes::GitSSH, RemoteHostTypes::Github, RemoteHostTypes::GoogleDoc, RemoteHostTypes::JDBC, RemoteHostTypes::Postgres, RemoteHostTypes::Redshift, RemoteHostTypes::S3Storage, and RemoteHostTypes::Salesforce}
#' \item{url}{string, The URL for the remote host.}
#' @export
remote_hosts_list <- function(type = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/"
  path_params  <- list()
  query_params <- list(type = type)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a Remote Host
#' @param name string required. The human readable name for the remote host.
#' @param url string required. The URL for the remote host.
#' @param type string required. The type of remote host. One of: RemoteHostTypes::Bigquery, RemoteHostTypes::Bitbucket, RemoteHostTypes::GitSSH, RemoteHostTypes::Github, RemoteHostTypes::GoogleDoc, RemoteHostTypes::JDBC, RemoteHostTypes::Postgres, RemoteHostTypes::Redshift, RemoteHostTypes::S3Storage, and RemoteHostTypes::Salesforce
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the remote host.}
#' \item{name}{string, The human readable name for the remote host.}
#' \item{type}{string, The type of remote host. One of: RemoteHostTypes::Bigquery, RemoteHostTypes::Bitbucket, RemoteHostTypes::GitSSH, RemoteHostTypes::Github, RemoteHostTypes::GoogleDoc, RemoteHostTypes::JDBC, RemoteHostTypes::Postgres, RemoteHostTypes::Redshift, RemoteHostTypes::S3Storage, and RemoteHostTypes::Salesforce}
#' \item{url}{string, The URL for the remote host.}
#' \item{description}{string, The description of the remote host.}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
remote_hosts_post <- function(name, url, type) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(name = name, url = url, type = type)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get a Remote Host
#' @param id integer required. 
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the remote host.}
#' \item{name}{string, The human readable name for the remote host.}
#' \item{type}{string, The type of remote host. One of: RemoteHostTypes::Bigquery, RemoteHostTypes::Bitbucket, RemoteHostTypes::GitSSH, RemoteHostTypes::Github, RemoteHostTypes::GoogleDoc, RemoteHostTypes::JDBC, RemoteHostTypes::Postgres, RemoteHostTypes::Redshift, RemoteHostTypes::S3Storage, and RemoteHostTypes::Salesforce}
#' \item{url}{string, The URL for the remote host.}
#' \item{description}{string, The description of the remote host.}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
remote_hosts_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Replace all attributes of this Remote Host
#' @param id integer required. The ID of the remote host.
#' @param name string required. The human readable name for the remote host.
#' @param type string required. The type of remote host. One of: RemoteHostTypes::Bigquery, RemoteHostTypes::Bitbucket, RemoteHostTypes::GitSSH, RemoteHostTypes::Github, RemoteHostTypes::GoogleDoc, RemoteHostTypes::JDBC, RemoteHostTypes::Postgres, RemoteHostTypes::Redshift, RemoteHostTypes::S3Storage, and RemoteHostTypes::Salesforce
#' @param url string required. The URL for the remote host.
#' @param description string required. The description of the remote host.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the remote host.}
#' \item{name}{string, The human readable name for the remote host.}
#' \item{type}{string, The type of remote host. One of: RemoteHostTypes::Bigquery, RemoteHostTypes::Bitbucket, RemoteHostTypes::GitSSH, RemoteHostTypes::Github, RemoteHostTypes::GoogleDoc, RemoteHostTypes::JDBC, RemoteHostTypes::Postgres, RemoteHostTypes::Redshift, RemoteHostTypes::S3Storage, and RemoteHostTypes::Salesforce}
#' \item{url}{string, The URL for the remote host.}
#' \item{description}{string, The description of the remote host.}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
remote_hosts_put <- function(id, name, type, url, description) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, type = type, url = url, description = description)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this Remote Host
#' @param id integer required. The ID of the remote host.
#' @param name string optional. The human readable name for the remote host.
#' @param type string optional. The type of remote host. One of: RemoteHostTypes::Bigquery, RemoteHostTypes::Bitbucket, RemoteHostTypes::GitSSH, RemoteHostTypes::Github, RemoteHostTypes::GoogleDoc, RemoteHostTypes::JDBC, RemoteHostTypes::Postgres, RemoteHostTypes::Redshift, RemoteHostTypes::S3Storage, and RemoteHostTypes::Salesforce
#' @param url string optional. The URL for the remote host.
#' @param description string optional. The description of the remote host.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of the remote host.}
#' \item{name}{string, The human readable name for the remote host.}
#' \item{type}{string, The type of remote host. One of: RemoteHostTypes::Bigquery, RemoteHostTypes::Bitbucket, RemoteHostTypes::GitSSH, RemoteHostTypes::Github, RemoteHostTypes::GoogleDoc, RemoteHostTypes::JDBC, RemoteHostTypes::Postgres, RemoteHostTypes::Redshift, RemoteHostTypes::S3Storage, and RemoteHostTypes::Salesforce}
#' \item{url}{string, The URL for the remote host.}
#' \item{description}{string, The description of the remote host.}
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' @export
remote_hosts_patch <- function(id, name = NULL, type = NULL, url = NULL, description = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, type = type, url = url, description = description)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
remote_hosts_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
remote_hosts_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
remote_hosts_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
remote_hosts_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
remote_hosts_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Authenticate against a remote host using either a credential or a user name and password
#' @param id integer required. The ID of the remote host.
#' @param credential_id integer optional. The credential ID.
#' @param username string optional. The user name for remote host.
#' @param password string optional. The password for remote host.
#' 
#' @return  An empty HTTP response
#' @export
remote_hosts_post_authenticate <- function(id, credential_id = NULL, username = NULL, password = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/{id}/authenticate"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(credentialId = credential_id, username = username, password = password)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List data sets available from a remote host
#' @param id integer required. The ID of the remote host.
#' @param credential_id integer optional. The credential ID.
#' @param username string optional. The user name for remote host.
#' @param password string optional. The password for remote host.
#' @param q string optional. The query string for data set.
#' @param s boolean optional. If true will only return schemas, otherwise, the results will be the full path.
#' 
#' @return  An array containing the following fields:
#' \item{name}{string, The path to a data_set.}
#' \item{fullPath}{boolean, Boolean that indicates whether further querying needs to be done before the table can be selected.}
#' @export
remote_hosts_list_data_sets <- function(id, credential_id = NULL, username = NULL, password = NULL, q = NULL, s = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/remote_hosts/{id}/data_sets"
  path_params  <- list(id = id)
  query_params <- list(credential_id = credential_id, username = username, password = password, q = q, s = s)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Reports
#' @param type string optional. If specified, return report of these types. It accepts a comma-separated list, possible values are 'tableau' or 'other'.
#' @param template_id integer optional. If specified, return reports using the provided Template.
#' @param author string optional. If specified, return items from any of these authors. It accepts a comma-separated list of user IDs.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' @param archived string optional. The archival status of the requested item(s).
#' @param limit integer optional. Number of results to return. Defaults to 20. Maximum allowed is 50.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to updated_at. Must be one of: updated_at, name, created_at.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to desc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
reports_list <- function(type = NULL, template_id = NULL, author = NULL, hidden = NULL, archived = NULL, limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/"
  path_params  <- list()
  query_params <- list(type = type, template_id = template_id, author = author, hidden = hidden, archived = archived, limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a report
#' @param script_id integer optional. The ID of the job (a script or a query) used to create this report.
#' @param name string optional. The name of the report.
#' @param code_body string optional. The code for the report visualization.
#' @param app_state list optional. Any application state blob for this report.
#' @param provide_api_key boolean optional. Allow the report to provide an API key to front-end code.
#' @param template_id integer optional. The ID of the template used for this report.
#' @param hidden boolean optional. The hidden status of the item.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{list, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
reports_post <- function(script_id = NULL, name = NULL, code_body = NULL, app_state = NULL, provide_api_key = NULL, template_id = NULL, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(scriptId = script_id, name = name, codeBody = code_body, appState = app_state, provideAPIKey = provide_api_key, templateId = template_id, hidden = hidden)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the git metadata attached to an item
#' @param id integer required. The ID of the file.
#' 
#' @return  A list containing the following elements:
#' \item{gitRef}{string, A git reference specifying an unambiguous version of the file. Can be a branch name, tag or the full or shortened SHA of a commit.}
#' \item{gitBranch}{string, The git branch that the file is on.}
#' \item{gitPath}{string, The path of the file in the repository.}
#' \item{gitRepo}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for this git repository.
#' \item repoUrl string, The URL for this git repository.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{gitRefType}{string, Specifies if the file is versioned by branch or tag.}
#' \item{pullFromGit}{boolean, Automatically pull latest commit from git. Only works for scripts and workflows (assuming you have the feature enabled)}
#' @export
reports_list_git <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/git"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Attach an item to a file in a git repo
#' @param id integer required. The ID of the file.
#' @param git_ref string optional. A git reference specifying an unambiguous version of the file. Can be a branch name, or the full or shortened SHA of a commit.
#' @param git_branch string optional. The git branch that the file is on.
#' @param git_path string optional. The path of the file in the repository.
#' @param git_repo_url string optional. The URL of the git repository.
#' @param git_ref_type string optional. Specifies if the file is versioned by branch or tag.
#' @param pull_from_git boolean optional. Automatically pull latest commit from git. Only works for scripts.
#' 
#' @return  A list containing the following elements:
#' \item{gitRef}{string, A git reference specifying an unambiguous version of the file. Can be a branch name, tag or the full or shortened SHA of a commit.}
#' \item{gitBranch}{string, The git branch that the file is on.}
#' \item{gitPath}{string, The path of the file in the repository.}
#' \item{gitRepo}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for this git repository.
#' \item repoUrl string, The URL for this git repository.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{gitRefType}{string, Specifies if the file is versioned by branch or tag.}
#' \item{pullFromGit}{boolean, Automatically pull latest commit from git. Only works for scripts and workflows (assuming you have the feature enabled)}
#' @export
reports_put_git <- function(id, git_ref = NULL, git_branch = NULL, git_path = NULL, git_repo_url = NULL, git_ref_type = NULL, pull_from_git = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/git"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(gitRef = git_ref, gitBranch = git_branch, gitPath = git_path, gitRepoUrl = git_repo_url, gitRefType = git_ref_type, pullFromGit = pull_from_git)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update an attached git file
#' @param id integer required. The ID of the file.
#' @param git_ref string optional. A git reference specifying an unambiguous version of the file. Can be a branch name, or the full or shortened SHA of a commit.
#' @param git_branch string optional. The git branch that the file is on.
#' @param git_path string optional. The path of the file in the repository.
#' @param git_repo_url string optional. The URL of the git repository.
#' @param git_ref_type string optional. Specifies if the file is versioned by branch or tag.
#' @param pull_from_git boolean optional. Automatically pull latest commit from git. Only works for scripts.
#' 
#' @return  A list containing the following elements:
#' \item{gitRef}{string, A git reference specifying an unambiguous version of the file. Can be a branch name, tag or the full or shortened SHA of a commit.}
#' \item{gitBranch}{string, The git branch that the file is on.}
#' \item{gitPath}{string, The path of the file in the repository.}
#' \item{gitRepo}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for this git repository.
#' \item repoUrl string, The URL for this git repository.
#' \item createdAt string, 
#' \item updatedAt string, 
#' }}
#' \item{gitRefType}{string, Specifies if the file is versioned by branch or tag.}
#' \item{pullFromGit}{boolean, Automatically pull latest commit from git. Only works for scripts and workflows (assuming you have the feature enabled)}
#' @export
reports_patch_git <- function(id, git_ref = NULL, git_branch = NULL, git_path = NULL, git_repo_url = NULL, git_ref_type = NULL, pull_from_git = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/git"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(gitRef = git_ref, gitBranch = git_branch, gitPath = git_path, gitRepoUrl = git_repo_url, gitRefType = git_ref_type, pullFromGit = pull_from_git)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get the git commits for an item on the current branch
#' @param id integer required. The ID of the file.
#' 
#' @return  A list containing the following elements:
#' \item{commitHash}{string, The SHA of the commit.}
#' \item{authorName}{string, The name of the commit's author.}
#' \item{date}{string, The commit's timestamp.}
#' \item{message}{string, The commit message.}
#' @export
reports_list_git_commits <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/git/commits"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Commit and push a new version of the file
#' @param id integer required. The ID of the file.
#' @param content string required. The contents to commit to the file.
#' @param message string required. A commit message describing the changes being made.
#' @param file_hash string required. The full SHA of the file being replaced.
#' 
#' @return  A list containing the following elements:
#' \item{content}{string, The file's contents.}
#' \item{type}{string, The file's type.}
#' \item{size}{integer, The file's size.}
#' \item{fileHash}{string, The SHA of the file.}
#' @export
reports_post_git_commits <- function(id, content, message, file_hash) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/git/commits"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(content = content, message = message, fileHash = file_hash)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Get file contents at git ref
#' @param id integer required. The ID of the file.
#' @param commit_hash string required. The SHA (full or shortened) of the desired git commit.
#' 
#' @return  A list containing the following elements:
#' \item{content}{string, The file's contents.}
#' \item{type}{string, The file's type.}
#' \item{size}{integer, The file's size.}
#' \item{fileHash}{string, The SHA of the file.}
#' @export
reports_get_git_commits <- function(id, commit_hash) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/git/commits/{commit_hash}"
  path_params  <- list(id = id, commit_hash = commit_hash)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Checkout latest commit on the current branch of a script or workflow
#' @param id integer required. The ID of the file.
#' 
#' @return  A list containing the following elements:
#' \item{content}{string, The file's contents.}
#' \item{type}{string, The file's type.}
#' \item{size}{integer, The file's size.}
#' \item{fileHash}{string, The SHA of the file.}
#' @export
reports_post_git_checkout_latest <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/git/checkout-latest"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Checkout content that the existing git_ref points to and save to the object
#' @param id integer required. The ID of the file.
#' 
#' @return  A list containing the following elements:
#' \item{content}{string, The file's contents.}
#' \item{type}{string, The file's type.}
#' \item{size}{integer, The file's size.}
#' \item{fileHash}{string, The SHA of the file.}
#' @export
reports_post_git_checkout <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/git/checkout"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show a single report
#' @param id integer required. The ID of this report.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{list, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
reports_get <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update a report
#' @param id integer required. The ID of the report to modify.
#' @param name string optional. The name of the report.
#' @param script_id integer optional. The ID of the job (a script or a query) used to create this report.
#' @param code_body string optional. The code for the report visualization.
#' @param config string optional. 
#' @param app_state list optional. The application state blob for this report.
#' @param provide_api_key boolean optional. Allow the report to provide an API key to front-end code.
#' @param template_id integer optional. The ID of the template used for this report. If null is passed, no template will back this report. Changes to the backing template will reset the report appState.
#' @param use_viewers_tableau_username boolean optional. Apply user level filtering on Tableau reports.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{list, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
reports_patch <- function(id, name = NULL, script_id = NULL, code_body = NULL, config = NULL, app_state = NULL, provide_api_key = NULL, template_id = NULL, use_viewers_tableau_username = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, scriptId = script_id, codeBody = code_body, config = config, appState = app_state, provideAPIKey = provide_api_key, templateId = template_id, useViewersTableauUsername = use_viewers_tableau_username)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Grant this report the ability to perform Civis platform API operations on your behalf
#' @param id integer required. The ID of this report.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{list, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
reports_post_grants <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/grants"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke permission for this report to perform Civis platform API operations on your behalf
#' @param id integer required. The ID of this report.
#' 
#' @return  An empty HTTP response
#' @export
reports_delete_grants <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/grants"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
reports_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
reports_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
reports_delete_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
reports_put_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
reports_delete_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
reports_list_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
reports_put_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a Report belongs to
#' @param id integer required. The ID of the Report.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
reports_list_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a Report to a project
#' @param id integer required. The ID of the Report.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
reports_put_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a Report from a project
#' @param id integer required. The ID of the Report.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
reports_delete_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{projects}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID for the project.
#' \item name string, The name of the project.
#' }}
#' \item{state}{string, The status of the report's last run.}
#' \item{finishedAt}{string, The time that the report's last run finished.}
#' \item{vizUpdatedAt}{string, The time that the report's visualization was last updated.}
#' \item{script}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID for the script.
#' \item name string, The name of the script.
#' \item sql string, The raw SQL query for the script.
#' }}
#' \item{jobPath}{string, The link to details of the job that backs this report.}
#' \item{tableauId}{integer, }
#' \item{type}{string, }
#' \item{templateId}{integer, The ID of the template used for this report.}
#' \item{authThumbnailUrl}{string, URL for a thumbnail of the report.}
#' \item{lastRun}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, 
#' \item state string, 
#' \item createdAt string, The time that the run was queued.
#' \item startedAt string, The time that the run started.
#' \item finishedAt string, The time that the run completed.
#' \item error string, The error message for this run, if present.
#' }}
#' \item{archived}{string, The archival status of the requested item(s).}
#' \item{hidden}{boolean, The hidden status of the item.}
#' \item{authDataUrl}{string, }
#' \item{authCodeUrl}{string, }
#' \item{config}{string, Any configuration metadata for this report.}
#' \item{validOutputFile}{boolean, Whether the job (a script or a query) that backs the report currently has a valid output file.}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{appState}{list, Any application state blob for this report.}
#' \item{useViewersTableauUsername}{boolean, Apply user level filtering on Tableau reports.}
#' @export
reports_put_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Show a single service report
#' @param id integer required. The ID of this report.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{host}{string, The host for the service report}
#' \item{displayUrl}{string, The URL to display the service report.}
#' \item{serviceId}{integer, The id of the backing service}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
reports_get_services <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update some attributes of this service report
#' @param id integer required. The ID of this report.
#' @param name string optional. The name of the service report.
#' @param provide_api_key boolean optional. Whether the report requests an API Key from the report viewer.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{host}{string, The host for the service report}
#' \item{displayUrl}{string, The URL to display the service report.}
#' \item{serviceId}{integer, The id of the backing service}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
reports_patch_services <- function(id, name = NULL, provide_api_key = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(name = name, provideAPIKey = provide_api_key)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PATCH", path, path_params, query_params, body_params)

  return(resp)

 }


#' Create a service report
#' @param service_id integer required. The id of the backing service
#' @param provide_api_key boolean optional. Whether the report requests an API Key from the report viewer.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{host}{string, The host for the service report}
#' \item{displayUrl}{string, The URL to display the service report.}
#' \item{serviceId}{integer, The id of the backing service}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
reports_post_services <- function(service_id, provide_api_key = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services"
  path_params  <- list()
  query_params <- list()
  body_params  <- list(serviceId = service_id, provideAPIKey = provide_api_key)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
reports_list_services_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
reports_put_services_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userIds = user_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a user has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. The ID of the user.
#' 
#' @return  An empty HTTP response
#' @export
reports_delete_services_shares_users <- function(id, user_id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/shares/users/{user_id}"
  path_params  <- list(id = id, user_id = user_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions groups has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_ids array required. An array of one or more group IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
reports_put_services_shares_groups <- function(id, group_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/shares/groups"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(groupIds = group_ids, permissionLevel = permission_level, shareEmailBody = share_email_body, sendSharedEmail = send_shared_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Revoke the permissions a group has on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param group_id integer required. The ID of the group.
#' 
#' @return  An empty HTTP response
#' @export
reports_delete_services_shares_groups <- function(id, group_id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/shares/groups/{group_id}"
  path_params  <- list(id = id, group_id = group_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' List dependent objects for this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer optional. ID of target user
#' 
#' @return  An array containing the following fields:
#' \item{objectType}{string, Dependent object type}
#' \item{fcoType}{string, Human readable dependent object type}
#' \item{id}{integer, Dependent object ID}
#' \item{name}{string, Dependent object name, or nil if the requesting user cannot read this object}
#' \item{permissionLevel}{string, Permission level of target user (not user's groups) for dependent object, or null if no target user}
#' \item{shareable}{boolean, Whether or not the requesting user can share this object.}
#' @export
reports_list_services_dependencies <- function(id, user_id = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/dependencies"
  path_params  <- list(id = id)
  query_params <- list(user_id = user_id)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Transfer ownership of this object to another user
#' @param id integer required. The ID of the resource that is shared.
#' @param user_id integer required. ID of target user
#' @param include_dependencies boolean required. Whether or not to give manage permissions on all dependencies
#' @param email_body string optional. Custom body text for e-mail sent on transfer.
#' @param send_email boolean optional. Send email to the target user of the transfer?
#' 
#' @return  A list containing the following elements:
#' \item{dependencies}{array, An array containing the following fields: 
#' \itemize{
#' \item objectType string, Dependent object type
#' \item fcoType string, Human readable dependent object type
#' \item id integer, Dependent object ID
#' \item name string, Dependent object name, or nil if the requesting user cannot read this object
#' \item permissionLevel string, Permission level of target user (not user's groups) for dependent object, or null if no target user
#' \item shared boolean, Whether dependent object was successfully shared with target user
#' }}
#' @export
reports_put_services_transfer <- function(id, user_id, include_dependencies, email_body = NULL, send_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/transfer"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(userId = user_id, includeDependencies = include_dependencies, emailBody = email_body, sendEmail = send_email)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' List the projects a Service Report belongs to
#' @param id integer required. The ID of the Service Report.
#' @param hidden boolean optional. If specified to be true, returns hidden items. Defaults to false, returning non-hidden items.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, The ID for this project.}
#' \item{author}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{name}{string, The name of this project.}
#' \item{description}{string, A description of the project.}
#' \item{users}{array, An array containing the following fields: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{autoShare}{boolean, }
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
reports_list_services_projects <- function(id, hidden = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/projects"
  path_params  <- list(id = id)
  query_params <- list(hidden = hidden)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Add a Service Report to a project
#' @param id integer required. The ID of the Service Report.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
reports_put_services_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Remove a Service Report from a project
#' @param id integer required. The ID of the Service Report.
#' @param project_id integer required. The ID of the project.
#' 
#' @return  An empty HTTP response
#' @export
reports_delete_services_projects <- function(id, project_id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/projects/{project_id}"
  path_params  <- list(id = id, project_id = project_id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("DELETE", path, path_params, query_params, body_params)

  return(resp)

 }


#' Update the archive status of this object
#' @param id integer required. The ID of the object.
#' @param status boolean required. The desired archived status of the object.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{name}{string, The name of the report.}
#' \item{user}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this user.
#' \item name string, This user's name.
#' \item username string, This user's username.
#' \item initials string, This user's initials.
#' \item online boolean, Whether this user is online.
#' }}
#' \item{createdAt}{string, }
#' \item{updatedAt}{string, }
#' \item{myPermissionLevel}{string, Your permission level on the object. One of "read", "write", or "manage".}
#' \item{host}{string, The host for the service report}
#' \item{displayUrl}{string, The URL to display the service report.}
#' \item{serviceId}{integer, The id of the backing service}
#' \item{provideAPIKey}{boolean, Whether the report requests an API Key from the report viewer.}
#' \item{apiKey}{string, A Civis API key that can be used by this report.}
#' \item{apiKeyId}{integer, The ID of the API key. Can be used for auditing API use by this report.}
#' \item{archived}{string, The archival status of the requested item(s).}
#' @export
reports_put_services_archive <- function(id, status) {

  args <- as.list(match.call())[-1]
  path <- "/reports/services/{id}/archive"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list(status = status)
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("PUT", path, path_params, query_params, body_params)

  return(resp)

 }


#' Refresh the data in this Tableau report
#' @param id integer required. The ID of this report.
#' 
#' @return  A list containing the following elements:
#' \item{id}{integer, The ID of this report.}
#' \item{organization}{list, A list containing the following elements: 
#' \itemize{
#' \item id integer, The ID of this organization.
#' \item tableauRefreshUsage integer, The number of tableau refreshes used this month.
#' \item tableauRefreshLimit integer, The number of monthly tableau refreshes permitted to this organization.
#' \item tableauRefreshHistory array, The number of tableau refreshes used this month.
#' }}
#' @export
reports_post_refresh <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/reports/{id}/refresh"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("POST", path, path_params, query_params, body_params)

  return(resp)

 }


#' List Roles
#' @param limit integer optional. Number of results to return. Defaults to 50. Maximum allowed is 1000.
#' @param page_num integer optional. Page number of the results to return. Defaults to the first page, 1.
#' @param order string optional. The field on which to order the result set. Defaults to id. Must be one of: id.
#' @param order_dir string optional. Direction in which to sort, either asc (ascending) or desc (descending) defaulting to asc.
#' 
#' @return  An array containing the following fields:
#' \item{id}{integer, ID of the Role.}
#' \item{name}{string, The name of the Role.}
#' \item{slug}{string, The slug.}
#' \item{description}{string, The description of the Role.}
#' @export
roles_list <- function(limit = NULL, page_num = NULL, order = NULL, order_dir = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/roles/"
  path_params  <- list()
  query_params <- list(limit = limit, page_num = page_num, order = order, order_dir = order_dir)
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' List users and groups permissioned on this object
#' @param id integer required. The ID of the resource that is shared.
#' 
#' @return  An array containing the following fields:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
saml_service_providers_list_shares <- function(id) {

  args <- as.list(match.call())[-1]
  path <- "/saml_service_providers/{id}/shares"
  path_params  <- list(id = id)
  query_params <- list()
  body_params  <- list()
  path_params  <- path_params[match_params(path_params, args)]
  query_params <- query_params[match_params(query_params, args)]
  body_params  <- body_params[match_params(body_params, args)]
  resp <- call_api("GET", path, path_params, query_params, body_params)

  return(resp)

 }


#' Set the permissions users have on this object
#' @param id integer required. The ID of the resource that is shared.
#' @param user_ids array required. An array of one or more user IDs.
#' @param permission_level string required. Options are: "read", "write", or "manage".
#' @param share_email_body string optional. Custom body text for e-mail sent on a share.
#' @param send_shared_email boolean optional. Send email to the recipients of a share.
#' 
#' @return  A list containing the following elements:
#' \item{readers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{writers}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{owners}{list, A list containing the following elements: 
#' \itemize{
#' \item users array, 
#' \item groups array, 
#' }}
#' \item{totalUserShares}{integer, For owners, the number of total users shared. For writers and readers, the number of visible users shared.}
#' \item{totalGroupShares}{integer, For owners, the number of total groups shared. For writers and readers, the number of visible groups shared.}
#' @export
saml_service_providers_put_shares_users <- function(id, user_ids, permission_level, share_email_body = NULL, send_shared_email = NULL) {

  args <- as.list(match.call())[-1]
  path <- "/saml_service_providers/{id}/shares/users"
  path_params  <- list(id = id)
  query_params <- list