% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pot.R
\name{fit_pot}
\alias{fit_pot}
\title{Fit a peaks-over-threshold model to exceedances over a threshold}
\usage{
fit_pot(y, x = NULL, threshold, locationFun = NULL, scaleFun = NULL,
  shapeFun = NULL, nBlocks = nrow(x), blockIndex = NULL, firstBlock = 1,
  index = NULL, nReplicates = 1, replicateIndex = NULL, weights = NULL,
  proportionMissing = NULL, returnPeriod = NULL, returnValue = NULL,
  getParams = FALSE, getFit = FALSE, xNew = NULL, xContrast = NULL,
  declustering = NULL, upperTail = TRUE, scaling = 1, bootSE = FALSE,
  bootControl = NULL, optimArgs = NULL, initial = NULL, logScale = NULL,
  .normalizeX = TRUE, .getInputs = FALSE)
}
\arguments{
\item{y}{a numeric vector of exceedance values (values of the outcome variable above the threshold).}

\item{x}{a data frame, or object that can be converted to a data frame with columns corresponding to covariate/predictor/feature variables and each row containing the values of the variable for a block (e.g., often a year with climate data). The number of rows must equal the number of blocks.}

\item{threshold}{a single numeric value for constant threshold or a numeric vector with length equal to the number of blocks, indicating the threshold for each block.}

\item{locationFun}{formula, vector of character strings, or indices describing a linear model (i.e., regression function) for the location parameter using columns from \code{x}.  \code{x} must be supplied if this is anything other than NULL or ~1.}

\item{scaleFun}{formula, vector of character strings, or indices describing a linear model (i.e., regression function) for the log of the scale parameter using columns from \code{x}.  \code{x} must be supplied if this is anything other than NULL or ~1.}

\item{shapeFun}{formula, vector of character strings, or indices describing a linear model (i.e., regression function) for the shape parameter using columns from \code{x}.  \code{x} must be supplied if this is anything other than NULL or ~1.}

\item{nBlocks}{number of blocks (e.g., a block will often be a year with climate data); note this value determines the interpretation of return values/periods/probabilities; see \code{returnPeriod} and \code{returnValue}.}

\item{blockIndex}{numeric vector providing the index of the block corresponding to each element of \code{y}. Used only when \code{x} is provided to match exceedances to the covariate/predictor/feature value for the exceedance or when using bootstrapping with the resampling based on blocks based on the \code{by} element of \code{bootControl}. If \code{firstBlock} is not equal to one, then \code{blockIndex} need not have one as its smallest possible value.}

\item{firstBlock}{single numeric value indicating the numeric value of the first possible block of \code{blockIndex}. For example the values in \code{blockIndex} might indicate the year of each exceedance with the first year of data being 1969, in which case \code{firstBlock} would be 1969. Note that the first block may not have any exceedances so it may not be represented in \code{blockIndex}. Used only to adjust \code{blockIndex} so that the block indices start at one and therefore correspond to the rows of \code{x}.}

\item{index}{numeric vector providing the integer-valued index (e.g., julian day for daily climate data) corresponding to each element of \code{y}. For example if there are 10 original observations and the third, fourth, and seventh values are exceedances, then \code{index} would be the vector 3,4,7. Used only when \code{declustering} is provided to determine which exceedances occur sequentially or within a contiguous set of values of a given length. The actual values are arbitrary; only the lags between the values are used.}

\item{nReplicates}{numeric value indicating the number of replicates.}

\item{replicateIndex}{numeric vector providing the index of the replicate corresponding to each element of \code{y}. Used for three purposes: (1) when using bootstrapping with the resampling based on replicates based on the \code{by} element of \code{bootControl}, (2) to avoid treating values in different replicates as potentially being sequential or within a short interval when removing values based on \code{declustering}, and (3) to match outcomes to \code{weights} or \code{proportionMissing} when either vary by replicate.}

\item{weights}{a vector or matrix providing the weights by block. When there is only one replicate or the weights do not vary by replicate, a vector of length equal to the number of blocks. When weights vary by replicate, a matrix with rows corresponding to blocks and columns to replicates. Likelihood contribution of each block is multiplied by the corresponding weight.}

\item{proportionMissing}{a numeric value, vector or matrix indicating the proportion of missing values in the original dataset before exceedances were selected. When the proportion missing is the same for all blocks and replicates, a single value. When there is only one replicate or the weights do not vary by replicate, a vector of length equal to the number of blocks. When weights vary by replicate, a matrix with rows corresponding to blocks and columns to replicates.}

\item{returnPeriod}{numeric value giving the number of blocks for which return values should be calculated. For example a \code{returnPeriod} equal to 20 will result in calculation of the value of an event that occurs with probability 1/20 in any block and therefore occurs on average every 20 blocks. Often blocks will correspond to years.}

\item{returnValue}{numeric value giving the value for which return probabilities/periods should be calculated, where the resulting period will be the average number of blocks until the value is exceeded and the probability the probability of exceeding the value in any single block.}

\item{getParams}{logical indicating whether to return the fitted parameter values and their standard errors; WARNING: parameter values for models with covariates for the scale parameter must interpreted based on the value of \code{logScale}.}

\item{getFit}{logical indicating whether to return the full fitted model (potentially useful for model evaluation and for understanding optimization problems); note that estimated parameters in the fit object for nonstationary models will not generally match the MLE provided when \code{getParams} is \code{TRUE} because covariates are normalized before fitting and the fit object is based on the normalized covariates. Similarly, parameters will not match if \code{scaling} is not 1.}

\item{xNew}{object of the same form as \code{x}, providing covariate/predictor/feature values for which return values/periods/probabilities are desired.}

\item{xContrast}{object of the same form and dimensions as \code{xNew}, providing covariate/predictor/feature values for which to calculate the differences of the return values and/or log return probabilities relative to the values in \code{xNew}. This provides a way to estimate differences in return value or log return probabilities (i.e., log risk ratios).}

\item{declustering}{one of \code{NULL}, \code{"noruns"}, or a number. If 'noruns' is specified, only the maximum (or minimum if upperTail = FALSE) value within a set of exceedances corresponding to successive indices is included. If a number, this should indicate the size of the interval (which will be used with the \code{index} argument) within which to allow only the largest (or smallest if upperTail = FALSE) value.}

\item{upperTail}{logical indicating whether one is working with exceedances over a high threshold (TRUE) or exceedances under a low threshold (FALSE); in the latter case, the function works with the negative of the values and the threshold, changing the sign of the resulting location parameters.}

\item{scaling}{positive-valued scalar used to scale the data values for more robust optimization performance. When multiplied by the values, it should produce values with magnitude around 1.}

\item{bootSE}{logical indicating whether to use the bootstrap to estimate standard errors.}

\item{bootControl}{a list of control parameters for the bootstrapping. See \sQuote{Details}.}

\item{optimArgs}{a list with named components matching exactly any arguments that the user wishes to pass to R's \code{optim} function. See \code{help(optim)} for details. Of particular note, \code{'method'} can be used to choose the optimization method used for maximizing the log-likelihood to fit the model and \code{control = list(maxit=VALUE)} for a user-specified VALUE can be used to increase the number of iterations if the optimization is converging slowly.}

\item{initial}{a list with components named \code{'location'}, \code{'scale'}, and \code{'shape'} providing initial parameter values, intended for use in speeding up or enabling optimization when the default initial values are resulting in failure of the optimization; note that use of \code{scaling}, \code{logScale} and \code{.normalizeX = TRUE} cause numerical changes in some of the parameters.}

\item{logScale}{logical indicating whether optimization for the scale parameter should be done on the log scale. By default this is FALSE when the scale is not a function of covariates and TRUE when the scale is a function of covariates (to ensure the scale is positive regardless of the regression coefficients).}

\item{.normalizeX}{logical indicating whether to normalize \code{x} values for better numerical performance; default is TRUE.}

\item{.getInputs}{logical indicating whether to return intermediate objects used in fitting. Defaults to \code{FALSE} and intended for internal use only}
}
\value{
The primary outputs of this function are as follows, depending on what is requested via \code{returnPeriod}, \code{returnValue}, \code{getParams} and \code{xContrast}:

when \code{returnPeriod} is given: for the period given in \code{returnPeriod} the return value(s) (\code{returnValue}) and its corresponding asymptotic standard error (\code{se_returnValue}) and, when \code{bootSE=TRUE}, also the bootstrapped standard error (\code{se_returnValue_boot}). For nonstationary models, these correspond to the covariate values given in \code{x}.

when \code{returnValue} is given: for the value given in \code{returnValue}, the log exceedance probability (\code{logReturnProb}) and the corresponding asymptotic standard error (\code{se_logReturnProb}) and, when \code{bootSE=TRUE}, also the bootstrapped standard error (\code{se_logReturnProb_boot}). This exceedance probability is the probability of exceedance for a single block. Also returned are the log return period (\code{logReturnPeriod}) and its corresponding asymptotic standard error (\code{se_logReturnPeriod}) and, when \code{bootSE=TRUE}, also the bootstrapped standard error (\code{se_logReturnPeriod_boot}). For nonstationary models, these correspond to the covariate values given in \code{x}. Note that results are on the log scale as probabilities and return times are likely to be closer to normally distributed on the log scale and therefore standard errors are more naturally given on this scale. Confidence intervals for return probabilities/periods can be obtained by exponentiating the interval obtained from plus/minus twice the standard error of the log probabilities/periods. 

when \code{getParams=TRUE}: the MLE for the model parameters (\code{mle}) and corresponding asymptotic standard error (\code{se_mle}) and, when \code{bootSE=TRUE}, also the bootstrapped standard error (\code{se_mle_boot}).

when \code{xContrast} is specified for nonstationary models: the difference in return values (\code{returnValueDiff}) and its corresponding asymptotic standard error (\code{se_returnValueDiff}) and,  when \code{bootSE=TRUE}, bootstrapped standard error (\code{se_returnValueDiff_boot}). These differences correspond to the differences when contrasting each row in \code{x} with the corresponding row in \code{xContrast}. Also returned are the difference in log return probabilities (i.e., the log risk ratio) (\code{logReturnProbDiff}) and its corresponding asymptotic standard error (\code{se_logReturnProbDiff}) and,  when \code{bootSE=TRUE}, bootstrapped standard error (\code{se_logReturnProbDiff_boot}).
}
\description{
Fit a peaks-over-threshold model, designed specifically for climate data, to exceedance-only data, using the point process approach. Any covariates/predictors/features assumed to vary only between and not within blocks of observations. It includes options for variable weights (useful for local likelihood) and variable proportions of missing data, as well as for bootstrapping to estimate uncertainties. Results can be returned in terms of parameter values, return values, return periods, return probabilities, and differences in either return values or log return probabilities (i.e., log risk ratios).
}
\details{
This function allows one to fit stationary or nonstationary peaks-over-threshold models using the point process approach. The function can return parameter estimates (which are asymptotically equivalent to GEV model parameters for block maxima data), return value/level for a given return period (number of blocks),  and return probabilities/periods for a given return value/level. The function provides standard errors based on the usual MLE asymptotics, with delta-method-based standard errors for functionals of the parameters, but also standard errors based on the nonparametric bootstrap, either resampling by block or by replicate or both.
}
\section{Analyzing aggregated observations, such as yearly averages}{


Aggregated average or summed data such as yearly or seasonal averages can be fit using this function. The best way to do this is to specify \code{nBlocks} to be the number of observations (i.e., the length of the observation period, not the number of exceedances). Then any return probabilities can be interpreted as the probabilities for a single block (e.g., a year). If instead \code{nBlocks} were one (i.e., a single block) then probabilities would be interpreted as the probability of occurrence in a multi-year block.
}

\section{Blocks and replicates}{


Note that blocks and replicates are related concepts. Blocks are the grouping of values such that return values and return periods are calculated based on the equivalent block maxima (or minima) generalized extreme value model. For example if a block is a year, the return period is the average number of years before the given value is seen, while the return value when \code{returnPeriod} is, say, 20, is the value exceeded on average once in 20 years. A given dataset will generally have multiple blocks. In some cases a block may contain only a single value, such as when analyzing yearly sums or averages.

Replicates are repeated datasets, each with the same structure, including the same number of blocks. The additional blocks in multiple replicates could simply be treated as additional blocks without replication, but when the predictor variables and weights are the same across replicates, it is simpler to make use of \code{nReplicates} and \code{replicateIndex} (see next paragraph). A given replicate might only contain a single block, such as with an ensemble of short climate model runs that are run only for the length of a single block (e.g., a single year). In this case \code{nBlocks} should be set to one.

When using multiple replicates (e.g., multiple members of a climate model initial condition ensemble), the standard input format is to append outcome values for additional replicates to the \code{y} argument and indicate the replicate ID for each exceedance in \code{replicateIndex}. However, if one has different covariate values or thresholds for different replicates, then one needs to treat the additional replicates as providing additional blocks, with only a single replicate. The covariate values can then be included as additional rows in \code{x}, with \code{nBlocks} reflecting the product of the number of blocks per replicate and the number of replicates and \code{nReplicates} set to 1. In this situation, if \code{declustering} is specified, make sure to set \code{index} such that the values for separate replicates do not overlap with each other, to avoid treating exceedances from different replicates as being sequential or from a contiguous set of values. Similarly, if there is a varying number of replicates by block, then all block-replicate pairs should be treated as individual blocks with a corresponding row in \code{x}.
}

\section{\code{bootControl} arguments}{


The \code{bootControl} argument is a list (or dictionary when calling from Python) that can supply any of the following components:
\itemize{
\item seed. Value of the random number seed to set before doing resampling. Defaults to \code{1}.
\item n. Number of bootstrap samples. Defaults to \code{250}.
\item by. Character string, one of \code{'block'}, \code{'replicate'}, or \code{'joint'}, indicating the basis for the resampling. If \code{'block'}, resampled datasets will consist of blocks drawn at random from the original set of blocks; if there are replicates, each replicate will occur once for every resampled block. If \code{'replicate'}, resampled datasets will consist of replicates drawn at random from the original set of replicates; all blocks from a replicate will occur in each resampled replicate. Note that this preserves any dependence across blocks rather than assuming independence between blocks. If \code{'joint'} resampled datasets will consist of block-replicate pairs drawn at random from the original set of block-replicate pairs. Defaults to \code{'block'}. 
\item getSample. Logical/boolean indicating whether the user wants the full bootstrap sample of parameter estimates and/or return value/period/probability information provided for use in subsequent calculations; if FALSE (the default), only the bootstrap-based estimated standard errors are returned.
}
}

\section{Optimization failures}{


It is not uncommon for maximization of the log-likelihood to fail for extreme value models. Users should carefully check the \code{info} element of the return object to ensure that the optimization converged. When there is a convergence failure, one can try a different optimization method, more iterations, or different starting values -- see \code{optimArgs} and \code{initial}. In particular, the Nelder-Mead method is used; users may want to try the BFGS method by setting \code{optimArgs = list(method = 'BFGS')} (or \code{optimArgs = {'method': 'BFGS'}} if calling from Python).

When using the bootstrap, users should check that the number of convergence failures when fitting to the boostrapped datasets is small, as it is not clear how to interpret the bootstrap results when there are convergence failures for some bootstrapped datasets.
}
\examples{
# setup Fort precipitation data
data(Fort, package = 'extRemes')
firstBlock <- min(Fort$year)
years <- min(Fort$year):max(Fort$year)
nYears <- length(years)
threshold <- 0.395
ord <- order(Fort$year, Fort$month, Fort$day) 
Fort <- Fort[ord, ]
ind <- Fort$Prec > threshold
FortExc <- Fort[ind, ]

# stationary fit
out <- fit_pot(FortExc$Prec, threshold = threshold, nBlocks = nYears, 
        returnPeriod = 20, returnValue = 3.5,
        getParams = TRUE, bootSE = FALSE)

# fit with location linear in year
out <- fit_pot(FortExc$Prec, x = data.frame(years = years), threshold = threshold,
        locationFun = ~years, nBlocks = nYears, 
        blockIndex = FortExc$year, firstBlock = firstBlock,
        returnPeriod = 20, returnValue = 3.5,
        getParams = TRUE, xNew = data.frame(years = range(Fort$year)), bootSE = FALSE)

# with declustering
index <- seq_len(nrow(Fort))
out <- fit_pot(FortExc$Prec, x = data.frame(years = years), threshold = threshold,
        locationFun = ~years, nBlocks = nYears, 
        blockIndex = FortExc$year, firstBlock = firstBlock, index = index[ind],
        returnPeriod = 20, returnValue = 3.5,
        getParams = TRUE, xNew = data.frame(years = range(Fort$year)),
        declustering = 'noruns', bootSE = FALSE)

# with replicates; for illustration here, I'll just duplicate the Fort data
Fort2x <- rbind(FortExc, FortExc)
n <- nrow(FortExc)
out <- fit_pot(Fort2x$Prec, x = data.frame(years = years), threshold = threshold,
        locationFun = ~years, nBlocks = nYears,
        blockIndex = Fort2x$year, firstBlock = firstBlock,
        nReplicates = 2, replicateIndex = c(rep(1, n), rep(2, n)),
        returnPeriod = 20, returnValue = 3.5,
        getParams = TRUE, xNew = data.frame(years = range(Fort$year)), bootSE = FALSE)

# analysis of seasonal total precipitation
FortSummer <- Fort[Fort$month \%in\% 6:8, ]  # summer precipitation
FortSummerSum <- aggregate(Prec ~ year, data = FortSummer, sum)
threshold <- quantile(FortSummerSum$Prec, 0.8)
FortSummerSumExc <- FortSummerSum[FortSummerSum$Prec > threshold, ]
# each year (single observation) treated as a block, so return probability
# can be interpreted as probability of exceeding a value in a single year
out <- fit_pot(FortSummerSumExc$Prec, x = data.frame(years = years), threshold = threshold,
        nBlocks = nYears, blockIndex = FortSummerSumExc$year, firstBlock = firstBlock,
        locationFun = ~years, returnPeriod = 20,
        returnValue = 10, getParams = TRUE, xNew = data.frame(years = range(Fort$year)),
        bootSE = FALSE)

}
\author{
Christopher J. Paciorek
}
\references{
Coles, S. 2001. An Introduction to Statistical Modeling of Extreme Values. Springer.

Paciorek, C.J., D.A. Stone, and M.F. Wehner. Quantifying uncertainty in the attribution of human influence on severe weather. http://arxiv.org/abs/1706.03388
}

