% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/time-point.R
\name{time_point_count_between}
\alias{time_point_count_between}
\title{Counting: time point}
\usage{
time_point_count_between(start, end, precision, ..., n = 1L)
}
\arguments{
\item{start, end}{\verb{[clock_time_point]}

A pair of time points. These will be recycled to their common size.}

\item{precision}{\verb{[character(1)]}

One of:
\itemize{
\item \code{"week"}
\item \code{"day"}
\item \code{"hour"}
\item \code{"minute"}
\item \code{"second"}
\item \code{"millisecond"}
\item \code{"microsecond"}
\item \code{"nanosecond"}
}}

\item{...}{These dots are for future extensions and must be empty.}

\item{n}{\verb{[positive integer(1)]}

A single positive integer specifying a multiple of \code{precision} to use.}
}
\value{
An integer representing the number of \code{precision} units between
\code{start} and \code{end}.
}
\description{
\code{time_point_count_between()} counts the number of \code{precision} units
between \code{start} and \code{end} (i.e., the number of days or hours). This count
corresponds to the \emph{whole number} of units, and will never return a
fractional value.

This is suitable for, say, computing the whole number of days between two
time points, accounting for the time of day.
}
\details{
Remember that \code{time_point_count_between()} returns an integer vector.
With extremely fine precisions, such as nanoseconds, the count can quickly
exceed the maximum value that is allowed in an integer. In this case, an
\code{NA} will be returned with a warning.
}
\section{Comparison Direction}{

The computed count has the property that if \code{start <= end}, then
\verb{start + <count> <= end}. Similarly, if \code{start >= end}, then
\verb{start + <count> >= end}. In other words, the comparison direction between
\code{start} and \code{end} will never change after adding the count to \code{start}. This
makes this function useful for repeated count computations at
increasingly fine precisions.
}

\examples{
x <- as_naive_time(year_month_day(2019, 2, 3))
y <- as_naive_time(year_month_day(2019, 2, 10))

# Whole number of days or hours between two time points
time_point_count_between(x, y, "day")
time_point_count_between(x, y, "hour")

# Whole number of 2-day units
time_point_count_between(x, y, "day", n = 2)

# Leap years are taken into account
x <- as_naive_time(year_month_day(c(2020, 2021), 2, 28))
y <- as_naive_time(year_month_day(c(2020, 2021), 3, 01))
time_point_count_between(x, y, "day")

# Time of day is taken into account.
# `2020-02-02T04 -> 2020-02-03T03` is not a whole day (because of the hour)
# `2020-02-02T04 -> 2020-02-03T05` is a whole day
x <- as_naive_time(year_month_day(2020, 2, 2, 4))
y <- as_naive_time(year_month_day(2020, 2, 3, c(3, 5)))
time_point_count_between(x, y, "day")
time_point_count_between(x, y, "hour")

# Can compute negative counts (using the same example from above)
time_point_count_between(y, x, "day")
time_point_count_between(y, x, "hour")

# Repeated computation at increasingly fine precisions
x <- as_naive_time(year_month_day(
  2020, 2, 2, 4, 5, 6, 200,
  subsecond_precision = "microsecond"
))
y <- as_naive_time(year_month_day(
  2020, 3, 1, 8, 9, 10, 100,
  subsecond_precision = "microsecond"
))

days <- time_point_count_between(x, y, "day")
x <- x + duration_days(days)

hours <- time_point_count_between(x, y, "hour")
x <- x + duration_hours(hours)

minutes <- time_point_count_between(x, y, "minute")
x <- x + duration_minutes(minutes)

seconds <- time_point_count_between(x, y, "second")
x <- x + duration_seconds(seconds)

microseconds <- time_point_count_between(x, y, "microsecond")
x <- x + duration_microseconds(microseconds)

data.frame(
  days = days,
  hours = hours,
  minutes = minutes,
  seconds = seconds,
  microseconds = microseconds
)
}
