#!/usr/bin/env python3

import json

# Some references:
# 
#   * http://www.unicode.org/reports/tr35/tr35-general.html#Segmentation_Exceptions
#
#   * http://icu-project.org/apiref/icu4j/com/ibm/icu/text/FilteredBreakIteratorBuilder.html
# 
#   * https://github.com/stanfordnlp/CoreNLP/blob/master/src/edu/stanford/nlp/process/PTBLexer.flex#L707
# 
#   * https://tech.grammarly.com/blog/posts/How-to-Split-Sentences.html
# 
# 
# The [CLDR](http://cldr.unicode.org/) data is at
# 
#     http://www.unicode.org/Public/cldr/31.0.1/
# 
#     http://unicode.org/Public/cldr/latest
# 
# Download `core.zip`. The files are in the `common/segments` directory.
# 
# Basically, you have a list of of "suppressions" (Mr., Dr., Nov., ...) and
# you make sure never to break a sentence after one of these suppressions. The
# data seems a little incomplete (e.g., some months are there, but not "Apr.",
# "Jun.", "Jul.", "Oct."). Maybe take some abbreviations from the CoreNLP lexer.
# 
# ICU implementation:
# 
#    * icu4c/source/common/filteredbrk.cpp
#

suppressions = {
        'english': set([
            # quanteda
            'Mr.' ,'Mrs.', 'Ms.', 'Dr.', 'Jr.', 'Prof.', 'Ph.D.', 'M.',
            'MM.', 'St.', 'etc.',

            # time
            'A.M.', 'P.M.', 'a.m.', 'p.m.',

            # months
            'Jan.', 'Feb.', 'Mar.', 'Apr.', 'Jun.', 'Jul.', 'Aug.', 'Sep.',
            'Sept.', 'Oct.', 'Nov.', 'Dec.',

            # other
            'Mx.', 'e.g.', 'i.e.', 'cf.'
            ])
        }

removals = {
        'english': set([
            # common words that rarely get used as abbreviations, and
            # should be removed
            'Act.', 'All.', 'As.', 'By.', 'Cap.', 'Do.', 'Go.', 'Hat.',
            'Is.', 'Link.', 'Long.', 'Ok.', 'OK.', 'On.', 'Or.', 'Pro.',
            'To.', 'Up.'
            ])
    }

locales = {
        'de': 'german', 'en': 'english', 'es': 'spanish', 'fr': 'french',
        'it': 'italian', 'pt': 'portuguese', 'ru': 'russian'
        }


for key in sorted(locales.keys()):
    filename = 'data/cldr/segments/' + key + '/suppressions.json'
    name = locales[key]
    try:
        f = open(filename)
    except:
        f = open('../' + filename)

    with f:
        obj = json.load(f)

    if 'segments' not in obj:
        continue
    segments = obj['segments']
    if 'segmentations' not in segments:
        continue
    segmentations = segments['segmentations']
    if ('SentenceBreak' not in segmentations
            or 'standard' not in segmentations['SentenceBreak']):
        continue
    sentbreak = segmentations['SentenceBreak']['standard']

    if name not in suppressions:
        suppressions[name] = set()
    supps = suppressions[name]

    for brk in sentbreak:
        if 'suppression' not in brk:
            continue
        s = brk['suppression']
        supps.add(s)

# remove terms from the lists
for name in removals:
    if name not in suppressions:
        continue
    supps = suppressions[name]
    for rm in removals[name]:
        if rm in supps:
            supps.remove(rm)

# delete empty lists
for nm in suppressions:
    if len(suppressions[nm]) == 0:
        del suppressions[nm]

names = sorted(suppressions.keys())

def encode(word):
    enc = ['"']
    for c in bytes(word, 'utf-8'):
        if c < 128:
            enc.append(chr(c))
        else:
            enc.append('\\{0:o}'.format(c))
    enc.append('"')
    return ''.join(enc)


print("/* This file is automatically generated. DO NOT EDIT!")
print("   Instead, edit gen-sentsuppress.py and re-run.  */")
print("")
print("#ifndef SENTSUPPRESS_H")
print("#define SENTSUPPRESS_H")
print("")
print("#include <stddef.h>")
print("#include <stdint.h>")
print("#include <string.h>")
print("")
print("struct sentsuppress_list {")
print("\tconst char *name;")
print("\tint offset;")
print("\tint length;")
print("};")
print("")
print("static const char *sentsuppress_list_names[] = {")
for i in range(len(names)):
    name = names[i]
    print("\t\"", name, "\"", sep="", end="")
    if i + 1 != len(names):
        print(",")
    else:
        print(",")
        print("\tNULL")
print("};")
print("")
print("static struct sentsuppress_list sentsuppress_lists[] = {")
off = 0
for i in range(len(names)):
    name = names[i]
    words = sorted(suppressions[name])
    print("\t{\"", name, "\", ", off, ", ", len(words), "}", sep="", end="")
    if i + 1 != len(names):
        print(",")
    else:
        print(",")
        print("\t{NULL, 0, 0}")
    off += len(words) + 1
print("};")
print("")
print("static const char * sentsuppress_strings[] = {")
for i in range(len(names)):
    name = names[i]
    words = sorted(suppressions[name])
    j = 0
    if i > 0:
        print("\n")
    print("\t/*", name, "*/", end="")
    for w in words:
        if j % 1 == 0:
            print("\n\t", end="")
        else:
            print(" ", end="")
        print(encode(w), ",", sep="", end="")
        j += 1
    print("\n\tNULL", end="")
    if i + 1 < len(names):
        print(",", end="")
    else:
        print("")
print("};")
print("")
print("static const char **sentsuppress_names(void)")
print("{")
print("\treturn (const char **)sentsuppress_list_names;")
print("}")
print("")
print("static const uint8_t **sentsuppress_list(const char *name, int *lenptr)")
print("{")
print("\tconst struct sentsuppress_list *ptr = sentsuppress_lists;")
print("")
print("\twhile (ptr->name != NULL && strcmp(ptr->name, name) != 0) {")
print("\t\tptr++;")
print("\t}")
print("")
print("\tif (ptr->name == NULL) {")
print("\t\tif(lenptr) {")
print("\t\t\t*lenptr = 0;")
print("\t\t}")
print("\t\treturn NULL;")
print("\t}")
print("")
print("\tif(lenptr) {")
print("\t\t*lenptr = ptr->length;")
print("\t}")
print("\treturn (const uint8_t **)(sentsuppress_strings + ptr->offset);")
print("}")
print("")
print("#endif /* SENTSUPPRESS_H */")
