% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coxed.R
\name{coxed}
\alias{coxed}
\title{Expected durations and marginal changes in expected duration from the
Cox proportional hazards model}
\usage{
coxed(cox.model, newdata = NULL, newdata2 = NULL, bootstrap = FALSE,
  method = "npsf", k = -1, ties.method = "random", B = 200,
  confidence = "studentized", level = 0.95, id = NULL, ...)
}
\arguments{
\item{cox.model}{The output from a Cox proportional hazards model estimated
with the \code{\link[survival]{coxph}} function in the \code{survival} package
or with the \code{\link[rms]{cph}} function in the \code{\link[rms]{rms}}
package}

\item{newdata}{An optional data frame in which to look for variables with
which to predict. If omitted, the fitted values are used}

\item{newdata2}{An optional data frame that can only be specified if
\code{newdata} is not omitted, and must have the same dimensions as \code{newdata}.
If specified, marginal changes are calculated by subtracting the expected
durations for \code{newdata2} from the expected durations for \code{newdata}}

\item{bootstrap}{Should bootstrapped standard errors and confidence intervals
be calculated?}

\item{method}{If "npsf" (the default), expected durations are calculated using
the non-parametric step function approach described in Kropko and Harden
(2018). If "gam", expected durations are calculated using the GAM method}

\item{k}{The number of knots in the GAM smoother. The default is -1, which
employs the \code{\link[mgcv]{choose.k}} function from the \code{\link{mgcv}} package
to choose the number of knots}

\item{ties.method}{A character string specifying how ties are treated,
see ‘Details’ in the documentation for \code{\link[base]{rank}}; can be
abbreviated}

\item{B}{Number of bootstrap simulation iterations}

\item{confidence}{If "studentized" (the default), bootstrapped CIs
are calculated from the tails of a normal distribution where the mean and
standard deviation are the point estimate and boostrapped SE of each duration
estimate. If "empirical", bootstrapped confidence intervals are calculated empirically.
If "bca", bootstrapped confidence intervals are calculated using the bias-correction
and acceleration method described by DiCiccio and Efron (1996).}

\item{level}{The level of the confidence interval to calculate (default is
.95 for a 95 percent confidence interval)}

\item{id}{Cluster variable if bootstrapping is to be done by clusters of
observations rather than individual observations. This must be filled in with
the case ID if the data are coded with time-varying covariates (using the \code{time2}
argument in the \code{\link[survival]{Surv}} function)}

\item{...}{Additional arguments to be passed to the \code{\link[coxed]{bootcov2}}
function, an adaptation of the \code{\link[rms]{bootcov}} function in the
\code{\link{rms}} package}
}
\value{
\code{coxed} returns an object of \code{\link[base]{class}} "coxedExpdur" or
"coxedMargin", which is a list containing some of the following components, depending
on the implementation of \code{coxed}:
\tabular{ll}{
\code{exp.dur} \tab A vector of predicted mean durations for the estimation sample
if \code{newdata} is omitted, or else for the specified new data.  If \code{bootstrap}
is \code{TRUE} bootstrapped standard errors are also provided, as well as the confidence
interval requested by \code{level}. \cr
\code{mean} \tab The mean of the predicted durations. If \code{bootstrap}
is \code{TRUE} bootstrapped standard errors are also provided, as well as the confidence
interval requested by \code{level}. \cr
\code{median} \tab The median of the predicted durations. If \code{bootstrap}
is \code{TRUE} bootstrapped standard errors are also provided, as well as the confidence
interval requested by \code{level}. \cr
\code{baseline.functions} \tab The estimated cumulative baseline hazard function and survivor function. \cr
\code{gam.model} \tab Output from the \code{\link[mgcv]{gam}} function in which the durations
are fit against the exponentiated linear predictors from the Cox model.\cr
\code{gam.data} \tab Fitted values and confidence intervals from the GAM model.\cr
\code{exp.dur1} \tab  A vector of predicted mean durations for the observations in \code{newdata1}
when calculating marginal effects. \cr
\code{exp.dur2} \tab A vector of predicted mean durations for the observations in \code{newdata2}
when calculating marginal effects. \cr
\code{mean1} \tab The mean of the predicted mean durations for the observations in \code{newdata1}
when calculating marginal effects. \cr
\code{mean2} \tab The mean of the predicted mean durations for the observations in \code{newdata2}
when calculating marginal effects. \cr
\code{median1} \tab The median of the predicted mean durations for the observations in \code{newdata1}
when calculating marginal effects. \cr
\code{median2} \tab The median of the predicted mean durations for the observations in \code{newdata2}
when calculating marginal effects. \cr
\code{diff} \tab A vector of the difference between the predicted mean durations for each
observation under the covariate profile in \code{newdata2} and the covariate profile in \code{newdata1}.\cr
\code{mean.diff} \tab The mean of the differences in duration across observations. \cr
\code{median.diff} \tab The median of the differences in duration across observations. \cr
}
}
\description{
\code{coxed()} returns expected durations for every observation in the data
used to fit the model, or in new data, or returns the mean or median of these
durations, or differences in duration for two pre-defined covariate profiles.
Standard errors and confidence intervals for all quantities produced by
\code{coxed()} are calculated via bootstrapping.
}
\details{
The \code{coxed} function generates expected durations for individual
observations and/or marginal changes in expected duration given a change in a covariate
from the Cox proportional hazards model. Specifically, the methods can compute (1) the
expected duration for each observation used to fit the Cox model, given the covariates,
(2) the expected duration for a "new" observation with a covariate profile set by the
analyst, or (3) the first difference, or change, in expected duration given two new data frames.

There are two different methods, described in Kropko and Harden (2018), of generating duration-based quantities in the package.
The first method calculates expected durations by using a nonparametric estimate of the
baseline hazard and survivor functions (see \code{\link[coxed]{coxed.npsf}} for details).
The second method employs a generalized additive model
(GAM) to map the model's estimated linear predictor values to duration times (see \code{\link[coxed]{coxed.gam}} for details). Both
methods are also implemented for data structures with time-varying covariates
(see \code{\link[coxed]{coxed.npsf.tvc}} and \code{\link[coxed]{coxed.gam.tvc}}).
}
\examples{
mv.surv <- Surv(martinvanberg$formdur, event = rep(1, nrow(martinvanberg)))
mv.cox <- coxph(mv.surv ~ postel + prevdef + cont + ident + rgovm + pgovno +
     tpgovno + minority, method = "breslow", data = martinvanberg)
summary(mv.cox)

# NPSF method
ed1 <- coxed(mv.cox, method="npsf")
ed1$baseline.functions
ed1$exp.dur
summary(ed1, stat="mean")
summary(ed1, stat="median")

\dontrun{ed1 <- coxed(mv.cox, method="npsf", bootstrap = TRUE)
ed1$exp.dur
summary(ed1, stat="mean")
summary(ed1, stat="median")
}

me <- coxed(mv.cox, method="npsf", bootstrap = FALSE,
            newdata = dplyr::mutate(martinvanberg, pgovno=1),
            newdata2 = dplyr::mutate(martinvanberg, pgovno=6))
summary(me, stat="mean")

# GAM method
ed2 <- coxed(mv.cox, method="gam")
summary(ed2$gam.data)
summary(ed2$gam.model)
ed2$exp.dur
summary(ed2, stat="mean")

\dontrun{me <- coxed(mv.cox, method="gam", bootstrap = TRUE,
            newdata = dplyr::mutate(martinvanberg, pgovno=1),
            newdata2 = dplyr::mutate(martinvanberg, pgovno=6))
summary(me, stat="mean")
summary(me, stat="median")
}

#Plotting the GAM fit
\dontrun{ggplot(ed2$gam.data, aes(x=rank.xb, y=y)) +
    geom_point() +
    geom_line(aes(x=rank.xb, y=gam_fit)) +
    geom_ribbon(aes(ymin=gam_fit_95lb, ymax=gam_fit_95ub), alpha=.5) +
    xlab("Cox model LP rank (smallest to largest)") +
    ylab("Duration")
}

#Time-varying covariates
bs.surv <- Surv(time = boxsteffensmeier$start, time2 = boxsteffensmeier$te,
     event = boxsteffensmeier$cut_hi)
bs.cox <- coxph(bs.surv ~ ec + dem + south + iv, data = boxsteffensmeier, method = "breslow")
summary(bs.cox)

ed1 <- coxed(bs.cox, method="npsf", id=boxsteffensmeier$caseid)
ed1$exp.dur
summary(ed1, stat="mean")
}
\references{
Kropko, J. and Harden, J. J. (2018). Beyond the Hazard Ratio: Generating Expected Durations from the
Cox Proportional Hazards Model. \emph{British Journal of Political Science} \url{https://doi.org/10.1017/S000712341700045X}

DiCiccio, T. J. and B. Efron. (1996). Bootstrap Confidence Intervals. \emph{Statistical Science}.
11(3): 189–212. \url{https://doi.org/10.1214/ss/1032280214}
}
\seealso{
\code{\link[survival]{coxph}}, \code{\link[rms]{cph}}, \code{\link[coxed]{bootcov2}},
\code{\link[coxed]{coxed.gam}}, \code{\link[coxed]{coxed.gam.tvc}}, \code{\link[coxed]{coxed.npsf}},
\code{\link[coxed]{coxed.npsf.tvc}}
}
\author{
Jonathan Kropko <jkropko@virginia.edu> and Jeffrey J. Harden <jharden2@nd.edu>
}
