% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GenFlexCureModel.R
\name{GenFlexCureModel}
\alias{GenFlexCureModel}
\title{Fit generalized mixture cure model}
\usage{
GenFlexCureModel(
  formula,
  data,
  smooth.formula = NULL,
  smooth.args = NULL,
  df = 3,
  tvc = NULL,
  tvc.formula = NULL,
  bhazard = NULL,
  cr.formula = ~1,
  type = "mixture",
  link.type.cr = c("logit", "loglog", "identity", "probit"),
  link.type = c("PH", "PO", "probit"),
  init = NULL,
  baseoff = FALSE,
  timeVar = "",
  time0Var = "",
  covariance = T,
  verbose = T,
  control = list(maxit = 10000),
  method = "Nelder-Mead",
  constraint = TRUE,
  ini.types = c("cure", "flexpara")
)
}
\arguments{
\item{formula}{Formula for modelling the survival of the uncured. Reponse has to be of the form \code{Surv(time, status)}.}

\item{data}{Data frame in which to interpret the variables names in \code{formula},
\code{smooth.formula}, \code{tvc.formula}, and \code{cr.formula}.}

\item{smooth.formula}{Formula for describing the time-effect of the survival of the uncured.
If \code{NULL} (default), the function uses the natural cubic splines modelled on the log-time scale.}

\item{smooth.args}{List. Optional arguments to the time-effect of the survival
of the uncured (default is \code{NULL}).}

\item{df}{Integer. Degrees of freedom (default is 3) for the time-effect of the survival of the uncured.
Is not used if \code{smooth.formula} is provided.}

\item{tvc}{Named list of integers. Specifies the degrees of freedom for time-varying covariate effects.
For instance, \code{tvc = list(x = 3)} creates a time-varying spline-effect of the covariate "x" with
3 degrees of freedom using the \code{rstpm2::nsx} function.}

\item{tvc.formula}{Formula for the time-varying covariate effects.
For time-varying effects, a linear term of the covariate has to be included in \code{formula}.}

\item{bhazard}{Background hazard.}

\item{cr.formula}{Formula for the cure proportion.
The left hand side of the formula is not used and should therefore not be specified.}

\item{type}{A character indicating the type of cure model.
Possible values are \code{mixture} for mixture cure models (default) and \code{nmixture}
for non-mixture cure models.}

\item{link.type.cr}{Character providing the link function for the cure proportion.
Possible values are \code{logit} (default), \code{loglog}, \code{identity}, and \code{probit}.}

\item{link.type}{Character providing the link function for the survival of the uncured.
Possible values are \code{PH} for a proportional hazards model (default), \code{PO} for a proportion odds model,
and \code{probit} for a probit model.}

\item{init}{Initial values for the optimization procedure.
If not specified, the function will create initial values internally.}

\item{baseoff}{Logical. If \code{TRUE}, the time-effect is modelled only using \code{tvc.formula} rather
than merging with \code{smooth.formula}.}

\item{timeVar}{Optional character giving the name of the variable specifying the time component of the \code{Surv} object.
Should currently not be used.}

\item{time0Var}{Optional character giving the name of the variable specifying the time start time component used for delayed entry.
Should currently not be used.}

\item{covariance}{Logical. If \code{TRUE} (default), the covariance matrix is computed.}

\item{verbose}{Logical. If \code{TRUE} status messages of the function is outputted.}

\item{control}{Named list with control arguments passed to \code{optim}.}

\item{method}{Character passed to \code{optim} indicating the method for optimization.
See \code{?optim} for details.}

\item{constraint}{Logical. Indicates whether non-negativity constraints should be forced upon
the hazard of the uncured patients (see details).}

\item{ini.types}{Character vector denoting the executed schemes for computing initial values (see details).}
}
\value{
An object of class \code{gfcm}.
}
\description{
The following function fits a generalized mixture or non-mixture cure model
using a link function for the cure rate and for the survival of the uncured.
For a mixture cure model, the model is specified by
\deqn{S(t|z) = \pi(z) + [1 - \pi(z)] S_u(t|z),}
where
\deqn{g_1[S_u(t|z)] = \eta_1(t, z)\qquad and \qquad g_2[\pi(z)] = \eta_2(z).}
The function implements multiple link functions for both \eqn{g_1} and \eqn{g_2}. The default time-effect
is natural cubic splines, but the function allows for the use of other smoothers.
}
\details{
The default smoother is natural cubic splines established by the \code{rstpm2::nsx} function.
Functions such as \code{ns}, \code{bs} are readily available for usage. Also the \code{cb} function in this package
can be used. Initial values are calculated by two procedures and the model is fitted under each set of initial values.
The model producing the highest likelihood is selected.\cr

Using \code{link.type = 'PH'}, the link function \eqn{g_1(x) = \log(-\log(x))} is used.
Using \code{link.type = 'PO'}, the link function \eqn{g_1(x) = \log(\frac{x}{1 - x})} is used.\cr

If \code{constraint = TRUE}, a non-negative hazard of the uncured is ensured by a general penalization scheme.
If \code{constraint = FALSE}, penalization is still employed, but on the all-cause hazard instead.
}
\examples{
##Use data cleaned version of the colon cancer data from the rstpm2 package
data("colonDC")
set.seed(2)
colonDC <- colonDC[sample(1:nrow(colonDC), 400), ]

##Extract general population hazards
colonDC$bhaz <- general.haz(time = "FU", rmap = list(age = "agedays", sex = "sex", year= "dx"),
                            data = colonDC, ratetable = survexp.dk)

###Without covariates
##Fit mixture cure model
fit <- GenFlexCureModel(Surv(FUyear, status) ~ 1, data = colonDC, df = 4, bhazard = "bhaz")

##Plot model
plot(fit)
plot(fit, time = seq(0.001, 40, length.out = 100))
plot(fit, type = "hazard")
plot(fit, type = "survuncured")
plot(fit, type = "probcure")

##Predict cure proportion
predict(fit, type = "curerate")


##Fit non-mixture cure model
fit <- GenFlexCureModel(Surv(FUyear, status) ~ 1, data = colonDC, df = 4,
                        bhazard = "bhaz", type = "nmixture")

##Plot relative survival
plot(fit)

##Predict cure proportion
predict(fit, type = "curerate")

###With covariates
##Fit mixture cure model
fit <- GenFlexCureModel(Surv(FUyear, status) ~ sex, data = colonDC, df = 4,
                        bhazard = "bhaz", cr.formula = ~ sex)

##Plot model
plot(fit, newdata = data.frame(sex = factor("female", levels = c("male", "female"))),
     time = seq(0.001, 15, length.out = 100), ci = FALSE)
plot(fit, newdata = data.frame(sex = factor("male", levels = c("male", "female"))),
     time = seq(0.001, 15, length.out = 100), col = 2, ci = FALSE, add = TRUE)


plot(fit, newdata = data.frame(sex = factor("female", levels = c("male", "female"))),
     time = seq(0.001, 15, length.out = 100), ci = FALSE, type = "survuncured")
plot(fit, newdata = data.frame(sex = factor("male", levels = c("male", "female"))),
     time = seq(0.001, 15, length.out = 100), col = 2, ci = FALSE,
     add = TRUE, type = "survuncured")

predict(fit, type = "curerate",
        data.frame(sex = factor(c("male", "female"),
                                levels = c("male", "female"))))


##Fit mixture cure model with time-varying covariates
colonDC$gender <- as.numeric(colonDC$sex) - 1
fit <- GenFlexCureModel(Surv(FUyear, status) ~ gender, data = colonDC, df = 6,
                        bhazard = "bhaz", cr.formula = ~ gender, tvc = list(gender = 2))

##Plot model
plot(fit, newdata = data.frame(gender = 0))
plot(fit, newdata = data.frame(gender = 1), add = TRUE, col = 2)

plot(fit, type = "hazard", newdata = data.frame(gender = 0), ci = FALSE)
plot(fit, type = "hazard", newdata = data.frame(gender = 1),
     add = TRUE, col = 2, ci = FALSE)

#Predict cure proportions for a male and female patients
predict(fit, type = "curerate", newdata = data.frame(gender = 0))
predict(fit, type = "curerate", newdata = data.frame(gender = 1))
}
