% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cv-mixed-models.R
\name{cv.merMod}
\alias{cv.merMod}
\alias{cv.lme}
\alias{cv.glmmTMB}
\title{Cross-Validate Mixed-Effects Model}
\usage{
\method{cv}{merMod}(
  model,
  data = insight::get_data(model),
  criterion = mse,
  k = NULL,
  reps = 1L,
  seed,
  details = NULL,
  ncores = 1L,
  clusterVariables,
  ...
)

\method{cv}{lme}(
  model,
  data = insight::get_data(model),
  criterion = mse,
  k = NULL,
  reps = 1L,
  seed,
  details = NULL,
  ncores = 1L,
  clusterVariables,
  ...
)

\method{cv}{glmmTMB}(
  model,
  data = insight::get_data(model),
  criterion = mse,
  k = NULL,
  reps = 1L,
  seed,
  details = NULL,
  ncores = 1L,
  clusterVariables,
  ...
)
}
\arguments{
\item{model}{a mixed-effects model object for which a \code{cv()} method is available.}

\item{data}{data frame to which the model was fit (not usually necessary).}

\item{criterion}{cross-validation ("cost" or lack-of-fit) criterion function of form \code{f(y, yhat)}
where \code{y} is the observed values of the response and
\code{yhat} the predicted values; the default is \code{\link{mse}}
(the mean-squared error).}

\item{k}{perform k-fold cross-validation; \code{k}
may be a number or \code{"loo"} or \code{"n"} for n-fold (leave-one-out)
cross-validation; the default is \code{10} if cross-validating individual
cases and \code{"loo"} if cross-validating clusters.}

\item{reps}{number of times to replicate k-fold CV (default is \code{1}),
or greater), compute a confidence interval for the bias-corrected CV
criterion, if the criterion is the average of casewise components.}

\item{seed}{for R's random number generator; optional, if not
supplied a random seed will be selected and saved; not needed
for n-fold cross-validation.}

\item{details}{if \code{TRUE} (the default if the number of
folds \code{k <= 10}), save detailed information about the value of the
CV criterion for the cases in each fold and the regression coefficients
with that fold deleted.}

\item{ncores}{number of cores to use for parallel computations
(default is \code{1}, i.e., computations aren't done in parallel).}

\item{clusterVariables}{a character vector of names of the variables
defining clusters for a mixed model with nested or crossed random effects;
if missing, cross-validation is performed for individual cases rather than
for clusters.}

\item{...}{for \code{cv()} methods, to match generic,
and for \code{cvMixed()}, arguments to be passed to \code{update()}.}
}
\value{
The methods \code{cv.merMod()}, \code{cv.lme()}, and \code{cv.glmmTMB()},
return objects of class \code{"cv"}, or,
if \code{reps > 1}, of class \code{"cvList"} (see \code{\link{cv}()}).
}
\description{
\code{\link{cv}()} methods for mixed-effect models of class \code{"merMod"}, fit
by the \code{\link[lme4]{lmer}()} and \code{\link[lme4]{glmer}()} functions
in the \pkg{lme4} package; for models of class \code{"lme"}
fit by the \code{\link[nlme]{lme}()} function in the \pkg{nlme}
package; and for models of class \code{"glmmTMB"} fit by the
\code{\link[glmmTMB]{glmmTMB}()} function in the \pkg{glmmTMB} package.
}
\details{
For mixed-effects models, cross-validation can be done by "clusters" or by
individual observations. If the former, predictions are based only on fixed
effects; if the latter, predictions include the random effects (i.e., are the
best linear unbiased predictors or "BLUPS").
}
\section{Functions}{
\itemize{
\item \code{cv(merMod)}: \code{cv()} method for \code{\link[lme4]{lmer}()} and
\code{\link[lme4]{glmer}()} models from the \pkg{lme4} package.

\item \code{cv(lme)}: \code{cv()} method for \code{\link[nlme]{lme}()}
models from the \pkg{nlme} package.

\item \code{cv(glmmTMB)}: \code{cv()} method for \code{\link[glmmTMB]{glmmTMB}()}
models from the \pkg{glmmTMB} package.

}}
\examples{
library("lme4")
# from ?lmer:
(fm1 <- lmer(Reaction ~ Days + (Days | Subject), sleepstudy))
summary(cv(fm1, clusterVariables="Subject")) # LOO CV of clusters
summary(cv(fm1, seed=447)) # 10-fold CV of cases
summary(cv(fm1, clusterVariables="Subject", k=5,
   seed=834, reps=3)) # 5-fold CV of clusters, repeated 3 times

library("nlme")
# from ?lme
(fm2 <- lme(distance ~ age + Sex, data = Orthodont,
            random = ~ 1))
summary(cv(fm2)) # LOO CV of cases
summary(cv(fm2, clusterVariables="Subject",
        k=5, seed=321)) # 5-fold CV of clusters

library("glmmTMB")
# from ?glmmTMB
(m1 <- glmmTMB(count ~ mined + (1|site),
               zi=~mined,
               family=poisson, data=Salamanders))
summary(cv(m1, seed=97816, k=5,
          clusterVariables="site")) # 5-fold CV of clusters
summary(cv(m1, seed=34506, k=5)) # 5-fold CV of cases
}
\seealso{
\code{\link{cv}}, \code{\link[lme4]{lmer}}, \code{\link[lme4]{glmer}},
\code{\link[nlme]{lme}}, \code{\link[glmmTMB]{glmmTMB}}
}
