% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_confusion_matrix.R
\name{plot_confusion_matrix}
\alias{plot_confusion_matrix}
\title{Plot a confusion matrix}
\usage{
plot_confusion_matrix(
  conf_matrix,
  target_col = "Target",
  prediction_col = "Prediction",
  counts_col = "N",
  sub_col = NULL,
  class_order = NULL,
  add_sums = FALSE,
  add_counts = TRUE,
  add_normalized = TRUE,
  add_row_percentages = TRUE,
  add_col_percentages = TRUE,
  diag_percentages_only = FALSE,
  rm_zero_percentages = TRUE,
  rm_zero_text = TRUE,
  add_zero_shading = TRUE,
  amount_3d_effect = 1,
  add_arrows = TRUE,
  counts_on_top = FALSE,
  palette = "Blues",
  intensity_by = "counts",
  intensity_lims = NULL,
  intensity_beyond_lims = "truncate",
  theme_fn = ggplot2::theme_minimal,
  place_x_axis_above = TRUE,
  rotate_y_text = TRUE,
  digits = 1,
  font_counts = font(),
  font_normalized = font(),
  font_row_percentages = font(),
  font_col_percentages = font(),
  dynamic_font_colors = dynamic_font_color_settings(),
  arrow_size = 0.048,
  arrow_color = "black",
  arrow_nudge_from_text = 0.065,
  tile_border_color = NA,
  tile_border_size = 0.1,
  tile_border_linetype = "solid",
  sums_settings = sum_tile_settings(),
  darkness = 0.8
)
}
\arguments{
\item{conf_matrix}{Confusion matrix \code{tibble} with each combination of
 targets and predictions along with their counts.

 E.g. for a binary classification:

 \tabular{rrr}{
  \strong{Target} \tab \strong{Prediction} \tab \strong{N} \cr
  class_1 \tab class_1 \tab 5 \cr
  class_1 \tab class_2 \tab 9 \cr
  class_2 \tab class_1 \tab 3 \cr
  class_2 \tab class_2 \tab 2 \cr
 }

 As created with the various evaluation functions in \code{cvms}, like
 \code{\link[cvms:evaluate]{evaluate()}}.

 An additional \code{`sub_col`} column (\code{character}) can be specified
 as well. Its content will replace the bottom text (`counts` by default or
 `normalized` when \code{`counts_on_top`} is enabled).

 \strong{Note}: If you supply the results from \code{\link[cvms:evaluate]{evaluate()}}
 or \code{\link[cvms:confusion_matrix]{confusion_matrix()}} directly,
 the confusion matrix \code{tibble} is extracted automatically, if possible.}

\item{target_col}{Name of column with target levels.}

\item{prediction_col}{Name of column with prediction levels.}

\item{counts_col}{Name of column with a count for each combination
of the target and prediction levels.}

\item{sub_col}{Name of column with text to replace the bottom text
 (`counts` by default or `normalized` when \code{`counts_on_top`} is enabled).

 It simply replaces the text, so all settings will still be called
 e.g. \code{`font_counts`} etc. When other settings make it so, that no
 bottom text is displayed (e.g. \code{`add_counts` = FALSE}),
 this text is not displayed either.}

\item{class_order}{Names of the classes in \code{`conf_matrix`} in the desired order.
When \code{NULL}, the classes are ordered alphabetically.
Note that the entire set of unique classes from both \code{`target_col`}
and \code{`prediction_col`} must be specified.}

\item{add_sums}{Add tiles with the row/column sums. Also adds a total count tile. (Logical)

 The appearance of these tiles can be specified in \code{`sums_settings`}.

 Note: Adding the sum tiles with a palette requires the \code{ggnewscale} package.}

\item{add_counts}{Add the counts to the middle of the tiles. (Logical)}

\item{add_normalized}{Normalize the counts to percentages and
add to the middle of the tiles. (Logical)}

\item{add_row_percentages}{Add the row percentages,
 i.e. how big a part of its row the tile makes up. (Logical)

 By default, the row percentage is placed to the right of the tile, rotated 90 degrees.}

\item{add_col_percentages}{Add the column percentages,
 i.e. how big a part of its column the tile makes up. (Logical)

 By default, the row percentage is placed at the bottom of the tile.}

\item{diag_percentages_only}{Whether to only have row and column percentages in the diagonal tiles. (Logical)}

\item{rm_zero_percentages}{Whether to remove row and column percentages when the count is \code{0}. (Logical)}

\item{rm_zero_text}{Whether to remove counts and normalized percentages when the count is \code{0}. (Logical)}

\item{add_zero_shading}{Add image of skewed lines to zero-tiles. (Logical)

 Note: Adding the zero-shading requires the \code{rsvg} and \code{ggimage} packages.

 Note: For large confusion matrices, this can be very slow. Consider turning
 off until the final plotting.}

\item{amount_3d_effect}{Amount of 3D effect (tile overlay) to add.
 Passed as whole number from \code{0} (no effect) up to \code{6} (biggest effect).
 This helps separate tiles with the same intensities.

 Note: The overlay may not fit the tiles in many-class cases that haven't been tested.
 If the boxes do not overlap properly, simply turn it off.

 Note: For large confusion matrices, this can be very slow. Consider turning
 off until the final plotting.}

\item{add_arrows}{Add the arrows to the row and col percentages. (Logical)

 Note: Adding the arrows requires the \code{rsvg} and \code{ggimage} packages.}

\item{counts_on_top}{Switch the counts and normalized counts,
such that the counts are on top. (Logical)}

\item{palette}{Color scheme. Passed directly to \code{`palette`} in
 \code{\link[ggplot2:scale_fill_distiller]{ggplot2::scale_fill_distiller}}.

 Try these palettes: \code{"Greens"}, \code{"Oranges"},
 \code{"Greys"}, \code{"Purples"}, \code{"Reds"},
 as well as the default \code{"Blues"}.

 Alternatively, pass a named list with limits of a custom gradient as e.g.
 \code{`list("low"="#B1F9E8", "high"="#239895")`}. These are passed to
 \code{\link[ggplot2:scale_fill_gradient]{ggplot2::scale_fill_gradient}}.}

\item{intensity_by}{The measure that should control the color intensity of the tiles.
 Either \code{`counts`}, \code{`normalized`}, \code{`row_percentages`}, \code{`col_percentages`},
 or one of \code{`log counts`, `log2 counts`, `log10 counts`, `arcsinh counts`}.

 For `normalized`, `row_percentages`, and `col_percentages`, the color limits
 become \code{0-100} (except when \code{`intensity_lims`} are specified),
 why the intensities can better be compared across plots.

 \strong{Note}: When \code{`add_sums=TRUE`}, the `row_percentages` and `col_percentages`
 options are only available for the main tiles. A separate intensity metric
 must be specified for the sum tiles (e.g., via
 \code{`sums_settings = sum_tile_settings(intensity_by='normalized')`}).

 For the `log*` and `arcsinh` versions, the log/arcsinh transformed counts are used.

 \strong{Note}: In `log*` transformed counts, 0-counts are set to `0`, why they
 won't be distinguishable from 1-counts.}

\item{intensity_lims}{A specific range of values for the color intensity of
 the tiles. Given as a numeric vector with \code{c(min, max)}.

 This allows having the same intensity scale across plots for better comparison
 of prediction sets.}

\item{intensity_beyond_lims}{What to do with values beyond the
\code{`intensity_lims`}. One of \code{"truncate", "grey"}.}

\item{theme_fn}{The \code{ggplot2} theme function to apply.}

\item{place_x_axis_above}{Move the x-axis text to the top and reverse the levels such that
the "correct" diagonal goes from top left to bottom right. (Logical)}

\item{rotate_y_text}{Whether to rotate the y-axis text to
be vertical instead of horizontal. (Logical)}

\item{digits}{Number of digits to round to (percentages only).
 Set to a negative number for no rounding.

 Can be set for each font individually via the \code{font_*} arguments.}

\item{font_counts}{\code{list} of font settings for the counts.
Can be provided with \code{\link[cvms:font]{font()}}.}

\item{font_normalized}{\code{list} of font settings for the normalized counts.
Can be provided with \code{\link[cvms:font]{font()}}.}

\item{font_row_percentages}{\code{list} of font settings for the row percentages.
Can be provided with \code{\link[cvms:font]{font()}}.}

\item{font_col_percentages}{\code{list} of font settings for the column percentages.
Can be provided with \code{\link[cvms:font]{font()}}.}

\item{dynamic_font_colors}{A list of settings for using dynamic font colors
 based on the value of the counts/normalized. Allows changing the font colors
 when the background tiles are too dark, etc.
 Can be provided with \code{\link[cvms:dynamic_font_color_settings]{dynamic_font_color_settings()}}.

 Individual thresholds can be set for the different fonts/values via the
 \code{`font_*`} arguments. Specifying colors in these arguments will overwrite
 this argument (for the specific font only).

 Specifying colors for specific fonts overrides the "all" values for those fonts.}

\item{arrow_size}{Size of arrow icons. (Numeric)

 Is divided by \code{sqrt(nrow(conf_matrix))} and passed on
 to \code{\link[ggimage:geom_icon]{ggimage::geom_icon()}}.}

\item{arrow_color}{Color of arrow icons. One of \code{"black", "white"}.}

\item{arrow_nudge_from_text}{Distance from the percentage text to the arrow. (Numeric)}

\item{tile_border_color}{Color of the tile borders. Passed as \emph{\code{`colour`}} to
\code{\link[ggplot2:geom_tile]{ggplot2::geom_tile}}.}

\item{tile_border_size}{Size of the tile borders. Passed as \emph{\code{`size`}} to
\code{\link[ggplot2:geom_tile]{ggplot2::geom_tile}}.}

\item{tile_border_linetype}{Linetype for the tile borders. Passed as \emph{\code{`linetype`}} to
\code{\link[ggplot2:geom_tile]{ggplot2::geom_tile}}.}

\item{sums_settings}{A list of settings for the appearance of the sum tiles.
Can be provided with \code{\link[cvms:sum_tile_settings]{sum_tile_settings()}}.}

\item{darkness}{How dark the darkest colors should be, between \code{0} and \code{1}, where \code{1} is darkest.

 Technically, a lower value increases the upper limit in
 \code{\link[ggplot2:scale_fill_distiller]{ggplot2::scale_fill_distiller}}.}
}
\value{
A \code{ggplot2} object representing a confusion matrix.
 Color intensity depends on either the counts (default) or the overall percentages.

 By default, each tile has the normalized count
 (overall percentage) and count in the middle, the
 column percentage at the bottom, and the
 row percentage to the right and rotated 90 degrees.

 In the "correct" diagonal (upper left to bottom right, by default),
 the column percentages are the class-level sensitivity scores,
 while the row percentages are the class-level positive predictive values.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

 Creates a \code{\link[ggplot2:ggplot]{ggplot2}} object representing a confusion matrix with counts,
 overall percentages, row percentages and column percentages. An extra row and column with sum tiles and the
 total count can be added.

 The confusion matrix can be created with \code{\link[cvms:evaluate]{evaluate()}}. See \code{`Examples`}.

 While this function is intended to be very flexible (hence the large number of arguments),
 the defaults should work in most cases for most users. See the \code{Examples}.

 \strong{Web-based}: Our
 \href{https://huggingface.co/spaces/ludvigolsen/plot_confusion_matrix}{
 \strong{Plot Confusion Matrix} web application}
 allows using this function without code. Select from multiple design templates
 or make your own.
}
\details{
Inspired by Antoine Sachet's answer at https://stackoverflow.com/a/53612391/11832955
}
\examples{
\donttest{
# Attach cvms
library(cvms)
library(ggplot2)

# Two classes

# Create targets and predictions data frame
data <- data.frame(
  "target" = c("A", "B", "A", "B", "A", "B", "A", "B",
               "A", "B", "A", "B", "A", "B", "A", "A"),
  "prediction" = c("B", "B", "A", "A", "A", "B", "B", "B",
                   "B", "B", "A", "B", "A", "A", "A", "A"),
  stringsAsFactors = FALSE
)

# Evaluate predictions and create confusion matrix
evaluation <- evaluate(
  data = data,
  target_col = "target",
  prediction_cols = "prediction",
  type = "binomial"
)

# Inspect confusion matrix tibble
evaluation[["Confusion Matrix"]][[1]]

# Plot confusion matrix
# Supply confusion matrix tibble directly
plot_confusion_matrix(evaluation[["Confusion Matrix"]][[1]])
# Plot first confusion matrix in evaluate() output
plot_confusion_matrix(evaluation)

}
\dontrun{
# Add sum tiles
plot_confusion_matrix(evaluation, add_sums = TRUE)
}
\donttest{

# Add labels to diagonal row and column percentages
# This example assumes "B" is the positive class
# but you could write anything as prefix to the percentages
plot_confusion_matrix(
    evaluation,
    font_row_percentages = font(prefix=c("NPV = ", "", "", "PPV = ")),
    font_col_percentages = font(prefix=c("Spec = ", "", "", "Sens = "))
)

# Dynamic font colors when background becomes too dark
# Also inverts the arrow colors
plot_confusion_matrix(
  evaluation[["Confusion Matrix"]][[1]],
  # Black and white theme
  palette = list("low"="#ffffff", "high"="#000000"),
  # Increase contrast
  darkness = 1.0,
  # Specify colors below and above threshold
  dynamic_font_colors = dynamic_font_color_settings(
    threshold = 30,
    by = "normalized",
    # Black at low values, white at high values
    all = c('#000', '#fff'),
    # White arrows above threshold
    invert_arrows = "at_and_above"
  )
)

# Three (or more) classes

# Create targets and predictions data frame
data <- data.frame(
  "target" = c("A", "B", "C", "B", "A", "B", "C",
               "B", "A", "B", "C", "B", "A"),
  "prediction" = c("C", "B", "A", "C", "A", "B", "B",
                   "C", "A", "B", "C", "A", "C"),
  stringsAsFactors = FALSE
)

# Evaluate predictions and create confusion matrix
evaluation <- evaluate(
  data = data,
  target_col = "target",
  prediction_cols = "prediction",
  type = "multinomial"
)

# Inspect confusion matrix tibble
evaluation[["Confusion Matrix"]][[1]]

# Plot confusion matrix
# Supply confusion matrix tibble directly
plot_confusion_matrix(evaluation[["Confusion Matrix"]][[1]])
# Plot first confusion matrix in evaluate() output
plot_confusion_matrix(evaluation)

}
\dontrun{
# Add sum tiles
plot_confusion_matrix(evaluation, add_sums = TRUE)
}
\donttest{

# Counts only
plot_confusion_matrix(
  evaluation[["Confusion Matrix"]][[1]],
  add_normalized = FALSE,
  add_row_percentages = FALSE,
  add_col_percentages = FALSE
)

# Change color palette to green
# Change theme to `theme_light`.
plot_confusion_matrix(
  evaluation[["Confusion Matrix"]][[1]],
  palette = "Greens",
  theme_fn = ggplot2::theme_light
)

}
\dontrun{
# Change colors palette to custom gradient
# with a different gradient for sum tiles
plot_confusion_matrix(
  evaluation[["Confusion Matrix"]][[1]],
  palette = list("low" = "#B1F9E8", "high" = "#239895"),
  sums_settings = sum_tile_settings(
    palette = list("low" = "#e9e1fc", "high" = "#BE94E6")
  ),
  add_sums = TRUE
)
}
\donttest{

# The output is a ggplot2 object
# that you can add layers to
# Here we change the axis labels
plot_confusion_matrix(evaluation[["Confusion Matrix"]][[1]]) +
  ggplot2::labs(x = "True", y = "Guess")

# Replace the bottom tile text
# with some information
# First extract confusion matrix
# Then add new column with text
cm <- evaluation[["Confusion Matrix"]][[1]]
cm[["Trials"]] <- c(
  "(8/9)", "(3/9)", "(1/9)",
  "(3/9)", "(7/9)", "(4/9)",
  "(1/9)", "(2/9)", "(8/9)"
 )

# Now plot with the `sub_col` argument specified
plot_confusion_matrix(cm, sub_col="Trials")

}
}
\seealso{
Other plotting functions: 
\code{\link{dynamic_font_color_settings}()},
\code{\link{font}()},
\code{\link{plot_metric_density}()},
\code{\link{plot_probabilities}()},
\code{\link{plot_probabilities_ecdf}()},
\code{\link{sum_tile_settings}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{plotting functions}
