% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eviSimulation.R
\name{eviSimulation}
\alias{eviSimulation}
\title{Expected Value of Information (EVI) Simulation.}
\usage{
eviSimulation(welfare, currentEstimate, prospectiveEstimate, numberOfModelRuns,
  randomMethod = "calculate", functionSyntax = "data.frameNames",
  relativeTolerance = 0.05, verbosity = 0)
}
\arguments{
\item{welfare}{either a \code{function} or a \code{list} with two \code{functions}, i.e.
\code{list(p1,p2)}. In the first case the function is the net benefit (or welfare) of project approval (PA) vs.
the status quo (SQ). In the second case the element \code{p1} is the function valuing the first
project and the element \code{p2} valuing the second project, viz. the welfare function of \code{p1}
and \code{p2} respectively.}

\item{currentEstimate}{\code{\link{estimate}}: describing the distribution of the input variables
as currently being estimated.}

\item{prospectiveEstimate}{\code{\link{estimate}} or \code{list} of \code{estimate} objects:
describing the prospective distribution of the input variables which could hypothetically
be achieved by collecting more information, viz. improving the measurement.}

\item{numberOfModelRuns}{\code{integer}: The number of running the welfare model for the 
underlying Monte Carlo simulation.}

\item{randomMethod}{\code{character}: The method to be used to sample the distribution
representing the input estimate. For details see option \code{method} in 
\code{\link{random.estimate}}.}

\item{functionSyntax}{\code{character}: function syntax used in the welfare function(s). For 
details see \code{\link{mcSimulation}}.}

\item{relativeTolerance}{\code{numeric}: the relative tolerance level of deviation of the
generated confidence interval from the specified interval. If this deviation is greater than
\code{relativeTolerance} a warning is given.}

\item{verbosity}{\code{integer}: if \code{0} the function is silent; the larger the value the
more verbose is output information.}
}
\value{
An object of class \code{eviSimulation} with the following elements:
 \describe{
			\item{\code{$current}}{
			   \code{\link{welfareDecisionAnalysis}} object for \code{currentEstimate}
			}
			\item{\code{$prospective}}{
			  \code{\link{welfareDecisionAnalysis}} object  for single \code{prospectiveEstimate} or a 
			  list of \code{\link{welfareDecisionAnalysis}} objects for \code{prospectiveEstimate} being
			  a list of \code{estimate}s.
			}
 		\item{\code{$evi}}{
 		  Expected Value of Information(s) (EVI)(s)  gained by the prospective estimate(s) w.r.t. the 
 		  current estimate.
 		}
  }
}
\description{
The Expected Value of Information (EVI) is calculated based on a Monte Carlo simulation of the
expected welfare (or values or benefits) of two different decision alternatives. The expected
welfare is calculated for the current estimate of variables determining welfare and a prospective
estimate of these variables. The prospective estimate resembles an improvement in information.
}
\details{
\subsection{The Expected Value of Information (EVI)}{
    The Expected Value of Information is the decrease in the \eqn{\textrm{EOL}}{EOL} for an information
    improvement from the current (\eqn{\rho_X^{current}}{\rho_X_current}) to a better prospective (hypothetical)
    information (\eqn{\rho_X^{prospective}}{\rho_X_prospective}):
    \deqn{
      \textrm{EVI} := \textrm{EOL}(\rho_X^{current}) - \textrm{EOL}(\rho_X^{prospective}).
      }{
         EVI := EOL(\rho_X_current) - EOL(\rho_X_prospective).
      }
  }
}
\examples{
#############################################################
# Example 1 Only one prospective estimate:
#############################################################
numberOfModelRuns=10000
# Create the estimate object:
variable=c("revenue","costs")
distribution=c("posnorm","posnorm")
lower=c(10000,  5000)
upper=c(100000, 50000)
currentEstimate<-as.estimate(variable, distribution, lower, upper)
prospectiveEstimate<-currentEstimate
revenueConst<-mean(c(currentEstimate$marginal["revenue","lower"],
                     currentEstimate$marginal["revenue","upper"]))
prospectiveEstimate$marginal["revenue","distribution"]<-"const"
prospectiveEstimate$marginal["revenue","lower"]<-revenueConst 
prospectiveEstimate$marginal["revenue","upper"]<-revenueConst 
# (a) Define the welfare function without name for the return value:
profit<-function(x){
	x$revenue-x$costs
}

# Calculate the Expected Value of Information:
eviSimulationResult<-eviSimulation(welfare=profit,
                                   currentEstimate=currentEstimate,
                                   prospectiveEstimate=prospectiveEstimate,
                                   numberOfModelRuns=numberOfModelRuns,
                                   functionSyntax="data.frameNames")
# Show the simulation results:
print(summary(eviSimulationResult))
#############################################################
# (b) Define the welfare function with a name for the return value:
profit<-function(x){
	list(Profit=x$revenue-x$costs)
}
# Calculate the Expected Value of Information:
eviSimulationResult<-eviSimulation(welfare=profit,
                                   currentEstimate=currentEstimate,
                                   prospectiveEstimate=prospectiveEstimate,
                                   numberOfModelRuns=numberOfModelRuns,
                                   functionSyntax="data.frameNames")
# Show the simulation results:
print(summary((eviSimulationResult)))
#############################################################
# (c) Two decision variables:
decisionModel<-function(x){
 list(Profit=x$revenue-x$costs,
      Costs=-x$costs)
}
# Calculate the Expected Value of Information:
eviSimulationResult<-eviSimulation(welfare=decisionModel,
                                   currentEstimate=currentEstimate,
                                   prospectiveEstimate=prospectiveEstimate,
                                   numberOfModelRuns=numberOfModelRuns,
                                   functionSyntax="data.frameNames")
# Show the simulation results:
print(summary((eviSimulationResult)))
#############################################################
# Example 2 A list of prospective estimates:
#############################################################
numberOfModelRuns=10000
#  Define the welfare function with a name for the return value:
profit<-function(x){
 list(Profit=x$revenue-x$costs)
}
# Create the estimate object:
variable=c("revenue","costs")
distribution=c("posnorm","posnorm")
lower=c(10000,  5000)
upper=c(100000, 50000)
currentEstimate<-as.estimate(variable, distribution, lower, upper)
perfectInformationRevenue<-currentEstimate
revenueConst<-mean(c(currentEstimate$marginal["revenue","lower"],
                     currentEstimate$marginal["revenue","upper"]))
perfectInformationRevenue$marginal["revenue","distribution"]<-"const"
perfectInformationRevenue$marginal["revenue","lower"]<-revenueConst 
perfectInformationRevenue$marginal["revenue","upper"]<-revenueConst
# (a) A list with one element
prospectiveEstimate<-list(perfectInformationRevenue=perfectInformationRevenue)
# Calculate the Expected Value of Information:
eviSimulationResult<-eviSimulation(welfare=profit,
                                   currentEstimate=currentEstimate,
                                   prospectiveEstimate=prospectiveEstimate,
                                   numberOfModelRuns=numberOfModelRuns,
                                   functionSyntax="data.frameNames")
# Show the simulation results:
print(summary(eviSimulationResult))
#############################################################
# (b) A list with two elements
perfectInformationCosts<-currentEstimate
costsConst<-mean(c(currentEstimate$marginal["costs","lower"], 
                   currentEstimate$marginal["costs","upper"]))
perfectInformationCosts$marginal["costs","distribution"]<-"const"
perfectInformationCosts$marginal["costs","lower"]<-costsConst 
perfectInformationCosts$marginal["costs","upper"]<-costsConst
prospectiveEstimate<-list(perfectInformationRevenue=perfectInformationRevenue,
                          perfectInformationCosts=perfectInformationCosts)
# Calculate the Expected Value of Information:
eviSimulationResult<-eviSimulation(welfare=profit,
                                   currentEstimate=currentEstimate,
                                   prospectiveEstimate=prospectiveEstimate,
                                   numberOfModelRuns=numberOfModelRuns,
                                   functionSyntax="data.frameNames")
# Show the simulation results:
print(summary(eviSimulationResult))
#############################################################
# Example 3 A list of prospective estimates and two decision variables:
#############################################################
numberOfModelRuns=10000
# Create the current estimate object:
variable=c("revenue","costs")
distribution=c("posnorm","posnorm")
lower=c(10000,  5000)
upper=c(100000, 50000)
currentEstimate<-as.estimate(variable, distribution, lower, upper)
# Create a list of two prospective estimates:
perfectInformationRevenue<-currentEstimate
revenueConst<-mean(c(currentEstimate$marginal["revenue","lower"],
                     currentEstimate$marginal["revenue","upper"]))
perfectInformationRevenue$marginal["revenue","distribution"]<-"const"
perfectInformationRevenue$marginal["revenue","lower"]<-revenueConst 
perfectInformationRevenue$marginal["revenue","upper"]<-revenueConst
perfectInformationCosts<-currentEstimate
costsConst<-mean(c(currentEstimate$marginal["costs","lower"], 
                   currentEstimate$marginal["costs","upper"]))
perfectInformationCosts$marginal["costs","distribution"]<-"const"
perfectInformationCosts$marginal["costs","lower"]<-costsConst 
perfectInformationCosts$marginal["costs","upper"]<-costsConst
prospectiveEstimate<-list(perfectInformationRevenue=perfectInformationRevenue,
                          perfectInformationCosts=perfectInformationCosts)
# Define the welfare function with two decision variables:
decisionModel<-function(x){
 list(Profit=x$revenue-x$costs,
      Costs=-x$costs)
}
# Calculate the Expected Value of Information:
eviSimulationResult<-eviSimulation(welfare=decisionModel,
                                   currentEstimate=currentEstimate,
                                   prospectiveEstimate=prospectiveEstimate,
                                   numberOfModelRuns=numberOfModelRuns,
                                   functionSyntax="data.frameNames")
# Show the simulation results:
print(sort(summary(eviSimulationResult)),decreasing=TRUE,along="Profit")
}
\references{
Hubbard, Douglas W., \emph{How to Measure Anything? - Finding the Value of "Intangibles" in Business},
  John Wiley & Sons, Hoboken, New Jersey, 2014, 3rd Ed, \url{http://www.howtomeasureanything.com/}.
  
  Gravelle, Hugh and Ray Rees, \emph{Microeconomics}, Pearson Education Limited, 3rd edition, 2004.
}
\seealso{
\code{\link{welfareDecisionAnalysis}}, \code{\link{mcSimulation}}, \code{\link{estimate}},
  \code{\link{summary.eviSimulation}}
}
