% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/statistics.R
\name{f2}
\alias{f2}
\title{Similarity factor f2 for dissolution data}
\usage{
f2(data, tcol, grouping, use_ema = "yes", bounds = c(1, 85), nsf = c(1, 2))
}
\arguments{
\item{data}{A data frame with the dissolution profile data in wide format.}

\item{tcol}{A vector of indices that specifies the columns in \code{data}
that contain the \% release values. The length of \code{tcol} must be
three or longer.}

\item{grouping}{A character string that specifies the column in \code{data}
that contains the group names (i.e. a factorial variable, e.g., for the
differentiation of batches or formulations of a drug product).}

\item{use_ema}{A character string indicating whether the dissimilarity
factor \eqn{f_1} should be calculated following the EMA guideline
\dQuote{On the investigation of bioequivalence} (\code{"yes"}, the
default) or not (\code{"no"}), i.e. the recommendations concerning the
similarity factor \eqn{f_2}. A third option is \code{"ignore"}. If
\code{use_ema} is \code{"yes"} or \code{"no"} the appropriate profile
portion is determined on the basis of the values of the parameter
\code{bounds}. If it is \code{"ignore"}, the complete profiles are used
as specified by the parameter \code{tcol}.}

\item{bounds}{A numeric vector of the form \code{c(lower, upper)} that
specifies the \dQuote{lower} and \dQuote{upper} limits, respectively, for
the \% drug release given that \code{use_ema} is \code{"no"}. The default
is \code{c(1, 85)}. Mean \% release values of any of the two groups being
compared that are smaller than or equal to the lower bound are ignored and
only the first mean \% release value that is greater than or equal to the
upper bound is included while all the subsequent values are ignored. If
\code{use_ema} is \code{"yes"} the \code{bounds} are \code{c(0, 85)} per
definition. If \code{use_ema} is \code{"ignore"} the \code{bounds} are
disregarded.}

\item{nsf}{A vector of positive integers that specify the \dQuote{number
of significant figures} (nsf) of the corresponding values of the
\code{bounds} parameter. It must thus have the same length as the
\code{bounds} parameter. Before the \% release values are compared with
the limits that are specified by the \code{bounds} parameter, they are
rounded to the corresponding number of significant figures as specified
by the \code{nsf} parameter.}
}
\value{
A list with the following elements is returned:
\item{f2}{A numeric value representing the similarity factor \eqn{f_2}.}
\item{Profile.TP}{A named numeric vector of the columns in \code{data}
  specified by \code{tcol} and depending on the selection of \code{use_ema}.
  Given that the column names contain extractable numeric information, e.g.,
  the testing time points of the dissolution profile, it contains the
  corresponding numeric values. Elements where no numeric information could
  be extracted are \code{NA}.}
}
\description{
The function \code{f2()} calculates the similarity factor \eqn{f_2}.
}
\details{
Similarity of dissolution profiles is assessed using the similarity
factor \eqn{f_2} according to the EMA guideline (European Medicines Agency
2010) \dQuote{On the investigation of bioequivalence}. The evaluation of the
similarity factor is based on the following constraints:
\enumerate{
  \item A minimum of three time points (zero excluded).
  \item The time points should be the same for the two formulations.
  \item Twelve individual values for every time point for each formulation.
  \item Not more than one mean value of > 85\% dissolved for any of the
    formulations.
  \item The relative standard deviation or coefficient of variation of any
    product should be less than 20\% for the first time point and less than
    10\% from the second to the last time point.
}

The similarity factor \eqn{f_2} is calculated by aid of the equation

\deqn{f_2 = 50 \times \log \left(\frac{100}{\sqrt{1 + \frac{\sum_{t=1}^{n}
  \left(\bar{R}(t) - \bar{T}(t) \right)^2}{n}}} \right) .}{%
  f_2 = 50 log(100 / (sqrt(1 + (sum((R.bar(t) - T.bar(t))^2) / n)))) .}

In this equation
\describe{
  \item{\eqn{f_2}}{is the similarity factor,}
  \item{\eqn{n}}{is the number of time points,}
  \item{\eqn{\bar{R}(t)}{R.bar(t)}}{is the mean percent reference drug
      dissolved at time \eqn{t} after initiation of the study, and}
  \item{\eqn{\bar{T}(t)}{T.bar(t)}}{is the mean percent test drug dissolved
      at time \eqn{t} after initiation of the study.}
}

Dissolution profiles are regarded as similar if the \eqn{f_2} value is
between \eqn{50} and \eqn{100}. \cr
}
\examples{
# Use of defaults, i.e. 'use_ema = "yes"', 'bounds = c(1, 85)'
# Comparison always involves only two groups.
f2(data = dip1, tcol = 3:10, grouping = "type")

# $f2
# [1] 40.83405
#
# $Profile.TP
# t.5 t.10 t.15 t.20 t.30 t.60 t.90
#   5   10   15   20   30   60   90

# Use of 'use_ema = "no"', 'bounds = c(5, 80)'
f2(data = dip1, tcol = 3:10, grouping = "type", use_ema = "no",
   bounds = c(5, 80), nsf = c(1, 2))

# $f2
# [1] 39.24385
#
# $Profile.TP
# t.5 t.10 t.15 t.20 t.30 t.60
#   5   10   15   20   30   60

# Use of 'use_ema = "no"', 'bounds = c(1, 95)'
f2(data = dip1, tcol = 3:10, grouping = "type", use_ema = "no",
   bounds = c(1, 95), nsf = c(1, 2))

# $f2
# [1] 42.11197
#
# $Profile.TP
# t.5  t.10  t.15  t.20  t.30  t.60  t.90 t.120
#   5    10    15    20    30    60    90   120

# In this case, the whole profiles are used. The same result is obtained
# when setting 'use_ema = "ignore"' (ignoring values passed to 'bounds').
f2(data = dip1, tcol = 3:10, grouping = "type", use_ema = "ignore")

# Passing in a data frame with a grouping variable with a number of levels that
# differs from two produces an error.
\dontrun{
  tmp <- rbind(dip1,
               data.frame(type = "T2",
                          tablet = as.factor(1:6),
                          dip1[7:12, 3:10]))

  tryCatch(
    f2(data = tmp, tcol = 3:10, grouping = "type"),
    error = function(e) message(e),
    finally = message("\nMaybe you want to remove unesed levels in data."))
}

# Error in f1(data = tmp, tcol = 3:10, grouping = "type") :
#   The number of levels in column type differs from 2.
}
\references{
United States Food and Drug Administration (FDA). Guidance for industry:
dissolution testing of immediate release solid oral dosage forms. 1997.\cr
\url{https://www.fda.gov/media/70936/download}

United States Food and Drug Administration (FDA). Guidance for industry:
immediate release solid oral dosage form: scale-up and post-approval
changes, chemistry, manufacturing and controls, \emph{in vitro} dissolution
testing, and \emph{in vivo} bioequivalence documentation (SUPAC-IR). 1995.\cr
\url{https://www.fda.gov/media/70949/download}

European Medicines Agency (EMA), Committee for Medicinal Products for
Human Use (CHMP). Guideline on the Investigation of Bioequivalence. 2010;
\href{https://www.ema.europa.eu/en/documents/scientific-guideline/guideline-investigation-bioequivalence-rev1_en.pdf}{
CPMP/EWP/QWP/1401/98 Rev. 1}.
}
\seealso{
\code{\link{f1}}.
}
